/*
Copyright 2022 The Flux authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package main

import "testing"

func TestInstall(t *testing.T) {
	// The pointer to kubeconfigArgs.Namespace is shared across
	// the tests. When a new value is set, it will linger and
	// impact subsequent tests.
	// Given that this test uses an invalid namespace, it ensures
	// to restore whatever value it had previously.
	currentNamespace := *kubeconfigArgs.Namespace
	t.Cleanup(func() { *kubeconfigArgs.Namespace = currentNamespace })

	tests := []struct {
		name   string
		args   string
		assert assertFunc
	}{
		{
			name:   "invalid namespace",
			args:   "install --namespace='@#[]'",
			assert: assertError("namespace must be a valid DNS label: \"@#[]\""),
		},
		{
			name:   "invalid sub-command",
			args:   "install unexpectedPosArg --namespace=example",
			assert: assertError(`unknown command "unexpectedPosArg" for "flux install"`),
		},
		{
			name:   "missing image pull secret",
			args:   "install --registry-creds=fluxcd:test",
			assert: assertError(`--registry-creds requires --image-pull-secret to be set`),
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			cmd := cmdTestCase{
				args:   tt.args,
				assert: tt.assert,
			}
			cmd.runTestCmd(t)
		})
	}
}
