#include <config.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#ifdef HAVE_SYS_TIME_H
# include <sys/time.h>
#else
# include <time.h>
#endif
#include <sys/param.h>
#include <stdio.h>
#include <pwd.h>
#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif
#include <stdlib.h>
#include <signal.h>
#include "compat.h"
#include "rcfile.h"
#include "sock.h"
#include "dial.h"


int main(int argc, char *argv[]);
int auth(const char *user, char *pass, char *buf);
int getsettings(int argc, char *argv[], int *dest_port, char **user,
		char **pass, char **server, char **command, 
		int *keep_alive_secs);
void helpinfo(char *progname);
void sighandler(int signo);
void versioninfo(void);
char *xstrdup(const char *s);
int command_result(char *buf);
void killchilds(void);

/* stdin and sok use read/write
 * stdout and stderr use stdio
 */

/* XXX: is this safe? */
static volatile sig_atomic_t runpid = -1;
static volatile sig_atomic_t alivepid = -1;

int main(int argc, char *argv[])
{
	char *buf;
	int dest_port;
	char *server;
	char *user;
	char *pass;
	char *command;
	int keep_alive_secs;
	int exitstatus = EXIT_SUCCESS;

	if (set_signals(sighandler))
		return EXIT_FAILURE;
	/* settings */
	if (getsettings(argc, argv, &dest_port, &user, &pass, &server, 
				&command, &keep_alive_secs))
		return EXIT_FAILURE;
	if (connect_to_host(dest_port, server))
		return EXIT_FAILURE;
	buf = malloc(MAXLINE+1);
	if (NULL==buf) {
		perror("Malloc failed");
		return EXIT_FAILURE;
	}
	if (recvline(buf, MAXLINE)) {
		free(buf);
		return EXIT_FAILURE;
	}
	puts(buf); /* #Hello DWUN-{version} */
	if (auth(user, pass, buf)) {
		free(buf);
		return EXIT_FAILURE;
	}
	/* do not write to sok after keepalive is started! */
	/* XXX: is it safe to pass alivepid to keepalive? */
#ifdef __BEOS__
	if (keepalive_beos(keep_alive_secs))
		exitstatus = EXIT_FAILURE;
#else
	if (keepalive(&alivepid, keep_alive_secs))
		exitstatus = EXIT_FAILURE;
#endif
	if (eventloop(buf, command))
		exitstatus = EXIT_FAILURE;
	free(buf);
	sock_close();
	putchar('\n');
	killchilds();
	return exitstatus;
}

int auth(const char *user, char *pass, char *buf)
{
	if (sendline("USER ") || sendline(user) || sendline("\n"))
		return 1;
	if (command_result(buf))
		return 1;
	if (!pass)
		pass = getpass("Password:");
	if (sendline("PASS ") || sendline(pass) || sendline("\n")) {
		memset(pass, 0, strlen(pass));
		return 1;
	}
	memset(pass, 0, strlen(pass));
	if (command_result(buf))
		return 1;
	printf("Authenticated\n");
	if (sendline("CONNECT\n"))
		return 1;
	return 0;
}

/* Does not handle running -c "command" for #Connected.
 * (i.e. don't use command_result after we have done CONNECT. */
int command_result(char *buf)
{
	int ret;
	do {
		ret = recv_okay(buf);
	} while (ret==RECV_STR_INFO);
	if (ret==RECV_ERR)
		return 1; /* error receiving data */
	if (ret==RECV_STR_PROBLEM)
		return 1; /* server reported error */
	return 0; /* RECV_STR_OKAY; got "OKAY\n" */
}

int getsettings(int argc, char *argv[], int *dest_port, char **user,
		char **pass, char **server, char **command, 
		int *keep_alive_secs) 
{
	int ret;
	char **pt;
	char *rcfile;
	char *home;
	int rcfile_must_exist = 0; /* if given on the command line */
	char *tmp;

	/* defaults */
	*dest_port = 5540;
	if (NULL==(*server = xstrdup("127.0.0.1")))
		return 1;
	tmp = getenv("USER");
	if (tmp) {
		if (NULL==(*user = xstrdup(tmp)))
			return 1;
	} else {
		*user = NULL;
	}
	if (NULL==(rcfile = malloc(MAXPATHLEN))) {
		perror("Malloc failed");
		return 1;
	}
	home = getenv("HOME");
	if (home) {
#ifdef HAVE_SNPRINTF
		snprintf(rcfile, MAXPATHLEN, "%s/.dialrc", home);
#else
		if (strlen(home) + strlen("/.dialrc") + 1 > MAXPATHLEN) {
			fprintf(stderr, "$HOME environment variable too 
					long\n");
					return 1;
		}
		sprintf(rcfile, "%s/.dialrc", home);
#endif
	}
	*keep_alive_secs = REMOTE_KEEP_ALIVE;
	*pass = NULL; /* only allowed in rcfile */
	*command = NULL;

	for (pt = argv; *pt; ++pt) {
		if (**pt=='-' && *(*pt+1)=='r') {
			free(rcfile);
			if (NULL==(rcfile = xstrdup(*(pt+1))))
				return 1;
			rcfile_must_exist = 1;
			break;
		}
	}
	/* use system-wide default file if available */
	if (get_rcfile_settings("/etc/dialrc", 0, dest_port, user, pass, server,
		      		keep_alive_secs)) {
		free(rcfile);
		return 1;
	}
	if (get_rcfile_settings(rcfile, rcfile_must_exist, dest_port, user, 
				pass, server, keep_alive_secs)) {
		free(rcfile);
		return 1;
	}
	free(rcfile);

	while ((ret=getopt(argc, argv, "p:s:u:k:r:c:vh"))!=EOF) {
		switch(ret) {
		case 'p': 
			*dest_port = atoi(optarg);
			break;
		case 's': 
			if (NULL==(*server = xstrdup(optarg)))
				return 1;
			break;
		case 'u': 
			free(*user);
			if (NULL==(*user = xstrdup(optarg)))
				return 1;
			break;
		case 'k': 
			*keep_alive_secs = atoi(optarg);
			break;
		case 'r':
			break; /* handled */
		case 'c': 
		      	free(*command);
	      		if (NULL==(*command = xstrdup(optarg)))
				return 1;
			break;
		case 'v':
			versioninfo();
			return 1;
		case 'h':
			versioninfo();
			putchar('\n');
			helpinfo(argv[0]);
			return 1;
		default:
			fprintf(stderr, "Invalid argument\n");
	     		helpinfo(argv[0]);
			return 1;
		}
	}
	
	/* check for sane values */
	if (0==*dest_port) {
		fprintf(stderr, "Invalid port: 0\n");
		return 1;
	}
	/* We gethostbyname on server later */
	if (NULL==*user) {
		fprintf(stderr, "$USER not found in environment and not "
				"supplied as an option.\n");
		return 1;
	}
	if (0==strcmp(*user, "admin")) {
		fprintf(stderr, "Dial cannot be used with the admin user, "
				"use telnet instead\n");
		return 1;
	}
	if (*keep_alive_secs < 0) {
		fprintf(stderr, "Keep alive timeout must be positive\n");
		return 1;
	} else if (*keep_alive_secs > 0) {
		*keep_alive_secs = (int)*keep_alive_secs * 0.5;
		if (*keep_alive_secs==0)
			*keep_alive_secs = 1; /* minimum for non-zero time */
	}
	return 0;
}

int run(char *command)
{
	char dial_pid[MAXLINE];
#ifdef HAVE_SNPRINTF
	snprintf(dial_pid, MAXLINE, "DIAL_PID=%lu", (unsigned long)getpid());
#else
	sprintf(dial_pid, "DIAL_PID=%lu", (unsigned long)getpid());
#endif
	runpid = xfork();
	if (runpid==-1) {
		perror("Fork failed");
		return 1;
	} else if (runpid==0) {
		set_signals(SIG_DFL);
		unblock_all_signals();
		sock_close();
		putenv(dial_pid);
		if (-1==execl("/bin/sh", "sh", "-c", command, NULL)) {
			fprintf(stderr, "Running '/bin/sh -c %s' failed: %s\n",
					command, strerror(errno));
			_exit(EXIT_FAILURE);
		}
		_exit(0);
	}
	return 0;
}

char *xstrdup(const char *s)
{
	char *d;
	d = strdup(s);
	if (d)
		return d;
	perror("Malloc failed");
	return NULL;
}

/* signals */
void sighandler(int signo)
{
	pid_t pid;
	if (signo==SIGCHLD) {
		while((pid = waitpid(0, NULL, WNOHANG)) > 0) {
			if (pid==alivepid)
				alivepid = -1;
			else if (pid==runpid)
				runpid = -1;
		}
	} else if (signo==SIGTERM || signo==SIGINT) {
		killchilds();
		_exit(EXIT_SUCCESS);
	}
}

void killchilds(void)
{
	if (alivepid!=-1)
		kill(alivepid, SIGTERM);
	if (runpid!=-1)
		kill(runpid, SIGTERM);
	return;
}

int set_signals(void (*handler)(int))
{
	if (handle_signal(SIGTERM, handler) || 
			handle_signal(SIGINT, handler) ||
			handle_signal(SIGCHLD, handler))
		return 1;
	return 0;
}

int handle_signal(int signum, void (*handler)(int))
{
	int ret;
	struct sigaction cursig;
	memset(&cursig, 0, sizeof(cursig));
	cursig.sa_handler = handler;
	do {
		ret = sigaction(signum, &cursig, NULL);
	} while (ret==-1 && errno==EINTR);
	if (ret==-1) {
		perror("Setting up signal handler failed");
		return 1;
	}
	return 0;
}

/* Blocks all signals, forks, then returns to previous block mask in parent.
 * Child calls unblock_all_signals when they've ready to receive signals */
pid_t xfork(void)
{
	pid_t pid;
	int fork_errno;
	sigset_t old;
	sigset_t full;

	sigfillset(&full);
	sigprocmask(SIG_BLOCK, &full, &old);
	pid = fork();
	fork_errno = errno;
	if (pid!=0) /* child chooses when to do unblock_all_signals */
		sigprocmask(SIG_SETMASK, &old, NULL);
	errno = fork_errno;
	return pid;
}

void unblock_all_signals(void)
{
	sigset_t full;
	sigfillset(&full);
	sigprocmask(SIG_UNBLOCK, &full, NULL);
	return;
}

void helpinfo(char *progname)
{
	fprintf(stderr, "Usage: %s [OPTION] ...\n\n", progname);
	fprintf(stderr, "-p\tserver port\n");
	fprintf(stderr, "-s\tserver address\n");
	fprintf(stderr, "-u\tusername\n");
	fprintf(stderr, "-k\tsend keepalive every ... seconds (set to below "
			"server's expectations)\n");
	fprintf(stderr, "-r\trc filename\n");
	fprintf(stderr, "-v\tversion\n");
	fprintf(stderr, "-h\tthis help\n");
	return;
}

void versioninfo(void)
{
	fprintf(stderr, "DWUN dial version %s\n", VERSION);
	fprintf(stderr, "Copyright (C) 1999, 2000 Tim Sutherland\n");
	fprintf(stderr, "See COPYING file in the distribution\n");
	return;
}
