pub const ext = @import("ext.zig");
const gtk = @This();

const std = @import("std");
const compat = @import("compat");
const xlib = @import("xlib2");
const gdk = @import("gdk3");
const cairo = @import("cairo1");
const gobject = @import("gobject2");
const glib = @import("glib2");
const pango = @import("pango1");
const harfbuzz = @import("harfbuzz0");
const freetype2 = @import("freetype22");
const gio = @import("gio2");
const gmodule = @import("gmodule2");
const gdkpixbuf = @import("gdkpixbuf2");
const atk = @import("atk1");
/// A `gtk.Allocation`-struct of a widget represents region
/// which has been allocated to the widget by its parent. It is a subregion
/// of its parents allocation. See
/// [GtkWidget’s geometry management section][geometry-management] for
/// more information.
pub const Allocation = gdk.Rectangle;

pub const Stock = [*:0]u8;

/// The GtkAboutDialog offers a simple way to display information about
/// a program like its logo, name, copyright, website and license. It is
/// also possible to give credits to the authors, documenters, translators
/// and artists who have worked on the program. An about dialog is typically
/// opened when the user selects the `About` option from the `Help` menu.
/// All parts of the dialog are optional.
///
/// About dialogs often contain links and email addresses. GtkAboutDialog
/// displays these as clickable links. By default, it calls `gtk.showUriOnWindow`
/// when a user clicks one. The behaviour can be overridden with the
/// `gtk.AboutDialog.signals.activate`-link signal.
///
/// To specify a person with an email address, use a string like
/// "Edgar Allan Poe <edgar\@poe.com>". To specify a website with a title,
/// use a string like "GTK+ team http://www.gtk.org".
///
/// To make constructing a GtkAboutDialog as convenient as possible, you can
/// use the function `gtk.showAboutDialog` which constructs and shows a dialog
/// and keeps it around so that it can be shown again.
///
/// Note that GTK+ sets a default title of `_("About `s`")` on the dialog
/// window (where \%s is replaced by the name of the application, but in
/// order to ensure proper translation of the title, applications should
/// set the title property explicitly when constructing a GtkAboutDialog,
/// as shown in the following example:
/// ```
/// GdkPixbuf *example_logo = gdk_pixbuf_new_from_file ("./logo.png", NULL);
/// gtk_show_about_dialog (NULL,
///                        "program-name", "ExampleCode",
///                        "logo", example_logo,
///                        "title", _("About ExampleCode"),
///                        NULL);
/// ```
///
/// It is also possible to show a `gtk.AboutDialog` like any other `gtk.Dialog`,
/// e.g. using `gtk.Dialog.run`. In this case, you might need to know that
/// the “Close” button returns the `GTK_RESPONSE_CANCEL` response id.
pub const AboutDialog = extern struct {
    pub const Parent = gtk.Dialog;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.AboutDialogClass;
    f_parent_instance: gtk.Dialog,
    f_priv: ?*gtk.AboutDialogPrivate,

    pub const virtual_methods = struct {
        pub const activate_link = struct {
            pub fn call(p_class: anytype, p_dialog: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_uri: [*:0]const u8) c_int {
                return gobject.ext.as(AboutDialog.Class, p_class).f_activate_link.?(gobject.ext.as(AboutDialog, p_dialog), p_uri);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_dialog: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_uri: [*:0]const u8) callconv(.C) c_int) void {
                gobject.ext.as(AboutDialog.Class, p_class).f_activate_link = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The people who contributed artwork to the program, as a `NULL`-terminated
        /// array of strings. Each string may contain email addresses and URLs, which
        /// will be displayed as links, see the introduction for more details.
        pub const artists = struct {
            pub const name = "artists";

            pub const Type = ?[*][*:0]u8;
        };

        /// The authors of the program, as a `NULL`-terminated array of strings.
        /// Each string may contain email addresses and URLs, which will be displayed
        /// as links, see the introduction for more details.
        pub const authors = struct {
            pub const name = "authors";

            pub const Type = ?[*][*:0]u8;
        };

        /// Comments about the program. This string is displayed in a label
        /// in the main dialog, thus it should be a short explanation of
        /// the main purpose of the program, not a detailed list of features.
        pub const comments = struct {
            pub const name = "comments";

            pub const Type = ?[*:0]u8;
        };

        /// Copyright information for the program.
        pub const copyright = struct {
            pub const name = "copyright";

            pub const Type = ?[*:0]u8;
        };

        /// The people documenting the program, as a `NULL`-terminated array of strings.
        /// Each string may contain email addresses and URLs, which will be displayed
        /// as links, see the introduction for more details.
        pub const documenters = struct {
            pub const name = "documenters";

            pub const Type = ?[*][*:0]u8;
        };

        /// The license of the program. This string is displayed in a
        /// text view in a secondary dialog, therefore it is fine to use
        /// a long multi-paragraph text. Note that the text is only wrapped
        /// in the text view if the "wrap-license" property is set to `TRUE`;
        /// otherwise the text itself must contain the intended linebreaks.
        /// When setting this property to a non-`NULL` value, the
        /// `gtk.AboutDialog.properties.license`-type property is set to `GTK_LICENSE_CUSTOM`
        /// as a side effect.
        pub const license = struct {
            pub const name = "license";

            pub const Type = ?[*:0]u8;
        };

        /// The license of the program, as a value of the `gtk.License` enumeration.
        ///
        /// The `gtk.AboutDialog` will automatically fill out a standard disclaimer
        /// and link the user to the appropriate online resource for the license
        /// text.
        ///
        /// If `GTK_LICENSE_UNKNOWN` is used, the link used will be the same
        /// specified in the `gtk.AboutDialog.properties.website` property.
        ///
        /// If `GTK_LICENSE_CUSTOM` is used, the current contents of the
        /// `gtk.AboutDialog.properties.license` property are used.
        ///
        /// For any other `gtk.License` value, the contents of the
        /// `gtk.AboutDialog.properties.license` property are also set by this property as
        /// a side effect.
        pub const license_type = struct {
            pub const name = "license-type";

            pub const Type = gtk.License;
        };

        /// A logo for the about box. If it is `NULL`, the default window icon
        /// set with `gtk.Window.setDefaultIcon` will be used.
        pub const logo = struct {
            pub const name = "logo";

            pub const Type = ?*gdkpixbuf.Pixbuf;
        };

        /// A named icon to use as the logo for the about box. This property
        /// overrides the `gtk.AboutDialog.properties.logo` property.
        pub const logo_icon_name = struct {
            pub const name = "logo-icon-name";

            pub const Type = ?[*:0]u8;
        };

        /// The name of the program.
        /// If this is not set, it defaults to `glib.getApplicationName`.
        pub const program_name = struct {
            pub const name = "program-name";

            pub const Type = ?[*:0]u8;
        };

        /// Credits to the translators. This string should be marked as translatable.
        /// The string may contain email addresses and URLs, which will be displayed
        /// as links, see the introduction for more details.
        pub const translator_credits = struct {
            pub const name = "translator-credits";

            pub const Type = ?[*:0]u8;
        };

        /// The version of the program.
        pub const version = struct {
            pub const name = "version";

            pub const Type = ?[*:0]u8;
        };

        /// The URL for the link to the website of the program.
        /// This should be a string starting with "http://.
        pub const website = struct {
            pub const name = "website";

            pub const Type = ?[*:0]u8;
        };

        /// The label for the link to the website of the program.
        pub const website_label = struct {
            pub const name = "website-label";

            pub const Type = ?[*:0]u8;
        };

        /// Whether to wrap the text in the license dialog.
        pub const wrap_license = struct {
            pub const name = "wrap-license";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The signal which gets emitted to activate a URI.
        /// Applications may connect to it to override the default behaviour,
        /// which is to call `gtk.showUriOnWindow`.
        pub const activate_link = struct {
            pub const name = "activate-link";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_uri: [*:0]u8, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AboutDialog, p_instance))),
                    gobject.signalLookup("activate-link", AboutDialog.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.AboutDialog`.
    extern fn gtk_about_dialog_new() *gtk.AboutDialog;
    pub const new = gtk_about_dialog_new;

    /// Creates a new section in the Credits page.
    extern fn gtk_about_dialog_add_credit_section(p_about: *AboutDialog, p_section_name: [*:0]const u8, p_people: [*][*:0]const u8) void;
    pub const addCreditSection = gtk_about_dialog_add_credit_section;

    /// Returns the string which are displayed in the artists tab
    /// of the secondary credits dialog.
    extern fn gtk_about_dialog_get_artists(p_about: *AboutDialog) [*]const [*:0]const u8;
    pub const getArtists = gtk_about_dialog_get_artists;

    /// Returns the string which are displayed in the authors tab
    /// of the secondary credits dialog.
    extern fn gtk_about_dialog_get_authors(p_about: *AboutDialog) [*]const [*:0]const u8;
    pub const getAuthors = gtk_about_dialog_get_authors;

    /// Returns the comments string.
    extern fn gtk_about_dialog_get_comments(p_about: *AboutDialog) [*:0]const u8;
    pub const getComments = gtk_about_dialog_get_comments;

    /// Returns the copyright string.
    extern fn gtk_about_dialog_get_copyright(p_about: *AboutDialog) [*:0]const u8;
    pub const getCopyright = gtk_about_dialog_get_copyright;

    /// Returns the string which are displayed in the documenters
    /// tab of the secondary credits dialog.
    extern fn gtk_about_dialog_get_documenters(p_about: *AboutDialog) [*]const [*:0]const u8;
    pub const getDocumenters = gtk_about_dialog_get_documenters;

    /// Returns the license information.
    extern fn gtk_about_dialog_get_license(p_about: *AboutDialog) [*:0]const u8;
    pub const getLicense = gtk_about_dialog_get_license;

    /// Retrieves the license set using `gtk.AboutDialog.setLicenseType`
    extern fn gtk_about_dialog_get_license_type(p_about: *AboutDialog) gtk.License;
    pub const getLicenseType = gtk_about_dialog_get_license_type;

    /// Returns the pixbuf displayed as logo in the about dialog.
    extern fn gtk_about_dialog_get_logo(p_about: *AboutDialog) *gdkpixbuf.Pixbuf;
    pub const getLogo = gtk_about_dialog_get_logo;

    /// Returns the icon name displayed as logo in the about dialog.
    extern fn gtk_about_dialog_get_logo_icon_name(p_about: *AboutDialog) [*:0]const u8;
    pub const getLogoIconName = gtk_about_dialog_get_logo_icon_name;

    /// Returns the program name displayed in the about dialog.
    extern fn gtk_about_dialog_get_program_name(p_about: *AboutDialog) [*:0]const u8;
    pub const getProgramName = gtk_about_dialog_get_program_name;

    /// Returns the translator credits string which is displayed
    /// in the translators tab of the secondary credits dialog.
    extern fn gtk_about_dialog_get_translator_credits(p_about: *AboutDialog) [*:0]const u8;
    pub const getTranslatorCredits = gtk_about_dialog_get_translator_credits;

    /// Returns the version string.
    extern fn gtk_about_dialog_get_version(p_about: *AboutDialog) [*:0]const u8;
    pub const getVersion = gtk_about_dialog_get_version;

    /// Returns the website URL.
    extern fn gtk_about_dialog_get_website(p_about: *AboutDialog) [*:0]const u8;
    pub const getWebsite = gtk_about_dialog_get_website;

    /// Returns the label used for the website link.
    extern fn gtk_about_dialog_get_website_label(p_about: *AboutDialog) [*:0]const u8;
    pub const getWebsiteLabel = gtk_about_dialog_get_website_label;

    /// Returns whether the license text in `about` is
    /// automatically wrapped.
    extern fn gtk_about_dialog_get_wrap_license(p_about: *AboutDialog) c_int;
    pub const getWrapLicense = gtk_about_dialog_get_wrap_license;

    /// Sets the strings which are displayed in the artists tab
    /// of the secondary credits dialog.
    extern fn gtk_about_dialog_set_artists(p_about: *AboutDialog, p_artists: [*][*:0]const u8) void;
    pub const setArtists = gtk_about_dialog_set_artists;

    /// Sets the strings which are displayed in the authors tab
    /// of the secondary credits dialog.
    extern fn gtk_about_dialog_set_authors(p_about: *AboutDialog, p_authors: [*][*:0]const u8) void;
    pub const setAuthors = gtk_about_dialog_set_authors;

    /// Sets the comments string to display in the about dialog.
    /// This should be a short string of one or two lines.
    extern fn gtk_about_dialog_set_comments(p_about: *AboutDialog, p_comments: ?[*:0]const u8) void;
    pub const setComments = gtk_about_dialog_set_comments;

    /// Sets the copyright string to display in the about dialog.
    /// This should be a short string of one or two lines.
    extern fn gtk_about_dialog_set_copyright(p_about: *AboutDialog, p_copyright: ?[*:0]const u8) void;
    pub const setCopyright = gtk_about_dialog_set_copyright;

    /// Sets the strings which are displayed in the documenters tab
    /// of the secondary credits dialog.
    extern fn gtk_about_dialog_set_documenters(p_about: *AboutDialog, p_documenters: [*][*:0]const u8) void;
    pub const setDocumenters = gtk_about_dialog_set_documenters;

    /// Sets the license information to be displayed in the secondary
    /// license dialog. If `license` is `NULL`, the license button is
    /// hidden.
    extern fn gtk_about_dialog_set_license(p_about: *AboutDialog, p_license: ?[*:0]const u8) void;
    pub const setLicense = gtk_about_dialog_set_license;

    /// Sets the license of the application showing the `about` dialog from a
    /// list of known licenses.
    ///
    /// This function overrides the license set using
    /// `gtk.AboutDialog.setLicense`.
    extern fn gtk_about_dialog_set_license_type(p_about: *AboutDialog, p_license_type: gtk.License) void;
    pub const setLicenseType = gtk_about_dialog_set_license_type;

    /// Sets the pixbuf to be displayed as logo in the about dialog.
    /// If it is `NULL`, the default window icon set with
    /// `gtk.Window.setDefaultIcon` will be used.
    extern fn gtk_about_dialog_set_logo(p_about: *AboutDialog, p_logo: ?*gdkpixbuf.Pixbuf) void;
    pub const setLogo = gtk_about_dialog_set_logo;

    /// Sets the pixbuf to be displayed as logo in the about dialog.
    /// If it is `NULL`, the default window icon set with
    /// `gtk.Window.setDefaultIcon` will be used.
    extern fn gtk_about_dialog_set_logo_icon_name(p_about: *AboutDialog, p_icon_name: ?[*:0]const u8) void;
    pub const setLogoIconName = gtk_about_dialog_set_logo_icon_name;

    /// Sets the name to display in the about dialog.
    /// If this is not set, it defaults to `glib.getApplicationName`.
    extern fn gtk_about_dialog_set_program_name(p_about: *AboutDialog, p_name: [*:0]const u8) void;
    pub const setProgramName = gtk_about_dialog_set_program_name;

    /// Sets the translator credits string which is displayed in
    /// the translators tab of the secondary credits dialog.
    ///
    /// The intended use for this string is to display the translator
    /// of the language which is currently used in the user interface.
    /// Using `gettext`, a simple way to achieve that is to mark the
    /// string for translation:
    /// ```
    /// GtkWidget *about = gtk_about_dialog_new ();
    /// gtk_about_dialog_set_translator_credits (GTK_ABOUT_DIALOG (about),
    ///                                          _("translator-credits"));
    /// ```
    /// It is a good idea to use the customary msgid “translator-credits” for this
    /// purpose, since translators will already know the purpose of that msgid, and
    /// since `gtk.AboutDialog` will detect if “translator-credits” is untranslated
    /// and hide the tab.
    extern fn gtk_about_dialog_set_translator_credits(p_about: *AboutDialog, p_translator_credits: ?[*:0]const u8) void;
    pub const setTranslatorCredits = gtk_about_dialog_set_translator_credits;

    /// Sets the version string to display in the about dialog.
    extern fn gtk_about_dialog_set_version(p_about: *AboutDialog, p_version: ?[*:0]const u8) void;
    pub const setVersion = gtk_about_dialog_set_version;

    /// Sets the URL to use for the website link.
    extern fn gtk_about_dialog_set_website(p_about: *AboutDialog, p_website: ?[*:0]const u8) void;
    pub const setWebsite = gtk_about_dialog_set_website;

    /// Sets the label to be used for the website link.
    extern fn gtk_about_dialog_set_website_label(p_about: *AboutDialog, p_website_label: [*:0]const u8) void;
    pub const setWebsiteLabel = gtk_about_dialog_set_website_label;

    /// Sets whether the license text in `about` is
    /// automatically wrapped.
    extern fn gtk_about_dialog_set_wrap_license(p_about: *AboutDialog, p_wrap_license: c_int) void;
    pub const setWrapLicense = gtk_about_dialog_set_wrap_license;

    extern fn gtk_about_dialog_get_type() usize;
    pub const getGObjectType = gtk_about_dialog_get_type;

    extern fn g_object_ref(p_self: *gtk.AboutDialog) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.AboutDialog) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *AboutDialog, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.AccelGroup` represents a group of keyboard accelerators,
/// typically attached to a toplevel `gtk.Window` (with
/// `gtk.Window.addAccelGroup`). Usually you won’t need to create a
/// `gtk.AccelGroup` directly; instead, when using `gtk.UIManager`, GTK+
/// automatically sets up the accelerators for your menus in the ui
/// manager’s `gtk.AccelGroup`.
///
/// Note that “accelerators” are different from
/// “mnemonics”. Accelerators are shortcuts for
/// activating a menu item; they appear alongside the menu item they’re a
/// shortcut for. For example “Ctrl+Q” might appear alongside the “Quit”
/// menu item. Mnemonics are shortcuts for GUI elements such as text
/// entries or buttons; they appear as underlined characters. See
/// `gtk.Label.newWithMnemonic`. Menu items can have both accelerators
/// and mnemonics, of course.
pub const AccelGroup = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtk.AccelGroupClass;
    f_parent: gobject.Object,
    f_priv: ?*gtk.AccelGroupPrivate,

    pub const virtual_methods = struct {
        /// Signal emitted when an entry is added to or removed
        ///    from the accel group.
        pub const accel_changed = struct {
            pub fn call(p_class: anytype, p_accel_group: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_keyval: c_uint, p_modifier: gdk.ModifierType, p_accel_closure: *gobject.Closure) void {
                return gobject.ext.as(AccelGroup.Class, p_class).f_accel_changed.?(gobject.ext.as(AccelGroup, p_accel_group), p_keyval, p_modifier, p_accel_closure);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_accel_group: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_keyval: c_uint, p_modifier: gdk.ModifierType, p_accel_closure: *gobject.Closure) callconv(.C) void) void {
                gobject.ext.as(AccelGroup.Class, p_class).f_accel_changed = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const is_locked = struct {
            pub const name = "is-locked";

            pub const Type = c_int;
        };

        pub const modifier_mask = struct {
            pub const name = "modifier-mask";

            pub const Type = gdk.ModifierType;
        };
    };

    pub const signals = struct {
        /// The accel-activate signal is an implementation detail of
        /// `gtk.AccelGroup` and not meant to be used by applications.
        pub const accel_activate = struct {
            pub const name = "accel-activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_acceleratable: *gobject.Object, p_keyval: c_uint, p_modifier: gdk.ModifierType, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AccelGroup, p_instance))),
                    gobject.signalLookup("accel-activate", AccelGroup.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The accel-changed signal is emitted when an entry
        /// is added to or removed from the accel group.
        ///
        /// Widgets like `gtk.AccelLabel` which display an associated
        /// accelerator should connect to this signal, and rebuild
        /// their visual representation if the `accel_closure` is theirs.
        pub const accel_changed = struct {
            pub const name = "accel-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_keyval: c_uint, p_modifier: gdk.ModifierType, p_accel_closure: *gobject.Closure, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AccelGroup, p_instance))),
                    gobject.signalLookup("accel-changed", AccelGroup.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Finds the `gtk.AccelGroup` to which `closure` is connected;
    /// see `gtk.AccelGroup.connect`.
    extern fn gtk_accel_group_from_accel_closure(p_closure: *gobject.Closure) ?*gtk.AccelGroup;
    pub const fromAccelClosure = gtk_accel_group_from_accel_closure;

    /// Creates a new `gtk.AccelGroup`.
    extern fn gtk_accel_group_new() *gtk.AccelGroup;
    pub const new = gtk_accel_group_new;

    /// Finds the first accelerator in `accel_group` that matches
    /// `accel_key` and `accel_mods`, and activates it.
    extern fn gtk_accel_group_activate(p_accel_group: *AccelGroup, p_accel_quark: glib.Quark, p_acceleratable: *gobject.Object, p_accel_key: c_uint, p_accel_mods: gdk.ModifierType) c_int;
    pub const activate = gtk_accel_group_activate;

    /// Installs an accelerator in this group. When `accel_group` is being
    /// activated in response to a call to `gtk.accelGroupsActivate`,
    /// `closure` will be invoked if the `accel_key` and `accel_mods` from
    /// `gtk.accelGroupsActivate` match those of this connection.
    ///
    /// The signature used for the `closure` is that of `gtk.AccelGroupActivate`.
    ///
    /// Note that, due to implementation details, a single closure can
    /// only be connected to one accelerator group.
    extern fn gtk_accel_group_connect(p_accel_group: *AccelGroup, p_accel_key: c_uint, p_accel_mods: gdk.ModifierType, p_accel_flags: gtk.AccelFlags, p_closure: *gobject.Closure) void;
    pub const connect = gtk_accel_group_connect;

    /// Installs an accelerator in this group, using an accelerator path
    /// to look up the appropriate key and modifiers (see
    /// `gtk.AccelMap.addEntry`). When `accel_group` is being activated
    /// in response to a call to `gtk.accelGroupsActivate`, `closure` will
    /// be invoked if the `accel_key` and `accel_mods` from
    /// `gtk.accelGroupsActivate` match the key and modifiers for the path.
    ///
    /// The signature used for the `closure` is that of `gtk.AccelGroupActivate`.
    ///
    /// Note that `accel_path` string will be stored in a `glib.Quark`. Therefore,
    /// if you pass a static string, you can save some memory by interning it
    /// first with `glib.internStaticString`.
    extern fn gtk_accel_group_connect_by_path(p_accel_group: *AccelGroup, p_accel_path: [*:0]const u8, p_closure: *gobject.Closure) void;
    pub const connectByPath = gtk_accel_group_connect_by_path;

    /// Removes an accelerator previously installed through
    /// `gtk.AccelGroup.connect`.
    ///
    /// Since 2.20 `closure` can be `NULL`.
    extern fn gtk_accel_group_disconnect(p_accel_group: *AccelGroup, p_closure: ?*gobject.Closure) c_int;
    pub const disconnect = gtk_accel_group_disconnect;

    /// Removes an accelerator previously installed through
    /// `gtk.AccelGroup.connect`.
    extern fn gtk_accel_group_disconnect_key(p_accel_group: *AccelGroup, p_accel_key: c_uint, p_accel_mods: gdk.ModifierType) c_int;
    pub const disconnectKey = gtk_accel_group_disconnect_key;

    /// Finds the first entry in an accelerator group for which
    /// `find_func` returns `TRUE` and returns its `gtk.AccelKey`.
    extern fn gtk_accel_group_find(p_accel_group: *AccelGroup, p_find_func: gtk.AccelGroupFindFunc, p_data: ?*anyopaque) *gtk.AccelKey;
    pub const find = gtk_accel_group_find;

    /// Locks are added and removed using `gtk.AccelGroup.lock` and
    /// `gtk.AccelGroup.unlock`.
    extern fn gtk_accel_group_get_is_locked(p_accel_group: *AccelGroup) c_int;
    pub const getIsLocked = gtk_accel_group_get_is_locked;

    /// Gets a `gdk.ModifierType` representing the mask for this
    /// `accel_group`. For example, `GDK_CONTROL_MASK`, `GDK_SHIFT_MASK`, etc.
    extern fn gtk_accel_group_get_modifier_mask(p_accel_group: *AccelGroup) gdk.ModifierType;
    pub const getModifierMask = gtk_accel_group_get_modifier_mask;

    /// Locks the given accelerator group.
    ///
    /// Locking an acelerator group prevents the accelerators contained
    /// within it to be changed during runtime. Refer to
    /// `gtk.AccelMap.changeEntry` about runtime accelerator changes.
    ///
    /// If called more than once, `accel_group` remains locked until
    /// `gtk.AccelGroup.unlock` has been called an equivalent number
    /// of times.
    extern fn gtk_accel_group_lock(p_accel_group: *AccelGroup) void;
    pub const lock = gtk_accel_group_lock;

    /// Queries an accelerator group for all entries matching `accel_key`
    /// and `accel_mods`.
    extern fn gtk_accel_group_query(p_accel_group: *AccelGroup, p_accel_key: c_uint, p_accel_mods: gdk.ModifierType, p_n_entries: ?*c_uint) ?[*]gtk.AccelGroupEntry;
    pub const query = gtk_accel_group_query;

    /// Undoes the last call to `gtk.AccelGroup.lock` on this `accel_group`.
    extern fn gtk_accel_group_unlock(p_accel_group: *AccelGroup) void;
    pub const unlock = gtk_accel_group_unlock;

    extern fn gtk_accel_group_get_type() usize;
    pub const getGObjectType = gtk_accel_group_get_type;

    extern fn g_object_ref(p_self: *gtk.AccelGroup) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.AccelGroup) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *AccelGroup, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.AccelLabel` widget is a subclass of `gtk.Label` that also displays an
/// accelerator key on the right of the label text, e.g. “Ctrl+S”.
/// It is commonly used in menus to show the keyboard short-cuts for commands.
///
/// The accelerator key to display is typically not set explicitly (although it
/// can be, with `gtk.AccelLabel.setAccel`). Instead, the `gtk.AccelLabel` displays
/// the accelerators which have been added to a particular widget. This widget is
/// set by calling `gtk.AccelLabel.setAccelWidget`.
///
/// For example, a `gtk.MenuItem` widget may have an accelerator added to emit
/// the “activate” signal when the “Ctrl+S” key combination is pressed.
/// A `gtk.AccelLabel` is created and added to the `gtk.MenuItem`, and
/// `gtk.AccelLabel.setAccelWidget` is called with the `gtk.MenuItem` as the
/// second argument. The `gtk.AccelLabel` will now display “Ctrl+S” after its label.
///
/// Note that creating a `gtk.MenuItem` with `gtk.MenuItem.newWithLabel` (or
/// one of the similar functions for `gtk.CheckMenuItem` and `gtk.RadioMenuItem`)
/// automatically adds a `gtk.AccelLabel` to the `gtk.MenuItem` and calls
/// `gtk.AccelLabel.setAccelWidget` to set it up for you.
///
/// A `gtk.AccelLabel` will only display accelerators which have `GTK_ACCEL_VISIBLE`
/// set (see `gtk.AccelFlags`).
/// A `gtk.AccelLabel` can display multiple accelerators and even signal names,
/// though it is almost always used to display just one accelerator key.
///
/// ## Creating a simple menu item with an accelerator key.
///
/// ```
///   GtkWidget *window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
///   GtkWidget *menu = gtk_menu_new ();
///   GtkWidget *save_item;
///   GtkAccelGroup *accel_group;
///
///   // Create a GtkAccelGroup and add it to the window.
///   accel_group = gtk_accel_group_new ();
///   gtk_window_add_accel_group (GTK_WINDOW (window), accel_group);
///
///   // Create the menu item using the convenience function.
///   save_item = gtk_menu_item_new_with_label ("Save");
///   gtk_widget_show (save_item);
///   gtk_container_add (GTK_CONTAINER (menu), save_item);
///
///   // Now add the accelerator to the GtkMenuItem. Note that since we
///   // called `gtk.MenuItem.newWithLabel` to create the GtkMenuItem
///   // the GtkAccelLabel is automatically set up to display the
///   // GtkMenuItem accelerators. We just need to make sure we use
///   // GTK_ACCEL_VISIBLE here.
///   gtk_widget_add_accelerator (save_item, "activate", accel_group,
///                               GDK_KEY_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
/// ```
///
/// # CSS nodes
///
/// ```
/// label
/// ╰── accelerator
/// ```
///
/// Like `gtk.Label`, GtkAccelLabel has a main CSS node with the name label.
/// It adds a subnode with name accelerator.
pub const AccelLabel = extern struct {
    pub const Parent = gtk.Label;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.AccelLabelClass;
    f_label: gtk.Label,
    f_priv: ?*gtk.AccelLabelPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const accel_closure = struct {
            pub const name = "accel-closure";

            pub const Type = ?*gobject.Closure;
        };

        pub const accel_widget = struct {
            pub const name = "accel-widget";

            pub const Type = ?*gtk.Widget;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.AccelLabel`.
    extern fn gtk_accel_label_new(p_string: [*:0]const u8) *gtk.AccelLabel;
    pub const new = gtk_accel_label_new;

    /// Gets the keyval and modifier mask set with
    /// `gtk.AccelLabel.setAccel`.
    extern fn gtk_accel_label_get_accel(p_accel_label: *AccelLabel, p_accelerator_key: *c_uint, p_accelerator_mods: *gdk.ModifierType) void;
    pub const getAccel = gtk_accel_label_get_accel;

    /// Fetches the widget monitored by this accelerator label. See
    /// `gtk.AccelLabel.setAccelWidget`.
    extern fn gtk_accel_label_get_accel_widget(p_accel_label: *AccelLabel) ?*gtk.Widget;
    pub const getAccelWidget = gtk_accel_label_get_accel_widget;

    /// Returns the width needed to display the accelerator key(s).
    /// This is used by menus to align all of the `gtk.MenuItem` widgets, and shouldn't
    /// be needed by applications.
    extern fn gtk_accel_label_get_accel_width(p_accel_label: *AccelLabel) c_uint;
    pub const getAccelWidth = gtk_accel_label_get_accel_width;

    /// Recreates the string representing the accelerator keys.
    /// This should not be needed since the string is automatically updated whenever
    /// accelerators are added or removed from the associated widget.
    extern fn gtk_accel_label_refetch(p_accel_label: *AccelLabel) c_int;
    pub const refetch = gtk_accel_label_refetch;

    /// Manually sets a keyval and modifier mask as the accelerator rendered
    /// by `accel_label`.
    ///
    /// If a keyval and modifier are explicitly set then these values are
    /// used regardless of any associated accel closure or widget.
    ///
    /// Providing an `accelerator_key` of 0 removes the manual setting.
    extern fn gtk_accel_label_set_accel(p_accel_label: *AccelLabel, p_accelerator_key: c_uint, p_accelerator_mods: gdk.ModifierType) void;
    pub const setAccel = gtk_accel_label_set_accel;

    /// Sets the closure to be monitored by this accelerator label. The closure
    /// must be connected to an accelerator group; see `gtk.AccelGroup.connect`.
    /// Passing `NULL` for `accel_closure` will dissociate `accel_label` from its
    /// current closure, if any.
    extern fn gtk_accel_label_set_accel_closure(p_accel_label: *AccelLabel, p_accel_closure: ?*gobject.Closure) void;
    pub const setAccelClosure = gtk_accel_label_set_accel_closure;

    /// Sets the widget to be monitored by this accelerator label. Passing `NULL` for
    /// `accel_widget` will dissociate `accel_label` from its current widget, if any.
    extern fn gtk_accel_label_set_accel_widget(p_accel_label: *AccelLabel, p_accel_widget: ?*gtk.Widget) void;
    pub const setAccelWidget = gtk_accel_label_set_accel_widget;

    extern fn gtk_accel_label_get_type() usize;
    pub const getGObjectType = gtk_accel_label_get_type;

    extern fn g_object_ref(p_self: *gtk.AccelLabel) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.AccelLabel) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *AccelLabel, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Accelerator maps are used to define runtime configurable accelerators.
/// Functions for manipulating them are are usually used by higher level
/// convenience mechanisms like `gtk.UIManager` and are thus considered
/// “low-level”. You’ll want to use them if you’re manually creating menus that
/// should have user-configurable accelerators.
///
/// An accelerator is uniquely defined by:
/// - accelerator path
/// - accelerator key
/// - accelerator modifiers
///
/// The accelerator path must consist of
/// “<WINDOWTYPE>/Category1/Category2/.../Action”, where WINDOWTYPE
/// should be a unique application-specific identifier that corresponds
/// to the kind of window the accelerator is being used in, e.g.
/// “Gimp-Image”, “Abiword-Document” or “Gnumeric-Settings”.
/// The “Category1/.../Action” portion is most appropriately chosen by
/// the action the accelerator triggers, i.e. for accelerators on menu
/// items, choose the item’s menu path, e.g. “File/Save As”,
/// “Image/View/Zoom” or “Edit/Select All”. So a full valid accelerator
/// path may look like: “<Gimp-Toolbox>/File/Dialogs/Tool Options...”.
///
/// All accelerators are stored inside one global `gtk.AccelMap` that can
/// be obtained using `gtk.AccelMap.get`. See
/// [Monitoring changes][monitoring-changes] for additional
/// details.
///
/// # Manipulating accelerators
///
/// New accelerators can be added using `gtk.AccelMap.addEntry`.
/// To search for specific accelerator, use `gtk.AccelMap.lookupEntry`.
/// Modifications of existing accelerators should be done using
/// `gtk.AccelMap.changeEntry`.
///
/// In order to avoid having some accelerators changed, they can be
/// locked using `gtk.AccelMap.lockPath`. Unlocking is done using
/// `gtk.AccelMap.unlockPath`.
///
/// # Saving and loading accelerator maps
///
/// Accelerator maps can be saved to and loaded from some external
/// resource. For simple saving and loading from file,
/// `gtk.AccelMap.save` and `gtk.AccelMap.load` are provided.
/// Saving and loading can also be done by providing file descriptor
/// to `gtk.AccelMap.saveFd` and `gtk.AccelMap.loadFd`.
///
/// # Monitoring changes
///
/// `gtk.AccelMap` object is only useful for monitoring changes of
/// accelerators. By connecting to `gtk.AccelMap.signals.changed` signal, one
/// can monitor changes of all accelerators. It is also possible to
/// monitor only single accelerator path by using it as a detail of
/// the `gtk.AccelMap.signals.changed` signal.
pub const AccelMap = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtk.AccelMapClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// Notifies of a change in the global accelerator map.
        /// The path is also used as the detail for the signal,
        /// so it is possible to connect to
        /// changed::`accel_path`.
        pub const changed = struct {
            pub const name = "changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_accel_path: [*:0]u8, p_accel_key: c_uint, p_accel_mods: gdk.ModifierType, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AccelMap, p_instance))),
                    gobject.signalLookup("changed", AccelMap.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Registers a new accelerator with the global accelerator map.
    /// This function should only be called once per `accel_path`
    /// with the canonical `accel_key` and `accel_mods` for this path.
    /// To change the accelerator during runtime programatically, use
    /// `gtk.AccelMap.changeEntry`.
    ///
    /// Set `accel_key` and `accel_mods` to 0 to request a removal of
    /// the accelerator.
    ///
    /// Note that `accel_path` string will be stored in a `glib.Quark`. Therefore, if you
    /// pass a static string, you can save some memory by interning it first with
    /// `glib.internStaticString`.
    extern fn gtk_accel_map_add_entry(p_accel_path: [*:0]const u8, p_accel_key: c_uint, p_accel_mods: gdk.ModifierType) void;
    pub const addEntry = gtk_accel_map_add_entry;

    /// Adds a filter to the global list of accel path filters.
    ///
    /// Accel map entries whose accel path matches one of the filters
    /// are skipped by `gtk.AccelMap.foreach`.
    ///
    /// This function is intended for GTK+ modules that create their own
    /// menus, but don’t want them to be saved into the applications accelerator
    /// map dump.
    extern fn gtk_accel_map_add_filter(p_filter_pattern: [*:0]const u8) void;
    pub const addFilter = gtk_accel_map_add_filter;

    /// Changes the `accel_key` and `accel_mods` currently associated with `accel_path`.
    /// Due to conflicts with other accelerators, a change may not always be possible,
    /// `replace` indicates whether other accelerators may be deleted to resolve such
    /// conflicts. A change will only occur if all conflicts could be resolved (which
    /// might not be the case if conflicting accelerators are locked). Successful
    /// changes are indicated by a `TRUE` return value.
    ///
    /// Note that `accel_path` string will be stored in a `glib.Quark`. Therefore, if you
    /// pass a static string, you can save some memory by interning it first with
    /// `glib.internStaticString`.
    extern fn gtk_accel_map_change_entry(p_accel_path: [*:0]const u8, p_accel_key: c_uint, p_accel_mods: gdk.ModifierType, p_replace: c_int) c_int;
    pub const changeEntry = gtk_accel_map_change_entry;

    /// Loops over the entries in the accelerator map whose accel path
    /// doesn’t match any of the filters added with `gtk.AccelMap.addFilter`,
    /// and execute `foreach_func` on each. The signature of `foreach_func` is
    /// that of `gtk.AccelMapForeach`, the `changed` parameter indicates whether
    /// this accelerator was changed during runtime (thus, would need
    /// saving during an accelerator map dump).
    extern fn gtk_accel_map_foreach(p_data: ?*anyopaque, p_foreach_func: gtk.AccelMapForeach) void;
    pub const foreach = gtk_accel_map_foreach;

    /// Loops over all entries in the accelerator map, and execute
    /// `foreach_func` on each. The signature of `foreach_func` is that of
    /// `gtk.AccelMapForeach`, the `changed` parameter indicates whether
    /// this accelerator was changed during runtime (thus, would need
    /// saving during an accelerator map dump).
    extern fn gtk_accel_map_foreach_unfiltered(p_data: ?*anyopaque, p_foreach_func: gtk.AccelMapForeach) void;
    pub const foreachUnfiltered = gtk_accel_map_foreach_unfiltered;

    /// Gets the singleton global `gtk.AccelMap` object. This object
    /// is useful only for notification of changes to the accelerator
    /// map via the ::changed signal; it isn’t a parameter to the
    /// other accelerator map functions.
    extern fn gtk_accel_map_get() *gtk.AccelMap;
    pub const get = gtk_accel_map_get;

    /// Parses a file previously saved with `gtk.AccelMap.save` for
    /// accelerator specifications, and propagates them accordingly.
    extern fn gtk_accel_map_load(p_file_name: [*:0]const u8) void;
    pub const load = gtk_accel_map_load;

    /// Filedescriptor variant of `gtk.AccelMap.load`.
    ///
    /// Note that the file descriptor will not be closed by this function.
    extern fn gtk_accel_map_load_fd(p_fd: c_int) void;
    pub const loadFd = gtk_accel_map_load_fd;

    /// `glib.Scanner` variant of `gtk.AccelMap.load`.
    extern fn gtk_accel_map_load_scanner(p_scanner: *glib.Scanner) void;
    pub const loadScanner = gtk_accel_map_load_scanner;

    /// Locks the given accelerator path. If the accelerator map doesn’t yet contain
    /// an entry for `accel_path`, a new one is created.
    ///
    /// Locking an accelerator path prevents its accelerator from being changed
    /// during runtime. A locked accelerator path can be unlocked by
    /// `gtk.AccelMap.unlockPath`. Refer to `gtk.AccelMap.changeEntry`
    /// for information about runtime accelerator changes.
    ///
    /// If called more than once, `accel_path` remains locked until
    /// `gtk.AccelMap.unlockPath` has been called an equivalent number
    /// of times.
    ///
    /// Note that locking of individual accelerator paths is independent from
    /// locking the `gtk.AccelGroup` containing them. For runtime accelerator
    /// changes to be possible, both the accelerator path and its `gtk.AccelGroup`
    /// have to be unlocked.
    extern fn gtk_accel_map_lock_path(p_accel_path: [*:0]const u8) void;
    pub const lockPath = gtk_accel_map_lock_path;

    /// Looks up the accelerator entry for `accel_path` and fills in `key`.
    extern fn gtk_accel_map_lookup_entry(p_accel_path: [*:0]const u8, p_key: ?*gtk.AccelKey) c_int;
    pub const lookupEntry = gtk_accel_map_lookup_entry;

    /// Saves current accelerator specifications (accelerator path, key
    /// and modifiers) to `file_name`.
    /// The file is written in a format suitable to be read back in by
    /// `gtk.AccelMap.load`.
    extern fn gtk_accel_map_save(p_file_name: [*:0]const u8) void;
    pub const save = gtk_accel_map_save;

    /// Filedescriptor variant of `gtk.AccelMap.save`.
    ///
    /// Note that the file descriptor will not be closed by this function.
    extern fn gtk_accel_map_save_fd(p_fd: c_int) void;
    pub const saveFd = gtk_accel_map_save_fd;

    /// Undoes the last call to `gtk.AccelMap.lockPath` on this `accel_path`.
    /// Refer to `gtk.AccelMap.lockPath` for information about accelerator path locking.
    extern fn gtk_accel_map_unlock_path(p_accel_path: [*:0]const u8) void;
    pub const unlockPath = gtk_accel_map_unlock_path;

    extern fn gtk_accel_map_get_type() usize;
    pub const getGObjectType = gtk_accel_map_get_type;

    extern fn g_object_ref(p_self: *gtk.AccelMap) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.AccelMap) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *AccelMap, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.Accessible` class is the base class for accessible
/// implementations for `gtk.Widget` subclasses. It is a thin
/// wrapper around `atk.Object`, which adds facilities for associating
/// a widget with its accessible object.
///
/// An accessible implementation for a third-party widget should
/// derive from `gtk.Accessible` and implement the suitable interfaces
/// from ATK, such as `atk.Text` or `atk.Selection`. To establish
/// the connection between the widget class and its corresponding
/// acccessible implementation, override the get_accessible vfunc
/// in `gtk.WidgetClass`.
pub const Accessible = extern struct {
    pub const Parent = atk.Object;
    pub const Implements = [_]type{};
    pub const Class = gtk.AccessibleClass;
    f_parent: atk.Object,
    f_priv: ?*gtk.AccessiblePrivate,

    pub const virtual_methods = struct {
        /// This function specifies the callback function to be called
        /// when the widget corresponding to a GtkAccessible is destroyed.
        pub const connect_widget_destroyed = struct {
            pub fn call(p_class: anytype, p_accessible: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Accessible.Class, p_class).f_connect_widget_destroyed.?(gobject.ext.as(Accessible, p_accessible));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_accessible: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Accessible.Class, p_class).f_connect_widget_destroyed = @ptrCast(p_implementation);
            }
        };

        pub const widget_set = struct {
            pub fn call(p_class: anytype, p_accessible: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Accessible.Class, p_class).f_widget_set.?(gobject.ext.as(Accessible, p_accessible));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_accessible: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Accessible.Class, p_class).f_widget_set = @ptrCast(p_implementation);
            }
        };

        pub const widget_unset = struct {
            pub fn call(p_class: anytype, p_accessible: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Accessible.Class, p_class).f_widget_unset.?(gobject.ext.as(Accessible, p_accessible));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_accessible: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Accessible.Class, p_class).f_widget_unset = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const widget = struct {
            pub const name = "widget";

            pub const Type = ?*gtk.Widget;
        };
    };

    pub const signals = struct {};

    /// This function specifies the callback function to be called
    /// when the widget corresponding to a GtkAccessible is destroyed.
    extern fn gtk_accessible_connect_widget_destroyed(p_accessible: *Accessible) void;
    pub const connectWidgetDestroyed = gtk_accessible_connect_widget_destroyed;

    /// Gets the `gtk.Widget` corresponding to the `gtk.Accessible`.
    /// The returned widget does not have a reference added, so
    /// you do not need to unref it.
    extern fn gtk_accessible_get_widget(p_accessible: *Accessible) ?*gtk.Widget;
    pub const getWidget = gtk_accessible_get_widget;

    /// Sets the `gtk.Widget` corresponding to the `gtk.Accessible`.
    ///
    /// `accessible` will not hold a reference to `widget`.
    /// It is the caller’s responsibility to ensure that when `widget`
    /// is destroyed, the widget is unset by calling this function
    /// again with `widget` set to `NULL`.
    extern fn gtk_accessible_set_widget(p_accessible: *Accessible, p_widget: ?*gtk.Widget) void;
    pub const setWidget = gtk_accessible_set_widget;

    extern fn gtk_accessible_get_type() usize;
    pub const getGObjectType = gtk_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.Accessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Accessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Accessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// > In GTK+ 3.10, GtkAction has been deprecated. Use `gio.Action`
/// > instead, and associate actions with `gtk.Actionable` widgets. Use
/// > `gio.MenuModel` for creating menus with `gtk.Menu.newFromModel`.
///
/// Actions represent operations that the user can be perform, along with
/// some information how it should be presented in the interface. Each action
/// provides methods to create icons, menu items and toolbar items
/// representing itself.
///
/// As well as the callback that is called when the action gets activated,
/// the following also gets associated with the action:
///
/// - a name (not translated, for path lookup)
///
/// - a label (translated, for display)
///
/// - an accelerator
///
/// - whether label indicates a stock id
///
/// - a tooltip (optional, translated)
///
/// - a toolbar label (optional, shorter than label)
///
///
/// The action will also have some state information:
///
/// - visible (shown/hidden)
///
/// - sensitive (enabled/disabled)
///
/// Apart from regular actions, there are [toggle actions][GtkToggleAction],
/// which can be toggled between two states and
/// [radio actions][GtkRadioAction], of which only one in a group
/// can be in the “active” state. Other actions can be implemented as `gtk.Action`
/// subclasses.
///
/// Each action can have one or more proxy widgets. To act as an action proxy,
/// widget needs to implement `gtk.Activatable` interface. Proxies mirror the state
/// of the action and should change when the action’s state changes. Properties
/// that are always mirrored by proxies are `gtk.Action.properties.sensitive` and
/// `gtk.Action.properties.visible`. `gtk.Action.properties.gicon`, `gtk.Action.properties.icon`-name, `gtk.Action.properties.label`,
/// `gtk.Action.properties.short`-label and `gtk.Action.properties.stock`-id properties are only mirorred
/// if proxy widget has `gtk.Activatable.properties.use`-action-appearance property set to
/// `TRUE`.
///
/// When the proxy is activated, it should activate its action.
pub const Action = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gtk.Buildable};
    pub const Class = gtk.ActionClass;
    f_object: gobject.Object,
    f_private_data: ?*gtk.ActionPrivate,

    pub const virtual_methods = struct {
        /// Emits the “activate” signal on the specified action, if it isn't
        /// insensitive. This gets called by the proxy widgets when they get
        /// activated.
        ///
        /// It can also be used to manually activate an action.
        pub const activate = struct {
            pub fn call(p_class: anytype, p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Action.Class, p_class).f_activate.?(gobject.ext.as(Action, p_action));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Action.Class, p_class).f_activate = @ptrCast(p_implementation);
            }
        };

        pub const connect_proxy = struct {
            pub fn call(p_class: anytype, p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_proxy: *gtk.Widget) void {
                return gobject.ext.as(Action.Class, p_class).f_connect_proxy.?(gobject.ext.as(Action, p_action), p_proxy);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_proxy: *gtk.Widget) callconv(.C) void) void {
                gobject.ext.as(Action.Class, p_class).f_connect_proxy = @ptrCast(p_implementation);
            }
        };

        /// If `action` provides a `gtk.Menu` widget as a submenu for the menu
        /// item or the toolbar item it creates, this function returns an
        /// instance of that menu.
        pub const create_menu = struct {
            pub fn call(p_class: anytype, p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *gtk.Widget {
                return gobject.ext.as(Action.Class, p_class).f_create_menu.?(gobject.ext.as(Action, p_action));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *gtk.Widget) void {
                gobject.ext.as(Action.Class, p_class).f_create_menu = @ptrCast(p_implementation);
            }
        };

        /// Creates a menu item widget that proxies for the given action.
        pub const create_menu_item = struct {
            pub fn call(p_class: anytype, p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *gtk.Widget {
                return gobject.ext.as(Action.Class, p_class).f_create_menu_item.?(gobject.ext.as(Action, p_action));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *gtk.Widget) void {
                gobject.ext.as(Action.Class, p_class).f_create_menu_item = @ptrCast(p_implementation);
            }
        };

        /// Creates a toolbar item widget that proxies for the given action.
        pub const create_tool_item = struct {
            pub fn call(p_class: anytype, p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *gtk.Widget {
                return gobject.ext.as(Action.Class, p_class).f_create_tool_item.?(gobject.ext.as(Action, p_action));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *gtk.Widget) void {
                gobject.ext.as(Action.Class, p_class).f_create_tool_item = @ptrCast(p_implementation);
            }
        };

        pub const disconnect_proxy = struct {
            pub fn call(p_class: anytype, p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_proxy: *gtk.Widget) void {
                return gobject.ext.as(Action.Class, p_class).f_disconnect_proxy.?(gobject.ext.as(Action, p_action), p_proxy);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_action: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_proxy: *gtk.Widget) callconv(.C) void) void {
                gobject.ext.as(Action.Class, p_class).f_disconnect_proxy = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The GtkActionGroup this GtkAction is associated with, or NULL
        /// (for internal use).
        pub const action_group = struct {
            pub const name = "action-group";

            pub const Type = ?*gtk.ActionGroup;
        };

        /// If `TRUE`, the action's menu item proxies will ignore the `gtk.Settings.properties.gtk`-menu-images
        /// setting and always show their image, if available.
        ///
        /// Use this property if the menu item would be useless or hard to use
        /// without their image.
        pub const always_show_image = struct {
            pub const name = "always-show-image";

            pub const Type = c_int;
        };

        /// The `gio.Icon` displayed in the `gtk.Action`.
        ///
        /// Note that the stock icon is preferred, if the `gtk.Action.properties.stock`-id
        /// property holds the id of an existing stock icon.
        ///
        /// This is an appearance property and thus only applies if
        /// `gtk.Activatable.properties.use`-action-appearance is `TRUE`.
        pub const gicon = struct {
            pub const name = "gicon";

            pub const Type = ?*gio.Icon;
        };

        /// When TRUE, empty menu proxies for this action are hidden.
        pub const hide_if_empty = struct {
            pub const name = "hide-if-empty";

            pub const Type = c_int;
        };

        /// The name of the icon from the icon theme.
        ///
        /// Note that the stock icon is preferred, if the `gtk.Action.properties.stock`-id
        /// property holds the id of an existing stock icon, and the `gio.Icon` is
        /// preferred if the `gtk.Action.properties.gicon` property is set.
        ///
        /// This is an appearance property and thus only applies if
        /// `gtk.Activatable.properties.use`-action-appearance is `TRUE`.
        pub const icon_name = struct {
            pub const name = "icon-name";

            pub const Type = ?[*:0]u8;
        };

        /// Whether the action is considered important. When TRUE, toolitem
        /// proxies for this action show text in GTK_TOOLBAR_BOTH_HORIZ mode.
        pub const is_important = struct {
            pub const name = "is-important";

            pub const Type = c_int;
        };

        /// The label used for menu items and buttons that activate
        /// this action. If the label is `NULL`, GTK+ uses the stock
        /// label specified via the stock-id property.
        ///
        /// This is an appearance property and thus only applies if
        /// `gtk.Activatable.properties.use`-action-appearance is `TRUE`.
        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };

        /// A unique name for the action.
        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };

        /// Whether the action is enabled.
        pub const sensitive = struct {
            pub const name = "sensitive";

            pub const Type = c_int;
        };

        /// A shorter label that may be used on toolbar buttons.
        ///
        /// This is an appearance property and thus only applies if
        /// `gtk.Activatable.properties.use`-action-appearance is `TRUE`.
        pub const short_label = struct {
            pub const name = "short-label";

            pub const Type = ?[*:0]u8;
        };

        /// The stock icon displayed in widgets representing this action.
        ///
        /// This is an appearance property and thus only applies if
        /// `gtk.Activatable.properties.use`-action-appearance is `TRUE`.
        pub const stock_id = struct {
            pub const name = "stock-id";

            pub const Type = ?[*:0]u8;
        };

        /// A tooltip for this action.
        pub const tooltip = struct {
            pub const name = "tooltip";

            pub const Type = ?[*:0]u8;
        };

        /// Whether the action is visible.
        pub const visible = struct {
            pub const name = "visible";

            pub const Type = c_int;
        };

        /// Whether the toolbar item is visible when the toolbar is in a horizontal orientation.
        pub const visible_horizontal = struct {
            pub const name = "visible-horizontal";

            pub const Type = c_int;
        };

        /// When `TRUE`, toolitem proxies for this action are represented in the
        /// toolbar overflow menu.
        pub const visible_overflown = struct {
            pub const name = "visible-overflown";

            pub const Type = c_int;
        };

        /// Whether the toolbar item is visible when the toolbar is in a vertical orientation.
        pub const visible_vertical = struct {
            pub const name = "visible-vertical";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The "activate" signal is emitted when the action is activated.
        pub const activate = struct {
            pub const name = "activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Action, p_instance))),
                    gobject.signalLookup("activate", Action.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.Action` object. To add the action to a
    /// `gtk.ActionGroup` and set the accelerator for the action,
    /// call `gtk.ActionGroup.addActionWithAccel`.
    /// See the [UI Definition section][XML-UI] for information on allowed action
    /// names.
    extern fn gtk_action_new(p_name: [*:0]const u8, p_label: ?[*:0]const u8, p_tooltip: ?[*:0]const u8, p_stock_id: ?[*:0]const u8) *gtk.Action;
    pub const new = gtk_action_new;

    /// Emits the “activate” signal on the specified action, if it isn't
    /// insensitive. This gets called by the proxy widgets when they get
    /// activated.
    ///
    /// It can also be used to manually activate an action.
    extern fn gtk_action_activate(p_action: *Action) void;
    pub const activate = gtk_action_activate;

    /// Disable activation signals from the action
    ///
    /// This is needed when updating the state of your proxy
    /// `gtk.Activatable` widget could result in calling `gtk.Action.activate`,
    /// this is a convenience function to avoid recursing in those
    /// cases (updating toggle state for instance).
    extern fn gtk_action_block_activate(p_action: *Action) void;
    pub const blockActivate = gtk_action_block_activate;

    /// Installs the accelerator for `action` if `action` has an
    /// accel path and group. See `gtk.Action.setAccelPath` and
    /// `gtk.Action.setAccelGroup`
    ///
    /// Since multiple proxies may independently trigger the installation
    /// of the accelerator, the `action` counts the number of times this
    /// function has been called and doesn’t remove the accelerator until
    /// `gtk.Action.disconnectAccelerator` has been called as many times.
    extern fn gtk_action_connect_accelerator(p_action: *Action) void;
    pub const connectAccelerator = gtk_action_connect_accelerator;

    /// This function is intended for use by action implementations to
    /// create icons displayed in the proxy widgets.
    extern fn gtk_action_create_icon(p_action: *Action, p_icon_size: c_int) *gtk.Widget;
    pub const createIcon = gtk_action_create_icon;

    /// If `action` provides a `gtk.Menu` widget as a submenu for the menu
    /// item or the toolbar item it creates, this function returns an
    /// instance of that menu.
    extern fn gtk_action_create_menu(p_action: *Action) *gtk.Widget;
    pub const createMenu = gtk_action_create_menu;

    /// Creates a menu item widget that proxies for the given action.
    extern fn gtk_action_create_menu_item(p_action: *Action) *gtk.Widget;
    pub const createMenuItem = gtk_action_create_menu_item;

    /// Creates a toolbar item widget that proxies for the given action.
    extern fn gtk_action_create_tool_item(p_action: *Action) *gtk.Widget;
    pub const createToolItem = gtk_action_create_tool_item;

    /// Undoes the effect of one call to `gtk.Action.connectAccelerator`.
    extern fn gtk_action_disconnect_accelerator(p_action: *Action) void;
    pub const disconnectAccelerator = gtk_action_disconnect_accelerator;

    /// Returns the accel closure for this action.
    extern fn gtk_action_get_accel_closure(p_action: *Action) *gobject.Closure;
    pub const getAccelClosure = gtk_action_get_accel_closure;

    /// Returns the accel path for this action.
    extern fn gtk_action_get_accel_path(p_action: *Action) [*:0]const u8;
    pub const getAccelPath = gtk_action_get_accel_path;

    /// Returns whether `action`'s menu item proxies will always
    /// show their image, if available.
    extern fn gtk_action_get_always_show_image(p_action: *Action) c_int;
    pub const getAlwaysShowImage = gtk_action_get_always_show_image;

    /// Gets the gicon of `action`.
    extern fn gtk_action_get_gicon(p_action: *Action) *gio.Icon;
    pub const getGicon = gtk_action_get_gicon;

    /// Gets the icon name of `action`.
    extern fn gtk_action_get_icon_name(p_action: *Action) [*:0]const u8;
    pub const getIconName = gtk_action_get_icon_name;

    /// Checks whether `action` is important or not
    extern fn gtk_action_get_is_important(p_action: *Action) c_int;
    pub const getIsImportant = gtk_action_get_is_important;

    /// Gets the label text of `action`.
    extern fn gtk_action_get_label(p_action: *Action) [*:0]const u8;
    pub const getLabel = gtk_action_get_label;

    /// Returns the name of the action.
    extern fn gtk_action_get_name(p_action: *Action) [*:0]const u8;
    pub const getName = gtk_action_get_name;

    /// Returns the proxy widgets for an action.
    /// See also `gtk.Activatable.getRelatedAction`.
    extern fn gtk_action_get_proxies(p_action: *Action) *glib.SList;
    pub const getProxies = gtk_action_get_proxies;

    /// Returns whether the action itself is sensitive. Note that this doesn’t
    /// necessarily mean effective sensitivity. See `gtk.Action.isSensitive`
    /// for that.
    extern fn gtk_action_get_sensitive(p_action: *Action) c_int;
    pub const getSensitive = gtk_action_get_sensitive;

    /// Gets the short label text of `action`.
    extern fn gtk_action_get_short_label(p_action: *Action) [*:0]const u8;
    pub const getShortLabel = gtk_action_get_short_label;

    /// Gets the stock id of `action`.
    extern fn gtk_action_get_stock_id(p_action: *Action) [*:0]const u8;
    pub const getStockId = gtk_action_get_stock_id;

    /// Gets the tooltip text of `action`.
    extern fn gtk_action_get_tooltip(p_action: *Action) [*:0]const u8;
    pub const getTooltip = gtk_action_get_tooltip;

    /// Returns whether the action itself is visible. Note that this doesn’t
    /// necessarily mean effective visibility. See `gtk.Action.isSensitive`
    /// for that.
    extern fn gtk_action_get_visible(p_action: *Action) c_int;
    pub const getVisible = gtk_action_get_visible;

    /// Checks whether `action` is visible when horizontal
    extern fn gtk_action_get_visible_horizontal(p_action: *Action) c_int;
    pub const getVisibleHorizontal = gtk_action_get_visible_horizontal;

    /// Checks whether `action` is visible when horizontal
    extern fn gtk_action_get_visible_vertical(p_action: *Action) c_int;
    pub const getVisibleVertical = gtk_action_get_visible_vertical;

    /// Returns whether the action is effectively sensitive.
    extern fn gtk_action_is_sensitive(p_action: *Action) c_int;
    pub const isSensitive = gtk_action_is_sensitive;

    /// Returns whether the action is effectively visible.
    extern fn gtk_action_is_visible(p_action: *Action) c_int;
    pub const isVisible = gtk_action_is_visible;

    /// Sets the `gtk.AccelGroup` in which the accelerator for this action
    /// will be installed.
    extern fn gtk_action_set_accel_group(p_action: *Action, p_accel_group: ?*gtk.AccelGroup) void;
    pub const setAccelGroup = gtk_action_set_accel_group;

    /// Sets the accel path for this action.  All proxy widgets associated
    /// with the action will have this accel path, so that their
    /// accelerators are consistent.
    ///
    /// Note that `accel_path` string will be stored in a `glib.Quark`. Therefore, if you
    /// pass a static string, you can save some memory by interning it first with
    /// `glib.internStaticString`.
    extern fn gtk_action_set_accel_path(p_action: *Action, p_accel_path: [*:0]const u8) void;
    pub const setAccelPath = gtk_action_set_accel_path;

    /// Sets whether `action`'s menu item proxies will ignore the
    /// `gtk.Settings.properties.gtk`-menu-images setting and always show their image, if available.
    ///
    /// Use this if the menu item would be useless or hard to use
    /// without their image.
    extern fn gtk_action_set_always_show_image(p_action: *Action, p_always_show: c_int) void;
    pub const setAlwaysShowImage = gtk_action_set_always_show_image;

    /// Sets the icon of `action`.
    extern fn gtk_action_set_gicon(p_action: *Action, p_icon: *gio.Icon) void;
    pub const setGicon = gtk_action_set_gicon;

    /// Sets the icon name on `action`
    extern fn gtk_action_set_icon_name(p_action: *Action, p_icon_name: [*:0]const u8) void;
    pub const setIconName = gtk_action_set_icon_name;

    /// Sets whether the action is important, this attribute is used
    /// primarily by toolbar items to decide whether to show a label
    /// or not.
    extern fn gtk_action_set_is_important(p_action: *Action, p_is_important: c_int) void;
    pub const setIsImportant = gtk_action_set_is_important;

    /// Sets the label of `action`.
    extern fn gtk_action_set_label(p_action: *Action, p_label: [*:0]const u8) void;
    pub const setLabel = gtk_action_set_label;

    /// Sets the :sensitive property of the action to `sensitive`. Note that
    /// this doesn’t necessarily mean effective sensitivity. See
    /// `gtk.Action.isSensitive`
    /// for that.
    extern fn gtk_action_set_sensitive(p_action: *Action, p_sensitive: c_int) void;
    pub const setSensitive = gtk_action_set_sensitive;

    /// Sets a shorter label text on `action`.
    extern fn gtk_action_set_short_label(p_action: *Action, p_short_label: [*:0]const u8) void;
    pub const setShortLabel = gtk_action_set_short_label;

    /// Sets the stock id on `action`
    extern fn gtk_action_set_stock_id(p_action: *Action, p_stock_id: [*:0]const u8) void;
    pub const setStockId = gtk_action_set_stock_id;

    /// Sets the tooltip text on `action`
    extern fn gtk_action_set_tooltip(p_action: *Action, p_tooltip: [*:0]const u8) void;
    pub const setTooltip = gtk_action_set_tooltip;

    /// Sets the :visible property of the action to `visible`. Note that
    /// this doesn’t necessarily mean effective visibility. See
    /// `gtk.Action.isVisible`
    /// for that.
    extern fn gtk_action_set_visible(p_action: *Action, p_visible: c_int) void;
    pub const setVisible = gtk_action_set_visible;

    /// Sets whether `action` is visible when horizontal
    extern fn gtk_action_set_visible_horizontal(p_action: *Action, p_visible_horizontal: c_int) void;
    pub const setVisibleHorizontal = gtk_action_set_visible_horizontal;

    /// Sets whether `action` is visible when vertical
    extern fn gtk_action_set_visible_vertical(p_action: *Action, p_visible_vertical: c_int) void;
    pub const setVisibleVertical = gtk_action_set_visible_vertical;

    /// Reenable activation signals from the action
    extern fn gtk_action_unblock_activate(p_action: *Action) void;
    pub const unblockActivate = gtk_action_unblock_activate;

    extern fn gtk_action_get_type() usize;
    pub const getGObjectType = gtk_action_get_type;

    extern fn g_object_ref(p_self: *gtk.Action) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Action) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Action, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// GtkActionBar is designed to present contextual actions. It is
/// expected to be displayed below the content and expand horizontally
/// to fill the area.
///
/// It allows placing children at the start or the end. In addition, it
/// contains an internal centered box which is centered with respect to
/// the full width of the box, even if the children at either side take
/// up different amounts of space.
///
/// # CSS nodes
///
/// GtkActionBar has a single CSS node with name actionbar.
pub const ActionBar = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.ActionBarClass;
    f_bin: gtk.Bin,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtk.ActionBar` widget.
    extern fn gtk_action_bar_new() *gtk.ActionBar;
    pub const new = gtk_action_bar_new;

    /// Retrieves the center bar widget of the bar.
    extern fn gtk_action_bar_get_center_widget(p_action_bar: *ActionBar) ?*gtk.Widget;
    pub const getCenterWidget = gtk_action_bar_get_center_widget;

    /// Adds `child` to `action_bar`, packed with reference to the
    /// end of the `action_bar`.
    extern fn gtk_action_bar_pack_end(p_action_bar: *ActionBar, p_child: *gtk.Widget) void;
    pub const packEnd = gtk_action_bar_pack_end;

    /// Adds `child` to `action_bar`, packed with reference to the
    /// start of the `action_bar`.
    extern fn gtk_action_bar_pack_start(p_action_bar: *ActionBar, p_child: *gtk.Widget) void;
    pub const packStart = gtk_action_bar_pack_start;

    /// Sets the center widget for the `gtk.ActionBar`.
    extern fn gtk_action_bar_set_center_widget(p_action_bar: *ActionBar, p_center_widget: ?*gtk.Widget) void;
    pub const setCenterWidget = gtk_action_bar_set_center_widget;

    extern fn gtk_action_bar_get_type() usize;
    pub const getGObjectType = gtk_action_bar_get_type;

    extern fn g_object_ref(p_self: *gtk.ActionBar) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ActionBar) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ActionBar, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Actions are organised into groups. An action group is essentially a
/// map from names to `gtk.Action` objects.
///
/// All actions that would make sense to use in a particular context
/// should be in a single group. Multiple action groups may be used for a
/// particular user interface. In fact, it is expected that most nontrivial
/// applications will make use of multiple groups. For example, in an
/// application that can edit multiple documents, one group holding global
/// actions (e.g. quit, about, new), and one group per document holding
/// actions that act on that document (eg. save, cut/copy/paste, etc). Each
/// window’s menus would be constructed from a combination of two action
/// groups.
///
/// ## Accelerators
///
/// Accelerators are handled by the GTK+ accelerator map. All actions are
/// assigned an accelerator path (which normally has the form
/// `<Actions>/group-name/action-name`) and a shortcut is associated with
/// this accelerator path. All menuitems and toolitems take on this accelerator
/// path. The GTK+ accelerator map code makes sure that the correct shortcut
/// is displayed next to the menu item.
///
/// # GtkActionGroup as GtkBuildable
///
/// The `gtk.ActionGroup` implementation of the `gtk.Buildable` interface accepts
/// `gtk.Action` objects as `<child>` elements in UI definitions.
///
/// Note that it is probably more common to define actions and action groups
/// in the code, since they are directly related to what the code can do.
///
/// The GtkActionGroup implementation of the GtkBuildable interface supports
/// a custom `<accelerator>` element, which has attributes named “key“ and
/// “modifiers“ and allows to specify accelerators. This is similar to the
/// `<accelerator>` element of `gtk.Widget`, the main difference is that
/// it doesn’t allow you to specify a signal.
///
/// ## A
///
/// ```
/// <object class="GtkActionGroup" id="actiongroup">
///   <child>
///       <object class="GtkAction" id="About">
///           <property name="name">About</property>
///           <property name="stock_id">gtk-about</property>
///           <signal handler="about_activate" name="activate"/>
///       </object>
///       <accelerator key="F1" modifiers="GDK_CONTROL_MASK | GDK_SHIFT_MASK"/>
///   </child>
/// </object>
/// ```
pub const ActionGroup = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gtk.Buildable};
    pub const Class = gtk.ActionGroupClass;
    f_parent: gobject.Object,
    f_priv: ?*gtk.ActionGroupPrivate,

    pub const virtual_methods = struct {
        /// Looks up an action in the action group by name.
        pub const get_action = struct {
            pub fn call(p_class: anytype, p_action_group: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_action_name: [*:0]const u8) *gtk.Action {
                return gobject.ext.as(ActionGroup.Class, p_class).f_get_action.?(gobject.ext.as(ActionGroup, p_action_group), p_action_name);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_action_group: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_action_name: [*:0]const u8) callconv(.C) *gtk.Action) void {
                gobject.ext.as(ActionGroup.Class, p_class).f_get_action = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The accelerator group the actions of this group should use.
        pub const accel_group = struct {
            pub const name = "accel-group";

            pub const Type = ?*gtk.AccelGroup;
        };

        /// A name for the action.
        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };

        /// Whether the action group is enabled.
        pub const sensitive = struct {
            pub const name = "sensitive";

            pub const Type = c_int;
        };

        /// Whether the action group is visible.
        pub const visible = struct {
            pub const name = "visible";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The ::connect-proxy signal is emitted after connecting a proxy to
        /// an action in the group. Note that the proxy may have been connected
        /// to a different action before.
        ///
        /// This is intended for simple customizations for which a custom action
        /// class would be too clumsy, e.g. showing tooltips for menuitems in the
        /// statusbar.
        ///
        /// `gtk.UIManager` proxies the signal and provides global notification
        /// just before any action is connected to a proxy, which is probably more
        /// convenient to use.
        pub const connect_proxy = struct {
            pub const name = "connect-proxy";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_action: *gtk.Action, p_proxy: *gtk.Widget, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ActionGroup, p_instance))),
                    gobject.signalLookup("connect-proxy", ActionGroup.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::disconnect-proxy signal is emitted after disconnecting a proxy
        /// from an action in the group.
        ///
        /// `gtk.UIManager` proxies the signal and provides global notification
        /// just before any action is connected to a proxy, which is probably more
        /// convenient to use.
        pub const disconnect_proxy = struct {
            pub const name = "disconnect-proxy";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_action: *gtk.Action, p_proxy: *gtk.Widget, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ActionGroup, p_instance))),
                    gobject.signalLookup("disconnect-proxy", ActionGroup.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::post-activate signal is emitted just after the `action` in the
        /// `action_group` is activated
        ///
        /// This is intended for `gtk.UIManager` to proxy the signal and provide global
        /// notification just after any action is activated.
        pub const post_activate = struct {
            pub const name = "post-activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_action: *gtk.Action, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ActionGroup, p_instance))),
                    gobject.signalLookup("post-activate", ActionGroup.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::pre-activate signal is emitted just before the `action` in the
        /// `action_group` is activated
        ///
        /// This is intended for `gtk.UIManager` to proxy the signal and provide global
        /// notification just before any action is activated.
        pub const pre_activate = struct {
            pub const name = "pre-activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_action: *gtk.Action, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ActionGroup, p_instance))),
                    gobject.signalLookup("pre-activate", ActionGroup.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.ActionGroup` object. The name of the action group
    /// is used when associating [keybindings][Action-Accel]
    /// with the actions.
    extern fn gtk_action_group_new(p_name: [*:0]const u8) *gtk.ActionGroup;
    pub const new = gtk_action_group_new;

    /// Adds an action object to the action group. Note that this function
    /// does not set up the accel path of the action, which can lead to problems
    /// if a user tries to modify the accelerator of a menuitem associated with
    /// the action. Therefore you must either set the accel path yourself with
    /// `gtk.Action.setAccelPath`, or use
    /// `gtk_action_group_add_action_with_accel (..., NULL)`.
    extern fn gtk_action_group_add_action(p_action_group: *ActionGroup, p_action: *gtk.Action) void;
    pub const addAction = gtk_action_group_add_action;

    /// Adds an action object to the action group and sets up the accelerator.
    ///
    /// If `accelerator` is `NULL`, attempts to use the accelerator associated
    /// with the stock_id of the action.
    ///
    /// Accel paths are set to `<Actions>/group-name/action-name`.
    extern fn gtk_action_group_add_action_with_accel(p_action_group: *ActionGroup, p_action: *gtk.Action, p_accelerator: ?[*:0]const u8) void;
    pub const addActionWithAccel = gtk_action_group_add_action_with_accel;

    /// This is a convenience function to create a number of actions and add them
    /// to the action group.
    ///
    /// The “activate” signals of the actions are connected to the callbacks
    /// and their accel paths are set to `<Actions>/group-name/action-name`.
    extern fn gtk_action_group_add_actions(p_action_group: *ActionGroup, p_entries: [*]const gtk.ActionEntry, p_n_entries: c_uint, p_user_data: ?*anyopaque) void;
    pub const addActions = gtk_action_group_add_actions;

    /// This variant of `gtk.ActionGroup.addActions` adds a `glib.DestroyNotify`
    /// callback for `user_data`.
    extern fn gtk_action_group_add_actions_full(p_action_group: *ActionGroup, p_entries: [*]const gtk.ActionEntry, p_n_entries: c_uint, p_user_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) void;
    pub const addActionsFull = gtk_action_group_add_actions_full;

    /// This is a convenience routine to create a group of radio actions and
    /// add them to the action group.
    ///
    /// The “changed” signal of the first radio action is connected to the
    /// `on_change` callback and the accel paths of the actions are set to
    /// `<Actions>/group-name/action-name`.
    extern fn gtk_action_group_add_radio_actions(p_action_group: *ActionGroup, p_entries: [*]const gtk.RadioActionEntry, p_n_entries: c_uint, p_value: c_int, p_on_change: gobject.Callback, p_user_data: ?*anyopaque) void;
    pub const addRadioActions = gtk_action_group_add_radio_actions;

    /// This variant of `gtk.ActionGroup.addRadioActions` adds a
    /// `glib.DestroyNotify` callback for `user_data`.
    extern fn gtk_action_group_add_radio_actions_full(p_action_group: *ActionGroup, p_entries: [*]const gtk.RadioActionEntry, p_n_entries: c_uint, p_value: c_int, p_on_change: gobject.Callback, p_user_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) void;
    pub const addRadioActionsFull = gtk_action_group_add_radio_actions_full;

    /// This is a convenience function to create a number of toggle actions and add them
    /// to the action group.
    ///
    /// The “activate” signals of the actions are connected to the callbacks
    /// and their accel paths are set to `<Actions>/group-name/action-name`.
    extern fn gtk_action_group_add_toggle_actions(p_action_group: *ActionGroup, p_entries: [*]const gtk.ToggleActionEntry, p_n_entries: c_uint, p_user_data: ?*anyopaque) void;
    pub const addToggleActions = gtk_action_group_add_toggle_actions;

    /// This variant of `gtk.ActionGroup.addToggleActions` adds a
    /// `glib.DestroyNotify` callback for `user_data`.
    extern fn gtk_action_group_add_toggle_actions_full(p_action_group: *ActionGroup, p_entries: [*]const gtk.ToggleActionEntry, p_n_entries: c_uint, p_user_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) void;
    pub const addToggleActionsFull = gtk_action_group_add_toggle_actions_full;

    /// Gets the accelerator group.
    extern fn gtk_action_group_get_accel_group(p_action_group: *ActionGroup) *gtk.AccelGroup;
    pub const getAccelGroup = gtk_action_group_get_accel_group;

    /// Looks up an action in the action group by name.
    extern fn gtk_action_group_get_action(p_action_group: *ActionGroup, p_action_name: [*:0]const u8) *gtk.Action;
    pub const getAction = gtk_action_group_get_action;

    /// Gets the name of the action group.
    extern fn gtk_action_group_get_name(p_action_group: *ActionGroup) [*:0]const u8;
    pub const getName = gtk_action_group_get_name;

    /// Returns `TRUE` if the group is sensitive.  The constituent actions
    /// can only be logically sensitive (see `gtk.Action.isSensitive`) if
    /// they are sensitive (see `gtk.Action.getSensitive`) and their group
    /// is sensitive.
    extern fn gtk_action_group_get_sensitive(p_action_group: *ActionGroup) c_int;
    pub const getSensitive = gtk_action_group_get_sensitive;

    /// Returns `TRUE` if the group is visible.  The constituent actions
    /// can only be logically visible (see `gtk.Action.isVisible`) if
    /// they are visible (see `gtk.Action.getVisible`) and their group
    /// is visible.
    extern fn gtk_action_group_get_visible(p_action_group: *ActionGroup) c_int;
    pub const getVisible = gtk_action_group_get_visible;

    /// Lists the actions in the action group.
    extern fn gtk_action_group_list_actions(p_action_group: *ActionGroup) *glib.List;
    pub const listActions = gtk_action_group_list_actions;

    /// Removes an action object from the action group.
    extern fn gtk_action_group_remove_action(p_action_group: *ActionGroup, p_action: *gtk.Action) void;
    pub const removeAction = gtk_action_group_remove_action;

    /// Sets the accelerator group to be used by every action in this group.
    extern fn gtk_action_group_set_accel_group(p_action_group: *ActionGroup, p_accel_group: ?*gtk.AccelGroup) void;
    pub const setAccelGroup = gtk_action_group_set_accel_group;

    /// Changes the sensitivity of `action_group`
    extern fn gtk_action_group_set_sensitive(p_action_group: *ActionGroup, p_sensitive: c_int) void;
    pub const setSensitive = gtk_action_group_set_sensitive;

    /// Sets a function to be used for translating the `label` and `tooltip` of
    /// `GtkActionEntrys` added by `gtk.ActionGroup.addActions`.
    ///
    /// If you’re using `gettext`, it is enough to set the translation domain
    /// with `gtk.ActionGroup.setTranslationDomain`.
    extern fn gtk_action_group_set_translate_func(p_action_group: *ActionGroup, p_func: gtk.TranslateFunc, p_data: ?*anyopaque, p_notify: ?glib.DestroyNotify) void;
    pub const setTranslateFunc = gtk_action_group_set_translate_func;

    /// Sets the translation domain and uses `glib.dgettext` for translating the
    /// `label` and `tooltip` of `GtkActionEntrys` added by
    /// `gtk.ActionGroup.addActions`.
    ///
    /// If you’re not using `gettext` for localization, see
    /// `gtk.ActionGroup.setTranslateFunc`.
    extern fn gtk_action_group_set_translation_domain(p_action_group: *ActionGroup, p_domain: ?[*:0]const u8) void;
    pub const setTranslationDomain = gtk_action_group_set_translation_domain;

    /// Changes the visible of `action_group`.
    extern fn gtk_action_group_set_visible(p_action_group: *ActionGroup, p_visible: c_int) void;
    pub const setVisible = gtk_action_group_set_visible;

    /// Translates a string using the function set with
    /// `gtk.ActionGroup.setTranslateFunc`. This
    /// is mainly intended for language bindings.
    extern fn gtk_action_group_translate_string(p_action_group: *ActionGroup, p_string: [*:0]const u8) [*:0]const u8;
    pub const translateString = gtk_action_group_translate_string;

    extern fn gtk_action_group_get_type() usize;
    pub const getGObjectType = gtk_action_group_get_type;

    extern fn g_object_ref(p_self: *gtk.ActionGroup) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ActionGroup) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ActionGroup, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.Adjustment` object represents a value which has an associated lower
/// and upper bound, together with step and page increments, and a page size.
/// It is used within several GTK+ widgets, including `gtk.SpinButton`, `gtk.Viewport`,
/// and `gtk.Range` (which is a base class for `gtk.Scrollbar` and `gtk.Scale`).
///
/// The `gtk.Adjustment` object does not update the value itself. Instead
/// it is left up to the owner of the `gtk.Adjustment` to control the value.
pub const Adjustment = extern struct {
    pub const Parent = gobject.InitiallyUnowned;
    pub const Implements = [_]type{};
    pub const Class = gtk.AdjustmentClass;
    f_parent_instance: gobject.InitiallyUnowned,
    f_priv: ?*gtk.AdjustmentPrivate,

    pub const virtual_methods = struct {
        /// Emits a `gtk.Adjustment.signals.changed` signal from the `gtk.Adjustment`.
        /// This is typically called by the owner of the `gtk.Adjustment` after it has
        /// changed any of the `gtk.Adjustment` properties other than the value.
        pub const changed = struct {
            pub fn call(p_class: anytype, p_adjustment: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Adjustment.Class, p_class).f_changed.?(gobject.ext.as(Adjustment, p_adjustment));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_adjustment: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Adjustment.Class, p_class).f_changed = @ptrCast(p_implementation);
            }
        };

        /// Emits a `gtk.Adjustment.signals.value`-changed signal from the `gtk.Adjustment`.
        /// This is typically called by the owner of the `gtk.Adjustment` after it has
        /// changed the `gtk.Adjustment.properties.value` property.
        pub const value_changed = struct {
            pub fn call(p_class: anytype, p_adjustment: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Adjustment.Class, p_class).f_value_changed.?(gobject.ext.as(Adjustment, p_adjustment));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_adjustment: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Adjustment.Class, p_class).f_value_changed = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The minimum value of the adjustment.
        pub const lower = struct {
            pub const name = "lower";

            pub const Type = f64;
        };

        /// The page increment of the adjustment.
        pub const page_increment = struct {
            pub const name = "page-increment";

            pub const Type = f64;
        };

        /// The page size of the adjustment.
        /// Note that the page-size is irrelevant and should be set to zero
        /// if the adjustment is used for a simple scalar value, e.g. in a
        /// `gtk.SpinButton`.
        pub const page_size = struct {
            pub const name = "page-size";

            pub const Type = f64;
        };

        /// The step increment of the adjustment.
        pub const step_increment = struct {
            pub const name = "step-increment";

            pub const Type = f64;
        };

        /// The maximum value of the adjustment.
        /// Note that values will be restricted by
        /// `upper - page-size` if the page-size
        /// property is nonzero.
        pub const upper = struct {
            pub const name = "upper";

            pub const Type = f64;
        };

        /// The value of the adjustment.
        pub const value = struct {
            pub const name = "value";

            pub const Type = f64;
        };
    };

    pub const signals = struct {
        /// Emitted when one or more of the `gtk.Adjustment` properties have been
        /// changed, other than the `gtk.Adjustment.properties.value` property.
        pub const changed = struct {
            pub const name = "changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Adjustment, p_instance))),
                    gobject.signalLookup("changed", Adjustment.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the `gtk.Adjustment.properties.value` property has been changed.
        pub const value_changed = struct {
            pub const name = "value-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Adjustment, p_instance))),
                    gobject.signalLookup("value-changed", Adjustment.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.Adjustment`.
    extern fn gtk_adjustment_new(p_value: f64, p_lower: f64, p_upper: f64, p_step_increment: f64, p_page_increment: f64, p_page_size: f64) *gtk.Adjustment;
    pub const new = gtk_adjustment_new;

    /// Emits a `gtk.Adjustment.signals.changed` signal from the `gtk.Adjustment`.
    /// This is typically called by the owner of the `gtk.Adjustment` after it has
    /// changed any of the `gtk.Adjustment` properties other than the value.
    extern fn gtk_adjustment_changed(p_adjustment: *Adjustment) void;
    pub const changed = gtk_adjustment_changed;

    /// Updates the `gtk.Adjustment.properties.value` property to ensure that the range
    /// between `lower` and `upper` is in the current page (i.e. between
    /// `gtk.Adjustment.properties.value` and `gtk.Adjustment.properties.value` + `gtk.Adjustment.properties.page`-size).
    /// If the range is larger than the page size, then only the start of it will
    /// be in the current page.
    ///
    /// A `gtk.Adjustment.signals.value`-changed signal will be emitted if the value is changed.
    extern fn gtk_adjustment_clamp_page(p_adjustment: *Adjustment, p_lower: f64, p_upper: f64) void;
    pub const clampPage = gtk_adjustment_clamp_page;

    /// Sets all properties of the adjustment at once.
    ///
    /// Use this function to avoid multiple emissions of the
    /// `gtk.Adjustment.signals.changed` signal. See `gtk.Adjustment.setLower`
    /// for an alternative way of compressing multiple emissions of
    /// `gtk.Adjustment.signals.changed` into one.
    extern fn gtk_adjustment_configure(p_adjustment: *Adjustment, p_value: f64, p_lower: f64, p_upper: f64, p_step_increment: f64, p_page_increment: f64, p_page_size: f64) void;
    pub const configure = gtk_adjustment_configure;

    /// Retrieves the minimum value of the adjustment.
    extern fn gtk_adjustment_get_lower(p_adjustment: *Adjustment) f64;
    pub const getLower = gtk_adjustment_get_lower;

    /// Gets the smaller of step increment and page increment.
    extern fn gtk_adjustment_get_minimum_increment(p_adjustment: *Adjustment) f64;
    pub const getMinimumIncrement = gtk_adjustment_get_minimum_increment;

    /// Retrieves the page increment of the adjustment.
    extern fn gtk_adjustment_get_page_increment(p_adjustment: *Adjustment) f64;
    pub const getPageIncrement = gtk_adjustment_get_page_increment;

    /// Retrieves the page size of the adjustment.
    extern fn gtk_adjustment_get_page_size(p_adjustment: *Adjustment) f64;
    pub const getPageSize = gtk_adjustment_get_page_size;

    /// Retrieves the step increment of the adjustment.
    extern fn gtk_adjustment_get_step_increment(p_adjustment: *Adjustment) f64;
    pub const getStepIncrement = gtk_adjustment_get_step_increment;

    /// Retrieves the maximum value of the adjustment.
    extern fn gtk_adjustment_get_upper(p_adjustment: *Adjustment) f64;
    pub const getUpper = gtk_adjustment_get_upper;

    /// Gets the current value of the adjustment.
    /// See `gtk.Adjustment.setValue`.
    extern fn gtk_adjustment_get_value(p_adjustment: *Adjustment) f64;
    pub const getValue = gtk_adjustment_get_value;

    /// Sets the minimum value of the adjustment.
    ///
    /// When setting multiple adjustment properties via their individual
    /// setters, multiple `gtk.Adjustment.signals.changed` signals will be emitted.
    /// However, since the emission of the `gtk.Adjustment.signals.changed` signal
    /// is tied to the emission of the `gobject.Object.signals.notify` signals of the changed
    /// properties, it’s possible to compress the `gtk.Adjustment.signals.changed`
    /// signals into one by calling `gobject.Object.freezeNotify` and
    /// `gobject.Object.thawNotify` around the calls to the individual setters.
    ///
    /// Alternatively, using a single `gobject.Object.set` for all the properties
    /// to change, or using `gtk.Adjustment.configure` has the same effect
    /// of compressing `gtk.Adjustment.signals.changed` emissions.
    extern fn gtk_adjustment_set_lower(p_adjustment: *Adjustment, p_lower: f64) void;
    pub const setLower = gtk_adjustment_set_lower;

    /// Sets the page increment of the adjustment.
    ///
    /// See `gtk.Adjustment.setLower` about how to compress multiple
    /// emissions of the `gtk.Adjustment.signals.changed` signal when setting
    /// multiple adjustment properties.
    extern fn gtk_adjustment_set_page_increment(p_adjustment: *Adjustment, p_page_increment: f64) void;
    pub const setPageIncrement = gtk_adjustment_set_page_increment;

    /// Sets the page size of the adjustment.
    ///
    /// See `gtk.Adjustment.setLower` about how to compress multiple
    /// emissions of the GtkAdjustment::changed signal when setting
    /// multiple adjustment properties.
    extern fn gtk_adjustment_set_page_size(p_adjustment: *Adjustment, p_page_size: f64) void;
    pub const setPageSize = gtk_adjustment_set_page_size;

    /// Sets the step increment of the adjustment.
    ///
    /// See `gtk.Adjustment.setLower` about how to compress multiple
    /// emissions of the `gtk.Adjustment.signals.changed` signal when setting
    /// multiple adjustment properties.
    extern fn gtk_adjustment_set_step_increment(p_adjustment: *Adjustment, p_step_increment: f64) void;
    pub const setStepIncrement = gtk_adjustment_set_step_increment;

    /// Sets the maximum value of the adjustment.
    ///
    /// Note that values will be restricted by `upper - page-size`
    /// if the page-size property is nonzero.
    ///
    /// See `gtk.Adjustment.setLower` about how to compress multiple
    /// emissions of the `gtk.Adjustment.signals.changed` signal when setting
    /// multiple adjustment properties.
    extern fn gtk_adjustment_set_upper(p_adjustment: *Adjustment, p_upper: f64) void;
    pub const setUpper = gtk_adjustment_set_upper;

    /// Sets the `gtk.Adjustment` value. The value is clamped to lie between
    /// `gtk.Adjustment.properties.lower` and `gtk.Adjustment.properties.upper`.
    ///
    /// Note that for adjustments which are used in a `gtk.Scrollbar`, the
    /// effective range of allowed values goes from `gtk.Adjustment.properties.lower` to
    /// `gtk.Adjustment.properties.upper` - `gtk.Adjustment.properties.page`-size.
    extern fn gtk_adjustment_set_value(p_adjustment: *Adjustment, p_value: f64) void;
    pub const setValue = gtk_adjustment_set_value;

    /// Emits a `gtk.Adjustment.signals.value`-changed signal from the `gtk.Adjustment`.
    /// This is typically called by the owner of the `gtk.Adjustment` after it has
    /// changed the `gtk.Adjustment.properties.value` property.
    extern fn gtk_adjustment_value_changed(p_adjustment: *Adjustment) void;
    pub const valueChanged = gtk_adjustment_value_changed;

    extern fn gtk_adjustment_get_type() usize;
    pub const getGObjectType = gtk_adjustment_get_type;

    extern fn g_object_ref(p_self: *gtk.Adjustment) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Adjustment) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Adjustment, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.Alignment` widget controls the alignment and size of its child widget.
/// It has four settings: xscale, yscale, xalign, and yalign.
///
/// The scale settings are used to specify how much the child widget should
/// expand to fill the space allocated to the `gtk.Alignment`.
/// The values can range from 0 (meaning the child doesn’t expand at all) to
/// 1 (meaning the child expands to fill all of the available space).
///
/// The align settings are used to place the child widget within the available
/// area. The values range from 0 (top or left) to 1 (bottom or right).
/// Of course, if the scale settings are both set to 1, the alignment settings
/// have no effect.
///
/// GtkAlignment has been deprecated in 3.14 and should not be used in
/// newly-written code. The desired effect can be achieved by using the
/// `gtk.Widget.properties.halign`, `gtk.Widget.properties.valign` and `gtk.Widget.properties.margin` properties on the
/// child widget.
pub const Alignment = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.AlignmentClass;
    f_bin: gtk.Bin,
    f_priv: ?*gtk.AlignmentPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The padding to insert at the bottom of the widget.
        pub const bottom_padding = struct {
            pub const name = "bottom-padding";

            pub const Type = c_uint;
        };

        /// The padding to insert at the left of the widget.
        pub const left_padding = struct {
            pub const name = "left-padding";

            pub const Type = c_uint;
        };

        /// The padding to insert at the right of the widget.
        pub const right_padding = struct {
            pub const name = "right-padding";

            pub const Type = c_uint;
        };

        /// The padding to insert at the top of the widget.
        pub const top_padding = struct {
            pub const name = "top-padding";

            pub const Type = c_uint;
        };

        /// Horizontal position of child in available space. A value of 0.0
        /// will flush the child left (or right, in RTL locales); a value
        /// of 1.0 will flush the child right (or left, in RTL locales).
        pub const xalign = struct {
            pub const name = "xalign";

            pub const Type = f32;
        };

        /// If available horizontal space is bigger than needed, how much
        /// of it to use for the child. A value of 0.0 means none; a value
        /// of 1.0 means all.
        pub const xscale = struct {
            pub const name = "xscale";

            pub const Type = f32;
        };

        /// Vertical position of child in available space. A value of 0.0
        /// will flush the child to the top; a value of 1.0 will flush the
        /// child to the bottom.
        pub const yalign = struct {
            pub const name = "yalign";

            pub const Type = f32;
        };

        /// If available vertical space is bigger than needed, how much
        /// of it to use for the child. A value of 0.0 means none; a value
        /// of 1.0 means all.
        pub const yscale = struct {
            pub const name = "yscale";

            pub const Type = f32;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.Alignment`.
    extern fn gtk_alignment_new(p_xalign: f32, p_yalign: f32, p_xscale: f32, p_yscale: f32) *gtk.Alignment;
    pub const new = gtk_alignment_new;

    /// Gets the padding on the different sides of the widget.
    /// See gtk_alignment_set_padding ().
    extern fn gtk_alignment_get_padding(p_alignment: *Alignment, p_padding_top: ?*c_uint, p_padding_bottom: ?*c_uint, p_padding_left: ?*c_uint, p_padding_right: ?*c_uint) void;
    pub const getPadding = gtk_alignment_get_padding;

    /// Sets the `gtk.Alignment` values.
    extern fn gtk_alignment_set(p_alignment: *Alignment, p_xalign: f32, p_yalign: f32, p_xscale: f32, p_yscale: f32) void;
    pub const set = gtk_alignment_set;

    /// Sets the padding on the different sides of the widget.
    /// The padding adds blank space to the sides of the widget. For instance,
    /// this can be used to indent the child widget towards the right by adding
    /// padding on the left.
    extern fn gtk_alignment_set_padding(p_alignment: *Alignment, p_padding_top: c_uint, p_padding_bottom: c_uint, p_padding_left: c_uint, p_padding_right: c_uint) void;
    pub const setPadding = gtk_alignment_set_padding;

    extern fn gtk_alignment_get_type() usize;
    pub const getGObjectType = gtk_alignment_get_type;

    extern fn g_object_ref(p_self: *gtk.Alignment) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Alignment) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Alignment, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.AppChooserButton` is a widget that lets the user select
/// an application. It implements the `gtk.AppChooser` interface.
///
/// Initially, a `gtk.AppChooserButton` selects the first application
/// in its list, which will either be the most-recently used application
/// or, if `gtk.AppChooserButton.properties.show`-default-item is `TRUE`, the
/// default application.
///
/// The list of applications shown in a `gtk.AppChooserButton` includes
/// the recommended applications for the given content type. When
/// `gtk.AppChooserButton.properties.show`-default-item is set, the default application
/// is also included. To let the user chooser other applications,
/// you can set the `gtk.AppChooserButton.properties.show`-dialog-item property,
/// which allows to open a full `gtk.AppChooserDialog`.
///
/// It is possible to add custom items to the list, using
/// `gtk.AppChooserButton.appendCustomItem`. These items cause
/// the `gtk.AppChooserButton.signals.custom`-item-activated signal to be
/// emitted when they are selected.
///
/// To track changes in the selected application, use the
/// `gtk.ComboBox.signals.changed` signal.
pub const AppChooserButton = extern struct {
    pub const Parent = gtk.ComboBox;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.AppChooser, gtk.Buildable, gtk.CellEditable, gtk.CellLayout };
    pub const Class = gtk.AppChooserButtonClass;
    f_parent: gtk.ComboBox,
    f_priv: ?*gtk.AppChooserButtonPrivate,

    pub const virtual_methods = struct {
        /// Signal emitted when a custom item,
        ///    previously added with `gtk.AppChooserButton.appendCustomItem`,
        ///    is activated from the dropdown menu.
        pub const custom_item_activated = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_name: [*:0]const u8) void {
                return gobject.ext.as(AppChooserButton.Class, p_class).f_custom_item_activated.?(gobject.ext.as(AppChooserButton, p_self), p_item_name);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_name: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(AppChooserButton.Class, p_class).f_custom_item_activated = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The text to show at the top of the dialog that can be
        /// opened from the button. The string may contain Pango markup.
        pub const heading = struct {
            pub const name = "heading";

            pub const Type = ?[*:0]u8;
        };

        /// The `gtk.AppChooserButton.properties.show`-default-item property determines
        /// whether the dropdown menu should show the default application
        /// on top for the provided content type.
        pub const show_default_item = struct {
            pub const name = "show-default-item";

            pub const Type = c_int;
        };

        /// The `gtk.AppChooserButton.properties.show`-dialog-item property determines
        /// whether the dropdown menu should show an item that triggers
        /// a `gtk.AppChooserDialog` when clicked.
        pub const show_dialog_item = struct {
            pub const name = "show-dialog-item";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// Emitted when a custom item, previously added with
        /// `gtk.AppChooserButton.appendCustomItem`, is activated from the
        /// dropdown menu.
        pub const custom_item_activated = struct {
            pub const name = "custom-item-activated";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_item_name: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AppChooserButton, p_instance))),
                    gobject.signalLookup("custom-item-activated", AppChooserButton.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.AppChooserButton` for applications
    /// that can handle content of the given type.
    extern fn gtk_app_chooser_button_new(p_content_type: [*:0]const u8) *gtk.AppChooserButton;
    pub const new = gtk_app_chooser_button_new;

    /// Appends a custom item to the list of applications that is shown
    /// in the popup; the item name must be unique per-widget.
    /// Clients can use the provided name as a detail for the
    /// `gtk.AppChooserButton.signals.custom`-item-activated signal, to add a
    /// callback for the activation of a particular custom item in the list.
    /// See also `gtk.AppChooserButton.appendSeparator`.
    extern fn gtk_app_chooser_button_append_custom_item(p_self: *AppChooserButton, p_name: [*:0]const u8, p_label: [*:0]const u8, p_icon: *gio.Icon) void;
    pub const appendCustomItem = gtk_app_chooser_button_append_custom_item;

    /// Appends a separator to the list of applications that is shown
    /// in the popup.
    extern fn gtk_app_chooser_button_append_separator(p_self: *AppChooserButton) void;
    pub const appendSeparator = gtk_app_chooser_button_append_separator;

    /// Returns the text to display at the top of the dialog.
    extern fn gtk_app_chooser_button_get_heading(p_self: *AppChooserButton) ?[*:0]const u8;
    pub const getHeading = gtk_app_chooser_button_get_heading;

    /// Returns the current value of the `gtk.AppChooserButton.properties.show`-default-item
    /// property.
    extern fn gtk_app_chooser_button_get_show_default_item(p_self: *AppChooserButton) c_int;
    pub const getShowDefaultItem = gtk_app_chooser_button_get_show_default_item;

    /// Returns the current value of the `gtk.AppChooserButton.properties.show`-dialog-item
    /// property.
    extern fn gtk_app_chooser_button_get_show_dialog_item(p_self: *AppChooserButton) c_int;
    pub const getShowDialogItem = gtk_app_chooser_button_get_show_dialog_item;

    /// Selects a custom item previously added with
    /// `gtk.AppChooserButton.appendCustomItem`.
    ///
    /// Use `gtk.AppChooser.refresh` to bring the selection
    /// to its initial state.
    extern fn gtk_app_chooser_button_set_active_custom_item(p_self: *AppChooserButton, p_name: [*:0]const u8) void;
    pub const setActiveCustomItem = gtk_app_chooser_button_set_active_custom_item;

    /// Sets the text to display at the top of the dialog.
    /// If the heading is not set, the dialog displays a default text.
    extern fn gtk_app_chooser_button_set_heading(p_self: *AppChooserButton, p_heading: [*:0]const u8) void;
    pub const setHeading = gtk_app_chooser_button_set_heading;

    /// Sets whether the dropdown menu of this button should show the
    /// default application for the given content type at top.
    extern fn gtk_app_chooser_button_set_show_default_item(p_self: *AppChooserButton, p_setting: c_int) void;
    pub const setShowDefaultItem = gtk_app_chooser_button_set_show_default_item;

    /// Sets whether the dropdown menu of this button should show an
    /// entry to trigger a `gtk.AppChooserDialog`.
    extern fn gtk_app_chooser_button_set_show_dialog_item(p_self: *AppChooserButton, p_setting: c_int) void;
    pub const setShowDialogItem = gtk_app_chooser_button_set_show_dialog_item;

    extern fn gtk_app_chooser_button_get_type() usize;
    pub const getGObjectType = gtk_app_chooser_button_get_type;

    extern fn g_object_ref(p_self: *gtk.AppChooserButton) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.AppChooserButton) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *AppChooserButton, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.AppChooserDialog` shows a `gtk.AppChooserWidget` inside a `gtk.Dialog`.
///
/// Note that `gtk.AppChooserDialog` does not have any interesting methods
/// of its own. Instead, you should get the embedded `gtk.AppChooserWidget`
/// using `gtk.AppChooserDialog.getWidget` and call its methods if
/// the generic `gtk.AppChooser` interface is not sufficient for your needs.
///
/// To set the heading that is shown above the `gtk.AppChooserWidget`,
/// use `gtk.AppChooserDialog.setHeading`.
pub const AppChooserDialog = extern struct {
    pub const Parent = gtk.Dialog;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.AppChooser, gtk.Buildable };
    pub const Class = gtk.AppChooserDialogClass;
    f_parent: gtk.Dialog,
    f_priv: ?*gtk.AppChooserDialogPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The GFile used by the `gtk.AppChooserDialog`.
        /// The dialog's `gtk.AppChooserWidget` content type will be guessed from the
        /// file, if present.
        pub const gfile = struct {
            pub const name = "gfile";

            pub const Type = ?*gio.File;
        };

        /// The text to show at the top of the dialog.
        /// The string may contain Pango markup.
        pub const heading = struct {
            pub const name = "heading";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.AppChooserDialog` for the provided `gio.File`,
    /// to allow the user to select an application for it.
    extern fn gtk_app_chooser_dialog_new(p_parent: ?*gtk.Window, p_flags: gtk.DialogFlags, p_file: *gio.File) *gtk.AppChooserDialog;
    pub const new = gtk_app_chooser_dialog_new;

    /// Creates a new `gtk.AppChooserDialog` for the provided content type,
    /// to allow the user to select an application for it.
    extern fn gtk_app_chooser_dialog_new_for_content_type(p_parent: ?*gtk.Window, p_flags: gtk.DialogFlags, p_content_type: [*:0]const u8) *gtk.AppChooserDialog;
    pub const newForContentType = gtk_app_chooser_dialog_new_for_content_type;

    /// Returns the text to display at the top of the dialog.
    extern fn gtk_app_chooser_dialog_get_heading(p_self: *AppChooserDialog) ?[*:0]const u8;
    pub const getHeading = gtk_app_chooser_dialog_get_heading;

    /// Returns the `gtk.AppChooserWidget` of this dialog.
    extern fn gtk_app_chooser_dialog_get_widget(p_self: *AppChooserDialog) *gtk.Widget;
    pub const getWidget = gtk_app_chooser_dialog_get_widget;

    /// Sets the text to display at the top of the dialog.
    /// If the heading is not set, the dialog displays a default text.
    extern fn gtk_app_chooser_dialog_set_heading(p_self: *AppChooserDialog, p_heading: [*:0]const u8) void;
    pub const setHeading = gtk_app_chooser_dialog_set_heading;

    extern fn gtk_app_chooser_dialog_get_type() usize;
    pub const getGObjectType = gtk_app_chooser_dialog_get_type;

    extern fn g_object_ref(p_self: *gtk.AppChooserDialog) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.AppChooserDialog) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *AppChooserDialog, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.AppChooserWidget` is a widget for selecting applications.
/// It is the main building block for `gtk.AppChooserDialog`. Most
/// applications only need to use the latter; but you can use
/// this widget as part of a larger widget if you have special needs.
///
/// `gtk.AppChooserWidget` offers detailed control over what applications
/// are shown, using the
/// `gtk.AppChooserWidget.properties.show`-default,
/// `gtk.AppChooserWidget.properties.show`-recommended,
/// `gtk.AppChooserWidget.properties.show`-fallback,
/// `gtk.AppChooserWidget.properties.show`-other and
/// `gtk.AppChooserWidget.properties.show`-all
/// properties. See the `gtk.AppChooser` documentation for more information
/// about these groups of applications.
///
/// To keep track of the selected application, use the
/// `gtk.AppChooserWidget.signals.application`-selected and `gtk.AppChooserWidget.signals.application`-activated signals.
///
/// # CSS nodes
///
/// GtkAppChooserWidget has a single CSS node with name appchooser.
pub const AppChooserWidget = extern struct {
    pub const Parent = gtk.Box;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.AppChooser, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.AppChooserWidgetClass;
    f_parent: gtk.Box,
    f_priv: ?*gtk.AppChooserWidgetPrivate,

    pub const virtual_methods = struct {
        /// Signal emitted when an application item is
        ///    activated from the widget’s list.
        pub const application_activated = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_app_info: *gio.AppInfo) void {
                return gobject.ext.as(AppChooserWidget.Class, p_class).f_application_activated.?(gobject.ext.as(AppChooserWidget, p_self), p_app_info);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_app_info: *gio.AppInfo) callconv(.C) void) void {
                gobject.ext.as(AppChooserWidget.Class, p_class).f_application_activated = @ptrCast(p_implementation);
            }
        };

        /// Signal emitted when an application item is
        ///    selected from the widget’s list.
        pub const application_selected = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_app_info: *gio.AppInfo) void {
                return gobject.ext.as(AppChooserWidget.Class, p_class).f_application_selected.?(gobject.ext.as(AppChooserWidget, p_self), p_app_info);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_app_info: *gio.AppInfo) callconv(.C) void) void {
                gobject.ext.as(AppChooserWidget.Class, p_class).f_application_selected = @ptrCast(p_implementation);
            }
        };

        /// Signal emitted when a context menu is about to
        ///    popup over an application item.
        pub const populate_popup = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_menu: *gtk.Menu, p_app_info: *gio.AppInfo) void {
                return gobject.ext.as(AppChooserWidget.Class, p_class).f_populate_popup.?(gobject.ext.as(AppChooserWidget, p_self), p_menu, p_app_info);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_menu: *gtk.Menu, p_app_info: *gio.AppInfo) callconv(.C) void) void {
                gobject.ext.as(AppChooserWidget.Class, p_class).f_populate_popup = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The `gtk.AppChooserWidget.properties.default`-text property determines the text
        /// that appears in the widget when there are no applications for the
        /// given content type.
        /// See also `gtk.AppChooserWidget.setDefaultText`.
        pub const default_text = struct {
            pub const name = "default-text";

            pub const Type = ?[*:0]u8;
        };

        /// If the `gtk.AppChooserWidget.properties.show`-all property is `TRUE`, the app
        /// chooser presents all applications in a single list, without
        /// subsections for default, recommended or related applications.
        pub const show_all = struct {
            pub const name = "show-all";

            pub const Type = c_int;
        };

        /// The ::show-default property determines whether the app chooser
        /// should show the default handler for the content type in a
        /// separate section. If `FALSE`, the default handler is listed
        /// among the recommended applications.
        pub const show_default = struct {
            pub const name = "show-default";

            pub const Type = c_int;
        };

        /// The `gtk.AppChooserWidget.properties.show`-fallback property determines whether
        /// the app chooser should show a section for fallback applications.
        /// If `FALSE`, the fallback applications are listed among the other
        /// applications.
        pub const show_fallback = struct {
            pub const name = "show-fallback";

            pub const Type = c_int;
        };

        /// The `gtk.AppChooserWidget.properties.show`-other property determines whether
        /// the app chooser should show a section for other applications.
        pub const show_other = struct {
            pub const name = "show-other";

            pub const Type = c_int;
        };

        /// The `gtk.AppChooserWidget.properties.show`-recommended property determines
        /// whether the app chooser should show a section for recommended
        /// applications. If `FALSE`, the recommended applications are listed
        /// among the other applications.
        pub const show_recommended = struct {
            pub const name = "show-recommended";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// Emitted when an application item is activated from the widget's list.
        ///
        /// This usually happens when the user double clicks an item, or an item
        /// is selected and the user presses one of the keys Space, Shift+Space,
        /// Return or Enter.
        pub const application_activated = struct {
            pub const name = "application-activated";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_application: *gio.AppInfo, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AppChooserWidget, p_instance))),
                    gobject.signalLookup("application-activated", AppChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when an application item is selected from the widget's list.
        pub const application_selected = struct {
            pub const name = "application-selected";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_application: *gio.AppInfo, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AppChooserWidget, p_instance))),
                    gobject.signalLookup("application-selected", AppChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a context menu is about to popup over an application item.
        /// Clients can insert menu items into the provided `gtk.Menu` object in the
        /// callback of this signal; the context menu will be shown over the item
        /// if at least one item has been added to the menu.
        pub const populate_popup = struct {
            pub const name = "populate-popup";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_menu: *gtk.Menu, p_application: *gio.AppInfo, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AppChooserWidget, p_instance))),
                    gobject.signalLookup("populate-popup", AppChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.AppChooserWidget` for applications
    /// that can handle content of the given type.
    extern fn gtk_app_chooser_widget_new(p_content_type: [*:0]const u8) *gtk.AppChooserWidget;
    pub const new = gtk_app_chooser_widget_new;

    /// Returns the text that is shown if there are not applications
    /// that can handle the content type.
    extern fn gtk_app_chooser_widget_get_default_text(p_self: *AppChooserWidget) [*:0]const u8;
    pub const getDefaultText = gtk_app_chooser_widget_get_default_text;

    /// Returns the current value of the `gtk.AppChooserWidget.properties.show`-all
    /// property.
    extern fn gtk_app_chooser_widget_get_show_all(p_self: *AppChooserWidget) c_int;
    pub const getShowAll = gtk_app_chooser_widget_get_show_all;

    /// Returns the current value of the `gtk.AppChooserWidget.properties.show`-default
    /// property.
    extern fn gtk_app_chooser_widget_get_show_default(p_self: *AppChooserWidget) c_int;
    pub const getShowDefault = gtk_app_chooser_widget_get_show_default;

    /// Returns the current value of the `gtk.AppChooserWidget.properties.show`-fallback
    /// property.
    extern fn gtk_app_chooser_widget_get_show_fallback(p_self: *AppChooserWidget) c_int;
    pub const getShowFallback = gtk_app_chooser_widget_get_show_fallback;

    /// Returns the current value of the `gtk.AppChooserWidget.properties.show`-other
    /// property.
    extern fn gtk_app_chooser_widget_get_show_other(p_self: *AppChooserWidget) c_int;
    pub const getShowOther = gtk_app_chooser_widget_get_show_other;

    /// Returns the current value of the `gtk.AppChooserWidget.properties.show`-recommended
    /// property.
    extern fn gtk_app_chooser_widget_get_show_recommended(p_self: *AppChooserWidget) c_int;
    pub const getShowRecommended = gtk_app_chooser_widget_get_show_recommended;

    /// Sets the text that is shown if there are not applications
    /// that can handle the content type.
    extern fn gtk_app_chooser_widget_set_default_text(p_self: *AppChooserWidget, p_text: [*:0]const u8) void;
    pub const setDefaultText = gtk_app_chooser_widget_set_default_text;

    /// Sets whether the app chooser should show all applications
    /// in a flat list.
    extern fn gtk_app_chooser_widget_set_show_all(p_self: *AppChooserWidget, p_setting: c_int) void;
    pub const setShowAll = gtk_app_chooser_widget_set_show_all;

    /// Sets whether the app chooser should show the default handler
    /// for the content type in a separate section.
    extern fn gtk_app_chooser_widget_set_show_default(p_self: *AppChooserWidget, p_setting: c_int) void;
    pub const setShowDefault = gtk_app_chooser_widget_set_show_default;

    /// Sets whether the app chooser should show related applications
    /// for the content type in a separate section.
    extern fn gtk_app_chooser_widget_set_show_fallback(p_self: *AppChooserWidget, p_setting: c_int) void;
    pub const setShowFallback = gtk_app_chooser_widget_set_show_fallback;

    /// Sets whether the app chooser should show applications
    /// which are unrelated to the content type.
    extern fn gtk_app_chooser_widget_set_show_other(p_self: *AppChooserWidget, p_setting: c_int) void;
    pub const setShowOther = gtk_app_chooser_widget_set_show_other;

    /// Sets whether the app chooser should show recommended applications
    /// for the content type in a separate section.
    extern fn gtk_app_chooser_widget_set_show_recommended(p_self: *AppChooserWidget, p_setting: c_int) void;
    pub const setShowRecommended = gtk_app_chooser_widget_set_show_recommended;

    extern fn gtk_app_chooser_widget_get_type() usize;
    pub const getGObjectType = gtk_app_chooser_widget_get_type;

    extern fn g_object_ref(p_self: *gtk.AppChooserWidget) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.AppChooserWidget) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *AppChooserWidget, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.Application` is a class that handles many important aspects
/// of a GTK+ application in a convenient fashion, without enforcing
/// a one-size-fits-all application model.
///
/// Currently, GtkApplication handles GTK+ initialization, application
/// uniqueness, session management, provides some basic scriptability and
/// desktop shell integration by exporting actions and menus and manages a
/// list of toplevel windows whose life-cycle is automatically tied to the
/// life-cycle of your application.
///
/// While GtkApplication works fine with plain `GtkWindows`, it is recommended
/// to use it together with `gtk.ApplicationWindow`.
///
/// When GDK threads are enabled, GtkApplication will acquire the GDK
/// lock when invoking actions that arrive from other processes.  The GDK
/// lock is not touched for local action invocations.  In order to have
/// actions invoked in a predictable context it is therefore recommended
/// that the GDK lock be held while invoking actions locally with
/// `gio.ActionGroup.activateAction`.  The same applies to actions
/// associated with `gtk.ApplicationWindow` and to the “activate” and
/// “open” `gio.Application` methods.
///
/// ## Automatic resources
///
/// `gtk.Application` will automatically load menus from the `gtk.Builder`
/// resource located at "gtk/menus.ui", relative to the application's
/// resource base path (see `gio.Application.setResourceBasePath`).  The
/// menu with the ID "app-menu" is taken as the application's app menu
/// and the menu with the ID "menubar" is taken as the application's
/// menubar.  Additional menus (most interesting submenus) can be named
/// and accessed via `gtk.Application.getMenuById` which allows for
/// dynamic population of a part of the menu structure.
///
/// If the resources "gtk/menus-appmenu.ui" or "gtk/menus-traditional.ui" are
/// present then these files will be used in preference, depending on the value
/// of `gtk.Application.prefersAppMenu`. If the resource "gtk/menus-common.ui"
/// is present it will be loaded as well. This is useful for storing items that
/// are referenced from both "gtk/menus-appmenu.ui" and
/// "gtk/menus-traditional.ui".
///
/// It is also possible to provide the menus manually using
/// `gtk.Application.setAppMenu` and `gtk.Application.setMenubar`.
///
/// `gtk.Application` will also automatically setup an icon search path for
/// the default icon theme by appending "icons" to the resource base
/// path.  This allows your application to easily store its icons as
/// resources.  See `gtk.IconTheme.addResourcePath` for more
/// information.
///
/// If there is a resource located at "gtk/help-overlay.ui" which
/// defines a `gtk.ShortcutsWindow` with ID "help_overlay" then GtkApplication
/// associates an instance of this shortcuts window with each
/// `gtk.ApplicationWindow` and sets up keyboard accelerators (Control-F1
/// and Control-?) to open it. To create a menu item that displays the
/// shortcuts window, associate the item with the action win.show-help-overlay.
///
/// ## A simple application
///
/// [A simple example](https://gitlab.gnome.org/GNOME/gtk/-/blob/gtk-3-24/examples/bp/bloatpad.c)
///
/// GtkApplication optionally registers with a session manager
/// of the users session (if you set the `gtk.Application.properties.register`-session
/// property) and offers various functionality related to the session
/// life-cycle.
///
/// An application can block various ways to end the session with
/// the `gtk.Application.inhibit` function. Typical use cases for
/// this kind of inhibiting are long-running, uninterruptible operations,
/// such as burning a CD or performing a disk backup. The session
/// manager may not honor the inhibitor, but it can be expected to
/// inform the user about the negative consequences of ending the
/// session while inhibitors are present.
///
/// ## See Also
/// [HowDoI: Using GtkApplication](https://wiki.gnome.org/HowDoI/GtkApplication),
/// [Getting Started with GTK+: Basics](https://developer.gnome.org/gtk3/stable/gtk-getting-started.html`id`-1.2.3.3)
pub const Application = extern struct {
    pub const Parent = gio.Application;
    pub const Implements = [_]type{ gio.ActionGroup, gio.ActionMap };
    pub const Class = gtk.ApplicationClass;
    f_parent: gio.Application,
    f_priv: ?*gtk.ApplicationPrivate,

    pub const virtual_methods = struct {
        /// Signal emitted when a `gtk.Window` is added to
        ///    application through `gtk.Application.addWindow`.
        pub const window_added = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_window: *gtk.Window) void {
                return gobject.ext.as(Application.Class, p_class).f_window_added.?(gobject.ext.as(Application, p_application), p_window);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_window: *gtk.Window) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_window_added = @ptrCast(p_implementation);
            }
        };

        /// Signal emitted when a `gtk.Window` is removed from
        ///    application, either as a side-effect of being destroyed or
        ///    explicitly through `gtk.Application.removeWindow`.
        pub const window_removed = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_window: *gtk.Window) void {
                return gobject.ext.as(Application.Class, p_class).f_window_removed.?(gobject.ext.as(Application, p_application), p_window);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_window: *gtk.Window) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_window_removed = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const active_window = struct {
            pub const name = "active-window";

            pub const Type = ?*gtk.Window;
        };

        pub const app_menu = struct {
            pub const name = "app-menu";

            pub const Type = ?*gio.MenuModel;
        };

        pub const menubar = struct {
            pub const name = "menubar";

            pub const Type = ?*gio.MenuModel;
        };

        /// Set this property to `TRUE` to register with the session manager.
        pub const register_session = struct {
            pub const name = "register-session";

            pub const Type = c_int;
        };

        /// This property is `TRUE` if GTK+ believes that the screensaver is
        /// currently active. GTK+ only tracks session state (including this)
        /// when `gtk.Application.signals.register`-session is set to `TRUE`.
        ///
        /// Tracking the screensaver state is supported on Linux.
        pub const screensaver_active = struct {
            pub const name = "screensaver-active";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// Emitted when the session manager is about to end the session, only
        /// if `gtk.Application.signals.register`-session is `TRUE`. Applications can
        /// connect to this signal and call `gtk.Application.inhibit` with
        /// `GTK_APPLICATION_INHIBIT_LOGOUT` to delay the end of the session
        /// until state has been saved.
        pub const query_end = struct {
            pub const name = "query-end";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Application, p_instance))),
                    gobject.signalLookup("query-end", Application.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a `gtk.Window` is added to `application` through
        /// `gtk.Application.addWindow`.
        pub const window_added = struct {
            pub const name = "window-added";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_window: *gtk.Window, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Application, p_instance))),
                    gobject.signalLookup("window-added", Application.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a `gtk.Window` is removed from `application`,
        /// either as a side-effect of being destroyed or explicitly
        /// through `gtk.Application.removeWindow`.
        pub const window_removed = struct {
            pub const name = "window-removed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_window: *gtk.Window, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Application, p_instance))),
                    gobject.signalLookup("window-removed", Application.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.Application` instance.
    ///
    /// When using `gtk.Application`, it is not necessary to call `gtk.init`
    /// manually. It is called as soon as the application gets registered as
    /// the primary instance.
    ///
    /// Concretely, `gtk.init` is called in the default handler for the
    /// `gio.Application.signals.startup` signal. Therefore, `gtk.Application` subclasses should
    /// chain up in their `gio.Application.signals.startup` handler before using any GTK+ API.
    ///
    /// Note that commandline arguments are not passed to `gtk.init`.
    /// All GTK+ functionality that is available via commandline arguments
    /// can also be achieved by setting suitable environment variables
    /// such as `G_DEBUG`, so this should not be a big
    /// problem. If you absolutely must support GTK+ commandline arguments,
    /// you can explicitly call `gtk.init` before creating the application
    /// instance.
    ///
    /// If non-`NULL`, the application ID must be valid.  See
    /// `gio.Application.idIsValid`.
    ///
    /// If no application ID is given then some features (most notably application
    /// uniqueness) will be disabled. A null application ID is only allowed with
    /// GTK+ 3.6 or later.
    extern fn gtk_application_new(p_application_id: ?[*:0]const u8, p_flags: gio.ApplicationFlags) *gtk.Application;
    pub const new = gtk_application_new;

    /// Installs an accelerator that will cause the named action
    /// to be activated when the key combination specificed by `accelerator`
    /// is pressed.
    ///
    /// `accelerator` must be a string that can be parsed by `gtk.acceleratorParse`,
    /// e.g. "<Primary>q" or “<Control><Alt>p”.
    ///
    /// `action_name` must be the name of an action as it would be used
    /// in the app menu, i.e. actions that have been added to the application
    /// are referred to with an “app.” prefix, and window-specific actions
    /// with a “win.” prefix.
    ///
    /// GtkApplication also extracts accelerators out of “accel” attributes
    /// in the `GMenuModels` passed to `gtk.Application.setAppMenu` and
    /// `gtk.Application.setMenubar`, which is usually more convenient
    /// than calling this function for each accelerator.
    extern fn gtk_application_add_accelerator(p_application: *Application, p_accelerator: [*:0]const u8, p_action_name: [*:0]const u8, p_parameter: ?*glib.Variant) void;
    pub const addAccelerator = gtk_application_add_accelerator;

    /// Adds a window to `application`.
    ///
    /// This call can only happen after the `application` has started;
    /// typically, you should add new application windows in response
    /// to the emission of the `gio.Application.signals.activate` signal.
    ///
    /// This call is equivalent to setting the `gtk.Window.properties.application`
    /// property of `window` to `application`.
    ///
    /// Normally, the connection between the application and the window
    /// will remain until the window is destroyed, but you can explicitly
    /// remove it with `gtk.Application.removeWindow`.
    ///
    /// GTK+ will keep the `application` running as long as it has
    /// any windows.
    extern fn gtk_application_add_window(p_application: *Application, p_window: *gtk.Window) void;
    pub const addWindow = gtk_application_add_window;

    /// Gets the accelerators that are currently associated with
    /// the given action.
    extern fn gtk_application_get_accels_for_action(p_application: *Application, p_detailed_action_name: [*:0]const u8) [*][*:0]u8;
    pub const getAccelsForAction = gtk_application_get_accels_for_action;

    /// Returns the list of actions (possibly empty) that `accel` maps to.
    /// Each item in the list is a detailed action name in the usual form.
    ///
    /// This might be useful to discover if an accel already exists in
    /// order to prevent installation of a conflicting accelerator (from
    /// an accelerator editor or a plugin system, for example). Note that
    /// having more than one action per accelerator may not be a bad thing
    /// and might make sense in cases where the actions never appear in the
    /// same context.
    ///
    /// In case there are no actions for a given accelerator, an empty array
    /// is returned.  `NULL` is never returned.
    ///
    /// It is a programmer error to pass an invalid accelerator string.
    /// If you are unsure, check it with `gtk.acceleratorParse` first.
    extern fn gtk_application_get_actions_for_accel(p_application: *Application, p_accel: [*:0]const u8) [*][*:0]u8;
    pub const getActionsForAccel = gtk_application_get_actions_for_accel;

    /// Gets the “active” window for the application.
    ///
    /// The active window is the one that was most recently focused (within
    /// the application).  This window may not have the focus at the moment
    /// if another application has it — this is just the most
    /// recently-focused window within this application.
    extern fn gtk_application_get_active_window(p_application: *Application) ?*gtk.Window;
    pub const getActiveWindow = gtk_application_get_active_window;

    /// Returns the menu model that has been set with
    /// `gtk.Application.setAppMenu`.
    extern fn gtk_application_get_app_menu(p_application: *Application) ?*gio.MenuModel;
    pub const getAppMenu = gtk_application_get_app_menu;

    /// Gets a menu from automatically loaded resources.
    /// See [Automatic resources][automatic-resources]
    /// for more information.
    extern fn gtk_application_get_menu_by_id(p_application: *Application, p_id: [*:0]const u8) *gio.Menu;
    pub const getMenuById = gtk_application_get_menu_by_id;

    /// Returns the menu model that has been set with
    /// `gtk.Application.setMenubar`.
    extern fn gtk_application_get_menubar(p_application: *Application) *gio.MenuModel;
    pub const getMenubar = gtk_application_get_menubar;

    /// Returns the `gtk.ApplicationWindow` with the given ID.
    ///
    /// The ID of a `gtk.ApplicationWindow` can be retrieved with
    /// `gtk.ApplicationWindow.getId`.
    extern fn gtk_application_get_window_by_id(p_application: *Application, p_id: c_uint) ?*gtk.Window;
    pub const getWindowById = gtk_application_get_window_by_id;

    /// Gets a list of the `GtkWindows` associated with `application`.
    ///
    /// The list is sorted by most recently focused window, such that the first
    /// element is the currently focused window. (Useful for choosing a parent
    /// for a transient window.)
    ///
    /// The list that is returned should not be modified in any way. It will
    /// only remain valid until the next focus change or window creation or
    /// deletion.
    extern fn gtk_application_get_windows(p_application: *Application) *glib.List;
    pub const getWindows = gtk_application_get_windows;

    /// Inform the session manager that certain types of actions should be
    /// inhibited. This is not guaranteed to work on all platforms and for
    /// all types of actions.
    ///
    /// Applications should invoke this method when they begin an operation
    /// that should not be interrupted, such as creating a CD or DVD. The
    /// types of actions that may be blocked are specified by the `flags`
    /// parameter. When the application completes the operation it should
    /// call `gtk.Application.uninhibit` to remove the inhibitor. Note that
    /// an application can have multiple inhibitors, and all of them must
    /// be individually removed. Inhibitors are also cleared when the
    /// application exits.
    ///
    /// Applications should not expect that they will always be able to block
    /// the action. In most cases, users will be given the option to force
    /// the action to take place.
    ///
    /// Reasons should be short and to the point.
    ///
    /// If `window` is given, the session manager may point the user to
    /// this window to find out more about why the action is inhibited.
    extern fn gtk_application_inhibit(p_application: *Application, p_window: ?*gtk.Window, p_flags: gtk.ApplicationInhibitFlags, p_reason: ?[*:0]const u8) c_uint;
    pub const inhibit = gtk_application_inhibit;

    /// Determines if any of the actions specified in `flags` are
    /// currently inhibited (possibly by another application).
    ///
    /// Note that this information may not be available (for example
    /// when the application is running in a sandbox).
    extern fn gtk_application_is_inhibited(p_application: *Application, p_flags: gtk.ApplicationInhibitFlags) c_int;
    pub const isInhibited = gtk_application_is_inhibited;

    /// Lists the detailed action names which have associated accelerators.
    /// See `gtk.Application.setAccelsForAction`.
    extern fn gtk_application_list_action_descriptions(p_application: *Application) [*][*:0]u8;
    pub const listActionDescriptions = gtk_application_list_action_descriptions;

    /// Determines if the desktop environment in which the application is
    /// running would prefer an application menu be shown.
    ///
    /// If this function returns `TRUE` then the application should call
    /// `gtk.Application.setAppMenu` with the contents of an application
    /// menu, which will be shown by the desktop environment.  If it returns
    /// `FALSE` then you should consider using an alternate approach, such as
    /// a menubar.
    ///
    /// The value returned by this function is purely advisory and you are
    /// free to ignore it.  If you call `gtk.Application.setAppMenu` even
    /// if the desktop environment doesn't support app menus, then a fallback
    /// will be provided.
    ///
    /// Applications are similarly free not to set an app menu even if the
    /// desktop environment wants to show one.  In that case, a fallback will
    /// also be created by the desktop environment (GNOME, for example, uses
    /// a menu with only a "Quit" item in it).
    ///
    /// The value returned by this function never changes.  Once it returns a
    /// particular value, it is guaranteed to always return the same value.
    ///
    /// You may only call this function after the application has been
    /// registered and after the base startup handler has run.  You're most
    /// likely to want to use this from your own startup handler.  It may
    /// also make sense to consult this function while constructing UI (in
    /// activate, open or an action activation handler) in order to determine
    /// if you should show a gear menu or not.
    ///
    /// This function will return `FALSE` on Mac OS and a default app menu
    /// will be created automatically with the "usual" contents of that menu
    /// typical to most Mac OS applications.  If you call
    /// `gtk.Application.setAppMenu` anyway, then this menu will be
    /// replaced with your own.
    extern fn gtk_application_prefers_app_menu(p_application: *Application) c_int;
    pub const prefersAppMenu = gtk_application_prefers_app_menu;

    /// Removes an accelerator that has been previously added
    /// with `gtk.Application.addAccelerator`.
    extern fn gtk_application_remove_accelerator(p_application: *Application, p_action_name: [*:0]const u8, p_parameter: ?*glib.Variant) void;
    pub const removeAccelerator = gtk_application_remove_accelerator;

    /// Remove a window from `application`.
    ///
    /// If `window` belongs to `application` then this call is equivalent to
    /// setting the `gtk.Window.properties.application` property of `window` to
    /// `NULL`.
    ///
    /// The application may stop running as a result of a call to this
    /// function.
    extern fn gtk_application_remove_window(p_application: *Application, p_window: *gtk.Window) void;
    pub const removeWindow = gtk_application_remove_window;

    /// Sets zero or more keyboard accelerators that will trigger the
    /// given action. The first item in `accels` will be the primary
    /// accelerator, which may be displayed in the UI.
    ///
    /// To remove all accelerators for an action, use an empty, zero-terminated
    /// array for `accels`.
    ///
    /// For the `detailed_action_name`, see `gio.actionParseDetailedName` and
    /// `gio.actionPrintDetailedName`.
    extern fn gtk_application_set_accels_for_action(p_application: *Application, p_detailed_action_name: [*:0]const u8, p_accels: [*]const [*:0]const u8) void;
    pub const setAccelsForAction = gtk_application_set_accels_for_action;

    /// Sets or unsets the application menu for `application`.
    ///
    /// This can only be done in the primary instance of the application,
    /// after it has been registered.  `gio.Application.signals.startup` is a good place
    /// to call this.
    ///
    /// The application menu is a single menu containing items that typically
    /// impact the application as a whole, rather than acting on a specific
    /// window or document.  For example, you would expect to see
    /// “Preferences” or “Quit” in an application menu, but not “Save” or
    /// “Print”.
    ///
    /// If supported, the application menu will be rendered by the desktop
    /// environment.
    ///
    /// Use the base `gio.ActionMap` interface to add actions, to respond to the user
    /// selecting these menu items.
    extern fn gtk_application_set_app_menu(p_application: *Application, p_app_menu: ?*gio.MenuModel) void;
    pub const setAppMenu = gtk_application_set_app_menu;

    /// Sets or unsets the menubar for windows of `application`.
    ///
    /// This is a menubar in the traditional sense.
    ///
    /// This can only be done in the primary instance of the application,
    /// after it has been registered.  `gio.Application.signals.startup` is a good place
    /// to call this.
    ///
    /// Depending on the desktop environment, this may appear at the top of
    /// each window, or at the top of the screen.  In some environments, if
    /// both the application menu and the menubar are set, the application
    /// menu will be presented as if it were the first item of the menubar.
    /// Other environments treat the two as completely separate — for example,
    /// the application menu may be rendered by the desktop shell while the
    /// menubar (if set) remains in each individual window.
    ///
    /// Use the base `gio.ActionMap` interface to add actions, to respond to the
    /// user selecting these menu items.
    extern fn gtk_application_set_menubar(p_application: *Application, p_menubar: ?*gio.MenuModel) void;
    pub const setMenubar = gtk_application_set_menubar;

    /// Removes an inhibitor that has been established with `gtk.Application.inhibit`.
    /// Inhibitors are also cleared when the application exits.
    extern fn gtk_application_uninhibit(p_application: *Application, p_cookie: c_uint) void;
    pub const uninhibit = gtk_application_uninhibit;

    extern fn gtk_application_get_type() usize;
    pub const getGObjectType = gtk_application_get_type;

    extern fn g_object_ref(p_self: *gtk.Application) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Application) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Application, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.ApplicationWindow` is a `gtk.Window` subclass that offers some
/// extra functionality for better integration with `gtk.Application`
/// features.  Notably, it can handle both the application menu as well
/// as the menubar. See `gtk.Application.setAppMenu` and
/// `gtk.Application.setMenubar`.
///
/// This class implements the `gio.ActionGroup` and `gio.ActionMap` interfaces,
/// to let you add window-specific actions that will be exported by the
/// associated `gtk.Application`, together with its application-wide
/// actions.  Window-specific actions are prefixed with the “win.”
/// prefix and application-wide actions are prefixed with the “app.”
/// prefix.  Actions must be addressed with the prefixed name when
/// referring to them from a `gio.MenuModel`.
///
/// Note that widgets that are placed inside a `gtk.ApplicationWindow`
/// can also activate these actions, if they implement the
/// `gtk.Actionable` interface.
///
/// As with `gtk.Application`, the GDK lock will be acquired when
/// processing actions arriving from other processes and should therefore
/// be held when activating actions locally (if GDK threads are enabled).
///
/// The settings `gtk.Settings.properties.gtk`-shell-shows-app-menu and
/// `gtk.Settings.properties.gtk`-shell-shows-menubar tell GTK+ whether the
/// desktop environment is showing the application menu and menubar
/// models outside the application as part of the desktop shell.
/// For instance, on OS X, both menus will be displayed remotely;
/// on Windows neither will be. gnome-shell (starting with version 3.4)
/// will display the application menu, but not the menubar.
///
/// If the desktop environment does not display the menubar, then
/// `gtk.ApplicationWindow` will automatically show a `gtk.MenuBar` for it.
/// This behaviour can be overridden with the `gtk.ApplicationWindow.properties.show`-menubar
/// property. If the desktop environment does not display the application
/// menu, then it will automatically be included in the menubar or in the
/// windows client-side decorations.
///
/// ## A GtkApplicationWindow with a menubar
///
/// ```
/// GtkApplication *app = gtk_application_new ("org.gtk.test", 0);
///
/// GtkBuilder *builder = gtk_builder_new_from_string (
///     "<interface>"
///     "  <menu id='menubar'>"
///     "    <submenu label='_Edit'>"
///     "      <item label='_Copy' action='win.copy'/>"
///     "      <item label='_Paste' action='win.paste'/>"
///     "    </submenu>"
///     "  </menu>"
///     "</interface>",
///     -1);
///
/// GMenuModel *menubar = G_MENU_MODEL (gtk_builder_get_object (builder,
///                                                             "menubar"));
/// gtk_application_set_menubar (GTK_APPLICATION (app), menubar);
/// g_object_unref (builder);
///
/// // ...
///
/// GtkWidget *window = gtk_application_window_new (app);
/// ```
///
/// ## Handling fallback yourself
///
/// [A simple example](https://git.gnome.org/browse/gtk+/tree/examples/sunny.c)
///
/// The XML format understood by `gtk.Builder` for `gio.MenuModel` consists
/// of a toplevel `<menu>` element, which contains one or more `<item>`
/// elements. Each `<item>` element contains `<attribute>` and `<link>`
/// elements with a mandatory name attribute. `<link>` elements have the
/// same content model as `<menu>`. Instead of `<link name="submenu>` or
/// `<link name="section">`, you can use `<submenu>` or `<section>`
/// elements.
///
/// Attribute values can be translated using gettext, like other `gtk.Builder`
/// content. `<attribute>` elements can be marked for translation with a
/// `translatable="yes"` attribute. It is also possible to specify message
/// context and translator comments, using the context and comments attributes.
/// To make use of this, the `gtk.Builder` must have been given the gettext
/// domain to use.
///
/// The following attributes are used when constructing menu items:
/// - "label": a user-visible string to display
/// - "action": the prefixed name of the action to trigger
/// - "target": the parameter to use when activating the action
/// - "icon" and "verb-icon": names of icons that may be displayed
/// - "submenu-action": name of an action that may be used to determine
///      if a submenu can be opened
/// - "hidden-when": a string used to determine when the item will be hidden.
///      Possible values include "action-disabled", "action-missing", "macos-menubar".
///
/// The following attributes are used when constructing sections:
/// - "label": a user-visible string to use as section heading
/// - "display-hint": a string used to determine special formatting for the section.
///     Possible values include "horizontal-buttons".
/// - "text-direction": a string used to determine the `gtk.TextDirection` to use
///     when "display-hint" is set to "horizontal-buttons". Possible values
///     include "rtl", "ltr", and "none".
///
/// The following attributes are used when constructing submenus:
/// - "label": a user-visible string to display
/// - "icon": icon name to display
pub const ApplicationWindow = extern struct {
    pub const Parent = gtk.Window;
    pub const Implements = [_]type{ atk.ImplementorIface, gio.ActionGroup, gio.ActionMap, gtk.Buildable };
    pub const Class = gtk.ApplicationWindowClass;
    f_parent_instance: gtk.Window,
    f_priv: ?*gtk.ApplicationWindowPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// If this property is `TRUE`, the window will display a menubar
        /// that includes the app menu and menubar, unless these are
        /// shown by the desktop shell. See `gtk.Application.setAppMenu`
        /// and `gtk.Application.setMenubar`.
        ///
        /// If `FALSE`, the window will not display a menubar, regardless
        /// of whether the desktop shell is showing the menus or not.
        pub const show_menubar = struct {
            pub const name = "show-menubar";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.ApplicationWindow`.
    extern fn gtk_application_window_new(p_application: *gtk.Application) *gtk.ApplicationWindow;
    pub const new = gtk_application_window_new;

    /// Gets the `gtk.ShortcutsWindow` that has been set up with
    /// a prior call to `gtk.ApplicationWindow.setHelpOverlay`.
    extern fn gtk_application_window_get_help_overlay(p_window: *ApplicationWindow) ?*gtk.ShortcutsWindow;
    pub const getHelpOverlay = gtk_application_window_get_help_overlay;

    /// Returns the unique ID of the window. If the window has not yet been added to
    /// a `gtk.Application`, returns `0`.
    extern fn gtk_application_window_get_id(p_window: *ApplicationWindow) c_uint;
    pub const getId = gtk_application_window_get_id;

    /// Returns whether the window will display a menubar for the app menu
    /// and menubar as needed.
    extern fn gtk_application_window_get_show_menubar(p_window: *ApplicationWindow) c_int;
    pub const getShowMenubar = gtk_application_window_get_show_menubar;

    /// Associates a shortcuts window with the application window, and
    /// sets up an action with the name win.show-help-overlay to present
    /// it.
    ///
    /// `window` takes resposibility for destroying `help_overlay`.
    extern fn gtk_application_window_set_help_overlay(p_window: *ApplicationWindow, p_help_overlay: ?*gtk.ShortcutsWindow) void;
    pub const setHelpOverlay = gtk_application_window_set_help_overlay;

    /// Sets whether the window will display a menubar for the app menu
    /// and menubar as needed.
    extern fn gtk_application_window_set_show_menubar(p_window: *ApplicationWindow, p_show_menubar: c_int) void;
    pub const setShowMenubar = gtk_application_window_set_show_menubar;

    extern fn gtk_application_window_get_type() usize;
    pub const getGObjectType = gtk_application_window_get_type;

    extern fn g_object_ref(p_self: *gtk.ApplicationWindow) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ApplicationWindow) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ApplicationWindow, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// GtkArrow should be used to draw simple arrows that need to point in
/// one of the four cardinal directions (up, down, left, or right).  The
/// style of the arrow can be one of shadow in, shadow out, etched in, or
/// etched out.  Note that these directions and style types may be
/// amended in versions of GTK+ to come.
///
/// GtkArrow will fill any space alloted to it, but since it is inherited
/// from `gtk.Misc`, it can be padded and/or aligned, to fill exactly the
/// space the programmer desires.
///
/// Arrows are created with a call to `gtk.Arrow.new`.  The direction or
/// style of an arrow can be changed after creation by using `gtk.Arrow.set`.
///
/// GtkArrow has been deprecated; you can simply use a `gtk.Image` with a
/// suitable icon name, such as “pan-down-symbolic“. When replacing
/// GtkArrow by an image, pay attention to the fact that GtkArrow is
/// doing automatic flipping between `GTK_ARROW_LEFT` and `GTK_ARROW_RIGHT`,
/// depending on the text direction. To get the same effect with an image,
/// use the icon names “pan-start-symbolic“ and “pan-end-symbolic“, which
/// react to the text direction.
pub const Arrow = extern struct {
    pub const Parent = gtk.Misc;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.ArrowClass;
    f_misc: gtk.Misc,
    f_priv: ?*gtk.ArrowPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const arrow_type = struct {
            pub const name = "arrow-type";

            pub const Type = gtk.ArrowType;
        };

        pub const shadow_type = struct {
            pub const name = "shadow-type";

            pub const Type = gtk.ShadowType;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.Arrow` widget.
    extern fn gtk_arrow_new(p_arrow_type: gtk.ArrowType, p_shadow_type: gtk.ShadowType) *gtk.Arrow;
    pub const new = gtk_arrow_new;

    /// Sets the direction and style of the `gtk.Arrow`, `arrow`.
    extern fn gtk_arrow_set(p_arrow: *Arrow, p_arrow_type: gtk.ArrowType, p_shadow_type: gtk.ShadowType) void;
    pub const set = gtk_arrow_set;

    extern fn gtk_arrow_get_type() usize;
    pub const getGObjectType = gtk_arrow_get_type;

    extern fn g_object_ref(p_self: *gtk.Arrow) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Arrow) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Arrow, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ArrowAccessible = extern struct {
    pub const Parent = gtk.WidgetAccessible;
    pub const Implements = [_]type{ atk.Component, atk.Image };
    pub const Class = gtk.ArrowAccessibleClass;
    f_parent: gtk.WidgetAccessible,
    f_priv: ?*gtk.ArrowAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_arrow_accessible_get_type() usize;
    pub const getGObjectType = gtk_arrow_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.ArrowAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ArrowAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ArrowAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.AspectFrame` is useful when you want
/// pack a widget so that it can resize but always retains
/// the same aspect ratio. For instance, one might be
/// drawing a small preview of a larger image. `gtk.AspectFrame`
/// derives from `gtk.Frame`, so it can draw a label and
/// a frame around the child. The frame will be
/// “shrink-wrapped” to the size of the child.
///
/// # CSS nodes
///
/// GtkAspectFrame uses a CSS node with name frame.
pub const AspectFrame = extern struct {
    pub const Parent = gtk.Frame;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.AspectFrameClass;
    f_frame: gtk.Frame,
    f_priv: ?*gtk.AspectFramePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const obey_child = struct {
            pub const name = "obey-child";

            pub const Type = c_int;
        };

        pub const ratio = struct {
            pub const name = "ratio";

            pub const Type = f32;
        };

        pub const xalign = struct {
            pub const name = "xalign";

            pub const Type = f32;
        };

        pub const yalign = struct {
            pub const name = "yalign";

            pub const Type = f32;
        };
    };

    pub const signals = struct {};

    /// Create a new `gtk.AspectFrame`.
    extern fn gtk_aspect_frame_new(p_label: ?[*:0]const u8, p_xalign: f32, p_yalign: f32, p_ratio: f32, p_obey_child: c_int) *gtk.AspectFrame;
    pub const new = gtk_aspect_frame_new;

    /// Set parameters for an existing `gtk.AspectFrame`.
    extern fn gtk_aspect_frame_set(p_aspect_frame: *AspectFrame, p_xalign: f32, p_yalign: f32, p_ratio: f32, p_obey_child: c_int) void;
    pub const set = gtk_aspect_frame_set;

    extern fn gtk_aspect_frame_get_type() usize;
    pub const getGObjectType = gtk_aspect_frame_get_type;

    extern fn g_object_ref(p_self: *gtk.AspectFrame) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.AspectFrame) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *AspectFrame, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.Assistant` is a widget used to represent a generally complex
/// operation splitted in several steps, guiding the user through its
/// pages and controlling the page flow to collect the necessary data.
///
/// The design of GtkAssistant is that it controls what buttons to show
/// and to make sensitive, based on what it knows about the page sequence
/// and the [type][GtkAssistantPageType] of each page,
/// in addition to state information like the page
/// [completion][gtk-assistant-set-page-complete]
/// and [committed][gtk-assistant-commit] status.
///
/// If you have a case that doesn’t quite fit in `GtkAssistants` way of
/// handling buttons, you can use the `GTK_ASSISTANT_PAGE_CUSTOM` page
/// type and handle buttons yourself.
///
/// # GtkAssistant as GtkBuildable
///
/// The GtkAssistant implementation of the `gtk.Buildable` interface
/// exposes the `action_area` as internal children with the name
/// “action_area”.
///
/// To add pages to an assistant in `gtk.Builder`, simply add it as a
/// child to the GtkAssistant object, and set its child properties
/// as necessary.
///
/// # CSS nodes
///
/// GtkAssistant has a single CSS node with the name assistant.
pub const Assistant = extern struct {
    pub const Parent = gtk.Window;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.AssistantClass;
    f_parent: gtk.Window,
    f_priv: ?*gtk.AssistantPrivate,

    pub const virtual_methods = struct {
        /// Signal emitted when the apply button is clicked.
        pub const apply = struct {
            pub fn call(p_class: anytype, p_assistant: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Assistant.Class, p_class).f_apply.?(gobject.ext.as(Assistant, p_assistant));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_assistant: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Assistant.Class, p_class).f_apply = @ptrCast(p_implementation);
            }
        };

        /// Signal emitted when the cancel button is clicked.
        pub const cancel = struct {
            pub fn call(p_class: anytype, p_assistant: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Assistant.Class, p_class).f_cancel.?(gobject.ext.as(Assistant, p_assistant));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_assistant: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Assistant.Class, p_class).f_cancel = @ptrCast(p_implementation);
            }
        };

        /// Signal emitted either when the close button or last page apply button is clicked.
        pub const close = struct {
            pub fn call(p_class: anytype, p_assistant: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Assistant.Class, p_class).f_close.?(gobject.ext.as(Assistant, p_assistant));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_assistant: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Assistant.Class, p_class).f_close = @ptrCast(p_implementation);
            }
        };

        /// Signal emitted when a new page is set as the assistant’s current page, before making the new page visible.
        pub const prepare = struct {
            pub fn call(p_class: anytype, p_assistant: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_page: *gtk.Widget) void {
                return gobject.ext.as(Assistant.Class, p_class).f_prepare.?(gobject.ext.as(Assistant, p_assistant), p_page);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_assistant: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_page: *gtk.Widget) callconv(.C) void) void {
                gobject.ext.as(Assistant.Class, p_class).f_prepare = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// `TRUE` if the assistant uses a `gtk.HeaderBar` for action buttons
        /// instead of the action-area.
        ///
        /// For technical reasons, this property is declared as an integer
        /// property, but you should only set it to `TRUE` or `FALSE`.
        pub const use_header_bar = struct {
            pub const name = "use-header-bar";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The ::apply signal is emitted when the apply button is clicked.
        ///
        /// The default behavior of the `gtk.Assistant` is to switch to the page
        /// after the current page, unless the current page is the last one.
        ///
        /// A handler for the ::apply signal should carry out the actions for
        /// which the wizard has collected data. If the action takes a long time
        /// to complete, you might consider putting a page of type
        /// `GTK_ASSISTANT_PAGE_PROGRESS` after the confirmation page and handle
        /// this operation within the `gtk.Assistant.signals.prepare` signal of the progress
        /// page.
        pub const apply = struct {
            pub const name = "apply";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Assistant, p_instance))),
                    gobject.signalLookup("apply", Assistant.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::cancel signal is emitted when then the cancel button is clicked.
        pub const cancel = struct {
            pub const name = "cancel";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Assistant, p_instance))),
                    gobject.signalLookup("cancel", Assistant.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::close signal is emitted either when the close button of
        /// a summary page is clicked, or when the apply button in the last
        /// page in the flow (of type `GTK_ASSISTANT_PAGE_CONFIRM`) is clicked.
        pub const close = struct {
            pub const name = "close";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Assistant, p_instance))),
                    gobject.signalLookup("close", Assistant.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const escape = struct {
            pub const name = "escape";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Assistant, p_instance))),
                    gobject.signalLookup("escape", Assistant.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::prepare signal is emitted when a new page is set as the
        /// assistant's current page, before making the new page visible.
        ///
        /// A handler for this signal can do any preparations which are
        /// necessary before showing `page`.
        pub const prepare = struct {
            pub const name = "prepare";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_page: *gtk.Widget, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Assistant, p_instance))),
                    gobject.signalLookup("prepare", Assistant.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.Assistant`.
    extern fn gtk_assistant_new() *gtk.Assistant;
    pub const new = gtk_assistant_new;

    /// Adds a widget to the action area of a `gtk.Assistant`.
    extern fn gtk_assistant_add_action_widget(p_assistant: *Assistant, p_child: *gtk.Widget) void;
    pub const addActionWidget = gtk_assistant_add_action_widget;

    /// Appends a page to the `assistant`.
    extern fn gtk_assistant_append_page(p_assistant: *Assistant, p_page: *gtk.Widget) c_int;
    pub const appendPage = gtk_assistant_append_page;

    /// Erases the visited page history so the back button is not
    /// shown on the current page, and removes the cancel button
    /// from subsequent pages.
    ///
    /// Use this when the information provided up to the current
    /// page is hereafter deemed permanent and cannot be modified
    /// or undone. For example, showing a progress page to track
    /// a long-running, unreversible operation after the user has
    /// clicked apply on a confirmation page.
    extern fn gtk_assistant_commit(p_assistant: *Assistant) void;
    pub const commit = gtk_assistant_commit;

    /// Returns the page number of the current page.
    extern fn gtk_assistant_get_current_page(p_assistant: *Assistant) c_int;
    pub const getCurrentPage = gtk_assistant_get_current_page;

    /// Returns the number of pages in the `assistant`
    extern fn gtk_assistant_get_n_pages(p_assistant: *Assistant) c_int;
    pub const getNPages = gtk_assistant_get_n_pages;

    /// Returns the child widget contained in page number `page_num`.
    extern fn gtk_assistant_get_nth_page(p_assistant: *Assistant, p_page_num: c_int) ?*gtk.Widget;
    pub const getNthPage = gtk_assistant_get_nth_page;

    /// Gets whether `page` is complete.
    extern fn gtk_assistant_get_page_complete(p_assistant: *Assistant, p_page: *gtk.Widget) c_int;
    pub const getPageComplete = gtk_assistant_get_page_complete;

    /// Gets whether page has padding.
    extern fn gtk_assistant_get_page_has_padding(p_assistant: *Assistant, p_page: *gtk.Widget) c_int;
    pub const getPageHasPadding = gtk_assistant_get_page_has_padding;

    /// Gets the header image for `page`.
    extern fn gtk_assistant_get_page_header_image(p_assistant: *Assistant, p_page: *gtk.Widget) *gdkpixbuf.Pixbuf;
    pub const getPageHeaderImage = gtk_assistant_get_page_header_image;

    /// Gets the side image for `page`.
    extern fn gtk_assistant_get_page_side_image(p_assistant: *Assistant, p_page: *gtk.Widget) *gdkpixbuf.Pixbuf;
    pub const getPageSideImage = gtk_assistant_get_page_side_image;

    /// Gets the title for `page`.
    extern fn gtk_assistant_get_page_title(p_assistant: *Assistant, p_page: *gtk.Widget) [*:0]const u8;
    pub const getPageTitle = gtk_assistant_get_page_title;

    /// Gets the page type of `page`.
    extern fn gtk_assistant_get_page_type(p_assistant: *Assistant, p_page: *gtk.Widget) gtk.AssistantPageType;
    pub const getPageType = gtk_assistant_get_page_type;

    /// Inserts a page in the `assistant` at a given position.
    extern fn gtk_assistant_insert_page(p_assistant: *Assistant, p_page: *gtk.Widget, p_position: c_int) c_int;
    pub const insertPage = gtk_assistant_insert_page;

    /// Navigate to the next page.
    ///
    /// It is a programming error to call this function when
    /// there is no next page.
    ///
    /// This function is for use when creating pages of the
    /// `GTK_ASSISTANT_PAGE_CUSTOM` type.
    extern fn gtk_assistant_next_page(p_assistant: *Assistant) void;
    pub const nextPage = gtk_assistant_next_page;

    /// Prepends a page to the `assistant`.
    extern fn gtk_assistant_prepend_page(p_assistant: *Assistant, p_page: *gtk.Widget) c_int;
    pub const prependPage = gtk_assistant_prepend_page;

    /// Navigate to the previous visited page.
    ///
    /// It is a programming error to call this function when
    /// no previous page is available.
    ///
    /// This function is for use when creating pages of the
    /// `GTK_ASSISTANT_PAGE_CUSTOM` type.
    extern fn gtk_assistant_previous_page(p_assistant: *Assistant) void;
    pub const previousPage = gtk_assistant_previous_page;

    /// Removes a widget from the action area of a `gtk.Assistant`.
    extern fn gtk_assistant_remove_action_widget(p_assistant: *Assistant, p_child: *gtk.Widget) void;
    pub const removeActionWidget = gtk_assistant_remove_action_widget;

    /// Removes the `page_num`’s page from `assistant`.
    extern fn gtk_assistant_remove_page(p_assistant: *Assistant, p_page_num: c_int) void;
    pub const removePage = gtk_assistant_remove_page;

    /// Switches the page to `page_num`.
    ///
    /// Note that this will only be necessary in custom buttons,
    /// as the `assistant` flow can be set with
    /// `gtk.Assistant.setForwardPageFunc`.
    extern fn gtk_assistant_set_current_page(p_assistant: *Assistant, p_page_num: c_int) void;
    pub const setCurrentPage = gtk_assistant_set_current_page;

    /// Sets the page forwarding function to be `page_func`.
    ///
    /// This function will be used to determine what will be
    /// the next page when the user presses the forward button.
    /// Setting `page_func` to `NULL` will make the assistant to
    /// use the default forward function, which just goes to the
    /// next visible page.
    extern fn gtk_assistant_set_forward_page_func(p_assistant: *Assistant, p_page_func: ?gtk.AssistantPageFunc, p_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) void;
    pub const setForwardPageFunc = gtk_assistant_set_forward_page_func;

    /// Sets whether `page` contents are complete.
    ///
    /// This will make `assistant` update the buttons state
    /// to be able to continue the task.
    extern fn gtk_assistant_set_page_complete(p_assistant: *Assistant, p_page: *gtk.Widget, p_complete: c_int) void;
    pub const setPageComplete = gtk_assistant_set_page_complete;

    /// Sets whether the assistant is adding padding around
    /// the page.
    extern fn gtk_assistant_set_page_has_padding(p_assistant: *Assistant, p_page: *gtk.Widget, p_has_padding: c_int) void;
    pub const setPageHasPadding = gtk_assistant_set_page_has_padding;

    /// Sets a header image for `page`.
    extern fn gtk_assistant_set_page_header_image(p_assistant: *Assistant, p_page: *gtk.Widget, p_pixbuf: ?*gdkpixbuf.Pixbuf) void;
    pub const setPageHeaderImage = gtk_assistant_set_page_header_image;

    /// Sets a side image for `page`.
    ///
    /// This image used to be displayed in the side area of the assistant
    /// when `page` is the current page.
    extern fn gtk_assistant_set_page_side_image(p_assistant: *Assistant, p_page: *gtk.Widget, p_pixbuf: ?*gdkpixbuf.Pixbuf) void;
    pub const setPageSideImage = gtk_assistant_set_page_side_image;

    /// Sets a title for `page`.
    ///
    /// The title is displayed in the header area of the assistant
    /// when `page` is the current page.
    extern fn gtk_assistant_set_page_title(p_assistant: *Assistant, p_page: *gtk.Widget, p_title: [*:0]const u8) void;
    pub const setPageTitle = gtk_assistant_set_page_title;

    /// Sets the page type for `page`.
    ///
    /// The page type determines the page behavior in the `assistant`.
    extern fn gtk_assistant_set_page_type(p_assistant: *Assistant, p_page: *gtk.Widget, p_type: gtk.AssistantPageType) void;
    pub const setPageType = gtk_assistant_set_page_type;

    /// Forces `assistant` to recompute the buttons state.
    ///
    /// GTK+ automatically takes care of this in most situations,
    /// e.g. when the user goes to a different page, or when the
    /// visibility or completeness of a page changes.
    ///
    /// One situation where it can be necessary to call this
    /// function is when changing a value on the current page
    /// affects the future page flow of the assistant.
    extern fn gtk_assistant_update_buttons_state(p_assistant: *Assistant) void;
    pub const updateButtonsState = gtk_assistant_update_buttons_state;

    extern fn gtk_assistant_get_type() usize;
    pub const getGObjectType = gtk_assistant_get_type;

    extern fn g_object_ref(p_self: *gtk.Assistant) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Assistant) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Assistant, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.Bin` widget is a container with just one child.
/// It is not very useful itself, but it is useful for deriving subclasses,
/// since it provides common code needed for handling a single child widget.
///
/// Many GTK+ widgets are subclasses of `gtk.Bin`, including `gtk.Window`,
/// `gtk.Button`, `gtk.Frame`, `gtk.HandleBox` or `gtk.ScrolledWindow`.
pub const Bin = extern struct {
    pub const Parent = gtk.Container;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.BinClass;
    f_container: gtk.Container,
    f_priv: ?*gtk.BinPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Gets the child of the `gtk.Bin`, or `NULL` if the bin contains
    /// no child widget. The returned widget does not have a reference
    /// added, so you do not need to unref it.
    extern fn gtk_bin_get_child(p_bin: *Bin) ?*gtk.Widget;
    pub const getChild = gtk_bin_get_child;

    extern fn gtk_bin_get_type() usize;
    pub const getGObjectType = gtk_bin_get_type;

    extern fn g_object_ref(p_self: *gtk.Bin) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Bin) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Bin, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BooleanCellAccessible = extern struct {
    pub const Parent = gtk.RendererCellAccessible;
    pub const Implements = [_]type{ atk.Action, atk.Component, atk.TableCell };
    pub const Class = gtk.BooleanCellAccessibleClass;
    f_parent: gtk.RendererCellAccessible,
    f_priv: ?*gtk.BooleanCellAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_boolean_cell_accessible_get_type() usize;
    pub const getGObjectType = gtk_boolean_cell_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.BooleanCellAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.BooleanCellAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *BooleanCellAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The GtkBox widget arranges child widgets into a single row or column,
/// depending upon the value of its `gtk.Orientable.properties.orientation` property. Within
/// the other dimension, all children are allocated the same size. Of course,
/// the `gtk.Widget.properties.halign` and `gtk.Widget.properties.valign` properties can be used on
/// the children to influence their allocation.
///
/// GtkBox uses a notion of packing. Packing refers
/// to adding widgets with reference to a particular position in a
/// `gtk.Container`. For a GtkBox, there are two reference positions: the
/// start and the end of the box.
/// For a vertical `gtk.Box`, the start is defined as the top of the box and
/// the end is defined as the bottom. For a horizontal `gtk.Box` the start
/// is defined as the left side and the end is defined as the right side.
///
/// Use repeated calls to `gtk.Box.packStart` to pack widgets into a
/// GtkBox from start to end. Use `gtk.Box.packEnd` to add widgets from
/// end to start. You may intersperse these calls and add widgets from
/// both ends of the same GtkBox.
///
/// Because GtkBox is a `gtk.Container`, you may also use `gtk.Container.add`
/// to insert widgets into the box, and they will be packed with the default
/// values for expand and fill child properties. Use `gtk.Container.remove`
/// to remove widgets from the GtkBox.
///
/// Use `gtk.Box.setHomogeneous` to specify whether or not all children
/// of the GtkBox are forced to get the same amount of space.
///
/// Use `gtk.Box.setSpacing` to determine how much space will be
/// minimally placed between all children in the GtkBox. Note that
/// spacing is added between the children, while
/// padding added by `gtk.Box.packStart` or `gtk.Box.packEnd` is added
/// on either side of the widget it belongs to.
///
/// Use `gtk.Box.reorderChild` to move a GtkBox child to a different
/// place in the box.
///
/// Use `gtk.Box.setChildPacking` to reset the expand,
/// fill and padding child properties.
/// Use `gtk.Box.queryChildPacking` to query these fields.
///
/// # CSS nodes
///
/// GtkBox uses a single CSS node with name box.
///
/// In horizontal orientation, the nodes of the children are always arranged
/// from left to right. So :first-child will always select the leftmost child,
/// regardless of text direction.
pub const Box = extern struct {
    pub const Parent = gtk.Container;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.BoxClass;
    f_container: gtk.Container,
    f_priv: ?*gtk.BoxPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const baseline_position = struct {
            pub const name = "baseline-position";

            pub const Type = gtk.BaselinePosition;
        };

        pub const homogeneous = struct {
            pub const name = "homogeneous";

            pub const Type = c_int;
        };

        pub const spacing = struct {
            pub const name = "spacing";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.Box`.
    extern fn gtk_box_new(p_orientation: gtk.Orientation, p_spacing: c_int) *gtk.Box;
    pub const new = gtk_box_new;

    /// Gets the value set by `gtk.Box.setBaselinePosition`.
    extern fn gtk_box_get_baseline_position(p_box: *Box) gtk.BaselinePosition;
    pub const getBaselinePosition = gtk_box_get_baseline_position;

    /// Retrieves the center widget of the box.
    extern fn gtk_box_get_center_widget(p_box: *Box) ?*gtk.Widget;
    pub const getCenterWidget = gtk_box_get_center_widget;

    /// Returns whether the box is homogeneous (all children are the
    /// same size). See `gtk.Box.setHomogeneous`.
    extern fn gtk_box_get_homogeneous(p_box: *Box) c_int;
    pub const getHomogeneous = gtk_box_get_homogeneous;

    /// Gets the value set by `gtk.Box.setSpacing`.
    extern fn gtk_box_get_spacing(p_box: *Box) c_int;
    pub const getSpacing = gtk_box_get_spacing;

    /// Adds `child` to `box`, packed with reference to the end of `box`.
    /// The `child` is packed after (away from end of) any other child
    /// packed with reference to the end of `box`.
    extern fn gtk_box_pack_end(p_box: *Box, p_child: *gtk.Widget, p_expand: c_int, p_fill: c_int, p_padding: c_uint) void;
    pub const packEnd = gtk_box_pack_end;

    /// Adds `child` to `box`, packed with reference to the start of `box`.
    /// The `child` is packed after any other child packed with reference
    /// to the start of `box`.
    extern fn gtk_box_pack_start(p_box: *Box, p_child: *gtk.Widget, p_expand: c_int, p_fill: c_int, p_padding: c_uint) void;
    pub const packStart = gtk_box_pack_start;

    /// Obtains information about how `child` is packed into `box`.
    extern fn gtk_box_query_child_packing(p_box: *Box, p_child: *gtk.Widget, p_expand: *c_int, p_fill: *c_int, p_padding: *c_uint, p_pack_type: *gtk.PackType) void;
    pub const queryChildPacking = gtk_box_query_child_packing;

    /// Moves `child` to a new `position` in the list of `box` children.
    /// The list contains widgets packed `GTK_PACK_START`
    /// as well as widgets packed `GTK_PACK_END`, in the order that these
    /// widgets were added to `box`.
    ///
    /// A widget’s position in the `box` children list determines where
    /// the widget is packed into `box`.  A child widget at some position
    /// in the list will be packed just after all other widgets of the
    /// same packing type that appear earlier in the list.
    extern fn gtk_box_reorder_child(p_box: *Box, p_child: *gtk.Widget, p_position: c_int) void;
    pub const reorderChild = gtk_box_reorder_child;

    /// Sets the baseline position of a box. This affects
    /// only horizontal boxes with at least one baseline aligned
    /// child. If there is more vertical space available than requested,
    /// and the baseline is not allocated by the parent then
    /// `position` is used to allocate the baseline wrt the
    /// extra space available.
    extern fn gtk_box_set_baseline_position(p_box: *Box, p_position: gtk.BaselinePosition) void;
    pub const setBaselinePosition = gtk_box_set_baseline_position;

    /// Sets a center widget; that is a child widget that will be
    /// centered with respect to the full width of the box, even
    /// if the children at either side take up different amounts
    /// of space.
    extern fn gtk_box_set_center_widget(p_box: *Box, p_widget: ?*gtk.Widget) void;
    pub const setCenterWidget = gtk_box_set_center_widget;

    /// Sets the way `child` is packed into `box`.
    extern fn gtk_box_set_child_packing(p_box: *Box, p_child: *gtk.Widget, p_expand: c_int, p_fill: c_int, p_padding: c_uint, p_pack_type: gtk.PackType) void;
    pub const setChildPacking = gtk_box_set_child_packing;

    /// Sets the `gtk.Box.properties.homogeneous` property of `box`, controlling
    /// whether or not all children of `box` are given equal space
    /// in the box.
    extern fn gtk_box_set_homogeneous(p_box: *Box, p_homogeneous: c_int) void;
    pub const setHomogeneous = gtk_box_set_homogeneous;

    /// Sets the `gtk.Box.properties.spacing` property of `box`, which is the
    /// number of pixels to place between children of `box`.
    extern fn gtk_box_set_spacing(p_box: *Box, p_spacing: c_int) void;
    pub const setSpacing = gtk_box_set_spacing;

    extern fn gtk_box_get_type() usize;
    pub const getGObjectType = gtk_box_get_type;

    extern fn g_object_ref(p_self: *gtk.Box) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Box) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Box, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A GtkBuilder is an auxiliary object that reads textual descriptions
/// of a user interface and instantiates the described objects. To create
/// a GtkBuilder from a user interface description, call
/// `gtk.Builder.newFromFile`, `gtk.Builder.newFromResource` or
/// `gtk.Builder.newFromString`.
///
/// In the (unusual) case that you want to add user interface
/// descriptions from multiple sources to the same GtkBuilder you can
/// call `gtk.Builder.new` to get an empty builder and populate it by
/// (multiple) calls to `gtk.Builder.addFromFile`,
/// `gtk.Builder.addFromResource` or `gtk.Builder.addFromString`.
///
/// A GtkBuilder holds a reference to all objects that it has constructed
/// and drops these references when it is finalized. This finalization can
/// cause the destruction of non-widget objects or widgets which are not
/// contained in a toplevel window. For toplevel windows constructed by a
/// builder, it is the responsibility of the user to call `gtk.Widget.destroy`
/// to get rid of them and all the widgets they contain.
///
/// The functions `gtk.Builder.getObject` and `gtk.Builder.getObjects`
/// can be used to access the widgets in the interface by the names assigned
/// to them inside the UI description. Toplevel windows returned by these
/// functions will stay around until the user explicitly destroys them
/// with `gtk.Widget.destroy`. Other widgets will either be part of a
/// larger hierarchy constructed by the builder (in which case you should
/// not have to worry about their lifecycle), or without a parent, in which
/// case they have to be added to some container to make use of them.
/// Non-widget objects need to be reffed with `gobject.Object.ref` to keep them
/// beyond the lifespan of the builder.
///
/// The function `gtk.Builder.connectSignals` and variants thereof can be
/// used to connect handlers to the named signals in the description.
///
/// # GtkBuilder UI Definitions
///
/// GtkBuilder parses textual descriptions of user interfaces which are
/// specified in an XML format which can be roughly described by the
/// RELAX NG schema below. We refer to these descriptions as “GtkBuilder
/// UI definitions” or just “UI definitions” if the context is clear.
/// Do not confuse GtkBuilder UI Definitions with
/// [GtkUIManager UI Definitions][XML-UI], which are more limited in scope.
/// It is common to use `.ui` as the filename extension for files containing
/// GtkBuilder UI definitions.
///
/// [RELAX NG Compact Syntax](https://gitlab.gnome.org/GNOME/gtk/-/blob/gtk-3-24/gtk/gtkbuilder.rnc)
///
/// The toplevel element is `<interface>`. It optionally takes a “domain”
/// attribute, which will make the builder look for translated strings
/// using `dgettext` in the domain specified. This can also be done by
/// calling `gtk.Builder.setTranslationDomain` on the builder.
/// Objects are described by `<object>` elements, which can contain
/// `<property>` elements to set properties, `<signal>` elements which
/// connect signals to handlers, and `<child>` elements, which describe
/// child objects (most often widgets inside a container, but also e.g.
/// actions in an action group, or columns in a tree model). A `<child>`
/// element contains an `<object>` element which describes the child object.
/// The target toolkit version(s) are described by `<requires>` elements,
/// the “lib” attribute specifies the widget library in question (currently
/// the only supported value is “gtk+”) and the “version” attribute specifies
/// the target version in the form `<major>.<minor>`. The builder will error
/// out if the version requirements are not met.
///
/// Typically, the specific kind of object represented by an `<object>`
/// element is specified by the “class” attribute. If the type has not
/// been loaded yet, GTK+ tries to find the ``get_type`` function from the
/// class name by applying heuristics. This works in most cases, but if
/// necessary, it is possible to specify the name of the `get_type` function
/// explictly with the "type-func" attribute. As a special case, GtkBuilder
/// allows to use an object that has been constructed by a `gtk.UIManager` in
/// another part of the UI definition by specifying the id of the `gtk.UIManager`
/// in the “constructor” attribute and the name of the object in the “id”
/// attribute.
///
/// Objects may be given a name with the “id” attribute, which allows the
/// application to retrieve them from the builder with `gtk.Builder.getObject`.
/// An id is also necessary to use the object as property value in other
/// parts of the UI definition. GTK+ reserves ids starting and ending
/// with `___` (3 underscores) for its own purposes.
///
/// Setting properties of objects is pretty straightforward with the
/// `<property>` element: the “name” attribute specifies the name of the
/// property, and the content of the element specifies the value.
/// If the “translatable” attribute is set to a true value, GTK+ uses
/// `gettext` (or `dgettext` if the builder has a translation domain set)
/// to find a translation for the value. This happens before the value
/// is parsed, so it can be used for properties of any type, but it is
/// probably most useful for string properties. It is also possible to
/// specify a context to disambiguate short strings, and comments which
/// may help the translators.
///
/// GtkBuilder can parse textual representations for the most common
/// property types: characters, strings, integers, floating-point numbers,
/// booleans (strings like “TRUE”, “t”, “yes”, “y”, “1” are interpreted
/// as `TRUE`, strings like “FALSE”, “f”, “no”, “n”, “0” are interpreted
/// as `FALSE`), enumerations (can be specified by their name, nick or
/// integer value), flags (can be specified by their name, nick, integer
/// value, optionally combined with “|”, e.g. “GTK_VISIBLE|GTK_REALIZED”)
/// and colors (in a format understood by `gdk.RGBA.parse`).
///
/// GVariants can be specified in the format understood by `glib.variantParse`,
/// and pixbufs can be specified as a filename of an image file to load.
///
/// Objects can be referred to by their name and by default refer to
/// objects declared in the local xml fragment and objects exposed via
/// `gtk.Builder.exposeObject`. In general, GtkBuilder allows forward
/// references to objects — declared in the local xml; an object doesn’t
/// have to be constructed before it can be referred to. The exception
/// to this rule is that an object has to be constructed before it can
/// be used as the value of a construct-only property.
///
/// It is also possible to bind a property value to another object's
/// property value using the attributes
/// "bind-source" to specify the source object of the binding,
/// "bind-property" to specify the source property and optionally
/// "bind-flags" to specify the binding flags.
/// Internally builder implements this using GBinding objects.
/// For more information see `gobject.Object.bindProperty`
///
/// Signal handlers are set up with the `<signal>` element. The “name”
/// attribute specifies the name of the signal, and the “handler” attribute
/// specifies the function to connect to the signal. By default, GTK+ tries
/// to find the handler using `gmodule.Module.symbol`, but this can be changed by
/// passing a custom `gtk.BuilderConnectFunc` to
/// `gtk.Builder.connectSignalsFull`. The remaining attributes, “after”,
/// “swapped” and “object”, have the same meaning as the corresponding
/// parameters of the `gobject.signalConnectObject` or
/// `gobject.signalConnectData` functions. A “last_modification_time”
/// attribute is also allowed, but it does not have a meaning to the
/// builder.
///
/// Sometimes it is necessary to refer to widgets which have implicitly
/// been constructed by GTK+ as part of a composite widget, to set
/// properties on them or to add further children (e.g. the `vbox` of
/// a `gtk.Dialog`). This can be achieved by setting the “internal-child”
/// property of the `<child>` element to a true value. Note that GtkBuilder
/// still requires an `<object>` element for the internal child, even if it
/// has already been constructed.
///
/// A number of widgets have different places where a child can be added
/// (e.g. tabs vs. page content in notebooks). This can be reflected in
/// a UI definition by specifying the “type” attribute on a `<child>`
/// The possible values for the “type” attribute are described in the
/// sections describing the widget-specific portions of UI definitions.
///
/// # A GtkBuilder UI Definition
///
/// ```
/// <interface>
///   <object class="GtkDialog" id="dialog1">
///     <child internal-child="vbox">
///       <object class="GtkBox" id="vbox1">
///         <property name="border-width">10</property>
///         <child internal-child="action_area">
///           <object class="GtkButtonBox" id="hbuttonbox1">
///             <property name="border-width">20</property>
///             <child>
///               <object class="GtkButton" id="ok_button">
///                 <property name="label">gtk-ok</property>
///                 <property name="use-stock">TRUE</property>
///                 <signal name="clicked" handler="ok_button_clicked"/>
///               </object>
///             </child>
///           </object>
///         </child>
///       </object>
///     </child>
///   </object>
/// </interface>
/// ```
///
/// Beyond this general structure, several object classes define their
/// own XML DTD fragments for filling in the ANY placeholders in the DTD
/// above. Note that a custom element in a `<child>` element gets parsed by
/// the custom tag handler of the parent object, while a custom element in
/// an `<object>` element gets parsed by the custom tag handler of the object.
///
/// These XML fragments are explained in the documentation of the
/// respective objects.
///
/// Additionally, since 3.10 a special `<template>` tag has been added
/// to the format allowing one to define a widget class’s components.
/// See the [GtkWidget documentation][composite-templates] for details.
pub const Builder = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtk.BuilderClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gtk.BuilderPrivate,

    pub const virtual_methods = struct {
        /// Looks up a type by name, using the virtual function that
        /// `gtk.Builder` has for that purpose. This is mainly used when
        /// implementing the `gtk.Buildable` interface on a type.
        pub const get_type_from_name = struct {
            pub fn call(p_class: anytype, p_builder: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_type_name: [*:0]const u8) usize {
                return gobject.ext.as(Builder.Class, p_class).f_get_type_from_name.?(gobject.ext.as(Builder, p_builder), p_type_name);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_builder: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_type_name: [*:0]const u8) callconv(.C) usize) void {
                gobject.ext.as(Builder.Class, p_class).f_get_type_from_name = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The translation domain used when translating property values that
        /// have been marked as translatable in interface descriptions.
        /// If the translation domain is `NULL`, `gtk.Builder` uses `gettext`,
        /// otherwise `glib.dgettext`.
        pub const translation_domain = struct {
            pub const name = "translation-domain";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new empty builder object.
    ///
    /// This function is only useful if you intend to make multiple calls
    /// to `gtk.Builder.addFromFile`, `gtk.Builder.addFromResource`
    /// or `gtk.Builder.addFromString` in order to merge multiple UI
    /// descriptions into a single builder.
    ///
    /// Most users will probably want to use `gtk.Builder.newFromFile`,
    /// `gtk.Builder.newFromResource` or `gtk.Builder.newFromString`.
    extern fn gtk_builder_new() *gtk.Builder;
    pub const new = gtk_builder_new;

    /// Builds the [GtkBuilder UI definition][BUILDER-UI]
    /// in the file `filename`.
    ///
    /// If there is an error opening the file or parsing the description then
    /// the program will be aborted.  You should only ever attempt to parse
    /// user interface descriptions that are shipped as part of your program.
    extern fn gtk_builder_new_from_file(p_filename: [*:0]const u8) *gtk.Builder;
    pub const newFromFile = gtk_builder_new_from_file;

    /// Builds the [GtkBuilder UI definition][BUILDER-UI]
    /// at `resource_path`.
    ///
    /// If there is an error locating the resource or parsing the
    /// description, then the program will be aborted.
    extern fn gtk_builder_new_from_resource(p_resource_path: [*:0]const u8) *gtk.Builder;
    pub const newFromResource = gtk_builder_new_from_resource;

    /// Builds the user interface described by `string` (in the
    /// [GtkBuilder UI definition][BUILDER-UI] format).
    ///
    /// If `string` is `NULL`-terminated, then `length` should be -1.
    /// If `length` is not -1, then it is the length of `string`.
    ///
    /// If there is an error parsing `string` then the program will be
    /// aborted. You should not attempt to parse user interface description
    /// from untrusted sources.
    extern fn gtk_builder_new_from_string(p_string: [*:0]const u8, p_length: isize) *gtk.Builder;
    pub const newFromString = gtk_builder_new_from_string;

    /// Adds the `callback_symbol` to the scope of `builder` under the given `callback_name`.
    ///
    /// Using this function overrides the behavior of `gtk.Builder.connectSignals`
    /// for any callback symbols that are added. Using this method allows for better
    /// encapsulation as it does not require that callback symbols be declared in
    /// the global namespace.
    extern fn gtk_builder_add_callback_symbol(p_builder: *Builder, p_callback_name: [*:0]const u8, p_callback_symbol: gobject.Callback) void;
    pub const addCallbackSymbol = gtk_builder_add_callback_symbol;

    /// A convenience function to add many callbacks instead of calling
    /// `gtk.Builder.addCallbackSymbol` for each symbol.
    extern fn gtk_builder_add_callback_symbols(p_builder: *Builder, p_first_callback_name: [*:0]const u8, p_first_callback_symbol: gobject.Callback, ...) void;
    pub const addCallbackSymbols = gtk_builder_add_callback_symbols;

    /// Parses a file containing a [GtkBuilder UI definition][BUILDER-UI]
    /// and merges it with the current contents of `builder`.
    ///
    /// Most users will probably want to use `gtk.Builder.newFromFile`.
    ///
    /// If an error occurs, 0 will be returned and `error` will be assigned a
    /// `glib.Error` from the `GTK_BUILDER_ERROR`, `G_MARKUP_ERROR` or `G_FILE_ERROR`
    /// domain.
    ///
    /// It’s not really reasonable to attempt to handle failures of this
    /// call. You should not use this function with untrusted files (ie:
    /// files that are not part of your application). Broken `gtk.Builder`
    /// files can easily crash your program, and it’s possible that memory
    /// was leaked leading up to the reported failure. The only reasonable
    /// thing to do when an error is detected is to call `g_error`.
    extern fn gtk_builder_add_from_file(p_builder: *Builder, p_filename: [*:0]const u8, p_error: ?*?*glib.Error) c_uint;
    pub const addFromFile = gtk_builder_add_from_file;

    /// Parses a resource file containing a [GtkBuilder UI definition][BUILDER-UI]
    /// and merges it with the current contents of `builder`.
    ///
    /// Most users will probably want to use `gtk.Builder.newFromResource`.
    ///
    /// If an error occurs, 0 will be returned and `error` will be assigned a
    /// `glib.Error` from the `GTK_BUILDER_ERROR`, `G_MARKUP_ERROR` or `G_RESOURCE_ERROR`
    /// domain.
    ///
    /// It’s not really reasonable to attempt to handle failures of this
    /// call.  The only reasonable thing to do when an error is detected is
    /// to call `g_error`.
    extern fn gtk_builder_add_from_resource(p_builder: *Builder, p_resource_path: [*:0]const u8, p_error: ?*?*glib.Error) c_uint;
    pub const addFromResource = gtk_builder_add_from_resource;

    /// Parses a string containing a [GtkBuilder UI definition][BUILDER-UI]
    /// and merges it with the current contents of `builder`.
    ///
    /// Most users will probably want to use `gtk.Builder.newFromString`.
    ///
    /// Upon errors 0 will be returned and `error` will be assigned a
    /// `glib.Error` from the `GTK_BUILDER_ERROR`, `G_MARKUP_ERROR` or
    /// `G_VARIANT_PARSE_ERROR` domain.
    ///
    /// It’s not really reasonable to attempt to handle failures of this
    /// call.  The only reasonable thing to do when an error is detected is
    /// to call `g_error`.
    extern fn gtk_builder_add_from_string(p_builder: *Builder, p_buffer: [*:0]const u8, p_length: usize, p_error: ?*?*glib.Error) c_uint;
    pub const addFromString = gtk_builder_add_from_string;

    /// Parses a file containing a [GtkBuilder UI definition][BUILDER-UI]
    /// building only the requested objects and merges
    /// them with the current contents of `builder`.
    ///
    /// Upon errors 0 will be returned and `error` will be assigned a
    /// `glib.Error` from the `GTK_BUILDER_ERROR`, `G_MARKUP_ERROR` or `G_FILE_ERROR`
    /// domain.
    ///
    /// If you are adding an object that depends on an object that is not
    /// its child (for instance a `gtk.TreeView` that depends on its
    /// `gtk.TreeModel`), you have to explicitly list all of them in `object_ids`.
    extern fn gtk_builder_add_objects_from_file(p_builder: *Builder, p_filename: [*:0]const u8, p_object_ids: [*][*:0]u8, p_error: ?*?*glib.Error) c_uint;
    pub const addObjectsFromFile = gtk_builder_add_objects_from_file;

    /// Parses a resource file containing a [GtkBuilder UI definition][BUILDER-UI]
    /// building only the requested objects and merges
    /// them with the current contents of `builder`.
    ///
    /// Upon errors 0 will be returned and `error` will be assigned a
    /// `glib.Error` from the `GTK_BUILDER_ERROR`, `G_MARKUP_ERROR` or `G_RESOURCE_ERROR`
    /// domain.
    ///
    /// If you are adding an object that depends on an object that is not
    /// its child (for instance a `gtk.TreeView` that depends on its
    /// `gtk.TreeModel`), you have to explicitly list all of them in `object_ids`.
    extern fn gtk_builder_add_objects_from_resource(p_builder: *Builder, p_resource_path: [*:0]const u8, p_object_ids: [*][*:0]u8, p_error: ?*?*glib.Error) c_uint;
    pub const addObjectsFromResource = gtk_builder_add_objects_from_resource;

    /// Parses a string containing a [GtkBuilder UI definition][BUILDER-UI]
    /// building only the requested objects and merges
    /// them with the current contents of `builder`.
    ///
    /// Upon errors 0 will be returned and `error` will be assigned a
    /// `glib.Error` from the `GTK_BUILDER_ERROR` or `G_MARKUP_ERROR` domain.
    ///
    /// If you are adding an object that depends on an object that is not
    /// its child (for instance a `gtk.TreeView` that depends on its
    /// `gtk.TreeModel`), you have to explicitly list all of them in `object_ids`.
    extern fn gtk_builder_add_objects_from_string(p_builder: *Builder, p_buffer: [*:0]const u8, p_length: usize, p_object_ids: [*][*:0]u8, p_error: ?*?*glib.Error) c_uint;
    pub const addObjectsFromString = gtk_builder_add_objects_from_string;

    /// This method is a simpler variation of `gtk.Builder.connectSignalsFull`.
    /// It uses symbols explicitly added to `builder` with prior calls to
    /// `gtk.Builder.addCallbackSymbol`. In the case that symbols are not
    /// explicitly added; it uses `gmodule.Module`’s introspective features (by opening the module `NULL`)
    /// to look at the application’s symbol table. From here it tries to match
    /// the signal handler names given in the interface description with
    /// symbols in the application and connects the signals. Note that this
    /// function can only be called once, subsequent calls will do nothing.
    ///
    /// Note that unless `gtk.Builder.addCallbackSymbol` is called for
    /// all signal callbacks which are referenced by the loaded XML, this
    /// function will require that `gmodule.Module` be supported on the platform.
    ///
    /// If you rely on `gmodule.Module` support to lookup callbacks in the symbol table,
    /// the following details should be noted:
    ///
    /// When compiling applications for Windows, you must declare signal callbacks
    /// with `G_MODULE_EXPORT`, or they will not be put in the symbol table.
    /// On Linux and Unices, this is not necessary; applications should instead
    /// be compiled with the -Wl,--export-dynamic CFLAGS, and linked against
    /// gmodule-export-2.0.
    extern fn gtk_builder_connect_signals(p_builder: *Builder, p_user_data: ?*anyopaque) void;
    pub const connectSignals = gtk_builder_connect_signals;

    /// This function can be thought of the interpreted language binding
    /// version of `gtk.Builder.connectSignals`, except that it does not
    /// require GModule to function correctly.
    extern fn gtk_builder_connect_signals_full(p_builder: *Builder, p_func: gtk.BuilderConnectFunc, p_user_data: ?*anyopaque) void;
    pub const connectSignalsFull = gtk_builder_connect_signals_full;

    /// Add `object` to the `builder` object pool so it can be referenced just like any
    /// other object built by builder.
    extern fn gtk_builder_expose_object(p_builder: *Builder, p_name: [*:0]const u8, p_object: *gobject.Object) void;
    pub const exposeObject = gtk_builder_expose_object;

    /// Main private entry point for building composite container
    /// components from template XML.
    ///
    /// This is exported purely to let gtk-builder-tool validate
    /// templates, applications have no need to call this function.
    extern fn gtk_builder_extend_with_template(p_builder: *Builder, p_widget: *gtk.Widget, p_template_type: usize, p_buffer: [*:0]const u8, p_length: usize, p_error: ?*?*glib.Error) c_uint;
    pub const extendWithTemplate = gtk_builder_extend_with_template;

    /// Gets the `gtk.Application` associated with the builder.
    ///
    /// The `gtk.Application` is used for creating action proxies as requested
    /// from XML that the builder is loading.
    ///
    /// By default, the builder uses the default application: the one from
    /// `gio.Application.getDefault`. If you want to use another application
    /// for constructing proxies, use `gtk.Builder.setApplication`.
    extern fn gtk_builder_get_application(p_builder: *Builder) ?*gtk.Application;
    pub const getApplication = gtk_builder_get_application;

    /// Gets the object named `name`. Note that this function does not
    /// increment the reference count of the returned object.
    extern fn gtk_builder_get_object(p_builder: *Builder, p_name: [*:0]const u8) ?*gobject.Object;
    pub const getObject = gtk_builder_get_object;

    /// Gets all objects that have been constructed by `builder`. Note that
    /// this function does not increment the reference counts of the returned
    /// objects.
    extern fn gtk_builder_get_objects(p_builder: *Builder) *glib.SList;
    pub const getObjects = gtk_builder_get_objects;

    /// Gets the translation domain of `builder`.
    extern fn gtk_builder_get_translation_domain(p_builder: *Builder) [*:0]const u8;
    pub const getTranslationDomain = gtk_builder_get_translation_domain;

    /// Looks up a type by name, using the virtual function that
    /// `gtk.Builder` has for that purpose. This is mainly used when
    /// implementing the `gtk.Buildable` interface on a type.
    extern fn gtk_builder_get_type_from_name(p_builder: *Builder, p_type_name: [*:0]const u8) usize;
    pub const getTypeFromName = gtk_builder_get_type_from_name;

    /// Fetches a symbol previously added to `builder`
    /// with `gtk.Builder.addCallbackSymbols`
    ///
    /// This function is intended for possible use in language bindings
    /// or for any case that one might be cusomizing signal connections
    /// using `gtk.Builder.connectSignalsFull`
    extern fn gtk_builder_lookup_callback_symbol(p_builder: *Builder, p_callback_name: [*:0]const u8) ?gobject.Callback;
    pub const lookupCallbackSymbol = gtk_builder_lookup_callback_symbol;

    /// Sets the application associated with `builder`.
    ///
    /// You only need this function if there is more than one `gio.Application`
    /// in your process. `application` cannot be `NULL`.
    extern fn gtk_builder_set_application(p_builder: *Builder, p_application: *gtk.Application) void;
    pub const setApplication = gtk_builder_set_application;

    /// Sets the translation domain of `builder`.
    /// See `gtk.Builder.properties.translation`-domain.
    extern fn gtk_builder_set_translation_domain(p_builder: *Builder, p_domain: ?[*:0]const u8) void;
    pub const setTranslationDomain = gtk_builder_set_translation_domain;

    /// This function demarshals a value from a string. This function
    /// calls `gobject.Value.init` on the `value` argument, so it need not be
    /// initialised beforehand.
    ///
    /// This function can handle char, uchar, boolean, int, uint, long,
    /// ulong, enum, flags, float, double, string, `gdk.Color`, `gdk.RGBA` and
    /// `gtk.Adjustment` type values. Support for `gtk.Widget` type values is
    /// still to come.
    ///
    /// Upon errors `FALSE` will be returned and `error` will be assigned a
    /// `glib.Error` from the `GTK_BUILDER_ERROR` domain.
    extern fn gtk_builder_value_from_string(p_builder: *Builder, p_pspec: *gobject.ParamSpec, p_string: [*:0]const u8, p_value: *gobject.Value, p_error: ?*?*glib.Error) c_int;
    pub const valueFromString = gtk_builder_value_from_string;

    /// Like `gtk.Builder.valueFromString`, this function demarshals
    /// a value from a string, but takes a `gobject.Type` instead of `gobject.ParamSpec`.
    /// This function calls `gobject.Value.init` on the `value` argument, so it
    /// need not be initialised beforehand.
    ///
    /// Upon errors `FALSE` will be returned and `error` will be assigned a
    /// `glib.Error` from the `GTK_BUILDER_ERROR` domain.
    extern fn gtk_builder_value_from_string_type(p_builder: *Builder, p_type: usize, p_string: [*:0]const u8, p_value: *gobject.Value, p_error: ?*?*glib.Error) c_int;
    pub const valueFromStringType = gtk_builder_value_from_string_type;

    extern fn gtk_builder_get_type() usize;
    pub const getGObjectType = gtk_builder_get_type;

    extern fn g_object_ref(p_self: *gtk.Builder) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Builder) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Builder, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.Button` widget is generally used to trigger a callback function that is
/// called when the button is pressed.  The various signals and how to use them
/// are outlined below.
///
/// The `gtk.Button` widget can hold any valid child widget.  That is, it can hold
/// almost any other standard `gtk.Widget`.  The most commonly used child is the
/// `gtk.Label`.
///
/// # CSS nodes
///
/// GtkButton has a single CSS node with name button. The node will get the
/// style classes .image-button or .text-button, if the content is just an
/// image or label, respectively. It may also receive the .flat style class.
///
/// Other style classes that are commonly used with GtkButton include
/// .suggested-action and .destructive-action. In special cases, buttons
/// can be made round by adding the .circular style class.
///
/// Button-like widgets like `gtk.ToggleButton`, `gtk.MenuButton`, `gtk.VolumeButton`,
/// `gtk.LockButton`, `gtk.ColorButton`, `gtk.FontButton` or `gtk.FileChooserButton` use
/// style classes such as .toggle, .popup, .scale, .lock, .color, .font, .file
/// to differentiate themselves from a plain GtkButton.
pub const Button = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Actionable, gtk.Activatable, gtk.Buildable };
    pub const Class = gtk.ButtonClass;
    f_bin: gtk.Bin,
    f_priv: ?*gtk.ButtonPrivate,

    pub const virtual_methods = struct {
        /// Signal that causes the button to animate press then
        ///    release. Applications should never connect to this signal, but use
        ///    the `clicked` signal.
        pub const activate = struct {
            pub fn call(p_class: anytype, p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Button.Class, p_class).f_activate.?(gobject.ext.as(Button, p_button));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Button.Class, p_class).f_activate = @ptrCast(p_implementation);
            }
        };

        /// Emits a `gtk.Button.signals.clicked` signal to the given `gtk.Button`.
        pub const clicked = struct {
            pub fn call(p_class: anytype, p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Button.Class, p_class).f_clicked.?(gobject.ext.as(Button, p_button));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Button.Class, p_class).f_clicked = @ptrCast(p_implementation);
            }
        };

        /// Emits a `gtk.Button.signals.enter` signal to the given `gtk.Button`.
        pub const enter = struct {
            pub fn call(p_class: anytype, p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Button.Class, p_class).f_enter.?(gobject.ext.as(Button, p_button));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Button.Class, p_class).f_enter = @ptrCast(p_implementation);
            }
        };

        /// Emits a `gtk.Button.signals.leave` signal to the given `gtk.Button`.
        pub const leave = struct {
            pub fn call(p_class: anytype, p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Button.Class, p_class).f_leave.?(gobject.ext.as(Button, p_button));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Button.Class, p_class).f_leave = @ptrCast(p_implementation);
            }
        };

        /// Emits a `gtk.Button.signals.pressed` signal to the given `gtk.Button`.
        pub const pressed = struct {
            pub fn call(p_class: anytype, p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Button.Class, p_class).f_pressed.?(gobject.ext.as(Button, p_button));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Button.Class, p_class).f_pressed = @ptrCast(p_implementation);
            }
        };

        /// Emits a `gtk.Button.signals.released` signal to the given `gtk.Button`.
        pub const released = struct {
            pub fn call(p_class: anytype, p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Button.Class, p_class).f_released.?(gobject.ext.as(Button, p_button));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Button.Class, p_class).f_released = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// If `TRUE`, the button will ignore the `gtk.Settings.properties.gtk`-button-images
        /// setting and always show the image, if available.
        ///
        /// Use this property if the button would be useless or hard to use
        /// without the image.
        pub const always_show_image = struct {
            pub const name = "always-show-image";

            pub const Type = c_int;
        };

        /// The child widget to appear next to the button text.
        pub const image = struct {
            pub const name = "image";

            pub const Type = ?*gtk.Widget;
        };

        /// The position of the image relative to the text inside the button.
        pub const image_position = struct {
            pub const name = "image-position";

            pub const Type = gtk.PositionType;
        };

        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };

        pub const relief = struct {
            pub const name = "relief";

            pub const Type = gtk.ReliefStyle;
        };

        pub const use_stock = struct {
            pub const name = "use-stock";

            pub const Type = c_int;
        };

        pub const use_underline = struct {
            pub const name = "use-underline";

            pub const Type = c_int;
        };

        /// If the child of the button is a `gtk.Misc` or `gtk.Alignment`, this property
        /// can be used to control its horizontal alignment. 0.0 is left aligned,
        /// 1.0 is right aligned.
        pub const xalign = struct {
            pub const name = "xalign";

            pub const Type = f32;
        };

        /// If the child of the button is a `gtk.Misc` or `gtk.Alignment`, this property
        /// can be used to control its vertical alignment. 0.0 is top aligned,
        /// 1.0 is bottom aligned.
        pub const yalign = struct {
            pub const name = "yalign";

            pub const Type = f32;
        };
    };

    pub const signals = struct {
        /// The ::activate signal on GtkButton is an action signal and
        /// emitting it causes the button to animate press then release.
        /// Applications should never connect to this signal, but use the
        /// `gtk.Button.signals.clicked` signal.
        pub const activate = struct {
            pub const name = "activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Button, p_instance))),
                    gobject.signalLookup("activate", Button.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the button has been activated (pressed and released).
        pub const clicked = struct {
            pub const name = "clicked";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Button, p_instance))),
                    gobject.signalLookup("clicked", Button.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the pointer enters the button.
        pub const enter = struct {
            pub const name = "enter";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Button, p_instance))),
                    gobject.signalLookup("enter", Button.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the pointer leaves the button.
        pub const leave = struct {
            pub const name = "leave";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Button, p_instance))),
                    gobject.signalLookup("leave", Button.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the button is pressed.
        pub const pressed = struct {
            pub const name = "pressed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Button, p_instance))),
                    gobject.signalLookup("pressed", Button.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the button is released.
        pub const released = struct {
            pub const name = "released";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Button, p_instance))),
                    gobject.signalLookup("released", Button.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.Button` widget. To add a child widget to the button,
    /// use `gtk.Container.add`.
    extern fn gtk_button_new() *gtk.Button;
    pub const new = gtk_button_new;

    /// Creates a new button containing an icon from the current icon theme.
    ///
    /// If the icon name isn’t known, a “broken image” icon will be
    /// displayed instead. If the current icon theme is changed, the icon
    /// will be updated appropriately.
    ///
    /// This function is a convenience wrapper around `gtk.Button.new` and
    /// `gtk.Button.setImage`.
    extern fn gtk_button_new_from_icon_name(p_icon_name: ?[*:0]const u8, p_size: c_int) *gtk.Button;
    pub const newFromIconName = gtk_button_new_from_icon_name;

    /// Creates a new `gtk.Button` containing the image and text from a
    /// [stock item][gtkstock].
    /// Some stock ids have preprocessor macros like `GTK_STOCK_OK` and
    /// `GTK_STOCK_APPLY`.
    ///
    /// If `stock_id` is unknown, then it will be treated as a mnemonic
    /// label (as for `gtk.Button.newWithMnemonic`).
    extern fn gtk_button_new_from_stock(p_stock_id: [*:0]const u8) *gtk.Button;
    pub const newFromStock = gtk_button_new_from_stock;

    /// Creates a `gtk.Button` widget with a `gtk.Label` child containing the given
    /// text.
    extern fn gtk_button_new_with_label(p_label: [*:0]const u8) *gtk.Button;
    pub const newWithLabel = gtk_button_new_with_label;

    /// Creates a new `gtk.Button` containing a label.
    /// If characters in `label` are preceded by an underscore, they are underlined.
    /// If you need a literal underscore character in a label, use “__” (two
    /// underscores). The first underlined character represents a keyboard
    /// accelerator called a mnemonic.
    /// Pressing Alt and that key activates the button.
    extern fn gtk_button_new_with_mnemonic(p_label: [*:0]const u8) *gtk.Button;
    pub const newWithMnemonic = gtk_button_new_with_mnemonic;

    /// Emits a `gtk.Button.signals.clicked` signal to the given `gtk.Button`.
    extern fn gtk_button_clicked(p_button: *Button) void;
    pub const clicked = gtk_button_clicked;

    /// Emits a `gtk.Button.signals.enter` signal to the given `gtk.Button`.
    extern fn gtk_button_enter(p_button: *Button) void;
    pub const enter = gtk_button_enter;

    /// Gets the alignment of the child in the button.
    extern fn gtk_button_get_alignment(p_button: *Button, p_xalign: *f32, p_yalign: *f32) void;
    pub const getAlignment = gtk_button_get_alignment;

    /// Returns whether the button will ignore the `gtk.Settings.properties.gtk`-button-images
    /// setting and always show the image, if available.
    extern fn gtk_button_get_always_show_image(p_button: *Button) c_int;
    pub const getAlwaysShowImage = gtk_button_get_always_show_image;

    /// Returns the button’s event window if it is realized, `NULL` otherwise.
    /// This function should be rarely needed.
    extern fn gtk_button_get_event_window(p_button: *Button) *gdk.Window;
    pub const getEventWindow = gtk_button_get_event_window;

    /// Returns whether the button grabs focus when it is clicked with the mouse.
    /// See `gtk.Button.setFocusOnClick`.
    extern fn gtk_button_get_focus_on_click(p_button: *Button) c_int;
    pub const getFocusOnClick = gtk_button_get_focus_on_click;

    /// Gets the widget that is currenty set as the image of `button`.
    /// This may have been explicitly set by `gtk.Button.setImage`
    /// or constructed by `gtk.Button.newFromStock`.
    extern fn gtk_button_get_image(p_button: *Button) ?*gtk.Widget;
    pub const getImage = gtk_button_get_image;

    /// Gets the position of the image relative to the text
    /// inside the button.
    extern fn gtk_button_get_image_position(p_button: *Button) gtk.PositionType;
    pub const getImagePosition = gtk_button_get_image_position;

    /// Fetches the text from the label of the button, as set by
    /// `gtk.Button.setLabel`. If the label text has not
    /// been set the return value will be `NULL`. This will be the
    /// case if you create an empty button with `gtk.Button.new` to
    /// use as a container.
    extern fn gtk_button_get_label(p_button: *Button) [*:0]const u8;
    pub const getLabel = gtk_button_get_label;

    /// Returns the current relief style of the given `gtk.Button`.
    extern fn gtk_button_get_relief(p_button: *Button) gtk.ReliefStyle;
    pub const getRelief = gtk_button_get_relief;

    /// Returns whether the button label is a stock item.
    extern fn gtk_button_get_use_stock(p_button: *Button) c_int;
    pub const getUseStock = gtk_button_get_use_stock;

    /// Returns whether an embedded underline in the button label indicates a
    /// mnemonic. See gtk_button_set_use_underline ().
    extern fn gtk_button_get_use_underline(p_button: *Button) c_int;
    pub const getUseUnderline = gtk_button_get_use_underline;

    /// Emits a `gtk.Button.signals.leave` signal to the given `gtk.Button`.
    extern fn gtk_button_leave(p_button: *Button) void;
    pub const leave = gtk_button_leave;

    /// Emits a `gtk.Button.signals.pressed` signal to the given `gtk.Button`.
    extern fn gtk_button_pressed(p_button: *Button) void;
    pub const pressed = gtk_button_pressed;

    /// Emits a `gtk.Button.signals.released` signal to the given `gtk.Button`.
    extern fn gtk_button_released(p_button: *Button) void;
    pub const released = gtk_button_released;

    /// Sets the alignment of the child. This property has no effect unless
    /// the child is a `gtk.Misc` or a `gtk.Alignment`.
    extern fn gtk_button_set_alignment(p_button: *Button, p_xalign: f32, p_yalign: f32) void;
    pub const setAlignment = gtk_button_set_alignment;

    /// If `TRUE`, the button will ignore the `gtk.Settings.properties.gtk`-button-images
    /// setting and always show the image, if available.
    ///
    /// Use this property if the button  would be useless or hard to use
    /// without the image.
    extern fn gtk_button_set_always_show_image(p_button: *Button, p_always_show: c_int) void;
    pub const setAlwaysShowImage = gtk_button_set_always_show_image;

    /// Sets whether the button will grab focus when it is clicked with the mouse.
    /// Making mouse clicks not grab focus is useful in places like toolbars where
    /// you don’t want the keyboard focus removed from the main area of the
    /// application.
    extern fn gtk_button_set_focus_on_click(p_button: *Button, p_focus_on_click: c_int) void;
    pub const setFocusOnClick = gtk_button_set_focus_on_click;

    /// Set the image of `button` to the given widget. The image will be
    /// displayed if the label text is `NULL` or if
    /// `gtk.Button.properties.always`-show-image is `TRUE`. You don’t have to call
    /// `gtk.Widget.show` on `image` yourself.
    extern fn gtk_button_set_image(p_button: *Button, p_image: ?*gtk.Widget) void;
    pub const setImage = gtk_button_set_image;

    /// Sets the position of the image relative to the text
    /// inside the button.
    extern fn gtk_button_set_image_position(p_button: *Button, p_position: gtk.PositionType) void;
    pub const setImagePosition = gtk_button_set_image_position;

    /// Sets the text of the label of the button to `str`. This text is
    /// also used to select the stock item if `gtk.Button.setUseStock`
    /// is used.
    ///
    /// This will also clear any previously set labels.
    extern fn gtk_button_set_label(p_button: *Button, p_label: [*:0]const u8) void;
    pub const setLabel = gtk_button_set_label;

    /// Sets the relief style of the edges of the given `gtk.Button` widget.
    /// Two styles exist, `GTK_RELIEF_NORMAL` and `GTK_RELIEF_NONE`.
    /// The default style is, as one can guess, `GTK_RELIEF_NORMAL`.
    /// The deprecated value `GTK_RELIEF_HALF` behaves the same as
    /// `GTK_RELIEF_NORMAL`.
    extern fn gtk_button_set_relief(p_button: *Button, p_relief: gtk.ReliefStyle) void;
    pub const setRelief = gtk_button_set_relief;

    /// If `TRUE`, the label set on the button is used as a
    /// stock id to select the stock item for the button.
    extern fn gtk_button_set_use_stock(p_button: *Button, p_use_stock: c_int) void;
    pub const setUseStock = gtk_button_set_use_stock;

    /// If true, an underline in the text of the button label indicates
    /// the next character should be used for the mnemonic accelerator key.
    extern fn gtk_button_set_use_underline(p_button: *Button, p_use_underline: c_int) void;
    pub const setUseUnderline = gtk_button_set_use_underline;

    extern fn gtk_button_get_type() usize;
    pub const getGObjectType = gtk_button_get_type;

    extern fn g_object_ref(p_self: *gtk.Button) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Button) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Button, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ButtonAccessible = extern struct {
    pub const Parent = gtk.ContainerAccessible;
    pub const Implements = [_]type{ atk.Action, atk.Component, atk.Image };
    pub const Class = gtk.ButtonAccessibleClass;
    f_parent: gtk.ContainerAccessible,
    f_priv: ?*gtk.ButtonAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_button_accessible_get_type() usize;
    pub const getGObjectType = gtk_button_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.ButtonAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ButtonAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ButtonAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ButtonBox = extern struct {
    pub const Parent = gtk.Box;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.ButtonBoxClass;
    f_box: gtk.Box,
    f_priv: ?*gtk.ButtonBoxPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const layout_style = struct {
            pub const name = "layout-style";

            pub const Type = gtk.ButtonBoxStyle;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.ButtonBox`.
    extern fn gtk_button_box_new(p_orientation: gtk.Orientation) *gtk.ButtonBox;
    pub const new = gtk_button_box_new;

    /// Returns whether the child is exempted from homogenous
    /// sizing.
    extern fn gtk_button_box_get_child_non_homogeneous(p_widget: *ButtonBox, p_child: *gtk.Widget) c_int;
    pub const getChildNonHomogeneous = gtk_button_box_get_child_non_homogeneous;

    /// Returns whether `child` should appear in a secondary group of children.
    extern fn gtk_button_box_get_child_secondary(p_widget: *ButtonBox, p_child: *gtk.Widget) c_int;
    pub const getChildSecondary = gtk_button_box_get_child_secondary;

    /// Retrieves the method being used to arrange the buttons in a button box.
    extern fn gtk_button_box_get_layout(p_widget: *ButtonBox) gtk.ButtonBoxStyle;
    pub const getLayout = gtk_button_box_get_layout;

    /// Sets whether the child is exempted from homogeous sizing.
    extern fn gtk_button_box_set_child_non_homogeneous(p_widget: *ButtonBox, p_child: *gtk.Widget, p_non_homogeneous: c_int) void;
    pub const setChildNonHomogeneous = gtk_button_box_set_child_non_homogeneous;

    /// Sets whether `child` should appear in a secondary group of children.
    /// A typical use of a secondary child is the help button in a dialog.
    ///
    /// This group appears after the other children if the style
    /// is `GTK_BUTTONBOX_START`, `GTK_BUTTONBOX_SPREAD` or
    /// `GTK_BUTTONBOX_EDGE`, and before the other children if the style
    /// is `GTK_BUTTONBOX_END`. For horizontal button boxes, the definition
    /// of before/after depends on direction of the widget (see
    /// `gtk.Widget.setDirection`). If the style is `GTK_BUTTONBOX_START`
    /// or `GTK_BUTTONBOX_END`, then the secondary children are aligned at
    /// the other end of the button box from the main children. For the
    /// other styles, they appear immediately next to the main children.
    extern fn gtk_button_box_set_child_secondary(p_widget: *ButtonBox, p_child: *gtk.Widget, p_is_secondary: c_int) void;
    pub const setChildSecondary = gtk_button_box_set_child_secondary;

    /// Changes the way buttons are arranged in their container.
    extern fn gtk_button_box_set_layout(p_widget: *ButtonBox, p_layout_style: gtk.ButtonBoxStyle) void;
    pub const setLayout = gtk_button_box_set_layout;

    extern fn gtk_button_box_get_type() usize;
    pub const getGObjectType = gtk_button_box_get_type;

    extern fn g_object_ref(p_self: *gtk.ButtonBox) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ButtonBox) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ButtonBox, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.Calendar` is a widget that displays a Gregorian calendar, one month
/// at a time. It can be created with `gtk.Calendar.new`.
///
/// The month and year currently displayed can be altered with
/// `gtk.Calendar.selectMonth`. The exact day can be selected from the
/// displayed month using `gtk.Calendar.selectDay`.
///
/// To place a visual marker on a particular day, use `gtk.Calendar.markDay`
/// and to remove the marker, `gtk.Calendar.unmarkDay`. Alternative, all
/// marks can be cleared with `gtk.Calendar.clearMarks`.
///
/// The way in which the calendar itself is displayed can be altered using
/// `gtk.Calendar.setDisplayOptions`.
///
/// The selected date can be retrieved from a `gtk.Calendar` using
/// `gtk.Calendar.getDate`.
///
/// Users should be aware that, although the Gregorian calendar is the
/// legal calendar in most countries, it was adopted progressively
/// between 1582 and 1929. Display before these dates is likely to be
/// historically incorrect.
pub const Calendar = extern struct {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.CalendarClass;
    f_widget: gtk.Widget,
    f_priv: ?*gtk.CalendarPrivate,

    pub const virtual_methods = struct {
        pub const day_selected = struct {
            pub fn call(p_class: anytype, p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Calendar.Class, p_class).f_day_selected.?(gobject.ext.as(Calendar, p_calendar));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Calendar.Class, p_class).f_day_selected = @ptrCast(p_implementation);
            }
        };

        pub const day_selected_double_click = struct {
            pub fn call(p_class: anytype, p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Calendar.Class, p_class).f_day_selected_double_click.?(gobject.ext.as(Calendar, p_calendar));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Calendar.Class, p_class).f_day_selected_double_click = @ptrCast(p_implementation);
            }
        };

        pub const month_changed = struct {
            pub fn call(p_class: anytype, p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Calendar.Class, p_class).f_month_changed.?(gobject.ext.as(Calendar, p_calendar));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Calendar.Class, p_class).f_month_changed = @ptrCast(p_implementation);
            }
        };

        pub const next_month = struct {
            pub fn call(p_class: anytype, p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Calendar.Class, p_class).f_next_month.?(gobject.ext.as(Calendar, p_calendar));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Calendar.Class, p_class).f_next_month = @ptrCast(p_implementation);
            }
        };

        pub const next_year = struct {
            pub fn call(p_class: anytype, p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Calendar.Class, p_class).f_next_year.?(gobject.ext.as(Calendar, p_calendar));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Calendar.Class, p_class).f_next_year = @ptrCast(p_implementation);
            }
        };

        pub const prev_month = struct {
            pub fn call(p_class: anytype, p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Calendar.Class, p_class).f_prev_month.?(gobject.ext.as(Calendar, p_calendar));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Calendar.Class, p_class).f_prev_month = @ptrCast(p_implementation);
            }
        };

        pub const prev_year = struct {
            pub fn call(p_class: anytype, p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Calendar.Class, p_class).f_prev_year.?(gobject.ext.as(Calendar, p_calendar));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_calendar: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Calendar.Class, p_class).f_prev_year = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The selected day (as a number between 1 and 31, or 0
        /// to unselect the currently selected day).
        /// This property gets initially set to the current day.
        pub const day = struct {
            pub const name = "day";

            pub const Type = c_int;
        };

        /// Height of a detail cell, in rows.
        /// A value of 0 allows any width. See `gtk.Calendar.setDetailFunc`.
        pub const detail_height_rows = struct {
            pub const name = "detail-height-rows";

            pub const Type = c_int;
        };

        /// Width of a detail cell, in characters.
        /// A value of 0 allows any width. See `gtk.Calendar.setDetailFunc`.
        pub const detail_width_chars = struct {
            pub const name = "detail-width-chars";

            pub const Type = c_int;
        };

        /// The selected month (as a number between 0 and 11).
        /// This property gets initially set to the current month.
        pub const month = struct {
            pub const name = "month";

            pub const Type = c_int;
        };

        /// Determines whether the selected month can be changed.
        pub const no_month_change = struct {
            pub const name = "no-month-change";

            pub const Type = c_int;
        };

        /// Determines whether day names are displayed.
        pub const show_day_names = struct {
            pub const name = "show-day-names";

            pub const Type = c_int;
        };

        /// Determines whether details are shown directly in the widget, or if they are
        /// available only as tooltip. When this property is set days with details are
        /// marked.
        pub const show_details = struct {
            pub const name = "show-details";

            pub const Type = c_int;
        };

        /// Determines whether a heading is displayed.
        pub const show_heading = struct {
            pub const name = "show-heading";

            pub const Type = c_int;
        };

        /// Determines whether week numbers are displayed.
        pub const show_week_numbers = struct {
            pub const name = "show-week-numbers";

            pub const Type = c_int;
        };

        /// The selected year.
        /// This property gets initially set to the current year.
        pub const year = struct {
            pub const name = "year";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// Emitted when the user selects a day.
        pub const day_selected = struct {
            pub const name = "day-selected";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Calendar, p_instance))),
                    gobject.signalLookup("day-selected", Calendar.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the user double-clicks a day.
        pub const day_selected_double_click = struct {
            pub const name = "day-selected-double-click";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Calendar, p_instance))),
                    gobject.signalLookup("day-selected-double-click", Calendar.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the user clicks a button to change the selected month on a
        /// calendar.
        pub const month_changed = struct {
            pub const name = "month-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Calendar, p_instance))),
                    gobject.signalLookup("month-changed", Calendar.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the user switched to the next month.
        pub const next_month = struct {
            pub const name = "next-month";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Calendar, p_instance))),
                    gobject.signalLookup("next-month", Calendar.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when user switched to the next year.
        pub const next_year = struct {
            pub const name = "next-year";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Calendar, p_instance))),
                    gobject.signalLookup("next-year", Calendar.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the user switched to the previous month.
        pub const prev_month = struct {
            pub const name = "prev-month";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Calendar, p_instance))),
                    gobject.signalLookup("prev-month", Calendar.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when user switched to the previous year.
        pub const prev_year = struct {
            pub const name = "prev-year";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Calendar, p_instance))),
                    gobject.signalLookup("prev-year", Calendar.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new calendar, with the current date being selected.
    extern fn gtk_calendar_new() *gtk.Calendar;
    pub const new = gtk_calendar_new;

    /// Remove all visual markers.
    extern fn gtk_calendar_clear_marks(p_calendar: *Calendar) void;
    pub const clearMarks = gtk_calendar_clear_marks;

    /// Obtains the selected date from a `gtk.Calendar`.
    extern fn gtk_calendar_get_date(p_calendar: *Calendar, p_year: ?*c_uint, p_month: ?*c_uint, p_day: ?*c_uint) void;
    pub const getDate = gtk_calendar_get_date;

    /// Returns if the `day` of the `calendar` is already marked.
    extern fn gtk_calendar_get_day_is_marked(p_calendar: *Calendar, p_day: c_uint) c_int;
    pub const getDayIsMarked = gtk_calendar_get_day_is_marked;

    /// Queries the height of detail cells, in rows.
    /// See `gtk.Calendar.properties.detail`-width-chars.
    extern fn gtk_calendar_get_detail_height_rows(p_calendar: *Calendar) c_int;
    pub const getDetailHeightRows = gtk_calendar_get_detail_height_rows;

    /// Queries the width of detail cells, in characters.
    /// See `gtk.Calendar.properties.detail`-width-chars.
    extern fn gtk_calendar_get_detail_width_chars(p_calendar: *Calendar) c_int;
    pub const getDetailWidthChars = gtk_calendar_get_detail_width_chars;

    /// Returns the current display options of `calendar`.
    extern fn gtk_calendar_get_display_options(p_calendar: *Calendar) gtk.CalendarDisplayOptions;
    pub const getDisplayOptions = gtk_calendar_get_display_options;

    /// Places a visual marker on a particular day.
    extern fn gtk_calendar_mark_day(p_calendar: *Calendar, p_day: c_uint) void;
    pub const markDay = gtk_calendar_mark_day;

    /// Selects a day from the current month.
    extern fn gtk_calendar_select_day(p_calendar: *Calendar, p_day: c_uint) void;
    pub const selectDay = gtk_calendar_select_day;

    /// Shifts the calendar to a different month.
    extern fn gtk_calendar_select_month(p_calendar: *Calendar, p_month: c_uint, p_year: c_uint) void;
    pub const selectMonth = gtk_calendar_select_month;

    /// Installs a function which provides Pango markup with detail information
    /// for each day. Examples for such details are holidays or appointments. That
    /// information is shown below each day when `gtk.Calendar.properties.show`-details is set.
    /// A tooltip containing with full detail information is provided, if the entire
    /// text should not fit into the details area, or if `gtk.Calendar.properties.show`-details
    /// is not set.
    ///
    /// The size of the details area can be restricted by setting the
    /// `gtk.Calendar.properties.detail`-width-chars and `gtk.Calendar.properties.detail`-height-rows
    /// properties.
    extern fn gtk_calendar_set_detail_func(p_calendar: *Calendar, p_func: gtk.CalendarDetailFunc, p_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) void;
    pub const setDetailFunc = gtk_calendar_set_detail_func;

    /// Updates the height of detail cells.
    /// See `gtk.Calendar.properties.detail`-height-rows.
    extern fn gtk_calendar_set_detail_height_rows(p_calendar: *Calendar, p_rows: c_int) void;
    pub const setDetailHeightRows = gtk_calendar_set_detail_height_rows;

    /// Updates the width of detail cells.
    /// See `gtk.Calendar.properties.detail`-width-chars.
    extern fn gtk_calendar_set_detail_width_chars(p_calendar: *Calendar, p_chars: c_int) void;
    pub const setDetailWidthChars = gtk_calendar_set_detail_width_chars;

    /// Sets display options (whether to display the heading and the month
    /// headings).
    extern fn gtk_calendar_set_display_options(p_calendar: *Calendar, p_flags: gtk.CalendarDisplayOptions) void;
    pub const setDisplayOptions = gtk_calendar_set_display_options;

    /// Removes the visual marker from a particular day.
    extern fn gtk_calendar_unmark_day(p_calendar: *Calendar, p_day: c_uint) void;
    pub const unmarkDay = gtk_calendar_unmark_day;

    extern fn gtk_calendar_get_type() usize;
    pub const getGObjectType = gtk_calendar_get_type;

    extern fn g_object_ref(p_self: *gtk.Calendar) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Calendar) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Calendar, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CellAccessible = extern struct {
    pub const Parent = gtk.Accessible;
    pub const Implements = [_]type{ atk.Action, atk.Component, atk.TableCell };
    pub const Class = gtk.CellAccessibleClass;
    f_parent: gtk.Accessible,
    f_priv: ?*gtk.CellAccessiblePrivate,

    pub const virtual_methods = struct {
        pub const update_cache = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_emit_signal: c_int) void {
                return gobject.ext.as(CellAccessible.Class, p_class).f_update_cache.?(gobject.ext.as(CellAccessible, p_cell), p_emit_signal);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_emit_signal: c_int) callconv(.C) void) void {
                gobject.ext.as(CellAccessible.Class, p_class).f_update_cache = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_cell_accessible_get_type() usize;
    pub const getGObjectType = gtk_cell_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.CellAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.CellArea` is an abstract class for `gtk.CellLayout` widgets
/// (also referred to as "layouting widgets") to interface with an
/// arbitrary number of `GtkCellRenderers` and interact with the user
/// for a given `gtk.TreeModel` row.
///
/// The cell area handles events, focus navigation, drawing and
/// size requests and allocations for a given row of data.
///
/// Usually users dont have to interact with the `gtk.CellArea` directly
/// unless they are implementing a cell-layouting widget themselves.
///
/// # Requesting area sizes
///
/// As outlined in
/// [GtkWidget’s geometry management section][geometry-management],
/// GTK+ uses a height-for-width
/// geometry management system to compute the sizes of widgets and user
/// interfaces. `gtk.CellArea` uses the same semantics to calculate the
/// size of an area for an arbitrary number of `gtk.TreeModel` rows.
///
/// When requesting the size of a cell area one needs to calculate
/// the size for a handful of rows, and this will be done differently by
/// different layouting widgets. For instance a `gtk.TreeViewColumn`
/// always lines up the areas from top to bottom while a `gtk.IconView`
/// on the other hand might enforce that all areas received the same
/// width and wrap the areas around, requesting height for more cell
/// areas when allocated less width.
///
/// It’s also important for areas to maintain some cell
/// alignments with areas rendered for adjacent rows (cells can
/// appear “columnized” inside an area even when the size of
/// cells are different in each row). For this reason the `gtk.CellArea`
/// uses a `gtk.CellAreaContext` object to store the alignments
/// and sizes along the way (as well as the overall largest minimum
/// and natural size for all the rows which have been calculated
/// with the said context).
///
/// The `gtk.CellAreaContext` is an opaque object specific to the
/// `gtk.CellArea` which created it (see `gtk.CellArea.createContext`).
/// The owning cell-layouting widget can create as many contexts as
/// it wishes to calculate sizes of rows which should receive the
/// same size in at least one orientation (horizontally or vertically),
/// However, it’s important that the same `gtk.CellAreaContext` which
/// was used to request the sizes for a given `gtk.TreeModel` row be
/// used when rendering or processing events for that row.
///
/// In order to request the width of all the rows at the root level
/// of a `gtk.TreeModel` one would do the following:
///
/// ```
/// GtkTreeIter iter;
/// gint        minimum_width;
/// gint        natural_width;
///
/// valid = gtk_tree_model_get_iter_first (model, &iter);
/// while (valid)
///   {
///     gtk_cell_area_apply_attributes (area, model, &iter, FALSE, FALSE);
///     gtk_cell_area_get_preferred_width (area, context, widget, NULL, NULL);
///
///     valid = gtk_tree_model_iter_next (model, &iter);
///   }
/// gtk_cell_area_context_get_preferred_width (context, &minimum_width, &natural_width);
/// ```
///
/// Note that in this example it’s not important to observe the
/// returned minimum and natural width of the area for each row
/// unless the cell-layouting object is actually interested in the
/// widths of individual rows. The overall width is however stored
/// in the accompanying `gtk.CellAreaContext` object and can be consulted
/// at any time.
///
/// This can be useful since `gtk.CellLayout` widgets usually have to
/// support requesting and rendering rows in treemodels with an
/// exceedingly large amount of rows. The `gtk.CellLayout` widget in
/// that case would calculate the required width of the rows in an
/// idle or timeout source (see `glib.timeoutAdd`) and when the widget
/// is requested its actual width in `gtk.WidgetClass.virtual_methods.get_preferred_width`
/// it can simply consult the width accumulated so far in the
/// `gtk.CellAreaContext` object.
///
/// A simple example where rows are rendered from top to bottom and
/// take up the full width of the layouting widget would look like:
///
/// ```
/// static void
/// foo_get_preferred_width (GtkWidget       *widget,
///                          gint            *minimum_size,
///                          gint            *natural_size)
/// {
///   Foo        *foo  = FOO (widget);
///   FooPrivate *priv = foo->priv;
///
///   foo_ensure_at_least_one_handfull_of_rows_have_been_requested (foo);
///
///   gtk_cell_area_context_get_preferred_width (priv->context, minimum_size, natural_size);
/// }
/// ```
///
/// In the above example the Foo widget has to make sure that some
/// row sizes have been calculated (the amount of rows that Foo judged
/// was appropriate to request space for in a single timeout iteration)
/// before simply returning the amount of space required by the area via
/// the `gtk.CellAreaContext`.
///
/// Requesting the height for width (or width for height) of an area is
/// a similar task except in this case the `gtk.CellAreaContext` does not
/// store the data (actually, it does not know how much space the layouting
/// widget plans to allocate it for every row. It’s up to the layouting
/// widget to render each row of data with the appropriate height and
/// width which was requested by the `gtk.CellArea`).
///
/// In order to request the height for width of all the rows at the
/// root level of a `gtk.TreeModel` one would do the following:
///
/// ```
/// GtkTreeIter iter;
/// gint        minimum_height;
/// gint        natural_height;
/// gint        full_minimum_height = 0;
/// gint        full_natural_height = 0;
///
/// valid = gtk_tree_model_get_iter_first (model, &iter);
/// while (valid)
///   {
///     gtk_cell_area_apply_attributes (area, model, &iter, FALSE, FALSE);
///     gtk_cell_area_get_preferred_height_for_width (area, context, widget,
///                                                   width, &minimum_height, &natural_height);
///
///     if (width_is_for_allocation)
///        cache_row_height (&iter, minimum_height, natural_height);
///
///     full_minimum_height += minimum_height;
///     full_natural_height += natural_height;
///
///     valid = gtk_tree_model_iter_next (model, &iter);
///   }
/// ```
///
/// Note that in the above example we would need to cache the heights
/// returned for each row so that we would know what sizes to render the
/// areas for each row. However we would only want to really cache the
/// heights if the request is intended for the layouting widgets real
/// allocation.
///
/// In some cases the layouting widget is requested the height for an
/// arbitrary for_width, this is a special case for layouting widgets
/// who need to request size for tens of thousands  of rows. For this
/// case it’s only important that the layouting widget calculate
/// one reasonably sized chunk of rows and return that height
/// synchronously. The reasoning here is that any layouting widget is
/// at least capable of synchronously calculating enough height to fill
/// the screen height (or scrolled window height) in response to a single
/// call to `gtk.WidgetClass.virtual_methods.get_preferred_height_for_width`. Returning
/// a perfect height for width that is larger than the screen area is
/// inconsequential since after the layouting receives an allocation
/// from a scrolled window it simply continues to drive the scrollbar
/// values while more and more height is required for the row heights
/// that are calculated in the background.
///
/// # Rendering Areas
///
/// Once area sizes have been aquired at least for the rows in the
/// visible area of the layouting widget they can be rendered at
/// `gtk.WidgetClass.virtual_methods.draw` time.
///
/// A crude example of how to render all the rows at the root level
/// runs as follows:
///
/// ```
/// GtkAllocation allocation;
/// GdkRectangle  cell_area = { 0, };
/// GtkTreeIter   iter;
/// gint          minimum_width;
/// gint          natural_width;
///
/// gtk_widget_get_allocation (widget, &allocation);
/// cell_area.width = allocation.width;
///
/// valid = gtk_tree_model_get_iter_first (model, &iter);
/// while (valid)
///   {
///     cell_area.height = get_cached_height_for_row (&iter);
///
///     gtk_cell_area_apply_attributes (area, model, &iter, FALSE, FALSE);
///     gtk_cell_area_render (area, context, widget, cr,
///                           &cell_area, &cell_area, state_flags, FALSE);
///
///     cell_area.y += cell_area.height;
///
///     valid = gtk_tree_model_iter_next (model, &iter);
///   }
/// ```
///
/// Note that the cached height in this example really depends on how
/// the layouting widget works. The layouting widget might decide to
/// give every row its minimum or natural height or, if the model content
/// is expected to fit inside the layouting widget without scrolling, it
/// would make sense to calculate the allocation for each row at
/// `gtk.Widget.signals.size`-allocate time using `gtk.distributeNaturalAllocation`.
///
/// # Handling Events and Driving Keyboard Focus
///
/// Passing events to the area is as simple as handling events on any
/// normal widget and then passing them to the `gtk.CellArea.event`
/// API as they come in. Usually `gtk.CellArea` is only interested in
/// button events, however some customized derived areas can be implemented
/// who are interested in handling other events. Handling an event can
/// trigger the `gtk.CellArea.signals.focus`-changed signal to fire; as well as
/// `gtk.CellArea.signals.add`-editable in the case that an editable cell was
/// clicked and needs to start editing. You can call
/// `gtk.CellArea.stopEditing` at any time to cancel any cell editing
/// that is currently in progress.
///
/// The `gtk.CellArea` drives keyboard focus from cell to cell in a way
/// similar to `gtk.Widget`. For layouting widgets that support giving
/// focus to cells it’s important to remember to pass `GTK_CELL_RENDERER_FOCUSED`
/// to the area functions for the row that has focus and to tell the
/// area to paint the focus at render time.
///
/// Layouting widgets that accept focus on cells should implement the
/// `gtk.WidgetClass.virtual_methods.focus` virtual method. The layouting widget is always
/// responsible for knowing where `gtk.TreeModel` rows are rendered inside
/// the widget, so at `gtk.WidgetClass.virtual_methods.focus` time the layouting widget
/// should use the `gtk.CellArea` methods to navigate focus inside the area
/// and then observe the GtkDirectionType to pass the focus to adjacent
/// rows and areas.
///
/// A basic example of how the `gtk.WidgetClass.virtual_methods.focus` virtual method
/// should be implemented:
///
/// ```
/// static gboolean
/// foo_focus (GtkWidget       *widget,
///            GtkDirectionType direction)
/// {
///   Foo        *foo  = FOO (widget);
///   FooPrivate *priv = foo->priv;
///   gint        focus_row;
///   gboolean    have_focus = FALSE;
///
///   focus_row = priv->focus_row;
///
///   if (!gtk_widget_has_focus (widget))
///     gtk_widget_grab_focus (widget);
///
///   valid = gtk_tree_model_iter_nth_child (priv->model, &iter, NULL, priv->focus_row);
///   while (valid)
///     {
///       gtk_cell_area_apply_attributes (priv->area, priv->model, &iter, FALSE, FALSE);
///
///       if (gtk_cell_area_focus (priv->area, direction))
///         {
///            priv->focus_row = focus_row;
///            have_focus = TRUE;
///            break;
///         }
///       else
///         {
///           if (direction == GTK_DIR_RIGHT ||
///               direction == GTK_DIR_LEFT)
///             break;
///           else if (direction == GTK_DIR_UP ||
///                    direction == GTK_DIR_TAB_BACKWARD)
///            {
///               if (focus_row == 0)
///                 break;
///               else
///                {
///                   focus_row--;
///                   valid = gtk_tree_model_iter_nth_child (priv->model, &iter, NULL, focus_row);
///                }
///             }
///           else
///             {
///               if (focus_row == last_row)
///                 break;
///               else
///                 {
///                   focus_row++;
///                   valid = gtk_tree_model_iter_next (priv->model, &iter);
///                 }
///             }
///         }
///     }
///     return have_focus;
/// }
/// ```
///
/// Note that the layouting widget is responsible for matching the
/// GtkDirectionType values to the way it lays out its cells.
///
/// # Cell Properties
///
/// The `gtk.CellArea` introduces cell properties for `GtkCellRenderers`
/// in very much the same way that `gtk.Container` introduces
/// [child properties][child-properties]
/// for `GtkWidgets`. This provides some general interfaces for defining
/// the relationship cell areas have with their cells. For instance in a
/// `gtk.CellAreaBox` a cell might “expand” and receive extra space when
/// the area is allocated more than its full natural request, or a cell
/// might be configured to “align” with adjacent rows which were requested
/// and rendered with the same `gtk.CellAreaContext`.
///
/// Use `gtk.CellAreaClass.installCellProperty` to install cell
/// properties for a cell area class and `gtk.CellAreaClass.findCellProperty`
/// or `gtk.CellAreaClass.listCellProperties` to get information about
/// existing cell properties.
///
/// To set the value of a cell property, use `gtk.CellArea.cellSetProperty`,
/// `gtk.CellArea.cellSet` or `gtk.CellArea.cellSetValist`. To obtain
/// the value of a cell property, use `gtk.CellArea.cellGetProperty`,
/// `gtk.CellArea.cellGet` or `gtk.CellArea.cellGetValist`.
pub const CellArea = extern struct {
    pub const Parent = gobject.InitiallyUnowned;
    pub const Implements = [_]type{ gtk.Buildable, gtk.CellLayout };
    pub const Class = gtk.CellAreaClass;
    f_parent_instance: gobject.InitiallyUnowned,
    f_priv: ?*gtk.CellAreaPrivate,

    pub const virtual_methods = struct {
        /// Activates `area`, usually by activating the currently focused
        /// cell, however some subclasses which embed widgets in the area
        /// can also activate a widget if it currently has the focus.
        pub const activate = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState, p_edit_only: c_int) c_int {
                return gobject.ext.as(CellArea.Class, p_class).f_activate.?(gobject.ext.as(CellArea, p_area), p_context, p_widget, p_cell_area, p_flags, p_edit_only);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState, p_edit_only: c_int) callconv(.C) c_int) void {
                gobject.ext.as(CellArea.Class, p_class).f_activate = @ptrCast(p_implementation);
            }
        };

        /// Adds `renderer` to `area` with the default child cell properties.
        pub const add = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gtk.CellRenderer) void {
                return gobject.ext.as(CellArea.Class, p_class).f_add.?(gobject.ext.as(CellArea, p_area), p_renderer);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gtk.CellRenderer) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_add = @ptrCast(p_implementation);
            }
        };

        /// Applies any connected attributes to the renderers in
        /// `area` by pulling the values from `tree_model`.
        pub const apply_attributes = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_tree_model: *gtk.TreeModel, p_iter: *gtk.TreeIter, p_is_expander: c_int, p_is_expanded: c_int) void {
                return gobject.ext.as(CellArea.Class, p_class).f_apply_attributes.?(gobject.ext.as(CellArea, p_area), p_tree_model, p_iter, p_is_expander, p_is_expanded);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_tree_model: *gtk.TreeModel, p_iter: *gtk.TreeIter, p_is_expander: c_int, p_is_expanded: c_int) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_apply_attributes = @ptrCast(p_implementation);
            }
        };

        /// This is sometimes needed for cases where rows need to share
        /// alignments in one orientation but may be separately grouped
        /// in the opposing orientation.
        ///
        /// For instance, `gtk.IconView` creates all icons (rows) to have
        /// the same width and the cells theirin to have the same
        /// horizontal alignments. However each row of icons may have
        /// a separate collective height. `gtk.IconView` uses this to
        /// request the heights of each row based on a context which
        /// was already used to request all the row widths that are
        /// to be displayed.
        pub const copy_context = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext) *gtk.CellAreaContext {
                return gobject.ext.as(CellArea.Class, p_class).f_copy_context.?(gobject.ext.as(CellArea, p_area), p_context);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext) callconv(.C) *gtk.CellAreaContext) void {
                gobject.ext.as(CellArea.Class, p_class).f_copy_context = @ptrCast(p_implementation);
            }
        };

        /// Creates a `gtk.CellAreaContext` to be used with `area` for
        /// all purposes. `gtk.CellAreaContext` stores geometry information
        /// for rows for which it was operated on, it is important to use
        /// the same context for the same row of data at all times (i.e.
        /// one should render and handle events with the same `gtk.CellAreaContext`
        /// which was used to request the size of those rows of data).
        pub const create_context = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *gtk.CellAreaContext {
                return gobject.ext.as(CellArea.Class, p_class).f_create_context.?(gobject.ext.as(CellArea, p_area));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *gtk.CellAreaContext) void {
                gobject.ext.as(CellArea.Class, p_class).f_create_context = @ptrCast(p_implementation);
            }
        };

        /// Delegates event handling to a `gtk.CellArea`.
        pub const event = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_event: *gdk.Event, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) c_int {
                return gobject.ext.as(CellArea.Class, p_class).f_event.?(gobject.ext.as(CellArea, p_area), p_context, p_widget, p_event, p_cell_area, p_flags);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_event: *gdk.Event, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) callconv(.C) c_int) void {
                gobject.ext.as(CellArea.Class, p_class).f_event = @ptrCast(p_implementation);
            }
        };

        /// This should be called by the `area`’s owning layout widget
        /// when focus is to be passed to `area`, or moved within `area`
        /// for a given `direction` and row data.
        ///
        /// Implementing `gtk.CellArea` classes should implement this
        /// method to receive and navigate focus in its own way particular
        /// to how it lays out cells.
        pub const focus = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_direction: gtk.DirectionType) c_int {
                return gobject.ext.as(CellArea.Class, p_class).f_focus.?(gobject.ext.as(CellArea, p_area), p_direction);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_direction: gtk.DirectionType) callconv(.C) c_int) void {
                gobject.ext.as(CellArea.Class, p_class).f_focus = @ptrCast(p_implementation);
            }
        };

        /// Calls `callback` for every `gtk.CellRenderer` in `area`.
        pub const foreach = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_callback: gtk.CellCallback, p_callback_data: ?*anyopaque) void {
                return gobject.ext.as(CellArea.Class, p_class).f_foreach.?(gobject.ext.as(CellArea, p_area), p_callback, p_callback_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_callback: gtk.CellCallback, p_callback_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_foreach = @ptrCast(p_implementation);
            }
        };

        /// Calls `callback` for every `gtk.CellRenderer` in `area` with the
        /// allocated rectangle inside `cell_area`.
        pub const foreach_alloc = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_cell_area: *const gdk.Rectangle, p_background_area: *const gdk.Rectangle, p_callback: gtk.CellAllocCallback, p_callback_data: ?*anyopaque) void {
                return gobject.ext.as(CellArea.Class, p_class).f_foreach_alloc.?(gobject.ext.as(CellArea, p_area), p_context, p_widget, p_cell_area, p_background_area, p_callback, p_callback_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_cell_area: *const gdk.Rectangle, p_background_area: *const gdk.Rectangle, p_callback: gtk.CellAllocCallback, p_callback_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_foreach_alloc = @ptrCast(p_implementation);
            }
        };

        /// This should be implemented to report the values of
        ///     child cell properties for a given child `gtk.CellRenderer`.
        pub const get_cell_property = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gtk.CellRenderer, p_property_id: c_uint, p_value: *gobject.Value, p_pspec: *gobject.ParamSpec) void {
                return gobject.ext.as(CellArea.Class, p_class).f_get_cell_property.?(gobject.ext.as(CellArea, p_area), p_renderer, p_property_id, p_value, p_pspec);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gtk.CellRenderer, p_property_id: c_uint, p_value: *gobject.Value, p_pspec: *gobject.ParamSpec) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_get_cell_property = @ptrCast(p_implementation);
            }
        };

        /// Retrieves a cell area’s initial minimum and natural height.
        ///
        /// `area` will store some geometrical information in `context` along the way;
        /// when requesting sizes over an arbitrary number of rows, it’s not important
        /// to check the `minimum_height` and `natural_height` of this call but rather to
        /// consult `gtk.CellAreaContext.getPreferredHeight` after a series of
        /// requests.
        pub const get_preferred_height = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) void {
                return gobject.ext.as(CellArea.Class, p_class).f_get_preferred_height.?(gobject.ext.as(CellArea, p_area), p_context, p_widget, p_minimum_height, p_natural_height);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_get_preferred_height = @ptrCast(p_implementation);
            }
        };

        /// Retrieves a cell area’s minimum and natural height if it would be given
        /// the specified `width`.
        ///
        /// `area` stores some geometrical information in `context` along the way
        /// while calling `gtk.CellArea.getPreferredWidth`. It’s important to
        /// perform a series of `gtk.CellArea.getPreferredWidth` requests with
        /// `context` first and then call `gtk.CellArea.getPreferredHeightForWidth`
        /// on each cell area individually to get the height for width of each
        /// fully requested row.
        ///
        /// If at some point, the width of a single row changes, it should be
        /// requested with `gtk.CellArea.getPreferredWidth` again and then
        /// the full width of the requested rows checked again with
        /// `gtk.CellAreaContext.getPreferredWidth`.
        pub const get_preferred_height_for_width = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_width: c_int, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) void {
                return gobject.ext.as(CellArea.Class, p_class).f_get_preferred_height_for_width.?(gobject.ext.as(CellArea, p_area), p_context, p_widget, p_width, p_minimum_height, p_natural_height);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_width: c_int, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_get_preferred_height_for_width = @ptrCast(p_implementation);
            }
        };

        /// Retrieves a cell area’s initial minimum and natural width.
        ///
        /// `area` will store some geometrical information in `context` along the way;
        /// when requesting sizes over an arbitrary number of rows, it’s not important
        /// to check the `minimum_width` and `natural_width` of this call but rather to
        /// consult `gtk.CellAreaContext.getPreferredWidth` after a series of
        /// requests.
        pub const get_preferred_width = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) void {
                return gobject.ext.as(CellArea.Class, p_class).f_get_preferred_width.?(gobject.ext.as(CellArea, p_area), p_context, p_widget, p_minimum_width, p_natural_width);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_get_preferred_width = @ptrCast(p_implementation);
            }
        };

        /// Retrieves a cell area’s minimum and natural width if it would be given
        /// the specified `height`.
        ///
        /// `area` stores some geometrical information in `context` along the way
        /// while calling `gtk.CellArea.getPreferredHeight`. It’s important to
        /// perform a series of `gtk.CellArea.getPreferredHeight` requests with
        /// `context` first and then call `gtk.CellArea.getPreferredWidthForHeight`
        /// on each cell area individually to get the height for width of each
        /// fully requested row.
        ///
        /// If at some point, the height of a single row changes, it should be
        /// requested with `gtk.CellArea.getPreferredHeight` again and then
        /// the full height of the requested rows checked again with
        /// `gtk.CellAreaContext.getPreferredHeight`.
        pub const get_preferred_width_for_height = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_height: c_int, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) void {
                return gobject.ext.as(CellArea.Class, p_class).f_get_preferred_width_for_height.?(gobject.ext.as(CellArea, p_area), p_context, p_widget, p_height, p_minimum_width, p_natural_width);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_height: c_int, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_get_preferred_width_for_height = @ptrCast(p_implementation);
            }
        };

        /// Gets whether the area prefers a height-for-width layout
        /// or a width-for-height layout.
        pub const get_request_mode = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) gtk.SizeRequestMode {
                return gobject.ext.as(CellArea.Class, p_class).f_get_request_mode.?(gobject.ext.as(CellArea, p_area));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) gtk.SizeRequestMode) void {
                gobject.ext.as(CellArea.Class, p_class).f_get_request_mode = @ptrCast(p_implementation);
            }
        };

        /// Returns whether the area can do anything when activated,
        /// after applying new attributes to `area`.
        pub const is_activatable = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(CellArea.Class, p_class).f_is_activatable.?(gobject.ext.as(CellArea, p_area));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(CellArea.Class, p_class).f_is_activatable = @ptrCast(p_implementation);
            }
        };

        /// Removes `renderer` from `area`.
        pub const remove = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gtk.CellRenderer) void {
                return gobject.ext.as(CellArea.Class, p_class).f_remove.?(gobject.ext.as(CellArea, p_area), p_renderer);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gtk.CellRenderer) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_remove = @ptrCast(p_implementation);
            }
        };

        /// Renders `area`’s cells according to `area`’s layout onto `widget` at
        /// the given coordinates.
        pub const render = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_cr: *cairo.Context, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState, p_paint_focus: c_int) void {
                return gobject.ext.as(CellArea.Class, p_class).f_render.?(gobject.ext.as(CellArea, p_area), p_context, p_widget, p_cr, p_background_area, p_cell_area, p_flags, p_paint_focus);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_cr: *cairo.Context, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState, p_paint_focus: c_int) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_render = @ptrCast(p_implementation);
            }
        };

        /// This should be implemented to handle changes in child
        ///     cell properties for a given `gtk.CellRenderer` that were previously
        ///     installed on the `gtk.CellAreaClass` with `gtk.CellAreaClass.installCellProperty`.
        pub const set_cell_property = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gtk.CellRenderer, p_property_id: c_uint, p_value: *const gobject.Value, p_pspec: *gobject.ParamSpec) void {
                return gobject.ext.as(CellArea.Class, p_class).f_set_cell_property.?(gobject.ext.as(CellArea, p_area), p_renderer, p_property_id, p_value, p_pspec);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_renderer: *gtk.CellRenderer, p_property_id: c_uint, p_value: *const gobject.Value, p_pspec: *gobject.ParamSpec) callconv(.C) void) void {
                gobject.ext.as(CellArea.Class, p_class).f_set_cell_property = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The widget currently editing the edited cell
        ///
        /// This property is read-only and only changes as
        /// a result of a call `gtk.CellArea.activateCell`.
        pub const edit_widget = struct {
            pub const name = "edit-widget";

            pub const Type = ?*gtk.CellEditable;
        };

        /// The cell in the area that is currently edited
        ///
        /// This property is read-only and only changes as
        /// a result of a call `gtk.CellArea.activateCell`.
        pub const edited_cell = struct {
            pub const name = "edited-cell";

            pub const Type = ?*gtk.CellRenderer;
        };

        /// The cell in the area that currently has focus
        pub const focus_cell = struct {
            pub const name = "focus-cell";

            pub const Type = ?*gtk.CellRenderer;
        };
    };

    pub const signals = struct {
        /// Indicates that editing has started on `renderer` and that `editable`
        /// should be added to the owning cell-layouting widget at `cell_area`.
        pub const add_editable = struct {
            pub const name = "add-editable";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_renderer: *gtk.CellRenderer, p_editable: *gtk.CellEditable, p_cell_area: *gdk.Rectangle, p_path: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CellArea, p_instance))),
                    gobject.signalLookup("add-editable", CellArea.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted whenever applying attributes to `area` from `model`
        pub const apply_attributes = struct {
            pub const name = "apply-attributes";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_model: *gtk.TreeModel, p_iter: *gtk.TreeIter, p_is_expander: c_int, p_is_expanded: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CellArea, p_instance))),
                    gobject.signalLookup("apply-attributes", CellArea.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Indicates that focus changed on this `area`. This signal
        /// is emitted either as a result of focus handling or event
        /// handling.
        ///
        /// It's possible that the signal is emitted even if the
        /// currently focused renderer did not change, this is
        /// because focus may change to the same renderer in the
        /// same cell area for a different row of data.
        pub const focus_changed = struct {
            pub const name = "focus-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_renderer: *gtk.CellRenderer, p_path: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CellArea, p_instance))),
                    gobject.signalLookup("focus-changed", CellArea.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Indicates that editing finished on `renderer` and that `editable`
        /// should be removed from the owning cell-layouting widget.
        pub const remove_editable = struct {
            pub const name = "remove-editable";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_renderer: *gtk.CellRenderer, p_editable: *gtk.CellEditable, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CellArea, p_instance))),
                    gobject.signalLookup("remove-editable", CellArea.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Activates `area`, usually by activating the currently focused
    /// cell, however some subclasses which embed widgets in the area
    /// can also activate a widget if it currently has the focus.
    extern fn gtk_cell_area_activate(p_area: *CellArea, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState, p_edit_only: c_int) c_int;
    pub const activate = gtk_cell_area_activate;

    /// This is used by `gtk.CellArea` subclasses when handling events
    /// to activate cells, the base `gtk.CellArea` class activates cells
    /// for keyboard events for free in its own GtkCellArea->`activate`
    /// implementation.
    extern fn gtk_cell_area_activate_cell(p_area: *CellArea, p_widget: *gtk.Widget, p_renderer: *gtk.CellRenderer, p_event: *gdk.Event, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) c_int;
    pub const activateCell = gtk_cell_area_activate_cell;

    /// Adds `renderer` to `area` with the default child cell properties.
    extern fn gtk_cell_area_add(p_area: *CellArea, p_renderer: *gtk.CellRenderer) void;
    pub const add = gtk_cell_area_add;

    /// Adds `sibling` to `renderer`’s focusable area, focus will be drawn
    /// around `renderer` and all of its siblings if `renderer` can
    /// focus for a given row.
    ///
    /// Events handled by focus siblings can also activate the given
    /// focusable `renderer`.
    extern fn gtk_cell_area_add_focus_sibling(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_sibling: *gtk.CellRenderer) void;
    pub const addFocusSibling = gtk_cell_area_add_focus_sibling;

    /// Adds `renderer` to `area`, setting cell properties at the same time.
    /// See `gtk.CellArea.add` and `gtk.CellArea.cellSet` for more details.
    extern fn gtk_cell_area_add_with_properties(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_first_prop_name: [*:0]const u8, ...) void;
    pub const addWithProperties = gtk_cell_area_add_with_properties;

    /// Applies any connected attributes to the renderers in
    /// `area` by pulling the values from `tree_model`.
    extern fn gtk_cell_area_apply_attributes(p_area: *CellArea, p_tree_model: *gtk.TreeModel, p_iter: *gtk.TreeIter, p_is_expander: c_int, p_is_expanded: c_int) void;
    pub const applyAttributes = gtk_cell_area_apply_attributes;

    /// Connects an `attribute` to apply values from `column` for the
    /// `gtk.TreeModel` in use.
    extern fn gtk_cell_area_attribute_connect(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_attribute: [*:0]const u8, p_column: c_int) void;
    pub const attributeConnect = gtk_cell_area_attribute_connect;

    /// Disconnects `attribute` for the `renderer` in `area` so that
    /// attribute will no longer be updated with values from the
    /// model.
    extern fn gtk_cell_area_attribute_disconnect(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_attribute: [*:0]const u8) void;
    pub const attributeDisconnect = gtk_cell_area_attribute_disconnect;

    /// Returns the model column that an attribute has been mapped to,
    /// or -1 if the attribute is not mapped.
    extern fn gtk_cell_area_attribute_get_column(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_attribute: [*:0]const u8) c_int;
    pub const attributeGetColumn = gtk_cell_area_attribute_get_column;

    /// Gets the values of one or more cell properties for `renderer` in `area`.
    extern fn gtk_cell_area_cell_get(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_first_prop_name: [*:0]const u8, ...) void;
    pub const cellGet = gtk_cell_area_cell_get;

    /// Gets the value of a cell property for `renderer` in `area`.
    extern fn gtk_cell_area_cell_get_property(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_property_name: [*:0]const u8, p_value: *gobject.Value) void;
    pub const cellGetProperty = gtk_cell_area_cell_get_property;

    /// Gets the values of one or more cell properties for `renderer` in `area`.
    extern fn gtk_cell_area_cell_get_valist(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_first_property_name: [*:0]const u8, p_var_args: std.builtin.VaList) void;
    pub const cellGetValist = gtk_cell_area_cell_get_valist;

    /// Sets one or more cell properties for `cell` in `area`.
    extern fn gtk_cell_area_cell_set(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_first_prop_name: [*:0]const u8, ...) void;
    pub const cellSet = gtk_cell_area_cell_set;

    /// Sets a cell property for `renderer` in `area`.
    extern fn gtk_cell_area_cell_set_property(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_property_name: [*:0]const u8, p_value: *const gobject.Value) void;
    pub const cellSetProperty = gtk_cell_area_cell_set_property;

    /// Sets one or more cell properties for `renderer` in `area`.
    extern fn gtk_cell_area_cell_set_valist(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_first_property_name: [*:0]const u8, p_var_args: std.builtin.VaList) void;
    pub const cellSetValist = gtk_cell_area_cell_set_valist;

    /// This is sometimes needed for cases where rows need to share
    /// alignments in one orientation but may be separately grouped
    /// in the opposing orientation.
    ///
    /// For instance, `gtk.IconView` creates all icons (rows) to have
    /// the same width and the cells theirin to have the same
    /// horizontal alignments. However each row of icons may have
    /// a separate collective height. `gtk.IconView` uses this to
    /// request the heights of each row based on a context which
    /// was already used to request all the row widths that are
    /// to be displayed.
    extern fn gtk_cell_area_copy_context(p_area: *CellArea, p_context: *gtk.CellAreaContext) *gtk.CellAreaContext;
    pub const copyContext = gtk_cell_area_copy_context;

    /// Creates a `gtk.CellAreaContext` to be used with `area` for
    /// all purposes. `gtk.CellAreaContext` stores geometry information
    /// for rows for which it was operated on, it is important to use
    /// the same context for the same row of data at all times (i.e.
    /// one should render and handle events with the same `gtk.CellAreaContext`
    /// which was used to request the size of those rows of data).
    extern fn gtk_cell_area_create_context(p_area: *CellArea) *gtk.CellAreaContext;
    pub const createContext = gtk_cell_area_create_context;

    /// Delegates event handling to a `gtk.CellArea`.
    extern fn gtk_cell_area_event(p_area: *CellArea, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_event: *gdk.Event, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) c_int;
    pub const event = gtk_cell_area_event;

    /// This should be called by the `area`’s owning layout widget
    /// when focus is to be passed to `area`, or moved within `area`
    /// for a given `direction` and row data.
    ///
    /// Implementing `gtk.CellArea` classes should implement this
    /// method to receive and navigate focus in its own way particular
    /// to how it lays out cells.
    extern fn gtk_cell_area_focus(p_area: *CellArea, p_direction: gtk.DirectionType) c_int;
    pub const focus = gtk_cell_area_focus;

    /// Calls `callback` for every `gtk.CellRenderer` in `area`.
    extern fn gtk_cell_area_foreach(p_area: *CellArea, p_callback: gtk.CellCallback, p_callback_data: ?*anyopaque) void;
    pub const foreach = gtk_cell_area_foreach;

    /// Calls `callback` for every `gtk.CellRenderer` in `area` with the
    /// allocated rectangle inside `cell_area`.
    extern fn gtk_cell_area_foreach_alloc(p_area: *CellArea, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_cell_area: *const gdk.Rectangle, p_background_area: *const gdk.Rectangle, p_callback: gtk.CellAllocCallback, p_callback_data: ?*anyopaque) void;
    pub const foreachAlloc = gtk_cell_area_foreach_alloc;

    /// Derives the allocation of `renderer` inside `area` if `area`
    /// were to be renderered in `cell_area`.
    extern fn gtk_cell_area_get_cell_allocation(p_area: *CellArea, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_renderer: *gtk.CellRenderer, p_cell_area: *const gdk.Rectangle, p_allocation: *gdk.Rectangle) void;
    pub const getCellAllocation = gtk_cell_area_get_cell_allocation;

    /// Gets the `gtk.CellRenderer` at `x` and `y` coordinates inside `area` and optionally
    /// returns the full cell allocation for it inside `cell_area`.
    extern fn gtk_cell_area_get_cell_at_position(p_area: *CellArea, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_cell_area: *const gdk.Rectangle, p_x: c_int, p_y: c_int, p_alloc_area: ?*gdk.Rectangle) *gtk.CellRenderer;
    pub const getCellAtPosition = gtk_cell_area_get_cell_at_position;

    /// Gets the current `gtk.TreePath` string for the currently
    /// applied `gtk.TreeIter`, this is implicitly updated when
    /// `gtk.CellArea.applyAttributes` is called and can be
    /// used to interact with renderers from `gtk.CellArea`
    /// subclasses.
    extern fn gtk_cell_area_get_current_path_string(p_area: *CellArea) [*:0]const u8;
    pub const getCurrentPathString = gtk_cell_area_get_current_path_string;

    /// Gets the `gtk.CellEditable` widget currently used
    /// to edit the currently edited cell.
    extern fn gtk_cell_area_get_edit_widget(p_area: *CellArea) *gtk.CellEditable;
    pub const getEditWidget = gtk_cell_area_get_edit_widget;

    /// Gets the `gtk.CellRenderer` in `area` that is currently
    /// being edited.
    extern fn gtk_cell_area_get_edited_cell(p_area: *CellArea) *gtk.CellRenderer;
    pub const getEditedCell = gtk_cell_area_get_edited_cell;

    /// Retrieves the currently focused cell for `area`
    extern fn gtk_cell_area_get_focus_cell(p_area: *CellArea) *gtk.CellRenderer;
    pub const getFocusCell = gtk_cell_area_get_focus_cell;

    /// Gets the `gtk.CellRenderer` which is expected to be focusable
    /// for which `renderer` is, or may be a sibling.
    ///
    /// This is handy for `gtk.CellArea` subclasses when handling events,
    /// after determining the renderer at the event location it can
    /// then chose to activate the focus cell for which the event
    /// cell may have been a sibling.
    extern fn gtk_cell_area_get_focus_from_sibling(p_area: *CellArea, p_renderer: *gtk.CellRenderer) ?*gtk.CellRenderer;
    pub const getFocusFromSibling = gtk_cell_area_get_focus_from_sibling;

    /// Gets the focus sibling cell renderers for `renderer`.
    extern fn gtk_cell_area_get_focus_siblings(p_area: *CellArea, p_renderer: *gtk.CellRenderer) *const glib.List;
    pub const getFocusSiblings = gtk_cell_area_get_focus_siblings;

    /// Retrieves a cell area’s initial minimum and natural height.
    ///
    /// `area` will store some geometrical information in `context` along the way;
    /// when requesting sizes over an arbitrary number of rows, it’s not important
    /// to check the `minimum_height` and `natural_height` of this call but rather to
    /// consult `gtk.CellAreaContext.getPreferredHeight` after a series of
    /// requests.
    extern fn gtk_cell_area_get_preferred_height(p_area: *CellArea, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) void;
    pub const getPreferredHeight = gtk_cell_area_get_preferred_height;

    /// Retrieves a cell area’s minimum and natural height if it would be given
    /// the specified `width`.
    ///
    /// `area` stores some geometrical information in `context` along the way
    /// while calling `gtk.CellArea.getPreferredWidth`. It’s important to
    /// perform a series of `gtk.CellArea.getPreferredWidth` requests with
    /// `context` first and then call `gtk.CellArea.getPreferredHeightForWidth`
    /// on each cell area individually to get the height for width of each
    /// fully requested row.
    ///
    /// If at some point, the width of a single row changes, it should be
    /// requested with `gtk.CellArea.getPreferredWidth` again and then
    /// the full width of the requested rows checked again with
    /// `gtk.CellAreaContext.getPreferredWidth`.
    extern fn gtk_cell_area_get_preferred_height_for_width(p_area: *CellArea, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_width: c_int, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) void;
    pub const getPreferredHeightForWidth = gtk_cell_area_get_preferred_height_for_width;

    /// Retrieves a cell area’s initial minimum and natural width.
    ///
    /// `area` will store some geometrical information in `context` along the way;
    /// when requesting sizes over an arbitrary number of rows, it’s not important
    /// to check the `minimum_width` and `natural_width` of this call but rather to
    /// consult `gtk.CellAreaContext.getPreferredWidth` after a series of
    /// requests.
    extern fn gtk_cell_area_get_preferred_width(p_area: *CellArea, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) void;
    pub const getPreferredWidth = gtk_cell_area_get_preferred_width;

    /// Retrieves a cell area’s minimum and natural width if it would be given
    /// the specified `height`.
    ///
    /// `area` stores some geometrical information in `context` along the way
    /// while calling `gtk.CellArea.getPreferredHeight`. It’s important to
    /// perform a series of `gtk.CellArea.getPreferredHeight` requests with
    /// `context` first and then call `gtk.CellArea.getPreferredWidthForHeight`
    /// on each cell area individually to get the height for width of each
    /// fully requested row.
    ///
    /// If at some point, the height of a single row changes, it should be
    /// requested with `gtk.CellArea.getPreferredHeight` again and then
    /// the full height of the requested rows checked again with
    /// `gtk.CellAreaContext.getPreferredHeight`.
    extern fn gtk_cell_area_get_preferred_width_for_height(p_area: *CellArea, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_height: c_int, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) void;
    pub const getPreferredWidthForHeight = gtk_cell_area_get_preferred_width_for_height;

    /// Gets whether the area prefers a height-for-width layout
    /// or a width-for-height layout.
    extern fn gtk_cell_area_get_request_mode(p_area: *CellArea) gtk.SizeRequestMode;
    pub const getRequestMode = gtk_cell_area_get_request_mode;

    /// Checks if `area` contains `renderer`.
    extern fn gtk_cell_area_has_renderer(p_area: *CellArea, p_renderer: *gtk.CellRenderer) c_int;
    pub const hasRenderer = gtk_cell_area_has_renderer;

    /// This is a convenience function for `gtk.CellArea` implementations
    /// to get the inner area where a given `gtk.CellRenderer` will be
    /// rendered. It removes any padding previously added by `gtk.CellArea.requestRenderer`.
    extern fn gtk_cell_area_inner_cell_area(p_area: *CellArea, p_widget: *gtk.Widget, p_cell_area: *const gdk.Rectangle, p_inner_area: *gdk.Rectangle) void;
    pub const innerCellArea = gtk_cell_area_inner_cell_area;

    /// Returns whether the area can do anything when activated,
    /// after applying new attributes to `area`.
    extern fn gtk_cell_area_is_activatable(p_area: *CellArea) c_int;
    pub const isActivatable = gtk_cell_area_is_activatable;

    /// Returns whether `sibling` is one of `renderer`’s focus siblings
    /// (see `gtk.CellArea.addFocusSibling`).
    extern fn gtk_cell_area_is_focus_sibling(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_sibling: *gtk.CellRenderer) c_int;
    pub const isFocusSibling = gtk_cell_area_is_focus_sibling;

    /// Removes `renderer` from `area`.
    extern fn gtk_cell_area_remove(p_area: *CellArea, p_renderer: *gtk.CellRenderer) void;
    pub const remove = gtk_cell_area_remove;

    /// Removes `sibling` from `renderer`’s focus sibling list
    /// (see `gtk.CellArea.addFocusSibling`).
    extern fn gtk_cell_area_remove_focus_sibling(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_sibling: *gtk.CellRenderer) void;
    pub const removeFocusSibling = gtk_cell_area_remove_focus_sibling;

    /// Renders `area`’s cells according to `area`’s layout onto `widget` at
    /// the given coordinates.
    extern fn gtk_cell_area_render(p_area: *CellArea, p_context: *gtk.CellAreaContext, p_widget: *gtk.Widget, p_cr: *cairo.Context, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState, p_paint_focus: c_int) void;
    pub const render = gtk_cell_area_render;

    /// This is a convenience function for `gtk.CellArea` implementations
    /// to request size for cell renderers. It’s important to use this
    /// function to request size and then use `gtk.CellArea.innerCellArea`
    /// at render and event time since this function will add padding
    /// around the cell for focus painting.
    extern fn gtk_cell_area_request_renderer(p_area: *CellArea, p_renderer: *gtk.CellRenderer, p_orientation: gtk.Orientation, p_widget: *gtk.Widget, p_for_size: c_int, p_minimum_size: ?*c_int, p_natural_size: ?*c_int) void;
    pub const requestRenderer = gtk_cell_area_request_renderer;

    /// Explicitly sets the currently focused cell to `renderer`.
    ///
    /// This is generally called by implementations of
    /// `gtk.CellAreaClass.virtual_methods.focus` or `gtk.CellAreaClass.virtual_methods.event`,
    /// however it can also be used to implement functions such
    /// as `gtk.TreeView.setCursorOnCell`.
    extern fn gtk_cell_area_set_focus_cell(p_area: *CellArea, p_renderer: *gtk.CellRenderer) void;
    pub const setFocusCell = gtk_cell_area_set_focus_cell;

    /// Explicitly stops the editing of the currently edited cell.
    ///
    /// If `canceled` is `TRUE`, the currently edited cell renderer
    /// will emit the ::editing-canceled signal, otherwise the
    /// the ::editing-done signal will be emitted on the current
    /// edit widget.
    ///
    /// See `gtk.CellArea.getEditedCell` and `gtk.CellArea.getEditWidget`.
    extern fn gtk_cell_area_stop_editing(p_area: *CellArea, p_canceled: c_int) void;
    pub const stopEditing = gtk_cell_area_stop_editing;

    extern fn gtk_cell_area_get_type() usize;
    pub const getGObjectType = gtk_cell_area_get_type;

    extern fn g_object_ref(p_self: *gtk.CellArea) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellArea) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellArea, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.CellAreaBox` renders cell renderers into a row or a column
/// depending on its `gtk.Orientation`.
///
/// GtkCellAreaBox uses a notion of packing. Packing
/// refers to adding cell renderers with reference to a particular position
/// in a `gtk.CellAreaBox`. There are two reference positions: the
/// start and the end of the box.
/// When the `gtk.CellAreaBox` is oriented in the `GTK_ORIENTATION_VERTICAL`
/// orientation, the start is defined as the top of the box and the end is
/// defined as the bottom. In the `GTK_ORIENTATION_HORIZONTAL` orientation
/// start is defined as the left side and the end is defined as the right
/// side.
///
/// Alignments of `GtkCellRenderers` rendered in adjacent rows can be
/// configured by configuring the `gtk.CellAreaBox` align child cell property
/// with `gtk.CellArea.cellSetProperty` or by specifying the "align"
/// argument to `gtk.CellAreaBox.packStart` and `gtk.CellAreaBox.packEnd`.
pub const CellAreaBox = extern struct {
    pub const Parent = gtk.CellArea;
    pub const Implements = [_]type{ gtk.Buildable, gtk.CellLayout, gtk.Orientable };
    pub const Class = gtk.CellAreaBoxClass;
    f_parent_instance: gtk.CellArea,
    f_priv: ?*gtk.CellAreaBoxPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The amount of space to reserve between cells.
        pub const spacing = struct {
            pub const name = "spacing";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.CellAreaBox`.
    extern fn gtk_cell_area_box_new() *gtk.CellAreaBox;
    pub const new = gtk_cell_area_box_new;

    /// Gets the spacing added between cell renderers.
    extern fn gtk_cell_area_box_get_spacing(p_box: *CellAreaBox) c_int;
    pub const getSpacing = gtk_cell_area_box_get_spacing;

    /// Adds `renderer` to `box`, packed with reference to the end of `box`.
    ///
    /// The `renderer` is packed after (away from end of) any other
    /// `gtk.CellRenderer` packed with reference to the end of `box`.
    extern fn gtk_cell_area_box_pack_end(p_box: *CellAreaBox, p_renderer: *gtk.CellRenderer, p_expand: c_int, p_align: c_int, p_fixed: c_int) void;
    pub const packEnd = gtk_cell_area_box_pack_end;

    /// Adds `renderer` to `box`, packed with reference to the start of `box`.
    ///
    /// The `renderer` is packed after any other `gtk.CellRenderer` packed
    /// with reference to the start of `box`.
    extern fn gtk_cell_area_box_pack_start(p_box: *CellAreaBox, p_renderer: *gtk.CellRenderer, p_expand: c_int, p_align: c_int, p_fixed: c_int) void;
    pub const packStart = gtk_cell_area_box_pack_start;

    /// Sets the spacing to add between cell renderers in `box`.
    extern fn gtk_cell_area_box_set_spacing(p_box: *CellAreaBox, p_spacing: c_int) void;
    pub const setSpacing = gtk_cell_area_box_set_spacing;

    extern fn gtk_cell_area_box_get_type() usize;
    pub const getGObjectType = gtk_cell_area_box_get_type;

    extern fn g_object_ref(p_self: *gtk.CellAreaBox) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellAreaBox) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellAreaBox, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.CellAreaContext` object is created by a given `gtk.CellArea`
/// implementation via its `gtk.CellAreaClass.virtual_methods.create_context` virtual
/// method and is used to store cell sizes and alignments for a series of
/// `gtk.TreeModel` rows that are requested and rendered in the same context.
///
/// `gtk.CellLayout` widgets can create any number of contexts in which to
/// request and render groups of data rows. However, it’s important that the
/// same context which was used to request sizes for a given `gtk.TreeModel`
/// row also be used for the same row when calling other `gtk.CellArea` APIs
/// such as `gtk.CellArea.render` and `gtk.CellArea.event`.
pub const CellAreaContext = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtk.CellAreaContextClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gtk.CellAreaContextPrivate,

    pub const virtual_methods = struct {
        /// Allocates a width and/or a height for all rows which are to be
        /// rendered with `context`.
        ///
        /// Usually allocation is performed only horizontally or sometimes
        /// vertically since a group of rows are usually rendered side by
        /// side vertically or horizontally and share either the same width
        /// or the same height. Sometimes they are allocated in both horizontal
        /// and vertical orientations producing a homogeneous effect of the
        /// rows. This is generally the case for `gtk.TreeView` when
        /// `gtk.TreeView.properties.fixed`-height-mode is enabled.
        ///
        /// Since 3.0
        pub const allocate = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_width: c_int, p_height: c_int) void {
                return gobject.ext.as(CellAreaContext.Class, p_class).f_allocate.?(gobject.ext.as(CellAreaContext, p_context), p_width, p_height);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_width: c_int, p_height: c_int) callconv(.C) void) void {
                gobject.ext.as(CellAreaContext.Class, p_class).f_allocate = @ptrCast(p_implementation);
            }
        };

        /// Gets the accumulative preferred height for `width` for all rows
        /// which have been requested for the same said `width` with this context.
        ///
        /// After `gtk.CellAreaContext.reset` is called and/or before ever
        /// requesting the size of a `gtk.CellArea`, the returned values are -1.
        pub const get_preferred_height_for_width = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_width: c_int, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) void {
                return gobject.ext.as(CellAreaContext.Class, p_class).f_get_preferred_height_for_width.?(gobject.ext.as(CellAreaContext, p_context), p_width, p_minimum_height, p_natural_height);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_width: c_int, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) callconv(.C) void) void {
                gobject.ext.as(CellAreaContext.Class, p_class).f_get_preferred_height_for_width = @ptrCast(p_implementation);
            }
        };

        /// Gets the accumulative preferred width for `height` for all rows which
        /// have been requested for the same said `height` with this context.
        ///
        /// After `gtk.CellAreaContext.reset` is called and/or before ever
        /// requesting the size of a `gtk.CellArea`, the returned values are -1.
        pub const get_preferred_width_for_height = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_height: c_int, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) void {
                return gobject.ext.as(CellAreaContext.Class, p_class).f_get_preferred_width_for_height.?(gobject.ext.as(CellAreaContext, p_context), p_height, p_minimum_width, p_natural_width);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_height: c_int, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) callconv(.C) void) void {
                gobject.ext.as(CellAreaContext.Class, p_class).f_get_preferred_width_for_height = @ptrCast(p_implementation);
            }
        };

        /// Resets any previously cached request and allocation
        /// data.
        ///
        /// When underlying `gtk.TreeModel` data changes its
        /// important to reset the context if the content
        /// size is allowed to shrink. If the content size
        /// is only allowed to grow (this is usually an option
        /// for views rendering large data stores as a measure
        /// of optimization), then only the row that changed
        /// or was inserted needs to be (re)requested with
        /// `gtk.CellArea.getPreferredWidth`.
        ///
        /// When the new overall size of the context requires
        /// that the allocated size changes (or whenever this
        /// allocation changes at all), the variable row
        /// sizes need to be re-requested for every row.
        ///
        /// For instance, if the rows are displayed all with
        /// the same width from top to bottom then a change
        /// in the allocated width necessitates a recalculation
        /// of all the displayed row heights using
        /// `gtk.CellArea.getPreferredHeightForWidth`.
        ///
        /// Since 3.0
        pub const reset = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(CellAreaContext.Class, p_class).f_reset.?(gobject.ext.as(CellAreaContext, p_context));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(CellAreaContext.Class, p_class).f_reset = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The `gtk.CellArea` this context was created by
        pub const area = struct {
            pub const name = "area";

            pub const Type = ?*gtk.CellArea;
        };

        /// The minimum height for the `gtk.CellArea` in this context
        /// for all `gtk.TreeModel` rows that this context was requested
        /// for using `gtk.CellArea.getPreferredHeight`.
        pub const minimum_height = struct {
            pub const name = "minimum-height";

            pub const Type = c_int;
        };

        /// The minimum width for the `gtk.CellArea` in this context
        /// for all `gtk.TreeModel` rows that this context was requested
        /// for using `gtk.CellArea.getPreferredWidth`.
        pub const minimum_width = struct {
            pub const name = "minimum-width";

            pub const Type = c_int;
        };

        /// The natural height for the `gtk.CellArea` in this context
        /// for all `gtk.TreeModel` rows that this context was requested
        /// for using `gtk.CellArea.getPreferredHeight`.
        pub const natural_height = struct {
            pub const name = "natural-height";

            pub const Type = c_int;
        };

        /// The natural width for the `gtk.CellArea` in this context
        /// for all `gtk.TreeModel` rows that this context was requested
        /// for using `gtk.CellArea.getPreferredWidth`.
        pub const natural_width = struct {
            pub const name = "natural-width";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Allocates a width and/or a height for all rows which are to be
    /// rendered with `context`.
    ///
    /// Usually allocation is performed only horizontally or sometimes
    /// vertically since a group of rows are usually rendered side by
    /// side vertically or horizontally and share either the same width
    /// or the same height. Sometimes they are allocated in both horizontal
    /// and vertical orientations producing a homogeneous effect of the
    /// rows. This is generally the case for `gtk.TreeView` when
    /// `gtk.TreeView.properties.fixed`-height-mode is enabled.
    ///
    /// Since 3.0
    extern fn gtk_cell_area_context_allocate(p_context: *CellAreaContext, p_width: c_int, p_height: c_int) void;
    pub const allocate = gtk_cell_area_context_allocate;

    /// Fetches the current allocation size for `context`.
    ///
    /// If the context was not allocated in width or height, or if the
    /// context was recently reset with `gtk.CellAreaContext.reset`,
    /// the returned value will be -1.
    extern fn gtk_cell_area_context_get_allocation(p_context: *CellAreaContext, p_width: ?*c_int, p_height: ?*c_int) void;
    pub const getAllocation = gtk_cell_area_context_get_allocation;

    /// Fetches the `gtk.CellArea` this `context` was created by.
    ///
    /// This is generally unneeded by layouting widgets; however,
    /// it is important for the context implementation itself to
    /// fetch information about the area it is being used for.
    ///
    /// For instance at `gtk.CellAreaContextClass.virtual_methods.allocate` time
    /// it’s important to know details about any cell spacing
    /// that the `gtk.CellArea` is configured with in order to
    /// compute a proper allocation.
    extern fn gtk_cell_area_context_get_area(p_context: *CellAreaContext) *gtk.CellArea;
    pub const getArea = gtk_cell_area_context_get_area;

    /// Gets the accumulative preferred height for all rows which have been
    /// requested with this context.
    ///
    /// After `gtk.CellAreaContext.reset` is called and/or before ever
    /// requesting the size of a `gtk.CellArea`, the returned values are 0.
    extern fn gtk_cell_area_context_get_preferred_height(p_context: *CellAreaContext, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) void;
    pub const getPreferredHeight = gtk_cell_area_context_get_preferred_height;

    /// Gets the accumulative preferred height for `width` for all rows
    /// which have been requested for the same said `width` with this context.
    ///
    /// After `gtk.CellAreaContext.reset` is called and/or before ever
    /// requesting the size of a `gtk.CellArea`, the returned values are -1.
    extern fn gtk_cell_area_context_get_preferred_height_for_width(p_context: *CellAreaContext, p_width: c_int, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) void;
    pub const getPreferredHeightForWidth = gtk_cell_area_context_get_preferred_height_for_width;

    /// Gets the accumulative preferred width for all rows which have been
    /// requested with this context.
    ///
    /// After `gtk.CellAreaContext.reset` is called and/or before ever
    /// requesting the size of a `gtk.CellArea`, the returned values are 0.
    extern fn gtk_cell_area_context_get_preferred_width(p_context: *CellAreaContext, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) void;
    pub const getPreferredWidth = gtk_cell_area_context_get_preferred_width;

    /// Gets the accumulative preferred width for `height` for all rows which
    /// have been requested for the same said `height` with this context.
    ///
    /// After `gtk.CellAreaContext.reset` is called and/or before ever
    /// requesting the size of a `gtk.CellArea`, the returned values are -1.
    extern fn gtk_cell_area_context_get_preferred_width_for_height(p_context: *CellAreaContext, p_height: c_int, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) void;
    pub const getPreferredWidthForHeight = gtk_cell_area_context_get_preferred_width_for_height;

    /// Causes the minimum and/or natural height to grow if the new
    /// proposed sizes exceed the current minimum and natural height.
    ///
    /// This is used by `gtk.CellAreaContext` implementations during
    /// the request process over a series of `gtk.TreeModel` rows to
    /// progressively push the requested height over a series of
    /// `gtk.CellArea.getPreferredHeight` requests.
    extern fn gtk_cell_area_context_push_preferred_height(p_context: *CellAreaContext, p_minimum_height: c_int, p_natural_height: c_int) void;
    pub const pushPreferredHeight = gtk_cell_area_context_push_preferred_height;

    /// Causes the minimum and/or natural width to grow if the new
    /// proposed sizes exceed the current minimum and natural width.
    ///
    /// This is used by `gtk.CellAreaContext` implementations during
    /// the request process over a series of `gtk.TreeModel` rows to
    /// progressively push the requested width over a series of
    /// `gtk.CellArea.getPreferredWidth` requests.
    extern fn gtk_cell_area_context_push_preferred_width(p_context: *CellAreaContext, p_minimum_width: c_int, p_natural_width: c_int) void;
    pub const pushPreferredWidth = gtk_cell_area_context_push_preferred_width;

    /// Resets any previously cached request and allocation
    /// data.
    ///
    /// When underlying `gtk.TreeModel` data changes its
    /// important to reset the context if the content
    /// size is allowed to shrink. If the content size
    /// is only allowed to grow (this is usually an option
    /// for views rendering large data stores as a measure
    /// of optimization), then only the row that changed
    /// or was inserted needs to be (re)requested with
    /// `gtk.CellArea.getPreferredWidth`.
    ///
    /// When the new overall size of the context requires
    /// that the allocated size changes (or whenever this
    /// allocation changes at all), the variable row
    /// sizes need to be re-requested for every row.
    ///
    /// For instance, if the rows are displayed all with
    /// the same width from top to bottom then a change
    /// in the allocated width necessitates a recalculation
    /// of all the displayed row heights using
    /// `gtk.CellArea.getPreferredHeightForWidth`.
    ///
    /// Since 3.0
    extern fn gtk_cell_area_context_reset(p_context: *CellAreaContext) void;
    pub const reset = gtk_cell_area_context_reset;

    extern fn gtk_cell_area_context_get_type() usize;
    pub const getGObjectType = gtk_cell_area_context_get_type;

    extern fn g_object_ref(p_self: *gtk.CellAreaContext) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellAreaContext) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellAreaContext, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.CellRenderer` is a base class of a set of objects used for
/// rendering a cell to a `cairo.Context`.  These objects are used primarily by
/// the `gtk.TreeView` widget, though they aren’t tied to them in any
/// specific way.  It is worth noting that `gtk.CellRenderer` is not a
/// `gtk.Widget` and cannot be treated as such.
///
/// The primary use of a `gtk.CellRenderer` is for drawing a certain graphical
/// elements on a `cairo.Context`. Typically, one cell renderer is used to
/// draw many cells on the screen.  To this extent, it isn’t expected that a
/// CellRenderer keep any permanent state around.  Instead, any state is set
/// just prior to use using `GObjects` property system.  Then, the
/// cell is measured using `gtk.CellRenderer.getSize`. Finally, the cell
/// is rendered in the correct location using `gtk.CellRenderer.render`.
///
/// There are a number of rules that must be followed when writing a new
/// `gtk.CellRenderer`.  First and foremost, it’s important that a certain set
/// of properties will always yield a cell renderer of the same size,
/// barring a `gtk.Style` change.  The `gtk.CellRenderer` also has a number of
/// generic properties that are expected to be honored by all children.
///
/// Beyond merely rendering a cell, cell renderers can optionally
/// provide active user interface elements. A cell renderer can be
/// “activatable” like `gtk.CellRendererToggle`,
/// which toggles when it gets activated by a mouse click, or it can be
/// “editable” like `gtk.CellRendererText`, which
/// allows the user to edit the text using a widget implementing the
/// `gtk.CellEditable` interface, e.g. `gtk.Entry`.
/// To make a cell renderer activatable or editable, you have to
/// implement the `gtk.CellRendererClass.activate` or
/// `gtk.CellRendererClass.start_editing` virtual functions, respectively.
///
/// Many properties of `gtk.CellRenderer` and its subclasses have a
/// corresponding “set” property, e.g. “cell-background-set” corresponds
/// to “cell-background”. These “set” properties reflect whether a property
/// has been set or not. You should not set them independently.
pub const CellRenderer = extern struct {
    pub const Parent = gobject.InitiallyUnowned;
    pub const Implements = [_]type{};
    pub const Class = gtk.CellRendererClass;
    f_parent_instance: gobject.InitiallyUnowned,
    f_priv: ?*gtk.CellRendererPrivate,

    pub const virtual_methods = struct {
        /// Passes an activate event to the cell renderer for possible processing.
        /// Some cell renderers may use events; for example, `gtk.CellRendererToggle`
        /// toggles when it gets a mouse click.
        pub const activate = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: *gdk.Event, p_widget: *gtk.Widget, p_path: [*:0]const u8, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) c_int {
                return gobject.ext.as(CellRenderer.Class, p_class).f_activate.?(gobject.ext.as(CellRenderer, p_cell), p_event, p_widget, p_path, p_background_area, p_cell_area, p_flags);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: *gdk.Event, p_widget: *gtk.Widget, p_path: [*:0]const u8, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) callconv(.C) c_int) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_activate = @ptrCast(p_implementation);
            }
        };

        /// Signal gets emitted when the user cancels the process of editing a cell.
        pub const editing_canceled = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(CellRenderer.Class, p_class).f_editing_canceled.?(gobject.ext.as(CellRenderer, p_cell));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_editing_canceled = @ptrCast(p_implementation);
            }
        };

        /// Signal gets emitted when a cell starts to be edited.
        pub const editing_started = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_editable: *gtk.CellEditable, p_path: [*:0]const u8) void {
                return gobject.ext.as(CellRenderer.Class, p_class).f_editing_started.?(gobject.ext.as(CellRenderer, p_cell), p_editable, p_path);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_editable: *gtk.CellEditable, p_path: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_editing_started = @ptrCast(p_implementation);
            }
        };

        /// Gets the aligned area used by `cell` inside `cell_area`. Used for finding
        /// the appropriate edit and focus rectangle.
        pub const get_aligned_area = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_flags: gtk.CellRendererState, p_cell_area: *const gdk.Rectangle, p_aligned_area: *gdk.Rectangle) void {
                return gobject.ext.as(CellRenderer.Class, p_class).f_get_aligned_area.?(gobject.ext.as(CellRenderer, p_cell), p_widget, p_flags, p_cell_area, p_aligned_area);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_flags: gtk.CellRendererState, p_cell_area: *const gdk.Rectangle, p_aligned_area: *gdk.Rectangle) callconv(.C) void) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_get_aligned_area = @ptrCast(p_implementation);
            }
        };

        /// Retreives a renderer’s natural size when rendered to `widget`.
        pub const get_preferred_height = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_minimum_size: ?*c_int, p_natural_size: ?*c_int) void {
                return gobject.ext.as(CellRenderer.Class, p_class).f_get_preferred_height.?(gobject.ext.as(CellRenderer, p_cell), p_widget, p_minimum_size, p_natural_size);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_minimum_size: ?*c_int, p_natural_size: ?*c_int) callconv(.C) void) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_get_preferred_height = @ptrCast(p_implementation);
            }
        };

        /// Retreives a cell renderers’s minimum and natural height if it were rendered to
        /// `widget` with the specified `width`.
        pub const get_preferred_height_for_width = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_width: c_int, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) void {
                return gobject.ext.as(CellRenderer.Class, p_class).f_get_preferred_height_for_width.?(gobject.ext.as(CellRenderer, p_cell), p_widget, p_width, p_minimum_height, p_natural_height);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_width: c_int, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) callconv(.C) void) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_get_preferred_height_for_width = @ptrCast(p_implementation);
            }
        };

        /// Retreives a renderer’s natural size when rendered to `widget`.
        pub const get_preferred_width = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_minimum_size: ?*c_int, p_natural_size: ?*c_int) void {
                return gobject.ext.as(CellRenderer.Class, p_class).f_get_preferred_width.?(gobject.ext.as(CellRenderer, p_cell), p_widget, p_minimum_size, p_natural_size);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_minimum_size: ?*c_int, p_natural_size: ?*c_int) callconv(.C) void) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_get_preferred_width = @ptrCast(p_implementation);
            }
        };

        /// Retreives a cell renderers’s minimum and natural width if it were rendered to
        /// `widget` with the specified `height`.
        pub const get_preferred_width_for_height = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_height: c_int, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) void {
                return gobject.ext.as(CellRenderer.Class, p_class).f_get_preferred_width_for_height.?(gobject.ext.as(CellRenderer, p_cell), p_widget, p_height, p_minimum_width, p_natural_width);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_height: c_int, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) callconv(.C) void) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_get_preferred_width_for_height = @ptrCast(p_implementation);
            }
        };

        /// Gets whether the cell renderer prefers a height-for-width layout
        /// or a width-for-height layout.
        pub const get_request_mode = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) gtk.SizeRequestMode {
                return gobject.ext.as(CellRenderer.Class, p_class).f_get_request_mode.?(gobject.ext.as(CellRenderer, p_cell));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) gtk.SizeRequestMode) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_get_request_mode = @ptrCast(p_implementation);
            }
        };

        /// Obtains the width and height needed to render the cell. Used by view
        /// widgets to determine the appropriate size for the cell_area passed to
        /// `gtk.CellRenderer.render`.  If `cell_area` is not `NULL`, fills in the
        /// x and y offsets (if set) of the cell relative to this location.
        ///
        /// Please note that the values set in `width` and `height`, as well as those
        /// in `x_offset` and `y_offset` are inclusive of the xpad and ypad properties.
        pub const get_size = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_cell_area: ?*const gdk.Rectangle, p_x_offset: ?*c_int, p_y_offset: ?*c_int, p_width: ?*c_int, p_height: ?*c_int) void {
                return gobject.ext.as(CellRenderer.Class, p_class).f_get_size.?(gobject.ext.as(CellRenderer, p_cell), p_widget, p_cell_area, p_x_offset, p_y_offset, p_width, p_height);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget, p_cell_area: ?*const gdk.Rectangle, p_x_offset: ?*c_int, p_y_offset: ?*c_int, p_width: ?*c_int, p_height: ?*c_int) callconv(.C) void) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_get_size = @ptrCast(p_implementation);
            }
        };

        /// Invokes the virtual render function of the `gtk.CellRenderer`. The three
        /// passed-in rectangles are areas in `cr`. Most renderers will draw within
        /// `cell_area`; the xalign, yalign, xpad, and ypad fields of the `gtk.CellRenderer`
        /// should be honored with respect to `cell_area`. `background_area` includes the
        /// blank space around the cell, and also the area containing the tree expander;
        /// so the `background_area` rectangles for all cells tile to cover the entire
        /// `window`.
        pub const render = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cr: *cairo.Context, p_widget: *gtk.Widget, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) void {
                return gobject.ext.as(CellRenderer.Class, p_class).f_render.?(gobject.ext.as(CellRenderer, p_cell), p_cr, p_widget, p_background_area, p_cell_area, p_flags);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cr: *cairo.Context, p_widget: *gtk.Widget, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) callconv(.C) void) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_render = @ptrCast(p_implementation);
            }
        };

        /// Starts editing the contents of this `cell`, through a new `gtk.CellEditable`
        /// widget created by the `gtk.CellRendererClass.start_editing` virtual function.
        pub const start_editing = struct {
            pub fn call(p_class: anytype, p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: ?*gdk.Event, p_widget: *gtk.Widget, p_path: [*:0]const u8, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) ?*gtk.CellEditable {
                return gobject.ext.as(CellRenderer.Class, p_class).f_start_editing.?(gobject.ext.as(CellRenderer, p_cell), p_event, p_widget, p_path, p_background_area, p_cell_area, p_flags);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: ?*gdk.Event, p_widget: *gtk.Widget, p_path: [*:0]const u8, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) callconv(.C) ?*gtk.CellEditable) void {
                gobject.ext.as(CellRenderer.Class, p_class).f_start_editing = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const cell_background = struct {
            pub const name = "cell-background";

            pub const Type = ?[*:0]u8;
        };

        /// Cell background as a `gdk.Color`
        pub const cell_background_gdk = struct {
            pub const name = "cell-background-gdk";

            pub const Type = ?*gdk.Color;
        };

        /// Cell background as a `gdk.RGBA`
        pub const cell_background_rgba = struct {
            pub const name = "cell-background-rgba";

            pub const Type = ?*gdk.RGBA;
        };

        pub const cell_background_set = struct {
            pub const name = "cell-background-set";

            pub const Type = c_int;
        };

        pub const editing = struct {
            pub const name = "editing";

            pub const Type = c_int;
        };

        pub const height = struct {
            pub const name = "height";

            pub const Type = c_int;
        };

        pub const is_expanded = struct {
            pub const name = "is-expanded";

            pub const Type = c_int;
        };

        pub const is_expander = struct {
            pub const name = "is-expander";

            pub const Type = c_int;
        };

        pub const mode = struct {
            pub const name = "mode";

            pub const Type = gtk.CellRendererMode;
        };

        pub const sensitive = struct {
            pub const name = "sensitive";

            pub const Type = c_int;
        };

        pub const visible = struct {
            pub const name = "visible";

            pub const Type = c_int;
        };

        pub const width = struct {
            pub const name = "width";

            pub const Type = c_int;
        };

        pub const xalign = struct {
            pub const name = "xalign";

            pub const Type = f32;
        };

        pub const xpad = struct {
            pub const name = "xpad";

            pub const Type = c_uint;
        };

        pub const yalign = struct {
            pub const name = "yalign";

            pub const Type = f32;
        };

        pub const ypad = struct {
            pub const name = "ypad";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {
        /// This signal gets emitted when the user cancels the process of editing a
        /// cell.  For example, an editable cell renderer could be written to cancel
        /// editing when the user presses Escape.
        ///
        /// See also: `gtk.CellRenderer.stopEditing`.
        pub const editing_canceled = struct {
            pub const name = "editing-canceled";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CellRenderer, p_instance))),
                    gobject.signalLookup("editing-canceled", CellRenderer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal gets emitted when a cell starts to be edited.
        /// The intended use of this signal is to do special setup
        /// on `editable`, e.g. adding a `gtk.EntryCompletion` or setting
        /// up additional columns in a `gtk.ComboBox`.
        ///
        /// See `gtk.CellEditable.startEditing` for information on the lifecycle of
        /// the `editable` and a way to do setup that doesn’t depend on the `renderer`.
        ///
        /// Note that GTK+ doesn't guarantee that cell renderers will
        /// continue to use the same kind of widget for editing in future
        /// releases, therefore you should check the type of `editable`
        /// before doing any specific setup, as in the following example:
        /// ```
        /// static void
        /// text_editing_started (GtkCellRenderer *cell,
        ///                       GtkCellEditable *editable,
        ///                       const gchar     *path,
        ///                       gpointer         data)
        /// {
        ///   if (GTK_IS_ENTRY (editable))
        ///     {
        ///       GtkEntry *entry = GTK_ENTRY (editable);
        ///
        ///       // ... create a GtkEntryCompletion
        ///
        ///       gtk_entry_set_completion (entry, completion);
        ///     }
        /// }
        /// ```
        pub const editing_started = struct {
            pub const name = "editing-started";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_editable: *gtk.CellEditable, p_path: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CellRenderer, p_instance))),
                    gobject.signalLookup("editing-started", CellRenderer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Passes an activate event to the cell renderer for possible processing.
    /// Some cell renderers may use events; for example, `gtk.CellRendererToggle`
    /// toggles when it gets a mouse click.
    extern fn gtk_cell_renderer_activate(p_cell: *CellRenderer, p_event: *gdk.Event, p_widget: *gtk.Widget, p_path: [*:0]const u8, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) c_int;
    pub const activate = gtk_cell_renderer_activate;

    /// Gets the aligned area used by `cell` inside `cell_area`. Used for finding
    /// the appropriate edit and focus rectangle.
    extern fn gtk_cell_renderer_get_aligned_area(p_cell: *CellRenderer, p_widget: *gtk.Widget, p_flags: gtk.CellRendererState, p_cell_area: *const gdk.Rectangle, p_aligned_area: *gdk.Rectangle) void;
    pub const getAlignedArea = gtk_cell_renderer_get_aligned_area;

    /// Fills in `xalign` and `yalign` with the appropriate values of `cell`.
    extern fn gtk_cell_renderer_get_alignment(p_cell: *CellRenderer, p_xalign: ?*f32, p_yalign: ?*f32) void;
    pub const getAlignment = gtk_cell_renderer_get_alignment;

    /// Fills in `width` and `height` with the appropriate size of `cell`.
    extern fn gtk_cell_renderer_get_fixed_size(p_cell: *CellRenderer, p_width: ?*c_int, p_height: ?*c_int) void;
    pub const getFixedSize = gtk_cell_renderer_get_fixed_size;

    /// Fills in `xpad` and `ypad` with the appropriate values of `cell`.
    extern fn gtk_cell_renderer_get_padding(p_cell: *CellRenderer, p_xpad: ?*c_int, p_ypad: ?*c_int) void;
    pub const getPadding = gtk_cell_renderer_get_padding;

    /// Retreives a renderer’s natural size when rendered to `widget`.
    extern fn gtk_cell_renderer_get_preferred_height(p_cell: *CellRenderer, p_widget: *gtk.Widget, p_minimum_size: ?*c_int, p_natural_size: ?*c_int) void;
    pub const getPreferredHeight = gtk_cell_renderer_get_preferred_height;

    /// Retreives a cell renderers’s minimum and natural height if it were rendered to
    /// `widget` with the specified `width`.
    extern fn gtk_cell_renderer_get_preferred_height_for_width(p_cell: *CellRenderer, p_widget: *gtk.Widget, p_width: c_int, p_minimum_height: ?*c_int, p_natural_height: ?*c_int) void;
    pub const getPreferredHeightForWidth = gtk_cell_renderer_get_preferred_height_for_width;

    /// Retrieves the minimum and natural size of a cell taking
    /// into account the widget’s preference for height-for-width management.
    extern fn gtk_cell_renderer_get_preferred_size(p_cell: *CellRenderer, p_widget: *gtk.Widget, p_minimum_size: ?*gtk.Requisition, p_natural_size: ?*gtk.Requisition) void;
    pub const getPreferredSize = gtk_cell_renderer_get_preferred_size;

    /// Retreives a renderer’s natural size when rendered to `widget`.
    extern fn gtk_cell_renderer_get_preferred_width(p_cell: *CellRenderer, p_widget: *gtk.Widget, p_minimum_size: ?*c_int, p_natural_size: ?*c_int) void;
    pub const getPreferredWidth = gtk_cell_renderer_get_preferred_width;

    /// Retreives a cell renderers’s minimum and natural width if it were rendered to
    /// `widget` with the specified `height`.
    extern fn gtk_cell_renderer_get_preferred_width_for_height(p_cell: *CellRenderer, p_widget: *gtk.Widget, p_height: c_int, p_minimum_width: ?*c_int, p_natural_width: ?*c_int) void;
    pub const getPreferredWidthForHeight = gtk_cell_renderer_get_preferred_width_for_height;

    /// Gets whether the cell renderer prefers a height-for-width layout
    /// or a width-for-height layout.
    extern fn gtk_cell_renderer_get_request_mode(p_cell: *CellRenderer) gtk.SizeRequestMode;
    pub const getRequestMode = gtk_cell_renderer_get_request_mode;

    /// Returns the cell renderer’s sensitivity.
    extern fn gtk_cell_renderer_get_sensitive(p_cell: *CellRenderer) c_int;
    pub const getSensitive = gtk_cell_renderer_get_sensitive;

    /// Obtains the width and height needed to render the cell. Used by view
    /// widgets to determine the appropriate size for the cell_area passed to
    /// `gtk.CellRenderer.render`.  If `cell_area` is not `NULL`, fills in the
    /// x and y offsets (if set) of the cell relative to this location.
    ///
    /// Please note that the values set in `width` and `height`, as well as those
    /// in `x_offset` and `y_offset` are inclusive of the xpad and ypad properties.
    extern fn gtk_cell_renderer_get_size(p_cell: *CellRenderer, p_widget: *gtk.Widget, p_cell_area: ?*const gdk.Rectangle, p_x_offset: ?*c_int, p_y_offset: ?*c_int, p_width: ?*c_int, p_height: ?*c_int) void;
    pub const getSize = gtk_cell_renderer_get_size;

    /// Translates the cell renderer state to `gtk.StateFlags`,
    /// based on the cell renderer and widget sensitivity, and
    /// the given `gtk.CellRendererState`.
    extern fn gtk_cell_renderer_get_state(p_cell: ?*CellRenderer, p_widget: ?*gtk.Widget, p_cell_state: gtk.CellRendererState) gtk.StateFlags;
    pub const getState = gtk_cell_renderer_get_state;

    /// Returns the cell renderer’s visibility.
    extern fn gtk_cell_renderer_get_visible(p_cell: *CellRenderer) c_int;
    pub const getVisible = gtk_cell_renderer_get_visible;

    /// Checks whether the cell renderer can do something when activated.
    extern fn gtk_cell_renderer_is_activatable(p_cell: *CellRenderer) c_int;
    pub const isActivatable = gtk_cell_renderer_is_activatable;

    /// Invokes the virtual render function of the `gtk.CellRenderer`. The three
    /// passed-in rectangles are areas in `cr`. Most renderers will draw within
    /// `cell_area`; the xalign, yalign, xpad, and ypad fields of the `gtk.CellRenderer`
    /// should be honored with respect to `cell_area`. `background_area` includes the
    /// blank space around the cell, and also the area containing the tree expander;
    /// so the `background_area` rectangles for all cells tile to cover the entire
    /// `window`.
    extern fn gtk_cell_renderer_render(p_cell: *CellRenderer, p_cr: *cairo.Context, p_widget: *gtk.Widget, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) void;
    pub const render = gtk_cell_renderer_render;

    /// Sets the renderer’s alignment within its available space.
    extern fn gtk_cell_renderer_set_alignment(p_cell: *CellRenderer, p_xalign: f32, p_yalign: f32) void;
    pub const setAlignment = gtk_cell_renderer_set_alignment;

    /// Sets the renderer size to be explicit, independent of the properties set.
    extern fn gtk_cell_renderer_set_fixed_size(p_cell: *CellRenderer, p_width: c_int, p_height: c_int) void;
    pub const setFixedSize = gtk_cell_renderer_set_fixed_size;

    /// Sets the renderer’s padding.
    extern fn gtk_cell_renderer_set_padding(p_cell: *CellRenderer, p_xpad: c_int, p_ypad: c_int) void;
    pub const setPadding = gtk_cell_renderer_set_padding;

    /// Sets the cell renderer’s sensitivity.
    extern fn gtk_cell_renderer_set_sensitive(p_cell: *CellRenderer, p_sensitive: c_int) void;
    pub const setSensitive = gtk_cell_renderer_set_sensitive;

    /// Sets the cell renderer’s visibility.
    extern fn gtk_cell_renderer_set_visible(p_cell: *CellRenderer, p_visible: c_int) void;
    pub const setVisible = gtk_cell_renderer_set_visible;

    /// Starts editing the contents of this `cell`, through a new `gtk.CellEditable`
    /// widget created by the `gtk.CellRendererClass.start_editing` virtual function.
    extern fn gtk_cell_renderer_start_editing(p_cell: *CellRenderer, p_event: ?*gdk.Event, p_widget: *gtk.Widget, p_path: [*:0]const u8, p_background_area: *const gdk.Rectangle, p_cell_area: *const gdk.Rectangle, p_flags: gtk.CellRendererState) ?*gtk.CellEditable;
    pub const startEditing = gtk_cell_renderer_start_editing;

    /// Informs the cell renderer that the editing is stopped.
    /// If `canceled` is `TRUE`, the cell renderer will emit the
    /// `gtk.CellRenderer.signals.editing`-canceled signal.
    ///
    /// This function should be called by cell renderer implementations
    /// in response to the `gtk.CellEditable.signals.editing`-done signal of
    /// `gtk.CellEditable`.
    extern fn gtk_cell_renderer_stop_editing(p_cell: *CellRenderer, p_canceled: c_int) void;
    pub const stopEditing = gtk_cell_renderer_stop_editing;

    extern fn gtk_cell_renderer_get_type() usize;
    pub const getGObjectType = gtk_cell_renderer_get_type;

    extern fn g_object_ref(p_self: *gtk.CellRenderer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellRenderer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellRenderer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.CellRendererAccel` displays a keyboard accelerator (i.e. a key
/// combination like `Control + a`). If the cell renderer is editable,
/// the accelerator can be changed by simply typing the new combination.
///
/// The `gtk.CellRendererAccel` cell renderer was added in GTK+ 2.10.
pub const CellRendererAccel = extern struct {
    pub const Parent = gtk.CellRendererText;
    pub const Implements = [_]type{};
    pub const Class = gtk.CellRendererAccelClass;
    f_parent: gtk.CellRendererText,
    f_priv: ?*gtk.CellRendererAccelPrivate,

    pub const virtual_methods = struct {
        pub const accel_cleared = struct {
            pub fn call(p_class: anytype, p_accel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path_string: [*:0]const u8) void {
                return gobject.ext.as(CellRendererAccel.Class, p_class).f_accel_cleared.?(gobject.ext.as(CellRendererAccel, p_accel), p_path_string);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_accel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path_string: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(CellRendererAccel.Class, p_class).f_accel_cleared = @ptrCast(p_implementation);
            }
        };

        pub const accel_edited = struct {
            pub fn call(p_class: anytype, p_accel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path_string: [*:0]const u8, p_accel_key: c_uint, p_accel_mods: gdk.ModifierType, p_hardware_keycode: c_uint) void {
                return gobject.ext.as(CellRendererAccel.Class, p_class).f_accel_edited.?(gobject.ext.as(CellRendererAccel, p_accel), p_path_string, p_accel_key, p_accel_mods, p_hardware_keycode);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_accel: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path_string: [*:0]const u8, p_accel_key: c_uint, p_accel_mods: gdk.ModifierType, p_hardware_keycode: c_uint) callconv(.C) void) void {
                gobject.ext.as(CellRendererAccel.Class, p_class).f_accel_edited = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The keyval of the accelerator.
        pub const accel_key = struct {
            pub const name = "accel-key";

            pub const Type = c_uint;
        };

        /// Determines if the edited accelerators are GTK+ accelerators. If
        /// they are, consumed modifiers are suppressed, only accelerators
        /// accepted by GTK+ are allowed, and the accelerators are rendered
        /// in the same way as they are in menus.
        pub const accel_mode = struct {
            pub const name = "accel-mode";

            pub const Type = gtk.CellRendererAccelMode;
        };

        /// The modifier mask of the accelerator.
        pub const accel_mods = struct {
            pub const name = "accel-mods";

            pub const Type = gdk.ModifierType;
        };

        /// The hardware keycode of the accelerator. Note that the hardware keycode is
        /// only relevant if the key does not have a keyval. Normally, the keyboard
        /// configuration should assign keyvals to all keys.
        pub const keycode = struct {
            pub const name = "keycode";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {
        /// Gets emitted when the user has removed the accelerator.
        pub const accel_cleared = struct {
            pub const name = "accel-cleared";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_path_string: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CellRendererAccel, p_instance))),
                    gobject.signalLookup("accel-cleared", CellRendererAccel.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Gets emitted when the user has selected a new accelerator.
        pub const accel_edited = struct {
            pub const name = "accel-edited";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_path_string: [*:0]u8, p_accel_key: c_uint, p_accel_mods: gdk.ModifierType, p_hardware_keycode: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CellRendererAccel, p_instance))),
                    gobject.signalLookup("accel-edited", CellRendererAccel.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.CellRendererAccel`.
    extern fn gtk_cell_renderer_accel_new() *gtk.CellRendererAccel;
    pub const new = gtk_cell_renderer_accel_new;

    extern fn gtk_cell_renderer_accel_get_type() usize;
    pub const getGObjectType = gtk_cell_renderer_accel_get_type;

    extern fn g_object_ref(p_self: *gtk.CellRendererAccel) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellRendererAccel) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellRendererAccel, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.CellRendererCombo` renders text in a cell like `gtk.CellRendererText` from
/// which it is derived. But while `gtk.CellRendererText` offers a simple entry to
/// edit the text, `gtk.CellRendererCombo` offers a `gtk.ComboBox`
/// widget to edit the text. The values to display in the combo box are taken from
/// the tree model specified in the `gtk.CellRendererCombo.properties.model` property.
///
/// The combo cell renderer takes care of adding a text cell renderer to the combo
/// box and sets it to display the column specified by its
/// `gtk.CellRendererCombo.properties.text`-column property. Further properties of the combo box
/// can be set in a handler for the `gtk.CellRenderer.signals.editing`-started signal.
///
/// The `gtk.CellRendererCombo` cell renderer was added in GTK+ 2.6.
pub const CellRendererCombo = extern struct {
    pub const Parent = gtk.CellRendererText;
    pub const Implements = [_]type{};
    pub const Class = gtk.CellRendererComboClass;
    f_parent: gtk.CellRendererText,
    f_priv: ?*gtk.CellRendererComboPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// If `TRUE`, the cell renderer will include an entry and allow to enter
        /// values other than the ones in the popup list.
        pub const has_entry = struct {
            pub const name = "has-entry";

            pub const Type = c_int;
        };

        /// Holds a tree model containing the possible values for the combo box.
        /// Use the text_column property to specify the column holding the values.
        pub const model = struct {
            pub const name = "model";

            pub const Type = ?*gtk.TreeModel;
        };

        /// Specifies the model column which holds the possible values for the
        /// combo box.
        ///
        /// Note that this refers to the model specified in the model property,
        /// not the model backing the tree view to which
        /// this cell renderer is attached.
        ///
        /// `gtk.CellRendererCombo` automatically adds a text cell renderer for
        /// this column to its combo box.
        pub const text_column = struct {
            pub const name = "text-column";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// This signal is emitted each time after the user selected an item in
        /// the combo box, either by using the mouse or the arrow keys.  Contrary
        /// to GtkComboBox, GtkCellRendererCombo::changed is not emitted for
        /// changes made to a selected item in the entry.  The argument `new_iter`
        /// corresponds to the newly selected item in the combo box and it is relative
        /// to the GtkTreeModel set via the model property on GtkCellRendererCombo.
        ///
        /// Note that as soon as you change the model displayed in the tree view,
        /// the tree view will immediately cease the editing operating.  This
        /// means that you most probably want to refrain from changing the model
        /// until the combo cell renderer emits the edited or editing_canceled signal.
        pub const changed = struct {
            pub const name = "changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_path_string: [*:0]u8, p_new_iter: *gtk.TreeIter, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CellRendererCombo, p_instance))),
                    gobject.signalLookup("changed", CellRendererCombo.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.CellRendererCombo`.
    /// Adjust how text is drawn using object properties.
    /// Object properties can be set globally (with `gobject.Object.set`).
    /// Also, with `gtk.TreeViewColumn`, you can bind a property to a value
    /// in a `gtk.TreeModel`. For example, you can bind the “text” property
    /// on the cell renderer to a string value in the model, thus rendering
    /// a different string in each row of the `gtk.TreeView`.
    extern fn gtk_cell_renderer_combo_new() *gtk.CellRendererCombo;
    pub const new = gtk_cell_renderer_combo_new;

    extern fn gtk_cell_renderer_combo_get_type() usize;
    pub const getGObjectType = gtk_cell_renderer_combo_get_type;

    extern fn g_object_ref(p_self: *gtk.CellRendererCombo) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellRendererCombo) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellRendererCombo, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.CellRendererPixbuf` can be used to render an image in a cell. It allows
/// to render either a given `gdkpixbuf.Pixbuf` (set via the
/// `gtk.CellRendererPixbuf.properties.pixbuf` property) or a named icon (set via the
/// `gtk.CellRendererPixbuf.properties.icon`-name property).
///
/// To support the tree view, `gtk.CellRendererPixbuf` also supports rendering two
/// alternative pixbufs, when the `gtk.CellRenderer.properties.is`-expander property is `TRUE`.
/// If the `gtk.CellRenderer.properties.is`-expanded property is `TRUE` and the
/// `gtk.CellRendererPixbuf.properties.pixbuf`-expander-open property is set to a pixbuf, it
/// renders that pixbuf, if the `gtk.CellRenderer.properties.is`-expanded property is `FALSE`
/// and the `gtk.CellRendererPixbuf.properties.pixbuf`-expander-closed property is set to a
/// pixbuf, it renders that one.
pub const CellRendererPixbuf = extern struct {
    pub const Parent = gtk.CellRenderer;
    pub const Implements = [_]type{};
    pub const Class = gtk.CellRendererPixbufClass;
    f_parent: gtk.CellRenderer,
    f_priv: ?*gtk.CellRendererPixbufPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Specifies whether the rendered pixbuf should be colorized
        /// according to the `gtk.CellRendererState`.
        pub const follow_state = struct {
            pub const name = "follow-state";

            pub const Type = c_int;
        };

        /// The GIcon representing the icon to display.
        /// If the icon theme is changed, the image will be updated
        /// automatically.
        pub const gicon = struct {
            pub const name = "gicon";

            pub const Type = ?*gio.Icon;
        };

        /// The name of the themed icon to display.
        /// This property only has an effect if not overridden by "stock_id"
        /// or "pixbuf" properties.
        pub const icon_name = struct {
            pub const name = "icon-name";

            pub const Type = ?[*:0]u8;
        };

        pub const pixbuf = struct {
            pub const name = "pixbuf";

            pub const Type = ?*gdkpixbuf.Pixbuf;
        };

        pub const pixbuf_expander_closed = struct {
            pub const name = "pixbuf-expander-closed";

            pub const Type = ?*gdkpixbuf.Pixbuf;
        };

        pub const pixbuf_expander_open = struct {
            pub const name = "pixbuf-expander-open";

            pub const Type = ?*gdkpixbuf.Pixbuf;
        };

        pub const stock_detail = struct {
            pub const name = "stock-detail";

            pub const Type = ?[*:0]u8;
        };

        pub const stock_id = struct {
            pub const name = "stock-id";

            pub const Type = ?[*:0]u8;
        };

        /// The `gtk.IconSize` value that specifies the size of the rendered icon.
        pub const stock_size = struct {
            pub const name = "stock-size";

            pub const Type = c_uint;
        };

        pub const surface = struct {
            pub const name = "surface";

            pub const Type = ?*cairo.Surface;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.CellRendererPixbuf`. Adjust rendering
    /// parameters using object properties. Object properties can be set
    /// globally (with `gobject.Object.set`). Also, with `gtk.TreeViewColumn`, you
    /// can bind a property to a value in a `gtk.TreeModel`. For example, you
    /// can bind the “pixbuf” property on the cell renderer to a pixbuf value
    /// in the model, thus rendering a different image in each row of the
    /// `gtk.TreeView`.
    extern fn gtk_cell_renderer_pixbuf_new() *gtk.CellRendererPixbuf;
    pub const new = gtk_cell_renderer_pixbuf_new;

    extern fn gtk_cell_renderer_pixbuf_get_type() usize;
    pub const getGObjectType = gtk_cell_renderer_pixbuf_get_type;

    extern fn g_object_ref(p_self: *gtk.CellRendererPixbuf) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellRendererPixbuf) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellRendererPixbuf, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.CellRendererProgress` renders a numeric value as a progress par in a cell.
/// Additionally, it can display a text on top of the progress bar.
///
/// The `gtk.CellRendererProgress` cell renderer was added in GTK+ 2.6.
pub const CellRendererProgress = extern struct {
    pub const Parent = gtk.CellRenderer;
    pub const Implements = [_]type{gtk.Orientable};
    pub const Class = gtk.CellRendererProgressClass;
    f_parent_instance: gtk.CellRenderer,
    f_priv: ?*gtk.CellRendererProgressPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const inverted = struct {
            pub const name = "inverted";

            pub const Type = c_int;
        };

        /// Setting this to a non-negative value causes the cell renderer to
        /// enter "activity mode", where a block bounces back and forth to
        /// indicate that some progress is made, without specifying exactly how
        /// much.
        ///
        /// Each increment of the property causes the block to move by a little
        /// bit.
        ///
        /// To indicate that the activity has not started yet, set the property
        /// to zero. To indicate completion, set the property to `G_MAXINT`.
        pub const pulse = struct {
            pub const name = "pulse";

            pub const Type = c_int;
        };

        /// The "text" property determines the label which will be drawn
        /// over the progress bar. Setting this property to `NULL` causes the default
        /// label to be displayed. Setting this property to an empty string causes
        /// no label to be displayed.
        pub const text = struct {
            pub const name = "text";

            pub const Type = ?[*:0]u8;
        };

        /// The "text-xalign" property controls the horizontal alignment of the
        /// text in the progress bar.  Valid values range from 0 (left) to 1
        /// (right).  Reserved for RTL layouts.
        pub const text_xalign = struct {
            pub const name = "text-xalign";

            pub const Type = f32;
        };

        /// The "text-yalign" property controls the vertical alignment of the
        /// text in the progress bar.  Valid values range from 0 (top) to 1
        /// (bottom).
        pub const text_yalign = struct {
            pub const name = "text-yalign";

            pub const Type = f32;
        };

        /// The "value" property determines the percentage to which the
        /// progress bar will be "filled in".
        pub const value = struct {
            pub const name = "value";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.CellRendererProgress`.
    extern fn gtk_cell_renderer_progress_new() *gtk.CellRendererProgress;
    pub const new = gtk_cell_renderer_progress_new;

    extern fn gtk_cell_renderer_progress_get_type() usize;
    pub const getGObjectType = gtk_cell_renderer_progress_get_type;

    extern fn g_object_ref(p_self: *gtk.CellRendererProgress) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellRendererProgress) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellRendererProgress, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.CellRendererSpin` renders text in a cell like `gtk.CellRendererText` from
/// which it is derived. But while `gtk.CellRendererText` offers a simple entry to
/// edit the text, `gtk.CellRendererSpin` offers a `gtk.SpinButton` widget. Of course,
/// that means that the text has to be parseable as a floating point number.
///
/// The range of the spinbutton is taken from the adjustment property of the
/// cell renderer, which can be set explicitly or mapped to a column in the
/// tree model, like all properties of cell renders. `gtk.CellRendererSpin`
/// also has properties for the `gtk.CellRendererSpin.properties.climb`-rate and the number
/// of `gtk.CellRendererSpin.properties.digits` to display. Other `gtk.SpinButton` properties
/// can be set in a handler for the `gtk.CellRenderer.signals.editing`-started signal.
///
/// The `gtk.CellRendererSpin` cell renderer was added in GTK+ 2.10.
pub const CellRendererSpin = extern struct {
    pub const Parent = gtk.CellRendererText;
    pub const Implements = [_]type{};
    pub const Class = gtk.CellRendererSpinClass;
    f_parent: gtk.CellRendererText,
    f_priv: ?*gtk.CellRendererSpinPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The adjustment that holds the value of the spinbutton.
        /// This must be non-`NULL` for the cell renderer to be editable.
        pub const adjustment = struct {
            pub const name = "adjustment";

            pub const Type = ?*gtk.Adjustment;
        };

        /// The acceleration rate when you hold down a button.
        pub const climb_rate = struct {
            pub const name = "climb-rate";

            pub const Type = f64;
        };

        /// The number of decimal places to display.
        pub const digits = struct {
            pub const name = "digits";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.CellRendererSpin`.
    extern fn gtk_cell_renderer_spin_new() *gtk.CellRendererSpin;
    pub const new = gtk_cell_renderer_spin_new;

    extern fn gtk_cell_renderer_spin_get_type() usize;
    pub const getGObjectType = gtk_cell_renderer_spin_get_type;

    extern fn g_object_ref(p_self: *gtk.CellRendererSpin) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellRendererSpin) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellRendererSpin, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// GtkCellRendererSpinner renders a spinning animation in a cell, very
/// similar to `gtk.Spinner`. It can often be used as an alternative
/// to a `gtk.CellRendererProgress` for displaying indefinite activity,
/// instead of actual progress.
///
/// To start the animation in a cell, set the `gtk.CellRendererSpinner.properties.active`
/// property to `TRUE` and increment the `gtk.CellRendererSpinner.properties.pulse` property
/// at regular intervals. The usual way to set the cell renderer properties
/// for each cell is to bind them to columns in your tree model using e.g.
/// `gtk.TreeViewColumn.addAttribute`.
pub const CellRendererSpinner = extern struct {
    pub const Parent = gtk.CellRenderer;
    pub const Implements = [_]type{};
    pub const Class = gtk.CellRendererSpinnerClass;
    f_parent: gtk.CellRenderer,
    f_priv: ?*gtk.CellRendererSpinnerPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const active = struct {
            pub const name = "active";

            pub const Type = c_int;
        };

        /// Pulse of the spinner. Increment this value to draw the next frame of the
        /// spinner animation. Usually, you would update this value in a timeout.
        ///
        /// By default, the `gtk.Spinner` widget draws one full cycle of the animation,
        /// consisting of 12 frames, in 750 milliseconds.
        pub const pulse = struct {
            pub const name = "pulse";

            pub const Type = c_uint;
        };

        /// The `gtk.IconSize` value that specifies the size of the rendered spinner.
        pub const size = struct {
            pub const name = "size";

            pub const Type = gtk.IconSize;
        };
    };

    pub const signals = struct {};

    /// Returns a new cell renderer which will show a spinner to indicate
    /// activity.
    extern fn gtk_cell_renderer_spinner_new() *gtk.CellRendererSpinner;
    pub const new = gtk_cell_renderer_spinner_new;

    extern fn gtk_cell_renderer_spinner_get_type() usize;
    pub const getGObjectType = gtk_cell_renderer_spinner_get_type;

    extern fn g_object_ref(p_self: *gtk.CellRendererSpinner) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellRendererSpinner) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellRendererSpinner, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.CellRendererText` renders a given text in its cell, using the font, color and
/// style information provided by its properties. The text will be ellipsized if it is
/// too long and the `gtk.CellRendererText.properties.ellipsize` property allows it.
///
/// If the `gtk.CellRenderer.properties.mode` is `GTK_CELL_RENDERER_MODE_EDITABLE`,
/// the `gtk.CellRendererText` allows to edit its text using an entry.
pub const CellRendererText = extern struct {
    pub const Parent = gtk.CellRenderer;
    pub const Implements = [_]type{};
    pub const Class = gtk.CellRendererTextClass;
    f_parent: gtk.CellRenderer,
    f_priv: ?*gtk.CellRendererTextPrivate,

    pub const virtual_methods = struct {
        pub const edited = struct {
            pub fn call(p_class: anytype, p_cell_renderer_text: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path: [*:0]const u8, p_new_text: [*:0]const u8) void {
                return gobject.ext.as(CellRendererText.Class, p_class).f_edited.?(gobject.ext.as(CellRendererText, p_cell_renderer_text), p_path, p_new_text);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell_renderer_text: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path: [*:0]const u8, p_new_text: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(CellRendererText.Class, p_class).f_edited = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const align_set = struct {
            pub const name = "align-set";

            pub const Type = c_int;
        };

        /// Specifies how to align the lines of text with respect to each other.
        ///
        /// Note that this property describes how to align the lines of text in
        /// case there are several of them. The "xalign" property of `gtk.CellRenderer`,
        /// on the other hand, sets the horizontal alignment of the whole text.
        pub const alignment = struct {
            pub const name = "alignment";

            pub const Type = pango.Alignment;
        };

        pub const attributes = struct {
            pub const name = "attributes";

            pub const Type = ?*pango.AttrList;
        };

        pub const background = struct {
            pub const name = "background";

            pub const Type = ?[*:0]u8;
        };

        /// Background color as a `gdk.Color`
        pub const background_gdk = struct {
            pub const name = "background-gdk";

            pub const Type = ?*gdk.Color;
        };

        /// Background color as a `gdk.RGBA`
        pub const background_rgba = struct {
            pub const name = "background-rgba";

            pub const Type = ?*gdk.RGBA;
        };

        pub const background_set = struct {
            pub const name = "background-set";

            pub const Type = c_int;
        };

        pub const editable = struct {
            pub const name = "editable";

            pub const Type = c_int;
        };

        pub const editable_set = struct {
            pub const name = "editable-set";

            pub const Type = c_int;
        };

        /// Specifies the preferred place to ellipsize the string, if the cell renderer
        /// does not have enough room to display the entire string. Setting it to
        /// `PANGO_ELLIPSIZE_NONE` turns off ellipsizing. See the wrap-width property
        /// for another way of making the text fit into a given width.
        pub const ellipsize = struct {
            pub const name = "ellipsize";

            pub const Type = pango.EllipsizeMode;
        };

        pub const ellipsize_set = struct {
            pub const name = "ellipsize-set";

            pub const Type = c_int;
        };

        pub const family = struct {
            pub const name = "family";

            pub const Type = ?[*:0]u8;
        };

        pub const family_set = struct {
            pub const name = "family-set";

            pub const Type = c_int;
        };

        pub const font = struct {
            pub const name = "font";

            pub const Type = ?[*:0]u8;
        };

        pub const font_desc = struct {
            pub const name = "font-desc";

            pub const Type = ?*pango.FontDescription;
        };

        pub const foreground = struct {
            pub const name = "foreground";

            pub const Type = ?[*:0]u8;
        };

        /// Foreground color as a `gdk.Color`
        pub const foreground_gdk = struct {
            pub const name = "foreground-gdk";

            pub const Type = ?*gdk.Color;
        };

        /// Foreground color as a `gdk.RGBA`
        pub const foreground_rgba = struct {
            pub const name = "foreground-rgba";

            pub const Type = ?*gdk.RGBA;
        };

        pub const foreground_set = struct {
            pub const name = "foreground-set";

            pub const Type = c_int;
        };

        pub const language = struct {
            pub const name = "language";

            pub const Type = ?[*:0]u8;
        };

        pub const language_set = struct {
            pub const name = "language-set";

            pub const Type = c_int;
        };

        pub const markup = struct {
            pub const name = "markup";

            pub const Type = ?[*:0]u8;
        };

        /// The desired maximum width of the cell, in characters. If this property
        /// is set to -1, the width will be calculated automatically.
        ///
        /// For cell renderers that ellipsize or wrap text; this property
        /// controls the maximum reported width of the cell. The
        /// cell should not receive any greater allocation unless it is
        /// set to expand in its `gtk.CellLayout` and all of the cell's siblings
        /// have received their natural width.
        pub const max_width_chars = struct {
            pub const name = "max-width-chars";

            pub const Type = c_int;
        };

        /// The text that will be displayed in the `gtk.CellRenderer` if
        /// `gtk.CellRendererText.properties.editable` is `TRUE` and the cell is empty.
        ///
        /// Since 3.6
        pub const placeholder_text = struct {
            pub const name = "placeholder-text";

            pub const Type = ?[*:0]u8;
        };

        pub const rise = struct {
            pub const name = "rise";

            pub const Type = c_int;
        };

        pub const rise_set = struct {
            pub const name = "rise-set";

            pub const Type = c_int;
        };

        pub const scale = struct {
            pub const name = "scale";

            pub const Type = f64;
        };

        pub const scale_set = struct {
            pub const name = "scale-set";

            pub const Type = c_int;
        };

        pub const single_paragraph_mode = struct {
            pub const name = "single-paragraph-mode";

            pub const Type = c_int;
        };

        pub const size = struct {
            pub const name = "size";

            pub const Type = c_int;
        };

        pub const size_points = struct {
            pub const name = "size-points";

            pub const Type = f64;
        };

        pub const size_set = struct {
            pub const name = "size-set";

            pub const Type = c_int;
        };

        pub const stretch = struct {
            pub const name = "stretch";

            pub const Type = pango.Stretch;
        };

        pub const stretch_set = struct {
            pub const name = "stretch-set";

            pub const Type = c_int;
        };

        pub const strikethrough = struct {
            pub const name = "strikethrough";

            pub const Type = c_int;
        };

        pub const strikethrough_set = struct {
            pub const name = "strikethrough-set";

            pub const Type = c_int;
        };

        pub const style = struct {
            pub const name = "style";

            pub const Type = pango.Style;
        };

        pub const style_set = struct {
            pub const name = "style-set";

            pub const Type = c_int;
        };

        pub const text = struct {
            pub const name = "text";

            pub const Type = ?[*:0]u8;
        };

        pub const underline = struct {
            pub const name = "underline";

            pub const Type = pango.Underline;
        };

        pub const underline_set = struct {
            pub const name = "underline-set";

            pub const Type = c_int;
        };

        pub const variant = struct {
            pub const name = "variant";

            pub const Type = pango.Variant;
        };

        pub const variant_set = struct {
            pub const name = "variant-set";

            pub const Type = c_int;
        };

        pub const weight = struct {
            pub const name = "weight";

            pub const Type = c_int;
        };

        pub const weight_set = struct {
            pub const name = "weight-set";

            pub const Type = c_int;
        };

        /// The desired width of the cell, in characters. If this property is set to
        /// -1, the width will be calculated automatically, otherwise the cell will
        /// request either 3 characters or the property value, whichever is greater.
        pub const width_chars = struct {
            pub const name = "width-chars";

            pub const Type = c_int;
        };

        /// Specifies how to break the string into multiple lines, if the cell
        /// renderer does not have enough room to display the entire string.
        /// This property has no effect unless the wrap-width property is set.
        pub const wrap_mode = struct {
            pub const name = "wrap-mode";

            pub const Type = pango.WrapMode;
        };

        /// Specifies the minimum width at which the text is wrapped. The wrap-mode property can
        /// be used to influence at what character positions the line breaks can be placed.
        /// Setting wrap-width to -1 turns wrapping off.
        pub const wrap_width = struct {
            pub const name = "wrap-width";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// This signal is emitted after `renderer` has been edited.
        ///
        /// It is the responsibility of the application to update the model
        /// and store `new_text` at the position indicated by `path`.
        pub const edited = struct {
            pub const name = "edited";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_path: [*:0]u8, p_new_text: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CellRendererText, p_instance))),
                    gobject.signalLookup("edited", CellRendererText.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.CellRendererText`. Adjust how text is drawn using
    /// object properties. Object properties can be
    /// set globally (with `gobject.Object.set`). Also, with `gtk.TreeViewColumn`,
    /// you can bind a property to a value in a `gtk.TreeModel`. For example,
    /// you can bind the “text” property on the cell renderer to a string
    /// value in the model, thus rendering a different string in each row
    /// of the `gtk.TreeView`
    extern fn gtk_cell_renderer_text_new() *gtk.CellRendererText;
    pub const new = gtk_cell_renderer_text_new;

    /// Sets the height of a renderer to explicitly be determined by the “font” and
    /// “y_pad” property set on it.  Further changes in these properties do not
    /// affect the height, so they must be accompanied by a subsequent call to this
    /// function.  Using this function is unflexible, and should really only be used
    /// if calculating the size of a cell is too slow (ie, a massive number of cells
    /// displayed).  If `number_of_rows` is -1, then the fixed height is unset, and
    /// the height is determined by the properties again.
    extern fn gtk_cell_renderer_text_set_fixed_height_from_font(p_renderer: *CellRendererText, p_number_of_rows: c_int) void;
    pub const setFixedHeightFromFont = gtk_cell_renderer_text_set_fixed_height_from_font;

    extern fn gtk_cell_renderer_text_get_type() usize;
    pub const getGObjectType = gtk_cell_renderer_text_get_type;

    extern fn g_object_ref(p_self: *gtk.CellRendererText) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellRendererText) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellRendererText, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.CellRendererToggle` renders a toggle button in a cell. The
/// button is drawn as a radio or a checkbutton, depending on the
/// `gtk.CellRendererToggle.properties.radio` property.
/// When activated, it emits the `gtk.CellRendererToggle.signals.toggled` signal.
pub const CellRendererToggle = extern struct {
    pub const Parent = gtk.CellRenderer;
    pub const Implements = [_]type{};
    pub const Class = gtk.CellRendererToggleClass;
    f_parent: gtk.CellRenderer,
    f_priv: ?*gtk.CellRendererTogglePrivate,

    pub const virtual_methods = struct {
        pub const toggled = struct {
            pub fn call(p_class: anytype, p_cell_renderer_toggle: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path: [*:0]const u8) void {
                return gobject.ext.as(CellRendererToggle.Class, p_class).f_toggled.?(gobject.ext.as(CellRendererToggle, p_cell_renderer_toggle), p_path);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cell_renderer_toggle: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(CellRendererToggle.Class, p_class).f_toggled = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const activatable = struct {
            pub const name = "activatable";

            pub const Type = c_int;
        };

        pub const active = struct {
            pub const name = "active";

            pub const Type = c_int;
        };

        pub const inconsistent = struct {
            pub const name = "inconsistent";

            pub const Type = c_int;
        };

        pub const indicator_size = struct {
            pub const name = "indicator-size";

            pub const Type = c_int;
        };

        pub const radio = struct {
            pub const name = "radio";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The ::toggled signal is emitted when the cell is toggled.
        ///
        /// It is the responsibility of the application to update the model
        /// with the correct value to store at `path`.  Often this is simply the
        /// opposite of the value currently stored at `path`.
        pub const toggled = struct {
            pub const name = "toggled";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_path: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CellRendererToggle, p_instance))),
                    gobject.signalLookup("toggled", CellRendererToggle.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.CellRendererToggle`. Adjust rendering
    /// parameters using object properties. Object properties can be set
    /// globally (with `gobject.Object.set`). Also, with `gtk.TreeViewColumn`, you
    /// can bind a property to a value in a `gtk.TreeModel`. For example, you
    /// can bind the “active” property on the cell renderer to a boolean value
    /// in the model, thus causing the check button to reflect the state of
    /// the model.
    extern fn gtk_cell_renderer_toggle_new() *gtk.CellRendererToggle;
    pub const new = gtk_cell_renderer_toggle_new;

    /// Returns whether the cell renderer is activatable. See
    /// `gtk.CellRendererToggle.setActivatable`.
    extern fn gtk_cell_renderer_toggle_get_activatable(p_toggle: *CellRendererToggle) c_int;
    pub const getActivatable = gtk_cell_renderer_toggle_get_activatable;

    /// Returns whether the cell renderer is active. See
    /// `gtk.CellRendererToggle.setActive`.
    extern fn gtk_cell_renderer_toggle_get_active(p_toggle: *CellRendererToggle) c_int;
    pub const getActive = gtk_cell_renderer_toggle_get_active;

    /// Returns whether we’re rendering radio toggles rather than checkboxes.
    extern fn gtk_cell_renderer_toggle_get_radio(p_toggle: *CellRendererToggle) c_int;
    pub const getRadio = gtk_cell_renderer_toggle_get_radio;

    /// Makes the cell renderer activatable.
    extern fn gtk_cell_renderer_toggle_set_activatable(p_toggle: *CellRendererToggle, p_setting: c_int) void;
    pub const setActivatable = gtk_cell_renderer_toggle_set_activatable;

    /// Activates or deactivates a cell renderer.
    extern fn gtk_cell_renderer_toggle_set_active(p_toggle: *CellRendererToggle, p_setting: c_int) void;
    pub const setActive = gtk_cell_renderer_toggle_set_active;

    /// If `radio` is `TRUE`, the cell renderer renders a radio toggle
    /// (i.e. a toggle in a group of mutually-exclusive toggles).
    /// If `FALSE`, it renders a check toggle (a standalone boolean option).
    /// This can be set globally for the cell renderer, or changed just
    /// before rendering each cell in the model (for `gtk.TreeView`, you set
    /// up a per-row setting using `gtk.TreeViewColumn` to associate model
    /// columns with cell renderer properties).
    extern fn gtk_cell_renderer_toggle_set_radio(p_toggle: *CellRendererToggle, p_radio: c_int) void;
    pub const setRadio = gtk_cell_renderer_toggle_set_radio;

    extern fn gtk_cell_renderer_toggle_get_type() usize;
    pub const getGObjectType = gtk_cell_renderer_toggle_get_type;

    extern fn g_object_ref(p_self: *gtk.CellRendererToggle) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellRendererToggle) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellRendererToggle, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.CellView` displays a single row of a `gtk.TreeModel` using a `gtk.CellArea`
/// and `gtk.CellAreaContext`. A `gtk.CellAreaContext` can be provided to the
/// `gtk.CellView` at construction time in order to keep the cellview in context
/// of a group of cell views, this ensures that the renderers displayed will
/// be properly aligned with eachother (like the aligned cells in the menus
/// of `gtk.ComboBox`).
///
/// `gtk.CellView` is `gtk.Orientable` in order to decide in which orientation
/// the underlying `gtk.CellAreaContext` should be allocated. Taking the `gtk.ComboBox`
/// menu as an example, cellviews should be oriented horizontally if the menus are
/// listed top-to-bottom and thus all share the same width but may have separate
/// individual heights (left-to-right menus should be allocated vertically since
/// they all share the same height but may have variable widths).
///
/// # CSS nodes
///
/// GtkCellView has a single CSS node with name cellview.
pub const CellView = extern struct {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.CellLayout, gtk.Orientable };
    pub const Class = gtk.CellViewClass;
    f_parent_instance: gtk.Widget,
    f_priv: ?*gtk.CellViewPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const background = struct {
            pub const name = "background";

            pub const Type = ?[*:0]u8;
        };

        /// The background color as a `gdk.Color`
        pub const background_gdk = struct {
            pub const name = "background-gdk";

            pub const Type = ?*gdk.Color;
        };

        /// The background color as a `gdk.RGBA`
        pub const background_rgba = struct {
            pub const name = "background-rgba";

            pub const Type = ?*gdk.RGBA;
        };

        pub const background_set = struct {
            pub const name = "background-set";

            pub const Type = c_int;
        };

        /// The `gtk.CellArea` rendering cells
        ///
        /// If no area is specified when creating the cell view with `gtk.CellView.newWithContext`
        /// a horizontally oriented `gtk.CellAreaBox` will be used.
        ///
        /// since 3.0
        pub const cell_area = struct {
            pub const name = "cell-area";

            pub const Type = ?*gtk.CellArea;
        };

        /// The `gtk.CellAreaContext` used to compute the geometry of the cell view.
        ///
        /// A group of cell views can be assigned the same context in order to
        /// ensure the sizes and cell alignments match across all the views with
        /// the same context.
        ///
        /// `gtk.ComboBox` menus uses this to assign the same context to all cell views
        /// in the menu items for a single menu (each submenu creates its own
        /// context since the size of each submenu does not depend on parent
        /// or sibling menus).
        ///
        /// since 3.0
        pub const cell_area_context = struct {
            pub const name = "cell-area-context";

            pub const Type = ?*gtk.CellAreaContext;
        };

        /// Whether all cells should be draw as sensitive for this view regardless
        /// of the actual cell properties (used to make menus with submenus appear
        /// sensitive when the items in submenus might be insensitive).
        ///
        /// since 3.0
        pub const draw_sensitive = struct {
            pub const name = "draw-sensitive";

            pub const Type = c_int;
        };

        /// Whether the view should request enough space to always fit
        /// the size of every row in the model (used by the combo box to
        /// ensure the combo box size doesnt change when different items
        /// are selected).
        ///
        /// since 3.0
        pub const fit_model = struct {
            pub const name = "fit-model";

            pub const Type = c_int;
        };

        /// The model for cell view
        ///
        /// since 2.10
        pub const model = struct {
            pub const name = "model";

            pub const Type = ?*gtk.TreeModel;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.CellView` widget.
    extern fn gtk_cell_view_new() *gtk.CellView;
    pub const new = gtk_cell_view_new;

    /// Creates a new `gtk.CellView` widget with a specific `gtk.CellArea`
    /// to layout cells and a specific `gtk.CellAreaContext`.
    ///
    /// Specifying the same context for a handfull of cells lets
    /// the underlying area synchronize the geometry for those cells,
    /// in this way alignments with cellviews for other rows are
    /// possible.
    extern fn gtk_cell_view_new_with_context(p_area: *gtk.CellArea, p_context: *gtk.CellAreaContext) *gtk.CellView;
    pub const newWithContext = gtk_cell_view_new_with_context;

    /// Creates a new `gtk.CellView` widget, adds a `gtk.CellRendererText`
    /// to it, and makes it show `markup`. The text can be
    /// marked up with the [Pango text markup language][PangoMarkupFormat].
    extern fn gtk_cell_view_new_with_markup(p_markup: [*:0]const u8) *gtk.CellView;
    pub const newWithMarkup = gtk_cell_view_new_with_markup;

    /// Creates a new `gtk.CellView` widget, adds a `gtk.CellRendererPixbuf`
    /// to it, and makes it show `pixbuf`.
    extern fn gtk_cell_view_new_with_pixbuf(p_pixbuf: *gdkpixbuf.Pixbuf) *gtk.CellView;
    pub const newWithPixbuf = gtk_cell_view_new_with_pixbuf;

    /// Creates a new `gtk.CellView` widget, adds a `gtk.CellRendererText`
    /// to it, and makes it show `text`.
    extern fn gtk_cell_view_new_with_text(p_text: [*:0]const u8) *gtk.CellView;
    pub const newWithText = gtk_cell_view_new_with_text;

    /// Returns a `gtk.TreePath` referring to the currently
    /// displayed row. If no row is currently displayed,
    /// `NULL` is returned.
    extern fn gtk_cell_view_get_displayed_row(p_cell_view: *CellView) ?*gtk.TreePath;
    pub const getDisplayedRow = gtk_cell_view_get_displayed_row;

    /// Gets whether `cell_view` is configured to draw all of its
    /// cells in a sensitive state.
    extern fn gtk_cell_view_get_draw_sensitive(p_cell_view: *CellView) c_int;
    pub const getDrawSensitive = gtk_cell_view_get_draw_sensitive;

    /// Gets whether `cell_view` is configured to request space
    /// to fit the entire `gtk.TreeModel`.
    extern fn gtk_cell_view_get_fit_model(p_cell_view: *CellView) c_int;
    pub const getFitModel = gtk_cell_view_get_fit_model;

    /// Returns the model for `cell_view`. If no model is used `NULL` is
    /// returned.
    extern fn gtk_cell_view_get_model(p_cell_view: *CellView) ?*gtk.TreeModel;
    pub const getModel = gtk_cell_view_get_model;

    /// Sets `requisition` to the size needed by `cell_view` to display
    /// the model row pointed to by `path`.
    extern fn gtk_cell_view_get_size_of_row(p_cell_view: *CellView, p_path: *gtk.TreePath, p_requisition: *gtk.Requisition) c_int;
    pub const getSizeOfRow = gtk_cell_view_get_size_of_row;

    /// Sets the background color of `view`.
    extern fn gtk_cell_view_set_background_color(p_cell_view: *CellView, p_color: *const gdk.Color) void;
    pub const setBackgroundColor = gtk_cell_view_set_background_color;

    /// Sets the background color of `cell_view`.
    extern fn gtk_cell_view_set_background_rgba(p_cell_view: *CellView, p_rgba: *const gdk.RGBA) void;
    pub const setBackgroundRgba = gtk_cell_view_set_background_rgba;

    /// Sets the row of the model that is currently displayed
    /// by the `gtk.CellView`. If the path is unset, then the
    /// contents of the cellview “stick” at their last value;
    /// this is not normally a desired result, but may be
    /// a needed intermediate state if say, the model for
    /// the `gtk.CellView` becomes temporarily empty.
    extern fn gtk_cell_view_set_displayed_row(p_cell_view: *CellView, p_path: ?*gtk.TreePath) void;
    pub const setDisplayedRow = gtk_cell_view_set_displayed_row;

    /// Sets whether `cell_view` should draw all of its
    /// cells in a sensitive state, this is used by `gtk.ComboBox` menus
    /// to ensure that rows with insensitive cells that contain
    /// children appear sensitive in the parent menu item.
    extern fn gtk_cell_view_set_draw_sensitive(p_cell_view: *CellView, p_draw_sensitive: c_int) void;
    pub const setDrawSensitive = gtk_cell_view_set_draw_sensitive;

    /// Sets whether `cell_view` should request space to fit the entire `gtk.TreeModel`.
    ///
    /// This is used by `gtk.ComboBox` to ensure that the cell view displayed on
    /// the combo box’s button always gets enough space and does not resize
    /// when selection changes.
    extern fn gtk_cell_view_set_fit_model(p_cell_view: *CellView, p_fit_model: c_int) void;
    pub const setFitModel = gtk_cell_view_set_fit_model;

    /// Sets the model for `cell_view`.  If `cell_view` already has a model
    /// set, it will remove it before setting the new model.  If `model` is
    /// `NULL`, then it will unset the old model.
    extern fn gtk_cell_view_set_model(p_cell_view: *CellView, p_model: ?*gtk.TreeModel) void;
    pub const setModel = gtk_cell_view_set_model;

    extern fn gtk_cell_view_get_type() usize;
    pub const getGObjectType = gtk_cell_view_get_type;

    extern fn g_object_ref(p_self: *gtk.CellView) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CellView) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CellView, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.CheckButton` places a discrete `gtk.ToggleButton` next to a widget,
/// (usually a `gtk.Label`). See the section on `gtk.ToggleButton` widgets for
/// more information about toggle/check buttons.
///
/// The important signal ( `gtk.ToggleButton.signals.toggled` ) is also inherited from
/// `gtk.ToggleButton`.
///
/// # CSS nodes
///
/// ```
/// checkbutton
/// ├── check
/// ╰── <child>
/// ```
///
/// A GtkCheckButton with indicator (see `gtk.ToggleButton.setMode`) has a
/// main CSS node with name checkbutton and a subnode with name check.
///
/// ```
/// button.check
/// ├── check
/// ╰── <child>
/// ```
///
/// A GtkCheckButton without indicator changes the name of its main node
/// to button and adds a .check style class to it. The subnode is invisible
/// in this case.
pub const CheckButton = extern struct {
    pub const Parent = gtk.ToggleButton;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Actionable, gtk.Activatable, gtk.Buildable };
    pub const Class = gtk.CheckButtonClass;
    f_toggle_button: gtk.ToggleButton,

    pub const virtual_methods = struct {
        pub const draw_indicator = struct {
            pub fn call(p_class: anytype, p_check_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cr: *cairo.Context) void {
                return gobject.ext.as(CheckButton.Class, p_class).f_draw_indicator.?(gobject.ext.as(CheckButton, p_check_button), p_cr);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_check_button: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cr: *cairo.Context) callconv(.C) void) void {
                gobject.ext.as(CheckButton.Class, p_class).f_draw_indicator = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtk.CheckButton`.
    extern fn gtk_check_button_new() *gtk.CheckButton;
    pub const new = gtk_check_button_new;

    /// Creates a new `gtk.CheckButton` with a `gtk.Label` to the right of it.
    extern fn gtk_check_button_new_with_label(p_label: [*:0]const u8) *gtk.CheckButton;
    pub const newWithLabel = gtk_check_button_new_with_label;

    /// Creates a new `gtk.CheckButton` containing a label. The label
    /// will be created using `gtk.Label.newWithMnemonic`, so underscores
    /// in `label` indicate the mnemonic for the check button.
    extern fn gtk_check_button_new_with_mnemonic(p_label: [*:0]const u8) *gtk.CheckButton;
    pub const newWithMnemonic = gtk_check_button_new_with_mnemonic;

    extern fn gtk_check_button_get_type() usize;
    pub const getGObjectType = gtk_check_button_get_type;

    extern fn g_object_ref(p_self: *gtk.CheckButton) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CheckButton) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CheckButton, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.CheckMenuItem` is a menu item that maintains the state of a boolean
/// value in addition to a `gtk.MenuItem` usual role in activating application
/// code.
///
/// A check box indicating the state of the boolean value is displayed
/// at the left side of the `gtk.MenuItem`.  Activating the `gtk.MenuItem`
/// toggles the value.
///
/// # CSS nodes
///
/// ```
/// menuitem
/// ├── check.left
/// ╰── <child>
/// ```
///
/// GtkCheckMenuItem has a main CSS node with name menuitem, and a subnode
/// with name check, which gets the .left or .right style class.
pub const CheckMenuItem = extern struct {
    pub const Parent = gtk.MenuItem;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Actionable, gtk.Activatable, gtk.Buildable };
    pub const Class = gtk.CheckMenuItemClass;
    f_menu_item: gtk.MenuItem,
    f_priv: ?*gtk.CheckMenuItemPrivate,

    pub const virtual_methods = struct {
        /// Called to draw the check indicator.
        pub const draw_indicator = struct {
            pub fn call(p_class: anytype, p_check_menu_item: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cr: *cairo.Context) void {
                return gobject.ext.as(CheckMenuItem.Class, p_class).f_draw_indicator.?(gobject.ext.as(CheckMenuItem, p_check_menu_item), p_cr);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_check_menu_item: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cr: *cairo.Context) callconv(.C) void) void {
                gobject.ext.as(CheckMenuItem.Class, p_class).f_draw_indicator = @ptrCast(p_implementation);
            }
        };

        /// Emits the `gtk.CheckMenuItem.signals.toggled` signal.
        pub const toggled = struct {
            pub fn call(p_class: anytype, p_check_menu_item: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(CheckMenuItem.Class, p_class).f_toggled.?(gobject.ext.as(CheckMenuItem, p_check_menu_item));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_check_menu_item: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(CheckMenuItem.Class, p_class).f_toggled = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const active = struct {
            pub const name = "active";

            pub const Type = c_int;
        };

        pub const draw_as_radio = struct {
            pub const name = "draw-as-radio";

            pub const Type = c_int;
        };

        pub const inconsistent = struct {
            pub const name = "inconsistent";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// This signal is emitted when the state of the check box is changed.
        ///
        /// A signal handler can use `gtk.CheckMenuItem.getActive`
        /// to discover the new state.
        pub const toggled = struct {
            pub const name = "toggled";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CheckMenuItem, p_instance))),
                    gobject.signalLookup("toggled", CheckMenuItem.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.CheckMenuItem`.
    extern fn gtk_check_menu_item_new() *gtk.CheckMenuItem;
    pub const new = gtk_check_menu_item_new;

    /// Creates a new `gtk.CheckMenuItem` with a label.
    extern fn gtk_check_menu_item_new_with_label(p_label: [*:0]const u8) *gtk.CheckMenuItem;
    pub const newWithLabel = gtk_check_menu_item_new_with_label;

    /// Creates a new `gtk.CheckMenuItem` containing a label. The label
    /// will be created using `gtk.Label.newWithMnemonic`, so underscores
    /// in `label` indicate the mnemonic for the menu item.
    extern fn gtk_check_menu_item_new_with_mnemonic(p_label: [*:0]const u8) *gtk.CheckMenuItem;
    pub const newWithMnemonic = gtk_check_menu_item_new_with_mnemonic;

    /// Returns whether the check menu item is active. See
    /// gtk_check_menu_item_set_active ().
    extern fn gtk_check_menu_item_get_active(p_check_menu_item: *CheckMenuItem) c_int;
    pub const getActive = gtk_check_menu_item_get_active;

    /// Returns whether `check_menu_item` looks like a `gtk.RadioMenuItem`
    extern fn gtk_check_menu_item_get_draw_as_radio(p_check_menu_item: *CheckMenuItem) c_int;
    pub const getDrawAsRadio = gtk_check_menu_item_get_draw_as_radio;

    /// Retrieves the value set by `gtk.CheckMenuItem.setInconsistent`.
    extern fn gtk_check_menu_item_get_inconsistent(p_check_menu_item: *CheckMenuItem) c_int;
    pub const getInconsistent = gtk_check_menu_item_get_inconsistent;

    /// Sets the active state of the menu item’s check box.
    extern fn gtk_check_menu_item_set_active(p_check_menu_item: *CheckMenuItem, p_is_active: c_int) void;
    pub const setActive = gtk_check_menu_item_set_active;

    /// Sets whether `check_menu_item` is drawn like a `gtk.RadioMenuItem`
    extern fn gtk_check_menu_item_set_draw_as_radio(p_check_menu_item: *CheckMenuItem, p_draw_as_radio: c_int) void;
    pub const setDrawAsRadio = gtk_check_menu_item_set_draw_as_radio;

    /// If the user has selected a range of elements (such as some text or
    /// spreadsheet cells) that are affected by a boolean setting, and the
    /// current values in that range are inconsistent, you may want to
    /// display the check in an “in between” state. This function turns on
    /// “in between” display.  Normally you would turn off the inconsistent
    /// state again if the user explicitly selects a setting. This has to be
    /// done manually, `gtk.CheckMenuItem.setInconsistent` only affects
    /// visual appearance, it doesn’t affect the semantics of the widget.
    extern fn gtk_check_menu_item_set_inconsistent(p_check_menu_item: *CheckMenuItem, p_setting: c_int) void;
    pub const setInconsistent = gtk_check_menu_item_set_inconsistent;

    /// Emits the `gtk.CheckMenuItem.signals.toggled` signal.
    extern fn gtk_check_menu_item_toggled(p_check_menu_item: *CheckMenuItem) void;
    pub const toggled = gtk_check_menu_item_toggled;

    extern fn gtk_check_menu_item_get_type() usize;
    pub const getGObjectType = gtk_check_menu_item_get_type;

    extern fn g_object_ref(p_self: *gtk.CheckMenuItem) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CheckMenuItem) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CheckMenuItem, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CheckMenuItemAccessible = extern struct {
    pub const Parent = gtk.MenuItemAccessible;
    pub const Implements = [_]type{ atk.Action, atk.Component, atk.Selection };
    pub const Class = gtk.CheckMenuItemAccessibleClass;
    f_parent: gtk.MenuItemAccessible,
    f_priv: ?*gtk.CheckMenuItemAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_check_menu_item_accessible_get_type() usize;
    pub const getGObjectType = gtk_check_menu_item_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.CheckMenuItemAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CheckMenuItemAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CheckMenuItemAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.Clipboard` object represents a clipboard of data shared
/// between different processes or between different widgets in
/// the same process. Each clipboard is identified by a name encoded as a
/// `gdk.Atom`. (Conversion to and from strings can be done with
/// `gdk.atomIntern` and `gdk.Atom.name`.) The default clipboard
/// corresponds to the “CLIPBOARD” atom; another commonly used clipboard
/// is the “PRIMARY” clipboard, which, in X, traditionally contains
/// the currently selected text.
///
/// To support having a number of different formats on the clipboard
/// at the same time, the clipboard mechanism allows providing
/// callbacks instead of the actual data.  When you set the contents
/// of the clipboard, you can either supply the data directly (via
/// functions like `gtk.Clipboard.setText`), or you can supply a
/// callback to be called at a later time when the data is needed (via
/// `gtk.Clipboard.setWithData` or `gtk.Clipboard.setWithOwner`.)
/// Providing a callback also avoids having to make copies of the data
/// when it is not needed.
///
/// `gtk.Clipboard.setWithData` and `gtk.Clipboard.setWithOwner`
/// are quite similar; the choice between the two depends mostly on
/// which is more convenient in a particular situation.
/// The former is most useful when you want to have a blob of data
/// with callbacks to convert it into the various data types that you
/// advertise. When the `clear_func` you provided is called, you
/// simply free the data blob. The latter is more useful when the
/// contents of clipboard reflect the internal state of a `gobject.Object`
/// (As an example, for the PRIMARY clipboard, when an entry widget
/// provides the clipboard’s contents the contents are simply the
/// text within the selected region.) If the contents change, the
/// entry widget can call `gtk.Clipboard.setWithOwner` to update
/// the timestamp for clipboard ownership, without having to worry
/// about `clear_func` being called.
///
/// Requesting the data from the clipboard is essentially
/// asynchronous. If the contents of the clipboard are provided within
/// the same process, then a direct function call will be made to
/// retrieve the data, but if they are provided by another process,
/// then the data needs to be retrieved from the other process, which
/// may take some time. To avoid blocking the user interface, the call
/// to request the selection, `gtk.Clipboard.requestContents` takes a
/// callback that will be called when the contents are received (or
/// when the request fails.) If you don’t want to deal with providing
/// a separate callback, you can also use `gtk.Clipboard.waitForContents`.
/// What this does is run the GLib main loop recursively waiting for
/// the contents. This can simplify the code flow, but you still have
/// to be aware that other callbacks in your program can be called
/// while this recursive mainloop is running.
///
/// Along with the functions to get the clipboard contents as an
/// arbitrary data chunk, there are also functions to retrieve
/// it as text, `gtk.Clipboard.requestText` and
/// `gtk.Clipboard.waitForText`. These functions take care of
/// determining which formats are advertised by the clipboard
/// provider, asking for the clipboard in the best available format
/// and converting the results into the UTF-8 encoding. (The standard
/// form for representing strings in GTK+.)
pub const Clipboard = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = opaque {
        pub const Instance = Clipboard;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// The ::owner-change signal is emitted when GTK+ receives an
        /// event that indicates that the ownership of the selection
        /// associated with `clipboard` has changed.
        pub const owner_change = struct {
            pub const name = "owner-change";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_event: *gdk.EventOwnerChange, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Clipboard, p_instance))),
                    gobject.signalLookup("owner-change", Clipboard.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Returns the clipboard object for the given selection.
    /// See `gtk.Clipboard.getForDisplay` for complete details.
    extern fn gtk_clipboard_get(p_selection: gdk.Atom) *gtk.Clipboard;
    pub const get = gtk_clipboard_get;

    /// Returns the default clipboard object for use with cut/copy/paste menu items
    /// and keyboard shortcuts.
    extern fn gtk_clipboard_get_default(p_display: *gdk.Display) *gtk.Clipboard;
    pub const getDefault = gtk_clipboard_get_default;

    /// Returns the clipboard object for the given selection.
    /// Cut/copy/paste menu items and keyboard shortcuts should use
    /// the default clipboard, returned by passing `GDK_SELECTION_CLIPBOARD` for `selection`.
    /// (`GDK_NONE` is supported as a synonym for GDK_SELECTION_CLIPBOARD
    /// for backwards compatibility reasons.)
    /// The currently-selected object or text should be provided on the clipboard
    /// identified by `GDK_SELECTION_PRIMARY`. Cut/copy/paste menu items
    /// conceptually copy the contents of the `GDK_SELECTION_PRIMARY` clipboard
    /// to the default clipboard, i.e. they copy the selection to what the
    /// user sees as the clipboard.
    ///
    /// (Passing `GDK_NONE` is the same as using `gdk_atom_intern
    /// ("CLIPBOARD", FALSE)`.
    ///
    /// See the
    /// [FreeDesktop Clipboard Specification](http://www.freedesktop.org/Standards/clipboards-spec)
    /// for a detailed discussion of the “CLIPBOARD” vs. “PRIMARY”
    /// selections under the X window system. On Win32 the
    /// `GDK_SELECTION_PRIMARY` clipboard is essentially ignored.)
    ///
    /// It’s possible to have arbitrary named clipboards; if you do invent
    /// new clipboards, you should prefix the selection name with an
    /// underscore (because the ICCCM requires that nonstandard atoms are
    /// underscore-prefixed), and namespace it as well. For example,
    /// if your application called “Foo” has a special-purpose
    /// clipboard, you might call it “_FOO_SPECIAL_CLIPBOARD”.
    extern fn gtk_clipboard_get_for_display(p_display: *gdk.Display, p_selection: gdk.Atom) *gtk.Clipboard;
    pub const getForDisplay = gtk_clipboard_get_for_display;

    /// Clears the contents of the clipboard. Generally this should only
    /// be called between the time you call `gtk.Clipboard.setWithOwner`
    /// or `gtk.Clipboard.setWithData`,
    /// and when the `clear_func` you supplied is called. Otherwise, the
    /// clipboard may be owned by someone else.
    extern fn gtk_clipboard_clear(p_clipboard: *Clipboard) void;
    pub const clear = gtk_clipboard_clear;

    /// Gets the `gdk.Display` associated with `clipboard`
    extern fn gtk_clipboard_get_display(p_clipboard: *Clipboard) *gdk.Display;
    pub const getDisplay = gtk_clipboard_get_display;

    /// If the clipboard contents callbacks were set with
    /// `gtk.Clipboard.setWithOwner`, and the `gtk.Clipboard.setWithData` or
    /// `gtk.Clipboard.clear` has not subsequently called, returns the owner set
    /// by `gtk.Clipboard.setWithOwner`.
    extern fn gtk_clipboard_get_owner(p_clipboard: *Clipboard) ?*gobject.Object;
    pub const getOwner = gtk_clipboard_get_owner;

    /// Gets the selection that this clipboard is for.
    extern fn gtk_clipboard_get_selection(p_clipboard: *Clipboard) gdk.Atom;
    pub const getSelection = gtk_clipboard_get_selection;

    /// Requests the contents of clipboard as the given target.
    /// When the results of the result are later received the supplied callback
    /// will be called.
    extern fn gtk_clipboard_request_contents(p_clipboard: *Clipboard, p_target: gdk.Atom, p_callback: gtk.ClipboardReceivedFunc, p_user_data: ?*anyopaque) void;
    pub const requestContents = gtk_clipboard_request_contents;

    /// Requests the contents of the clipboard as image. When the image is
    /// later received, it will be converted to a `gdkpixbuf.Pixbuf`, and
    /// `callback` will be called.
    ///
    /// The `pixbuf` parameter to `callback` will contain the resulting
    /// `gdkpixbuf.Pixbuf` if the request succeeded, or `NULL` if it failed. This
    /// could happen for various reasons, in particular if the clipboard
    /// was empty or if the contents of the clipboard could not be
    /// converted into an image.
    extern fn gtk_clipboard_request_image(p_clipboard: *Clipboard, p_callback: gtk.ClipboardImageReceivedFunc, p_user_data: ?*anyopaque) void;
    pub const requestImage = gtk_clipboard_request_image;

    /// Requests the contents of the clipboard as rich text. When the rich
    /// text is later received, `callback` will be called.
    ///
    /// The `text` parameter to `callback` will contain the resulting rich
    /// text if the request succeeded, or `NULL` if it failed. The `length`
    /// parameter will contain `text`’s length. This function can fail for
    /// various reasons, in particular if the clipboard was empty or if the
    /// contents of the clipboard could not be converted into rich text form.
    extern fn gtk_clipboard_request_rich_text(p_clipboard: *Clipboard, p_buffer: *gtk.TextBuffer, p_callback: gtk.ClipboardRichTextReceivedFunc, p_user_data: ?*anyopaque) void;
    pub const requestRichText = gtk_clipboard_request_rich_text;

    /// Requests the contents of the clipboard as list of supported targets.
    /// When the list is later received, `callback` will be called.
    ///
    /// The `targets` parameter to `callback` will contain the resulting targets if
    /// the request succeeded, or `NULL` if it failed.
    extern fn gtk_clipboard_request_targets(p_clipboard: *Clipboard, p_callback: gtk.ClipboardTargetsReceivedFunc, p_user_data: ?*anyopaque) void;
    pub const requestTargets = gtk_clipboard_request_targets;

    /// Requests the contents of the clipboard as text. When the text is
    /// later received, it will be converted to UTF-8 if necessary, and
    /// `callback` will be called.
    ///
    /// The `text` parameter to `callback` will contain the resulting text if
    /// the request succeeded, or `NULL` if it failed. This could happen for
    /// various reasons, in particular if the clipboard was empty or if the
    /// contents of the clipboard could not be converted into text form.
    extern fn gtk_clipboard_request_text(p_clipboard: *Clipboard, p_callback: gtk.ClipboardTextReceivedFunc, p_user_data: ?*anyopaque) void;
    pub const requestText = gtk_clipboard_request_text;

    /// Requests the contents of the clipboard as URIs. When the URIs are
    /// later received `callback` will be called.
    ///
    /// The `uris` parameter to `callback` will contain the resulting array of
    /// URIs if the request succeeded, or `NULL` if it failed. This could happen
    /// for various reasons, in particular if the clipboard was empty or if the
    /// contents of the clipboard could not be converted into URI form.
    extern fn gtk_clipboard_request_uris(p_clipboard: *Clipboard, p_callback: gtk.ClipboardURIReceivedFunc, p_user_data: ?*anyopaque) void;
    pub const requestUris = gtk_clipboard_request_uris;

    /// Hints that the clipboard data should be stored somewhere when the
    /// application exits or when gtk_clipboard_store () is called.
    ///
    /// This value is reset when the clipboard owner changes.
    /// Where the clipboard data is stored is platform dependent,
    /// see gdk_display_store_clipboard () for more information.
    extern fn gtk_clipboard_set_can_store(p_clipboard: *Clipboard, p_targets: ?[*]const gtk.TargetEntry, p_n_targets: c_int) void;
    pub const setCanStore = gtk_clipboard_set_can_store;

    /// Sets the contents of the clipboard to the given `gdkpixbuf.Pixbuf`.
    /// GTK+ will take responsibility for responding for requests
    /// for the image, and for converting the image into the
    /// requested format.
    extern fn gtk_clipboard_set_image(p_clipboard: *Clipboard, p_pixbuf: *gdkpixbuf.Pixbuf) void;
    pub const setImage = gtk_clipboard_set_image;

    /// Sets the contents of the clipboard to the given UTF-8 string. GTK+ will
    /// make a copy of the text and take responsibility for responding
    /// for requests for the text, and for converting the text into
    /// the requested format.
    extern fn gtk_clipboard_set_text(p_clipboard: *Clipboard, p_text: [*:0]const u8, p_len: c_int) void;
    pub const setText = gtk_clipboard_set_text;

    /// Virtually sets the contents of the specified clipboard by providing
    /// a list of supported formats for the clipboard data and a function
    /// to call to get the actual data when it is requested.
    extern fn gtk_clipboard_set_with_data(p_clipboard: *Clipboard, p_targets: [*]const gtk.TargetEntry, p_n_targets: c_uint, p_get_func: gtk.ClipboardGetFunc, p_clear_func: gtk.ClipboardClearFunc, p_user_data: ?*anyopaque) c_int;
    pub const setWithData = gtk_clipboard_set_with_data;

    /// Virtually sets the contents of the specified clipboard by providing
    /// a list of supported formats for the clipboard data and a function
    /// to call to get the actual data when it is requested.
    ///
    /// The difference between this function and `gtk.Clipboard.setWithData`
    /// is that instead of an generic `user_data` pointer, a `gobject.Object` is passed
    /// in.
    extern fn gtk_clipboard_set_with_owner(p_clipboard: *Clipboard, p_targets: [*]const gtk.TargetEntry, p_n_targets: c_uint, p_get_func: gtk.ClipboardGetFunc, p_clear_func: gtk.ClipboardClearFunc, p_owner: *gobject.Object) c_int;
    pub const setWithOwner = gtk_clipboard_set_with_owner;

    /// Stores the current clipboard data somewhere so that it will stay
    /// around after the application has quit.
    extern fn gtk_clipboard_store(p_clipboard: *Clipboard) void;
    pub const store = gtk_clipboard_store;

    /// Requests the contents of the clipboard using the given target.
    /// This function waits for the data to be received using the main
    /// loop, so events, timeouts, etc, may be dispatched during the wait.
    extern fn gtk_clipboard_wait_for_contents(p_clipboard: *Clipboard, p_target: gdk.Atom) ?*gtk.SelectionData;
    pub const waitForContents = gtk_clipboard_wait_for_contents;

    /// Requests the contents of the clipboard as image and converts
    /// the result to a `gdkpixbuf.Pixbuf`. This function waits for
    /// the data to be received using the main loop, so events,
    /// timeouts, etc, may be dispatched during the wait.
    extern fn gtk_clipboard_wait_for_image(p_clipboard: *Clipboard) ?*gdkpixbuf.Pixbuf;
    pub const waitForImage = gtk_clipboard_wait_for_image;

    /// Requests the contents of the clipboard as rich text.  This function
    /// waits for the data to be received using the main loop, so events,
    /// timeouts, etc, may be dispatched during the wait.
    extern fn gtk_clipboard_wait_for_rich_text(p_clipboard: *Clipboard, p_buffer: *gtk.TextBuffer, p_format: *gdk.Atom, p_length: *usize) ?[*]u8;
    pub const waitForRichText = gtk_clipboard_wait_for_rich_text;

    /// Returns a list of targets that are present on the clipboard, or `NULL`
    /// if there aren’t any targets available. The returned list must be
    /// freed with `glib.free`.
    /// This function waits for the data to be received using the main
    /// loop, so events, timeouts, etc, may be dispatched during the wait.
    extern fn gtk_clipboard_wait_for_targets(p_clipboard: *Clipboard, p_targets: [*]*gdk.Atom, p_n_targets: *c_int) c_int;
    pub const waitForTargets = gtk_clipboard_wait_for_targets;

    /// Requests the contents of the clipboard as text and converts
    /// the result to UTF-8 if necessary. This function waits for
    /// the data to be received using the main loop, so events,
    /// timeouts, etc, may be dispatched during the wait.
    extern fn gtk_clipboard_wait_for_text(p_clipboard: *Clipboard) ?[*:0]u8;
    pub const waitForText = gtk_clipboard_wait_for_text;

    /// Requests the contents of the clipboard as URIs. This function waits
    /// for the data to be received using the main loop, so events,
    /// timeouts, etc, may be dispatched during the wait.
    extern fn gtk_clipboard_wait_for_uris(p_clipboard: *Clipboard) ?[*][*:0]u8;
    pub const waitForUris = gtk_clipboard_wait_for_uris;

    /// Test to see if there is an image available to be pasted
    /// This is done by requesting the TARGETS atom and checking
    /// if it contains any of the supported image targets. This function
    /// waits for the data to be received using the main loop, so events,
    /// timeouts, etc, may be dispatched during the wait.
    ///
    /// This function is a little faster than calling
    /// `gtk.Clipboard.waitForImage` since it doesn’t need to retrieve
    /// the actual image data.
    extern fn gtk_clipboard_wait_is_image_available(p_clipboard: *Clipboard) c_int;
    pub const waitIsImageAvailable = gtk_clipboard_wait_is_image_available;

    /// Test to see if there is rich text available to be pasted
    /// This is done by requesting the TARGETS atom and checking
    /// if it contains any of the supported rich text targets. This function
    /// waits for the data to be received using the main loop, so events,
    /// timeouts, etc, may be dispatched during the wait.
    ///
    /// This function is a little faster than calling
    /// `gtk.Clipboard.waitForRichText` since it doesn’t need to retrieve
    /// the actual text.
    extern fn gtk_clipboard_wait_is_rich_text_available(p_clipboard: *Clipboard, p_buffer: *gtk.TextBuffer) c_int;
    pub const waitIsRichTextAvailable = gtk_clipboard_wait_is_rich_text_available;

    /// Checks if a clipboard supports pasting data of a given type. This
    /// function can be used to determine if a “Paste” menu item should be
    /// insensitive or not.
    ///
    /// If you want to see if there’s text available on the clipboard, use
    /// gtk_clipboard_wait_is_text_available () instead.
    extern fn gtk_clipboard_wait_is_target_available(p_clipboard: *Clipboard, p_target: gdk.Atom) c_int;
    pub const waitIsTargetAvailable = gtk_clipboard_wait_is_target_available;

    /// Test to see if there is text available to be pasted
    /// This is done by requesting the TARGETS atom and checking
    /// if it contains any of the supported text targets. This function
    /// waits for the data to be received using the main loop, so events,
    /// timeouts, etc, may be dispatched during the wait.
    ///
    /// This function is a little faster than calling
    /// `gtk.Clipboard.waitForText` since it doesn’t need to retrieve
    /// the actual text.
    extern fn gtk_clipboard_wait_is_text_available(p_clipboard: *Clipboard) c_int;
    pub const waitIsTextAvailable = gtk_clipboard_wait_is_text_available;

    /// Test to see if there is a list of URIs available to be pasted
    /// This is done by requesting the TARGETS atom and checking
    /// if it contains the URI targets. This function
    /// waits for the data to be received using the main loop, so events,
    /// timeouts, etc, may be dispatched during the wait.
    ///
    /// This function is a little faster than calling
    /// `gtk.Clipboard.waitForUris` since it doesn’t need to retrieve
    /// the actual URI data.
    extern fn gtk_clipboard_wait_is_uris_available(p_clipboard: *Clipboard) c_int;
    pub const waitIsUrisAvailable = gtk_clipboard_wait_is_uris_available;

    extern fn gtk_clipboard_get_type() usize;
    pub const getGObjectType = gtk_clipboard_get_type;

    extern fn g_object_ref(p_self: *gtk.Clipboard) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Clipboard) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Clipboard, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.ColorButton` is a button which displays the currently selected
/// color and allows to open a color selection dialog to change the color.
/// It is suitable widget for selecting a color in a preference dialog.
///
/// # CSS nodes
///
/// GtkColorButton has a single CSS node with name button. To differentiate
/// it from a plain `gtk.Button`, it gets the .color style class.
pub const ColorButton = extern struct {
    pub const Parent = gtk.Button;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Actionable, gtk.Activatable, gtk.Buildable, gtk.ColorChooser };
    pub const Class = gtk.ColorButtonClass;
    f_button: gtk.Button,
    f_priv: ?*gtk.ColorButtonPrivate,

    pub const virtual_methods = struct {
        pub const color_set = struct {
            pub fn call(p_class: anytype, p_cp: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(ColorButton.Class, p_class).f_color_set.?(gobject.ext.as(ColorButton, p_cp));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cp: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(ColorButton.Class, p_class).f_color_set = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The selected opacity value (0 fully transparent, 65535 fully opaque).
        pub const alpha = struct {
            pub const name = "alpha";

            pub const Type = c_uint;
        };

        /// The selected color.
        pub const color = struct {
            pub const name = "color";

            pub const Type = ?*gdk.Color;
        };

        /// The RGBA color.
        pub const rgba = struct {
            pub const name = "rgba";

            pub const Type = ?*gdk.RGBA;
        };

        /// Set this property to `TRUE` to skip the palette
        /// in the dialog and go directly to the color editor.
        ///
        /// This property should be used in cases where the palette
        /// in the editor would be redundant, such as when the color
        /// button is already part of a palette.
        pub const show_editor = struct {
            pub const name = "show-editor";

            pub const Type = c_int;
        };

        /// The title of the color selection dialog
        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };

        /// If this property is set to `TRUE`, the color swatch on the button is
        /// rendered against a checkerboard background to show its opacity and
        /// the opacity slider is displayed in the color selection dialog.
        pub const use_alpha = struct {
            pub const name = "use-alpha";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The ::color-set signal is emitted when the user selects a color.
        /// When handling this signal, use `gtk.ColorButton.getRgba` to
        /// find out which color was just selected.
        ///
        /// Note that this signal is only emitted when the user
        /// changes the color. If you need to react to programmatic color changes
        /// as well, use the notify::color signal.
        pub const color_set = struct {
            pub const name = "color-set";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ColorButton, p_instance))),
                    gobject.signalLookup("color-set", ColorButton.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new color button.
    ///
    /// This returns a widget in the form of a small button containing
    /// a swatch representing the current selected color. When the button
    /// is clicked, a color-selection dialog will open, allowing the user
    /// to select a color. The swatch will be updated to reflect the new
    /// color when the user finishes.
    extern fn gtk_color_button_new() *gtk.ColorButton;
    pub const new = gtk_color_button_new;

    /// Creates a new color button.
    extern fn gtk_color_button_new_with_color(p_color: *const gdk.Color) *gtk.ColorButton;
    pub const newWithColor = gtk_color_button_new_with_color;

    /// Creates a new color button.
    extern fn gtk_color_button_new_with_rgba(p_rgba: *const gdk.RGBA) *gtk.ColorButton;
    pub const newWithRgba = gtk_color_button_new_with_rgba;

    /// Returns the current alpha value.
    extern fn gtk_color_button_get_alpha(p_button: *ColorButton) u16;
    pub const getAlpha = gtk_color_button_get_alpha;

    /// Sets `color` to be the current color in the `gtk.ColorButton` widget.
    extern fn gtk_color_button_get_color(p_button: *ColorButton, p_color: *gdk.Color) void;
    pub const getColor = gtk_color_button_get_color;

    /// Sets `rgba` to be the current color in the `gtk.ColorButton` widget.
    extern fn gtk_color_button_get_rgba(p_button: *ColorButton, p_rgba: *gdk.RGBA) void;
    pub const getRgba = gtk_color_button_get_rgba;

    /// Gets the title of the color selection dialog.
    extern fn gtk_color_button_get_title(p_button: *ColorButton) [*:0]const u8;
    pub const getTitle = gtk_color_button_get_title;

    /// Does the color selection dialog use the alpha channel ?
    extern fn gtk_color_button_get_use_alpha(p_button: *ColorButton) c_int;
    pub const getUseAlpha = gtk_color_button_get_use_alpha;

    /// Sets the current opacity to be `alpha`.
    extern fn gtk_color_button_set_alpha(p_button: *ColorButton, p_alpha: u16) void;
    pub const setAlpha = gtk_color_button_set_alpha;

    /// Sets the current color to be `color`.
    extern fn gtk_color_button_set_color(p_button: *ColorButton, p_color: *const gdk.Color) void;
    pub const setColor = gtk_color_button_set_color;

    /// Sets the current color to be `rgba`.
    extern fn gtk_color_button_set_rgba(p_button: *ColorButton, p_rgba: *const gdk.RGBA) void;
    pub const setRgba = gtk_color_button_set_rgba;

    /// Sets the title for the color selection dialog.
    extern fn gtk_color_button_set_title(p_button: *ColorButton, p_title: [*:0]const u8) void;
    pub const setTitle = gtk_color_button_set_title;

    /// Sets whether or not the color button should use the alpha channel.
    extern fn gtk_color_button_set_use_alpha(p_button: *ColorButton, p_use_alpha: c_int) void;
    pub const setUseAlpha = gtk_color_button_set_use_alpha;

    extern fn gtk_color_button_get_type() usize;
    pub const getGObjectType = gtk_color_button_get_type;

    extern fn g_object_ref(p_self: *gtk.ColorButton) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ColorButton) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ColorButton, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.ColorChooserDialog` widget is a dialog for choosing
/// a color. It implements the `gtk.ColorChooser` interface.
pub const ColorChooserDialog = extern struct {
    pub const Parent = gtk.Dialog;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.ColorChooser };
    pub const Class = gtk.ColorChooserDialogClass;
    f_parent_instance: gtk.Dialog,
    f_priv: ?*gtk.ColorChooserDialogPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const show_editor = struct {
            pub const name = "show-editor";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.ColorChooserDialog`.
    extern fn gtk_color_chooser_dialog_new(p_title: ?[*:0]const u8, p_parent: ?*gtk.Window) *gtk.ColorChooserDialog;
    pub const new = gtk_color_chooser_dialog_new;

    extern fn gtk_color_chooser_dialog_get_type() usize;
    pub const getGObjectType = gtk_color_chooser_dialog_get_type;

    extern fn g_object_ref(p_self: *gtk.ColorChooserDialog) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ColorChooserDialog) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ColorChooserDialog, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.ColorChooserWidget` widget lets the user select a
/// color. By default, the chooser presents a predefined palette
/// of colors, plus a small number of settable custom colors.
/// It is also possible to select a different color with the
/// single-color editor. To enter the single-color editing mode,
/// use the context menu of any color of the palette, or use the
/// '+' button to add a new custom color.
///
/// The chooser automatically remembers the last selection, as well
/// as custom colors.
///
/// To change the initially selected color, use `gtk.ColorChooser.setRgba`.
/// To get the selected color use `gtk.ColorChooser.getRgba`.
///
/// The `gtk.ColorChooserWidget` is used in the `gtk.ColorChooserDialog`
/// to provide a dialog for selecting colors.
///
/// # CSS names
///
/// GtkColorChooserWidget has a single CSS node with name colorchooser.
pub const ColorChooserWidget = extern struct {
    pub const Parent = gtk.Box;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.ColorChooser, gtk.Orientable };
    pub const Class = gtk.ColorChooserWidgetClass;
    f_parent_instance: gtk.Box,
    f_priv: ?*gtk.ColorChooserWidgetPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The ::show-editor property is `TRUE` when the color chooser
        /// is showing the single-color editor. It can be set to switch
        /// the color chooser into single-color editing mode.
        pub const show_editor = struct {
            pub const name = "show-editor";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.ColorChooserWidget`.
    extern fn gtk_color_chooser_widget_new() *gtk.ColorChooserWidget;
    pub const new = gtk_color_chooser_widget_new;

    extern fn gtk_color_chooser_widget_get_type() usize;
    pub const getGObjectType = gtk_color_chooser_widget_get_type;

    extern fn g_object_ref(p_self: *gtk.ColorChooserWidget) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ColorChooserWidget) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ColorChooserWidget, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ColorSelection = extern struct {
    pub const Parent = gtk.Box;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.ColorSelectionClass;
    f_parent_instance: gtk.Box,
    f_private_data: ?*gtk.ColorSelectionPrivate,

    pub const virtual_methods = struct {
        pub const color_changed = struct {
            pub fn call(p_class: anytype, p_color_selection: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(ColorSelection.Class, p_class).f_color_changed.?(gobject.ext.as(ColorSelection, p_color_selection));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_color_selection: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(ColorSelection.Class, p_class).f_color_changed = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const current_alpha = struct {
            pub const name = "current-alpha";

            pub const Type = c_uint;
        };

        /// The current GdkColor color.
        pub const current_color = struct {
            pub const name = "current-color";

            pub const Type = ?*gdk.Color;
        };

        /// The current RGBA color.
        pub const current_rgba = struct {
            pub const name = "current-rgba";

            pub const Type = ?*gdk.RGBA;
        };

        pub const has_opacity_control = struct {
            pub const name = "has-opacity-control";

            pub const Type = c_int;
        };

        pub const has_palette = struct {
            pub const name = "has-palette";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// This signal is emitted when the color changes in the `gtk.ColorSelection`
        /// according to its update policy.
        pub const color_changed = struct {
            pub const name = "color-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ColorSelection, p_instance))),
                    gobject.signalLookup("color-changed", ColorSelection.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Parses a color palette string; the string is a colon-separated
    /// list of color names readable by `gdk.colorParse`.
    extern fn gtk_color_selection_palette_from_string(p_str: [*:0]const u8, p_colors: [*]*gdk.Color, p_n_colors: *c_int) c_int;
    pub const paletteFromString = gtk_color_selection_palette_from_string;

    /// Encodes a palette as a string, useful for persistent storage.
    extern fn gtk_color_selection_palette_to_string(p_colors: [*]const gdk.Color, p_n_colors: c_int) [*:0]u8;
    pub const paletteToString = gtk_color_selection_palette_to_string;

    /// Installs a global function to be called whenever the user
    /// tries to modify the palette in a color selection.
    ///
    /// This function should save the new palette contents, and update
    /// the `gtk.Settings.properties.gtk`-color-palette GtkSettings property so all
    /// GtkColorSelection widgets will be modified.
    extern fn gtk_color_selection_set_change_palette_with_screen_hook(p_func: gtk.ColorSelectionChangePaletteWithScreenFunc) gtk.ColorSelectionChangePaletteWithScreenFunc;
    pub const setChangePaletteWithScreenHook = gtk_color_selection_set_change_palette_with_screen_hook;

    /// Creates a new GtkColorSelection.
    extern fn gtk_color_selection_new() *gtk.ColorSelection;
    pub const new = gtk_color_selection_new;

    /// Returns the current alpha value.
    extern fn gtk_color_selection_get_current_alpha(p_colorsel: *ColorSelection) u16;
    pub const getCurrentAlpha = gtk_color_selection_get_current_alpha;

    /// Sets `color` to be the current color in the GtkColorSelection widget.
    extern fn gtk_color_selection_get_current_color(p_colorsel: *ColorSelection, p_color: *gdk.Color) void;
    pub const getCurrentColor = gtk_color_selection_get_current_color;

    /// Sets `rgba` to be the current color in the GtkColorSelection widget.
    extern fn gtk_color_selection_get_current_rgba(p_colorsel: *ColorSelection, p_rgba: *gdk.RGBA) void;
    pub const getCurrentRgba = gtk_color_selection_get_current_rgba;

    /// Determines whether the colorsel has an opacity control.
    extern fn gtk_color_selection_get_has_opacity_control(p_colorsel: *ColorSelection) c_int;
    pub const getHasOpacityControl = gtk_color_selection_get_has_opacity_control;

    /// Determines whether the color selector has a color palette.
    extern fn gtk_color_selection_get_has_palette(p_colorsel: *ColorSelection) c_int;
    pub const getHasPalette = gtk_color_selection_get_has_palette;

    /// Returns the previous alpha value.
    extern fn gtk_color_selection_get_previous_alpha(p_colorsel: *ColorSelection) u16;
    pub const getPreviousAlpha = gtk_color_selection_get_previous_alpha;

    /// Fills `color` in with the original color value.
    extern fn gtk_color_selection_get_previous_color(p_colorsel: *ColorSelection, p_color: *gdk.Color) void;
    pub const getPreviousColor = gtk_color_selection_get_previous_color;

    /// Fills `rgba` in with the original color value.
    extern fn gtk_color_selection_get_previous_rgba(p_colorsel: *ColorSelection, p_rgba: *gdk.RGBA) void;
    pub const getPreviousRgba = gtk_color_selection_get_previous_rgba;

    /// Gets the current state of the `colorsel`.
    extern fn gtk_color_selection_is_adjusting(p_colorsel: *ColorSelection) c_int;
    pub const isAdjusting = gtk_color_selection_is_adjusting;

    /// Sets the current opacity to be `alpha`.
    ///
    /// The first time this is called, it will also set
    /// the original opacity to be `alpha` too.
    extern fn gtk_color_selection_set_current_alpha(p_colorsel: *ColorSelection, p_alpha: u16) void;
    pub const setCurrentAlpha = gtk_color_selection_set_current_alpha;

    /// Sets the current color to be `color`.
    ///
    /// The first time this is called, it will also set
    /// the original color to be `color` too.
    extern fn gtk_color_selection_set_current_color(p_colorsel: *ColorSelection, p_color: *const gdk.Color) void;
    pub const setCurrentColor = gtk_color_selection_set_current_color;

    /// Sets the current color to be `rgba`.
    ///
    /// The first time this is called, it will also set
    /// the original color to be `rgba` too.
    extern fn gtk_color_selection_set_current_rgba(p_colorsel: *ColorSelection, p_rgba: *const gdk.RGBA) void;
    pub const setCurrentRgba = gtk_color_selection_set_current_rgba;

    /// Sets the `colorsel` to use or not use opacity.
    extern fn gtk_color_selection_set_has_opacity_control(p_colorsel: *ColorSelection, p_has_opacity: c_int) void;
    pub const setHasOpacityControl = gtk_color_selection_set_has_opacity_control;

    /// Shows and hides the palette based upon the value of `has_palette`.
    extern fn gtk_color_selection_set_has_palette(p_colorsel: *ColorSelection, p_has_palette: c_int) void;
    pub const setHasPalette = gtk_color_selection_set_has_palette;

    /// Sets the “previous” alpha to be `alpha`.
    ///
    /// This function should be called with some hesitations,
    /// as it might seem confusing to have that alpha change.
    extern fn gtk_color_selection_set_previous_alpha(p_colorsel: *ColorSelection, p_alpha: u16) void;
    pub const setPreviousAlpha = gtk_color_selection_set_previous_alpha;

    /// Sets the “previous” color to be `color`.
    ///
    /// This function should be called with some hesitations,
    /// as it might seem confusing to have that color change.
    /// Calling `gtk.ColorSelection.setCurrentColor` will also
    /// set this color the first time it is called.
    extern fn gtk_color_selection_set_previous_color(p_colorsel: *ColorSelection, p_color: *const gdk.Color) void;
    pub const setPreviousColor = gtk_color_selection_set_previous_color;

    /// Sets the “previous” color to be `rgba`.
    ///
    /// This function should be called with some hesitations,
    /// as it might seem confusing to have that color change.
    /// Calling `gtk.ColorSelection.setCurrentRgba` will also
    /// set this color the first time it is called.
    extern fn gtk_color_selection_set_previous_rgba(p_colorsel: *ColorSelection, p_rgba: *const gdk.RGBA) void;
    pub const setPreviousRgba = gtk_color_selection_set_previous_rgba;

    extern fn gtk_color_selection_get_type() usize;
    pub const getGObjectType = gtk_color_selection_get_type;

    extern fn g_object_ref(p_self: *gtk.ColorSelection) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ColorSelection) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ColorSelection, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ColorSelectionDialog = extern struct {
    pub const Parent = gtk.Dialog;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.ColorSelectionDialogClass;
    f_parent_instance: gtk.Dialog,
    f_priv: ?*gtk.ColorSelectionDialogPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const cancel_button = struct {
            pub const name = "cancel-button";

            pub const Type = ?*gtk.Widget;
        };

        pub const color_selection = struct {
            pub const name = "color-selection";

            pub const Type = ?*gtk.Widget;
        };

        pub const help_button = struct {
            pub const name = "help-button";

            pub const Type = ?*gtk.Widget;
        };

        pub const ok_button = struct {
            pub const name = "ok-button";

            pub const Type = ?*gtk.Widget;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.ColorSelectionDialog`.
    extern fn gtk_color_selection_dialog_new(p_title: [*:0]const u8) *gtk.ColorSelectionDialog;
    pub const new = gtk_color_selection_dialog_new;

    /// Retrieves the `gtk.ColorSelection` widget embedded in the dialog.
    extern fn gtk_color_selection_dialog_get_color_selection(p_colorsel: *ColorSelectionDialog) *gtk.Widget;
    pub const getColorSelection = gtk_color_selection_dialog_get_color_selection;

    extern fn gtk_color_selection_dialog_get_type() usize;
    pub const getGObjectType = gtk_color_selection_dialog_get_type;

    extern fn g_object_ref(p_self: *gtk.ColorSelectionDialog) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ColorSelectionDialog) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ColorSelectionDialog, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A GtkComboBox is a widget that allows the user to choose from a list of
/// valid choices. The GtkComboBox displays the selected choice. When
/// activated, the GtkComboBox displays a popup which allows the user to
/// make a new choice. The style in which the selected value is displayed,
/// and the style of the popup is determined by the current theme. It may
/// be similar to a Windows-style combo box.
///
/// The GtkComboBox uses the model-view pattern; the list of valid choices
/// is specified in the form of a tree model, and the display of the choices
/// can be adapted to the data in the model by using cell renderers, as you
/// would in a tree view. This is possible since GtkComboBox implements the
/// `gtk.CellLayout` interface. The tree model holding the valid choices is
/// not restricted to a flat list, it can be a real tree, and the popup will
/// reflect the tree structure.
///
/// To allow the user to enter values not in the model, the “has-entry”
/// property allows the GtkComboBox to contain a `gtk.Entry`. This entry
/// can be accessed by calling `gtk.Bin.getChild` on the combo box.
///
/// For a simple list of textual choices, the model-view API of GtkComboBox
/// can be a bit overwhelming. In this case, `gtk.ComboBoxText` offers a
/// simple alternative. Both GtkComboBox and `gtk.ComboBoxText` can contain
/// an entry.
///
/// # CSS nodes
///
/// ```
/// combobox
/// ├── box.linked
/// │   ╰── button.combo
/// │       ╰── box
/// │           ├── cellview
/// │           ╰── arrow
/// ╰── window.popup
/// ```
///
/// A normal combobox contains a box with the .linked class, a button
/// with the .combo class and inside those buttons, there are a cellview and
/// an arrow.
///
/// ```
/// combobox
/// ├── box.linked
/// │   ├── entry.combo
/// │   ╰── button.combo
/// │       ╰── box
/// │           ╰── arrow
/// ╰── window.popup
/// ```
///
/// A GtkComboBox with an entry has a single CSS node with name combobox. It
/// contains a box with the .linked class. That box contains an entry and a
/// button, both with the .combo class added.
/// The button also contains another node with name arrow.
pub const ComboBox = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.CellEditable, gtk.CellLayout };
    pub const Class = gtk.ComboBoxClass;
    f_parent_instance: gtk.Bin,
    f_priv: ?*gtk.ComboBoxPrivate,

    pub const virtual_methods = struct {
        /// Signal is emitted when the active item is changed.
        pub const changed = struct {
            pub fn call(p_class: anytype, p_combo_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(ComboBox.Class, p_class).f_changed.?(gobject.ext.as(ComboBox, p_combo_box));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_combo_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(ComboBox.Class, p_class).f_changed = @ptrCast(p_implementation);
            }
        };

        /// Signal which allows you to change how the text
        ///    displayed in a combo box’s entry is displayed.
        pub const format_entry_text = struct {
            pub fn call(p_class: anytype, p_combo_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path: [*:0]const u8) [*:0]u8 {
                return gobject.ext.as(ComboBox.Class, p_class).f_format_entry_text.?(gobject.ext.as(ComboBox, p_combo_box), p_path);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_combo_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path: [*:0]const u8) callconv(.C) [*:0]u8) void {
                gobject.ext.as(ComboBox.Class, p_class).f_format_entry_text = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The item which is currently active. If the model is a non-flat treemodel,
        /// and the active item is not an immediate child of the root of the tree,
        /// this property has the value
        /// `gtk_tree_path_get_indices (path)[0]`,
        /// where `path` is the `gtk.TreePath` of the active item.
        pub const active = struct {
            pub const name = "active";

            pub const Type = c_int;
        };

        /// The value of the ID column of the active row.
        pub const active_id = struct {
            pub const name = "active-id";

            pub const Type = ?[*:0]u8;
        };

        /// The add-tearoffs property controls whether generated menus
        /// have tearoff menu items.
        ///
        /// Note that this only affects menu style combo boxes.
        pub const add_tearoffs = struct {
            pub const name = "add-tearoffs";

            pub const Type = c_int;
        };

        /// Whether the dropdown button is sensitive when
        /// the model is empty.
        pub const button_sensitivity = struct {
            pub const name = "button-sensitivity";

            pub const Type = gtk.SensitivityType;
        };

        /// The `gtk.CellArea` used to layout cell renderers for this combo box.
        ///
        /// If no area is specified when creating the combo box with `gtk.ComboBox.newWithArea`
        /// a horizontally oriented `gtk.CellAreaBox` will be used.
        pub const cell_area = struct {
            pub const name = "cell-area";

            pub const Type = ?*gtk.CellArea;
        };

        /// If this is set to a non-negative value, it must be the index of a column
        /// of type `G_TYPE_INT` in the model. The value in that column for each item
        /// will determine how many columns that item will span in the popup.
        /// Therefore, values in this column must be greater than zero, and the sum of
        /// an item’s column position + span should not exceed `gtk.ComboBox.properties.wrap`-width.
        pub const column_span_column = struct {
            pub const name = "column-span-column";

            pub const Type = c_int;
        };

        /// The column in the combo box's model to associate with strings from the entry
        /// if the combo was created with `gtk.ComboBox.properties.has`-entry = `TRUE`.
        pub const entry_text_column = struct {
            pub const name = "entry-text-column";

            pub const Type = c_int;
        };

        /// Whether the combo box has an entry.
        pub const has_entry = struct {
            pub const name = "has-entry";

            pub const Type = c_int;
        };

        /// The has-frame property controls whether a frame
        /// is drawn around the entry.
        pub const has_frame = struct {
            pub const name = "has-frame";

            pub const Type = c_int;
        };

        /// The column in the combo box's model that provides string
        /// IDs for the values in the model, if != -1.
        pub const id_column = struct {
            pub const name = "id-column";

            pub const Type = c_int;
        };

        /// The model from which the combo box takes the values shown
        /// in the list.
        pub const model = struct {
            pub const name = "model";

            pub const Type = ?*gtk.TreeModel;
        };

        /// Whether the popup's width should be a fixed width matching the
        /// allocated width of the combo box.
        pub const popup_fixed_width = struct {
            pub const name = "popup-fixed-width";

            pub const Type = c_int;
        };

        /// Whether the combo boxes dropdown is popped up.
        /// Note that this property is mainly useful, because
        /// it allows you to connect to notify::popup-shown.
        pub const popup_shown = struct {
            pub const name = "popup-shown";

            pub const Type = c_int;
        };

        /// If this is set to a non-negative value, it must be the index of a column
        /// of type `G_TYPE_INT` in the model. The value in that column for each item
        /// will determine how many rows that item will span in the popup. Therefore,
        /// values in this column must be greater than zero.
        pub const row_span_column = struct {
            pub const name = "row-span-column";

            pub const Type = c_int;
        };

        /// A title that may be displayed by the window manager
        /// when the popup is torn-off.
        pub const tearoff_title = struct {
            pub const name = "tearoff-title";

            pub const Type = ?[*:0]u8;
        };

        /// If wrap-width is set to a positive value, items in the popup will be laid
        /// out along multiple columns, starting a new row on reaching the wrap width.
        pub const wrap_width = struct {
            pub const name = "wrap-width";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The changed signal is emitted when the active
        /// item is changed. The can be due to the user selecting
        /// a different item from the list, or due to a
        /// call to `gtk.ComboBox.setActiveIter`.
        /// It will also be emitted while typing into the entry of a combo box
        /// with an entry.
        pub const changed = struct {
            pub const name = "changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ComboBox, p_instance))),
                    gobject.signalLookup("changed", ComboBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// For combo boxes that are created with an entry (See GtkComboBox:has-entry).
        ///
        /// A signal which allows you to change how the text displayed in a combo box's
        /// entry is displayed.
        ///
        /// Connect a signal handler which returns an allocated string representing
        /// `path`. That string will then be used to set the text in the combo box's entry.
        /// The default signal handler uses the text from the GtkComboBox::entry-text-column
        /// model column.
        ///
        /// Here's an example signal handler which fetches data from the model and
        /// displays it in the entry.
        /// ```
        /// static gchar*
        /// format_entry_text_callback (GtkComboBox *combo,
        ///                             const gchar *path,
        ///                             gpointer     user_data)
        /// {
        ///   GtkTreeIter iter;
        ///   GtkTreeModel model;
        ///   gdouble      value;
        ///
        ///   model = gtk_combo_box_get_model (combo);
        ///
        ///   gtk_tree_model_get_iter_from_string (model, &iter, path);
        ///   gtk_tree_model_get (model, &iter,
        ///                       THE_DOUBLE_VALUE_COLUMN, &value,
        ///                       -1);
        ///
        ///   return g_strdup_printf ("`g`", value);
        /// }
        /// ```
        pub const format_entry_text = struct {
            pub const name = "format-entry-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_path: [*:0]u8, P_Data) callconv(.C) [*:0]u8, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ComboBox, p_instance))),
                    gobject.signalLookup("format-entry-text", ComboBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::move-active signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted to move the active selection.
        pub const move_active = struct {
            pub const name = "move-active";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_scroll_type: gtk.ScrollType, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ComboBox, p_instance))),
                    gobject.signalLookup("move-active", ComboBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::popdown signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted to popdown the combo box list.
        ///
        /// The default bindings for this signal are Alt+Up and Escape.
        pub const popdown = struct {
            pub const name = "popdown";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ComboBox, p_instance))),
                    gobject.signalLookup("popdown", ComboBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::popup signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted to popup the combo box list.
        ///
        /// The default binding for this signal is Alt+Down.
        pub const popup = struct {
            pub const name = "popup";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(ComboBox, p_instance))),
                    gobject.signalLookup("popup", ComboBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new empty `gtk.ComboBox`.
    extern fn gtk_combo_box_new() *gtk.ComboBox;
    pub const new = gtk_combo_box_new;

    /// Creates a new empty `gtk.ComboBox` using `area` to layout cells.
    extern fn gtk_combo_box_new_with_area(p_area: *gtk.CellArea) *gtk.ComboBox;
    pub const newWithArea = gtk_combo_box_new_with_area;

    /// Creates a new empty `gtk.ComboBox` with an entry.
    ///
    /// The new combo box will use `area` to layout cells.
    extern fn gtk_combo_box_new_with_area_and_entry(p_area: *gtk.CellArea) *gtk.ComboBox;
    pub const newWithAreaAndEntry = gtk_combo_box_new_with_area_and_entry;

    /// Creates a new empty `gtk.ComboBox` with an entry.
    extern fn gtk_combo_box_new_with_entry() *gtk.ComboBox;
    pub const newWithEntry = gtk_combo_box_new_with_entry;

    /// Creates a new `gtk.ComboBox` with the model initialized to `model`.
    extern fn gtk_combo_box_new_with_model(p_model: *gtk.TreeModel) *gtk.ComboBox;
    pub const newWithModel = gtk_combo_box_new_with_model;

    /// Creates a new empty `gtk.ComboBox` with an entry
    /// and with the model initialized to `model`.
    extern fn gtk_combo_box_new_with_model_and_entry(p_model: *gtk.TreeModel) *gtk.ComboBox;
    pub const newWithModelAndEntry = gtk_combo_box_new_with_model_and_entry;

    /// Returns the index of the currently active item, or -1 if there’s no
    /// active item. If the model is a non-flat treemodel, and the active item
    /// is not an immediate child of the root of the tree, this function returns
    /// `gtk_tree_path_get_indices (path)[0]`, where
    /// `path` is the `gtk.TreePath` of the active item.
    extern fn gtk_combo_box_get_active(p_combo_box: *ComboBox) c_int;
    pub const getActive = gtk_combo_box_get_active;

    /// Returns the ID of the active row of `combo_box`.  This value is taken
    /// from the active row and the column specified by the `gtk.ComboBox.properties.id`-column
    /// property of `combo_box` (see `gtk.ComboBox.setIdColumn`).
    ///
    /// The returned value is an interned string which means that you can
    /// compare the pointer by value to other interned strings and that you
    /// must not free it.
    ///
    /// If the `gtk.ComboBox.properties.id`-column property of `combo_box` is not set, or if
    /// no row is active, or if the active row has a `NULL` ID value, then `NULL`
    /// is returned.
    extern fn gtk_combo_box_get_active_id(p_combo_box: *ComboBox) ?[*:0]const u8;
    pub const getActiveId = gtk_combo_box_get_active_id;

    /// Sets `iter` to point to the currently active item, if any item is active.
    /// Otherwise, `iter` is left unchanged.
    extern fn gtk_combo_box_get_active_iter(p_combo_box: *ComboBox, p_iter: *gtk.TreeIter) c_int;
    pub const getActiveIter = gtk_combo_box_get_active_iter;

    /// Gets the current value of the :add-tearoffs property.
    extern fn gtk_combo_box_get_add_tearoffs(p_combo_box: *ComboBox) c_int;
    pub const getAddTearoffs = gtk_combo_box_get_add_tearoffs;

    /// Returns whether the combo box sets the dropdown button
    /// sensitive or not when there are no items in the model.
    extern fn gtk_combo_box_get_button_sensitivity(p_combo_box: *ComboBox) gtk.SensitivityType;
    pub const getButtonSensitivity = gtk_combo_box_get_button_sensitivity;

    /// Returns the column with column span information for `combo_box`.
    extern fn gtk_combo_box_get_column_span_column(p_combo_box: *ComboBox) c_int;
    pub const getColumnSpanColumn = gtk_combo_box_get_column_span_column;

    /// Returns the column which `combo_box` is using to get the strings
    /// from to display in the internal entry.
    extern fn gtk_combo_box_get_entry_text_column(p_combo_box: *ComboBox) c_int;
    pub const getEntryTextColumn = gtk_combo_box_get_entry_text_column;

    /// Returns whether the combo box grabs focus when it is clicked
    /// with the mouse. See `gtk.ComboBox.setFocusOnClick`.
    extern fn gtk_combo_box_get_focus_on_click(p_combo: *ComboBox) c_int;
    pub const getFocusOnClick = gtk_combo_box_get_focus_on_click;

    /// Returns whether the combo box has an entry.
    extern fn gtk_combo_box_get_has_entry(p_combo_box: *ComboBox) c_int;
    pub const getHasEntry = gtk_combo_box_get_has_entry;

    /// Returns the column which `combo_box` is using to get string IDs
    /// for values from.
    extern fn gtk_combo_box_get_id_column(p_combo_box: *ComboBox) c_int;
    pub const getIdColumn = gtk_combo_box_get_id_column;

    /// Returns the `gtk.TreeModel` which is acting as data source for `combo_box`.
    extern fn gtk_combo_box_get_model(p_combo_box: *ComboBox) *gtk.TreeModel;
    pub const getModel = gtk_combo_box_get_model;

    /// Gets the accessible object corresponding to the combo box’s popup.
    ///
    /// This function is mostly intended for use by accessibility technologies;
    /// applications should have little use for it.
    extern fn gtk_combo_box_get_popup_accessible(p_combo_box: *ComboBox) *atk.Object;
    pub const getPopupAccessible = gtk_combo_box_get_popup_accessible;

    /// Gets whether the popup uses a fixed width matching
    /// the allocated width of the combo box.
    extern fn gtk_combo_box_get_popup_fixed_width(p_combo_box: *ComboBox) c_int;
    pub const getPopupFixedWidth = gtk_combo_box_get_popup_fixed_width;

    /// Returns the current row separator function.
    extern fn gtk_combo_box_get_row_separator_func(p_combo_box: *ComboBox) gtk.TreeViewRowSeparatorFunc;
    pub const getRowSeparatorFunc = gtk_combo_box_get_row_separator_func;

    /// Returns the column with row span information for `combo_box`.
    extern fn gtk_combo_box_get_row_span_column(p_combo_box: *ComboBox) c_int;
    pub const getRowSpanColumn = gtk_combo_box_get_row_span_column;

    /// Gets the current title of the menu in tearoff mode. See
    /// `gtk.ComboBox.setAddTearoffs`.
    extern fn gtk_combo_box_get_title(p_combo_box: *ComboBox) [*:0]const u8;
    pub const getTitle = gtk_combo_box_get_title;

    /// Returns the wrap width which is used to determine the number of columns
    /// for the popup menu. If the wrap width is larger than 1, the combo box
    /// is in table mode.
    extern fn gtk_combo_box_get_wrap_width(p_combo_box: *ComboBox) c_int;
    pub const getWrapWidth = gtk_combo_box_get_wrap_width;

    /// Hides the menu or dropdown list of `combo_box`.
    ///
    /// This function is mostly intended for use by accessibility technologies;
    /// applications should have little use for it.
    extern fn gtk_combo_box_popdown(p_combo_box: *ComboBox) void;
    pub const popdown = gtk_combo_box_popdown;

    /// Pops up the menu or dropdown list of `combo_box`.
    ///
    /// This function is mostly intended for use by accessibility technologies;
    /// applications should have little use for it.
    ///
    /// Before calling this, `combo_box` must be mapped, or nothing will happen.
    extern fn gtk_combo_box_popup(p_combo_box: *ComboBox) void;
    pub const popup = gtk_combo_box_popup;

    /// Pops up the menu or dropdown list of `combo_box`, the popup window
    /// will be grabbed so only `device` and its associated pointer/keyboard
    /// are the only `GdkDevices` able to send events to it.
    extern fn gtk_combo_box_popup_for_device(p_combo_box: *ComboBox, p_device: *gdk.Device) void;
    pub const popupForDevice = gtk_combo_box_popup_for_device;

    /// Sets the active item of `combo_box` to be the item at `index`.
    extern fn gtk_combo_box_set_active(p_combo_box: *ComboBox, p_index_: c_int) void;
    pub const setActive = gtk_combo_box_set_active;

    /// Changes the active row of `combo_box` to the one that has an ID equal to
    /// `active_id`, or unsets the active row if `active_id` is `NULL`.  Rows having
    /// a `NULL` ID string cannot be made active by this function.
    ///
    /// If the `gtk.ComboBox.properties.id`-column property of `combo_box` is unset or if no
    /// row has the given ID then the function does nothing and returns `FALSE`.
    extern fn gtk_combo_box_set_active_id(p_combo_box: *ComboBox, p_active_id: ?[*:0]const u8) c_int;
    pub const setActiveId = gtk_combo_box_set_active_id;

    /// Sets the current active item to be the one referenced by `iter`, or
    /// unsets the active item if `iter` is `NULL`.
    extern fn gtk_combo_box_set_active_iter(p_combo_box: *ComboBox, p_iter: ?*gtk.TreeIter) void;
    pub const setActiveIter = gtk_combo_box_set_active_iter;

    /// Sets whether the popup menu should have a tearoff
    /// menu item.
    extern fn gtk_combo_box_set_add_tearoffs(p_combo_box: *ComboBox, p_add_tearoffs: c_int) void;
    pub const setAddTearoffs = gtk_combo_box_set_add_tearoffs;

    /// Sets whether the dropdown button of the combo box should be
    /// always sensitive (`GTK_SENSITIVITY_ON`), never sensitive (`GTK_SENSITIVITY_OFF`)
    /// or only if there is at least one item to display (`GTK_SENSITIVITY_AUTO`).
    extern fn gtk_combo_box_set_button_sensitivity(p_combo_box: *ComboBox, p_sensitivity: gtk.SensitivityType) void;
    pub const setButtonSensitivity = gtk_combo_box_set_button_sensitivity;

    /// Sets the column with column span information for `combo_box` to be
    /// `column_span`. The column span column contains integers which indicate
    /// how many columns an item should span.
    extern fn gtk_combo_box_set_column_span_column(p_combo_box: *ComboBox, p_column_span: c_int) void;
    pub const setColumnSpanColumn = gtk_combo_box_set_column_span_column;

    /// Sets the model column which `combo_box` should use to get strings from
    /// to be `text_column`. The column `text_column` in the model of `combo_box`
    /// must be of type `G_TYPE_STRING`.
    ///
    /// This is only relevant if `combo_box` has been created with
    /// `gtk.ComboBox.properties.has`-entry as `TRUE`.
    extern fn gtk_combo_box_set_entry_text_column(p_combo_box: *ComboBox, p_text_column: c_int) void;
    pub const setEntryTextColumn = gtk_combo_box_set_entry_text_column;

    /// Sets whether the combo box will grab focus when it is clicked with
    /// the mouse. Making mouse clicks not grab focus is useful in places
    /// like toolbars where you don’t want the keyboard focus removed from
    /// the main area of the application.
    extern fn gtk_combo_box_set_focus_on_click(p_combo: *ComboBox, p_focus_on_click: c_int) void;
    pub const setFocusOnClick = gtk_combo_box_set_focus_on_click;

    /// Sets the model column which `combo_box` should use to get string IDs
    /// for values from. The column `id_column` in the model of `combo_box`
    /// must be of type `G_TYPE_STRING`.
    extern fn gtk_combo_box_set_id_column(p_combo_box: *ComboBox, p_id_column: c_int) void;
    pub const setIdColumn = gtk_combo_box_set_id_column;

    /// Sets the model used by `combo_box` to be `model`. Will unset a previously set
    /// model (if applicable). If model is `NULL`, then it will unset the model.
    ///
    /// Note that this function does not clear the cell renderers, you have to
    /// call `gtk.CellLayout.clear` yourself if you need to set up different
    /// cell renderers for the new model.
    extern fn gtk_combo_box_set_model(p_combo_box: *ComboBox, p_model: ?*gtk.TreeModel) void;
    pub const setModel = gtk_combo_box_set_model;

    /// Specifies whether the popup’s width should be a fixed width
    /// matching the allocated width of the combo box.
    extern fn gtk_combo_box_set_popup_fixed_width(p_combo_box: *ComboBox, p_fixed: c_int) void;
    pub const setPopupFixedWidth = gtk_combo_box_set_popup_fixed_width;

    /// Sets the row separator function, which is used to determine
    /// whether a row should be drawn as a separator. If the row separator
    /// function is `NULL`, no separators are drawn. This is the default value.
    extern fn gtk_combo_box_set_row_separator_func(p_combo_box: *ComboBox, p_func: gtk.TreeViewRowSeparatorFunc, p_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) void;
    pub const setRowSeparatorFunc = gtk_combo_box_set_row_separator_func;

    /// Sets the column with row span information for `combo_box` to be `row_span`.
    /// The row span column contains integers which indicate how many rows
    /// an item should span.
    extern fn gtk_combo_box_set_row_span_column(p_combo_box: *ComboBox, p_row_span: c_int) void;
    pub const setRowSpanColumn = gtk_combo_box_set_row_span_column;

    /// Sets the menu’s title in tearoff mode.
    extern fn gtk_combo_box_set_title(p_combo_box: *ComboBox, p_title: [*:0]const u8) void;
    pub const setTitle = gtk_combo_box_set_title;

    /// Sets the wrap width of `combo_box` to be `width`. The wrap width is basically
    /// the preferred number of columns when you want the popup to be layed out
    /// in a table.
    extern fn gtk_combo_box_set_wrap_width(p_combo_box: *ComboBox, p_width: c_int) void;
    pub const setWrapWidth = gtk_combo_box_set_wrap_width;

    extern fn gtk_combo_box_get_type() usize;
    pub const getGObjectType = gtk_combo_box_get_type;

    extern fn g_object_ref(p_self: *gtk.ComboBox) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ComboBox) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ComboBox, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ComboBoxAccessible = extern struct {
    pub const Parent = gtk.ContainerAccessible;
    pub const Implements = [_]type{ atk.Action, atk.Component, atk.Selection };
    pub const Class = gtk.ComboBoxAccessibleClass;
    f_parent: gtk.ContainerAccessible,
    f_priv: ?*gtk.ComboBoxAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_combo_box_accessible_get_type() usize;
    pub const getGObjectType = gtk_combo_box_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.ComboBoxAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ComboBoxAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ComboBoxAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A GtkComboBoxText is a simple variant of `gtk.ComboBox` that hides
/// the model-view complexity for simple text-only use cases.
///
/// To create a GtkComboBoxText, use `gtk.ComboBoxText.new` or
/// `gtk.ComboBoxText.newWithEntry`.
///
/// You can add items to a GtkComboBoxText with
/// `gtk.ComboBoxText.appendText`, `gtk.ComboBoxText.insertText`
/// or `gtk.ComboBoxText.prependText` and remove options with
/// `gtk.ComboBoxText.remove`.
///
/// If the GtkComboBoxText contains an entry (via the “has-entry” property),
/// its contents can be retrieved using `gtk.ComboBoxText.getActiveText`.
/// The entry itself can be accessed by calling `gtk.Bin.getChild` on the
/// combo box.
///
/// You should not call `gtk.ComboBox.setModel` or attempt to pack more cells
/// into this combo box via its GtkCellLayout interface.
///
/// # GtkComboBoxText as GtkBuildable
///
/// The GtkComboBoxText implementation of the GtkBuildable interface supports
/// adding items directly using the `<items>` element and specifying `<item>`
/// elements for each item. Each `<item>` element can specify the “id”
/// corresponding to the appended text and also supports the regular
/// translation attributes “translatable”, “context” and “comments”.
///
/// Here is a UI definition fragment specifying GtkComboBoxText items:
///
/// ```
/// <object class="GtkComboBoxText">
///   <items>
///     <item translatable="yes" id="factory">Factory</item>
///     <item translatable="yes" id="home">Home</item>
///     <item translatable="yes" id="subway">Subway</item>
///   </items>
/// </object>
/// ```
///
/// # CSS nodes
///
/// ```
/// combobox
/// ╰── box.linked
///     ├── entry.combo
///     ├── button.combo
///     ╰── window.popup
/// ```
///
/// GtkComboBoxText has a single CSS node with name combobox. It adds
/// the style class .combo to the main CSS nodes of its entry and button
/// children, and the .linked class to the node of its internal box.
pub const ComboBoxText = extern struct {
    pub const Parent = gtk.ComboBox;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.CellEditable, gtk.CellLayout };
    pub const Class = gtk.ComboBoxTextClass;
    f_parent_instance: gtk.ComboBox,
    f_priv: ?*gtk.ComboBoxTextPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtk.ComboBoxText`, which is a `gtk.ComboBox` just displaying
    /// strings.
    extern fn gtk_combo_box_text_new() *gtk.ComboBoxText;
    pub const new = gtk_combo_box_text_new;

    /// Creates a new `gtk.ComboBoxText`, which is a `gtk.ComboBox` just displaying
    /// strings. The combo box created by this function has an entry.
    extern fn gtk_combo_box_text_new_with_entry() *gtk.ComboBoxText;
    pub const newWithEntry = gtk_combo_box_text_new_with_entry;

    /// Appends `text` to the list of strings stored in `combo_box`.
    /// If `id` is non-`NULL` then it is used as the ID of the row.
    ///
    /// This is the same as calling `gtk.ComboBoxText.insert` with a
    /// position of -1.
    extern fn gtk_combo_box_text_append(p_combo_box: *ComboBoxText, p_id: ?[*:0]const u8, p_text: [*:0]const u8) void;
    pub const append = gtk_combo_box_text_append;

    /// Appends `text` to the list of strings stored in `combo_box`.
    ///
    /// This is the same as calling `gtk.ComboBoxText.insertText` with a
    /// position of -1.
    extern fn gtk_combo_box_text_append_text(p_combo_box: *ComboBoxText, p_text: [*:0]const u8) void;
    pub const appendText = gtk_combo_box_text_append_text;

    /// Returns the currently active string in `combo_box`, or `NULL`
    /// if none is selected. If `combo_box` contains an entry, this
    /// function will return its contents (which will not necessarily
    /// be an item from the list).
    extern fn gtk_combo_box_text_get_active_text(p_combo_box: *ComboBoxText) [*:0]u8;
    pub const getActiveText = gtk_combo_box_text_get_active_text;

    /// Inserts `text` at `position` in the list of strings stored in `combo_box`.
    /// If `id` is non-`NULL` then it is used as the ID of the row.  See
    /// `gtk.ComboBox.properties.id`-column.
    ///
    /// If `position` is negative then `text` is appended.
    extern fn gtk_combo_box_text_insert(p_combo_box: *ComboBoxText, p_position: c_int, p_id: ?[*:0]const u8, p_text: [*:0]const u8) void;
    pub const insert = gtk_combo_box_text_insert;

    /// Inserts `text` at `position` in the list of strings stored in `combo_box`.
    ///
    /// If `position` is negative then `text` is appended.
    ///
    /// This is the same as calling `gtk.ComboBoxText.insert` with a `NULL`
    /// ID string.
    extern fn gtk_combo_box_text_insert_text(p_combo_box: *ComboBoxText, p_position: c_int, p_text: [*:0]const u8) void;
    pub const insertText = gtk_combo_box_text_insert_text;

    /// Prepends `text` to the list of strings stored in `combo_box`.
    /// If `id` is non-`NULL` then it is used as the ID of the row.
    ///
    /// This is the same as calling `gtk.ComboBoxText.insert` with a
    /// position of 0.
    extern fn gtk_combo_box_text_prepend(p_combo_box: *ComboBoxText, p_id: ?[*:0]const u8, p_text: [*:0]const u8) void;
    pub const prepend = gtk_combo_box_text_prepend;

    /// Prepends `text` to the list of strings stored in `combo_box`.
    ///
    /// This is the same as calling `gtk.ComboBoxText.insertText` with a
    /// position of 0.
    extern fn gtk_combo_box_text_prepend_text(p_combo_box: *ComboBoxText, p_text: [*:0]const u8) void;
    pub const prependText = gtk_combo_box_text_prepend_text;

    /// Removes the string at `position` from `combo_box`.
    extern fn gtk_combo_box_text_remove(p_combo_box: *ComboBoxText, p_position: c_int) void;
    pub const remove = gtk_combo_box_text_remove;

    /// Removes all the text entries from the combo box.
    extern fn gtk_combo_box_text_remove_all(p_combo_box: *ComboBoxText) void;
    pub const removeAll = gtk_combo_box_text_remove_all;

    extern fn gtk_combo_box_text_get_type() usize;
    pub const getGObjectType = gtk_combo_box_text_get_type;

    extern fn g_object_ref(p_self: *gtk.ComboBoxText) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ComboBoxText) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ComboBoxText, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A GTK+ user interface is constructed by nesting widgets inside widgets.
/// Container widgets are the inner nodes in the resulting tree of widgets:
/// they contain other widgets. So, for example, you might have a `gtk.Window`
/// containing a `gtk.Frame` containing a `gtk.Label`. If you wanted an image instead
/// of a textual label inside the frame, you might replace the `gtk.Label` widget
/// with a `gtk.Image` widget.
///
/// There are two major kinds of container widgets in GTK+. Both are subclasses
/// of the abstract GtkContainer base class.
///
/// The first type of container widget has a single child widget and derives
/// from `gtk.Bin`. These containers are decorators, which
/// add some kind of functionality to the child. For example, a `gtk.Button` makes
/// its child into a clickable button; a `gtk.Frame` draws a frame around its child
/// and a `gtk.Window` places its child widget inside a top-level window.
///
/// The second type of container can have more than one child; its purpose is to
/// manage layout. This means that these containers assign
/// sizes and positions to their children. For example, a `gtk.HBox` arranges its
/// children in a horizontal row, and a `gtk.Grid` arranges the widgets it contains
/// in a two-dimensional grid.
///
/// For implementations of `gtk.Container` the virtual method `gtk.ContainerClass.virtual_methods.forall`
/// is always required, since it's used for drawing and other internal operations
/// on the children.
/// If the `gtk.Container` implementation expect to have non internal children
/// it's needed to implement both `gtk.ContainerClass.virtual_methods.add` and `gtk.ContainerClass.virtual_methods.remove`.
/// If the GtkContainer implementation has internal children, they should be added
/// with `gtk.Widget.setParent` on `init` and removed with `gtk.Widget.unparent`
/// in the `gtk.WidgetClass.virtual_methods.destroy` implementation.
/// See more about implementing custom widgets at https://wiki.gnome.org/HowDoI/CustomWidgets
///
/// # Height for width geometry management
///
/// GTK+ uses a height-for-width (and width-for-height) geometry management system.
/// Height-for-width means that a widget can change how much vertical space it needs,
/// depending on the amount of horizontal space that it is given (and similar for
/// width-for-height).
///
/// There are some things to keep in mind when implementing container widgets
/// that make use of GTK+’s height for width geometry management system. First,
/// it’s important to note that a container must prioritize one of its
/// dimensions, that is to say that a widget or container can only have a
/// `gtk.SizeRequestMode` that is `GTK_SIZE_REQUEST_HEIGHT_FOR_WIDTH` or
/// `GTK_SIZE_REQUEST_WIDTH_FOR_HEIGHT`. However, every widget and container
/// must be able to respond to the APIs for both dimensions, i.e. even if a
/// widget has a request mode that is height-for-width, it is possible that
/// its parent will request its sizes using the width-for-height APIs.
///
/// To ensure that everything works properly, here are some guidelines to follow
/// when implementing height-for-width (or width-for-height) containers.
///
/// Each request mode involves 2 virtual methods. Height-for-width apis run
/// through `gtk.Widget.getPreferredWidth` and then through `gtk.Widget.getPreferredHeightForWidth`.
/// When handling requests in the opposite `gtk.SizeRequestMode` it is important that
/// every widget request at least enough space to display all of its content at all times.
///
/// When `gtk.Widget.getPreferredHeight` is called on a container that is height-for-width,
/// the container must return the height for its minimum width. This is easily achieved by
/// simply calling the reverse apis implemented for itself as follows:
///
/// ```
/// static void
/// foo_container_get_preferred_height (GtkWidget *widget,
///                                     gint *min_height,
///                                     gint *nat_height)
/// {
///    if (i_am_in_height_for_width_mode)
///      {
///        gint min_width;
///
///        GTK_WIDGET_GET_CLASS (widget)->get_preferred_width (widget,
///                                                            &min_width,
///                                                            NULL);
///        GTK_WIDGET_GET_CLASS (widget)->get_preferred_height_for_width
///                                                           (widget,
///                                                            min_width,
///                                                            min_height,
///                                                            nat_height);
///      }
///    else
///      {
///        ... many containers support both request modes, execute the
///        real width-for-height request here by returning the
///        collective heights of all widgets that are stacked
///        vertically (or whatever is appropriate for this container)
///        ...
///      }
/// }
/// ```
///
/// Similarly, when `gtk.Widget.getPreferredWidthForHeight` is called for a container or widget
/// that is height-for-width, it then only needs to return the base minimum width like so:
///
/// ```
/// static void
/// foo_container_get_preferred_width_for_height (GtkWidget *widget,
///                                               gint for_height,
///                                               gint *min_width,
///                                               gint *nat_width)
/// {
///    if (i_am_in_height_for_width_mode)
///      {
///        GTK_WIDGET_GET_CLASS (widget)->get_preferred_width (widget,
///                                                            min_width,
///                                                            nat_width);
///      }
///    else
///      {
///        ... execute the real width-for-height request here based on
///        the required width of the children collectively if the
///        container were to be allocated the said height ...
///      }
/// }
/// ```
///
/// Height for width requests are generally implemented in terms of a virtual allocation
/// of widgets in the input orientation. Assuming an height-for-width request mode, a container
/// would implement the `get_preferred_height_for_width` virtual function by first calling
/// `gtk.Widget.getPreferredWidth` for each of its children.
///
/// For each potential group of children that are lined up horizontally, the values returned by
/// `gtk.Widget.getPreferredWidth` should be collected in an array of `gtk.RequestedSize` structures.
/// Any child spacing should be removed from the input `for_width` and then the collective size should be
/// allocated using the `gtk.distributeNaturalAllocation` convenience function.
///
/// The container will then move on to request the preferred height for each child by using
/// `gtk.Widget.getPreferredHeightForWidth` and using the sizes stored in the `gtk.RequestedSize` array.
///
/// To allocate a height-for-width container, it’s again important
/// to consider that a container must prioritize one dimension over the other. So if
/// a container is a height-for-width container it must first allocate all widgets horizontally
/// using a `gtk.RequestedSize` array and `gtk.distributeNaturalAllocation` and then add any
/// extra space (if and where appropriate) for the widget to expand.
///
/// After adding all the expand space, the container assumes it was allocated sufficient
/// height to fit all of its content. At this time, the container must use the total horizontal sizes
/// of each widget to request the height-for-width of each of its children and store the requests in a
/// `gtk.RequestedSize` array for any widgets that stack vertically (for tabular containers this can
/// be generalized into the heights and widths of rows and columns).
/// The vertical space must then again be distributed using `gtk.distributeNaturalAllocation`
/// while this time considering the allocated height of the widget minus any vertical spacing
/// that the container adds. Then vertical expand space should be added where appropriate and available
/// and the container should go on to actually allocating the child widgets.
///
/// See [GtkWidget’s geometry management section][geometry-management]
/// to learn more about implementing height-for-width geometry management for widgets.
///
/// # Child properties
///
/// GtkContainer introduces child properties.
/// These are object properties that are not specific
/// to either the container or the contained widget, but rather to their relation.
/// Typical examples of child properties are the position or pack-type of a widget
/// which is contained in a `gtk.Box`.
///
/// Use `gtk.ContainerClass.installChildProperty` to install child properties
/// for a container class and `gtk.ContainerClass.findChildProperty` or
/// `gtk.ContainerClass.listChildProperties` to get information about existing
/// child properties.
///
/// To set the value of a child property, use `gtk.Container.childSetProperty`,
/// `gtk.Container.childSet` or `gtk.Container.childSetValist`.
/// To obtain the value of a child property, use
/// `gtk.Container.childGetProperty`, `gtk.Container.childGet` or
/// `gtk.Container.childGetValist`. To emit notification about child property
/// changes, use `gtk.Widget.childNotify`.
///
/// # GtkContainer as GtkBuildable
///
/// The GtkContainer implementation of the GtkBuildable interface supports
/// a `<packing>` element for children, which can contain multiple `<property>`
/// elements that specify child properties for the child.
///
/// Since 2.16, child properties can also be marked as translatable using
/// the same “translatable”, “comments” and “context” attributes that are used
/// for regular properties.
///
/// Since 3.16, containers can have a `<focus-chain>` element containing multiple
/// `<widget>` elements, one for each child that should be added to the focus
/// chain. The ”name” attribute gives the id of the widget.
///
/// An example of these properties in UI definitions:
///
/// ```
/// <object class="GtkBox">
///   <child>
///     <object class="GtkEntry" id="entry1"/>
///     <packing>
///       <property name="pack-type">start</property>
///     </packing>
///   </child>
///   <child>
///     <object class="GtkEntry" id="entry2"/>
///   </child>
///   <focus-chain>
///     <widget name="entry1"/>
///     <widget name="entry2"/>
///   </focus-chain>
/// </object>
/// ```
pub const Container = extern struct {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.ContainerClass;
    f_widget: gtk.Widget,
    f_priv: ?*gtk.ContainerPrivate,

    pub const virtual_methods = struct {
        /// Adds `widget` to `container`. Typically used for simple containers
        /// such as `gtk.Window`, `gtk.Frame`, or `gtk.Button`; for more complicated
        /// layout containers such as `gtk.Box` or `gtk.Grid`, this function will
        /// pick default packing parameters that may not be correct.  So
        /// consider functions such as `gtk.Box.packStart` and
        /// `gtk.Grid.attach` as an alternative to `gtk.Container.add` in
        /// those cases. A widget may be added to only one container at a time;
        /// you can’t place the same widget inside two different containers.
        ///
        /// Note that some containers, such as `gtk.ScrolledWindow` or `gtk.ListBox`,
        /// may add intermediate children between the added widget and the
        /// container.
        pub const add = struct {
            pub fn call(p_class: anytype, p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget) void {
                return gobject.ext.as(Container.Class, p_class).f_add.?(gobject.ext.as(Container, p_container), p_widget);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget) callconv(.C) void) void {
                gobject.ext.as(Container.Class, p_class).f_add = @ptrCast(p_implementation);
            }
        };

        /// Signal emitted when a size recalculation is needed.
        pub const check_resize = struct {
            pub fn call(p_class: anytype, p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Container.Class, p_class).f_check_resize.?(gobject.ext.as(Container, p_container));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Container.Class, p_class).f_check_resize = @ptrCast(p_implementation);
            }
        };

        /// Returns the type of the children supported by the container.
        ///
        /// Note that this may return `G_TYPE_NONE` to indicate that no more
        /// children can be added, e.g. for a `gtk.Paned` which already has two
        /// children.
        pub const child_type = struct {
            pub fn call(p_class: anytype, p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) usize {
                return gobject.ext.as(Container.Class, p_class).f_child_type.?(gobject.ext.as(Container, p_container));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) usize) void {
                gobject.ext.as(Container.Class, p_class).f_child_type = @ptrCast(p_implementation);
            }
        };

        /// Gets a widget’s composite name. Deprecated: 3.10.
        pub const composite_name = struct {
            pub fn call(p_class: anytype, p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget) [*:0]u8 {
                return gobject.ext.as(Container.Class, p_class).f_composite_name.?(gobject.ext.as(Container, p_container), p_child);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget) callconv(.C) [*:0]u8) void {
                gobject.ext.as(Container.Class, p_class).f_composite_name = @ptrCast(p_implementation);
            }
        };

        /// Invokes `callback` on each direct child of `container`, including
        /// children that are considered “internal” (implementation details
        /// of the container). “Internal” children generally weren’t added
        /// by the user of the container, but were added by the container
        /// implementation itself.
        ///
        /// Most applications should use `gtk.Container.foreach`, rather
        /// than `gtk.Container.forall`.
        pub const forall = struct {
            pub fn call(p_class: anytype, p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_include_internals: c_int, p_callback: gtk.Callback, p_callback_data: ?*anyopaque) void {
                return gobject.ext.as(Container.Class, p_class).f_forall.?(gobject.ext.as(Container, p_container), p_include_internals, p_callback, p_callback_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_include_internals: c_int, p_callback: gtk.Callback, p_callback_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Container.Class, p_class).f_forall = @ptrCast(p_implementation);
            }
        };

        /// Get a property from a child of container.
        pub const get_child_property = struct {
            pub fn call(p_class: anytype, p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget, p_property_id: c_uint, p_value: *gobject.Value, p_pspec: *gobject.ParamSpec) void {
                return gobject.ext.as(Container.Class, p_class).f_get_child_property.?(gobject.ext.as(Container, p_container), p_child, p_property_id, p_value, p_pspec);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget, p_property_id: c_uint, p_value: *gobject.Value, p_pspec: *gobject.ParamSpec) callconv(.C) void) void {
                gobject.ext.as(Container.Class, p_class).f_get_child_property = @ptrCast(p_implementation);
            }
        };

        /// Returns a newly created widget path representing all the widget hierarchy
        /// from the toplevel down to and including `child`.
        pub const get_path_for_child = struct {
            pub fn call(p_class: anytype, p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget) *gtk.WidgetPath {
                return gobject.ext.as(Container.Class, p_class).f_get_path_for_child.?(gobject.ext.as(Container, p_container), p_child);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget) callconv(.C) *gtk.WidgetPath) void {
                gobject.ext.as(Container.Class, p_class).f_get_path_for_child = @ptrCast(p_implementation);
            }
        };

        /// Removes `widget` from `container`. `widget` must be inside `container`.
        /// Note that `container` will own a reference to `widget`, and that this
        /// may be the last reference held; so removing a widget from its
        /// container can destroy that widget. If you want to use `widget`
        /// again, you need to add a reference to it before removing it from
        /// a container, using `gobject.Object.ref`. If you don’t want to use `widget`
        /// again it’s usually more efficient to simply destroy it directly
        /// using `gtk.Widget.destroy` since this will remove it from the
        /// container and help break any circular reference count cycles.
        pub const remove = struct {
            pub fn call(p_class: anytype, p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget) void {
                return gobject.ext.as(Container.Class, p_class).f_remove.?(gobject.ext.as(Container, p_container), p_widget);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_widget: *gtk.Widget) callconv(.C) void) void {
                gobject.ext.as(Container.Class, p_class).f_remove = @ptrCast(p_implementation);
            }
        };

        /// Set a property on a child of container.
        pub const set_child_property = struct {
            pub fn call(p_class: anytype, p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget, p_property_id: c_uint, p_value: *const gobject.Value, p_pspec: *gobject.ParamSpec) void {
                return gobject.ext.as(Container.Class, p_class).f_set_child_property.?(gobject.ext.as(Container, p_container), p_child, p_property_id, p_value, p_pspec);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget, p_property_id: c_uint, p_value: *const gobject.Value, p_pspec: *gobject.ParamSpec) callconv(.C) void) void {
                gobject.ext.as(Container.Class, p_class).f_set_child_property = @ptrCast(p_implementation);
            }
        };

        /// Sets, or unsets if `child` is `NULL`, the focused child of `container`.
        ///
        /// This function emits the GtkContainer::set_focus_child signal of
        /// `container`. Implementations of `gtk.Container` can override the
        /// default behaviour by overriding the class closure of this signal.
        ///
        /// This is function is mostly meant to be used by widgets. Applications can use
        /// `gtk.Widget.grabFocus` to manually set the focus to a specific widget.
        pub const set_focus_child = struct {
            pub fn call(p_class: anytype, p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: ?*gtk.Widget) void {
                return gobject.ext.as(Container.Class, p_class).f_set_focus_child.?(gobject.ext.as(Container, p_container), p_child);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_container: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: ?*gtk.Widget) callconv(.C) void) void {
                gobject.ext.as(Container.Class, p_class).f_set_focus_child = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const border_width = struct {
            pub const name = "border-width";

            pub const Type = c_uint;
        };

        pub const child = struct {
            pub const name = "child";

            pub const Type = ?*gtk.Widget;
        };

        pub const resize_mode = struct {
            pub const name = "resize-mode";

            pub const Type = gtk.ResizeMode;
        };
    };

    pub const signals = struct {
        pub const add = struct {
            pub const name = "add";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: *gtk.Widget, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Container, p_instance))),
                    gobject.signalLookup("add", Container.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const check_resize = struct {
            pub const name = "check-resize";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Container, p_instance))),
                    gobject.signalLookup("check-resize", Container.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const remove = struct {
            pub const name = "remove";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: *gtk.Widget, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Container, p_instance))),
                    gobject.signalLookup("remove", Container.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const set_focus_child = struct {
            pub const name = "set-focus-child";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: *gtk.Widget, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Container, p_instance))),
                    gobject.signalLookup("set-focus-child", Container.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Adds `widget` to `container`. Typically used for simple containers
    /// such as `gtk.Window`, `gtk.Frame`, or `gtk.Button`; for more complicated
    /// layout containers such as `gtk.Box` or `gtk.Grid`, this function will
    /// pick default packing parameters that may not be correct.  So
    /// consider functions such as `gtk.Box.packStart` and
    /// `gtk.Grid.attach` as an alternative to `gtk.Container.add` in
    /// those cases. A widget may be added to only one container at a time;
    /// you can’t place the same widget inside two different containers.
    ///
    /// Note that some containers, such as `gtk.ScrolledWindow` or `gtk.ListBox`,
    /// may add intermediate children between the added widget and the
    /// container.
    extern fn gtk_container_add(p_container: *Container, p_widget: *gtk.Widget) void;
    pub const add = gtk_container_add;

    /// Adds `widget` to `container`, setting child properties at the same time.
    /// See `gtk.Container.add` and `gtk.Container.childSet` for more details.
    extern fn gtk_container_add_with_properties(p_container: *Container, p_widget: *gtk.Widget, p_first_prop_name: [*:0]const u8, ...) void;
    pub const addWithProperties = gtk_container_add_with_properties;

    extern fn gtk_container_check_resize(p_container: *Container) void;
    pub const checkResize = gtk_container_check_resize;

    /// Gets the values of one or more child properties for `child` and `container`.
    extern fn gtk_container_child_get(p_container: *Container, p_child: *gtk.Widget, p_first_prop_name: [*:0]const u8, ...) void;
    pub const childGet = gtk_container_child_get;

    /// Gets the value of a child property for `child` and `container`.
    extern fn gtk_container_child_get_property(p_container: *Container, p_child: *gtk.Widget, p_property_name: [*:0]const u8, p_value: *gobject.Value) void;
    pub const childGetProperty = gtk_container_child_get_property;

    /// Gets the values of one or more child properties for `child` and `container`.
    extern fn gtk_container_child_get_valist(p_container: *Container, p_child: *gtk.Widget, p_first_property_name: [*:0]const u8, p_var_args: std.builtin.VaList) void;
    pub const childGetValist = gtk_container_child_get_valist;

    /// Emits a `gtk.Widget.signals.child`-notify signal for the
    /// [child property][child-properties]
    /// `child_property` on the child.
    ///
    /// This is an analogue of `gobject.Object.notify` for child properties.
    ///
    /// Also see `gtk.Widget.childNotify`.
    extern fn gtk_container_child_notify(p_container: *Container, p_child: *gtk.Widget, p_child_property: [*:0]const u8) void;
    pub const childNotify = gtk_container_child_notify;

    /// Emits a `gtk.Widget.signals.child`-notify signal for the
    /// [child property][child-properties] specified by
    /// `pspec` on the child.
    ///
    /// This is an analogue of `gobject.Object.notifyByPspec` for child properties.
    extern fn gtk_container_child_notify_by_pspec(p_container: *Container, p_child: *gtk.Widget, p_pspec: *gobject.ParamSpec) void;
    pub const childNotifyByPspec = gtk_container_child_notify_by_pspec;

    /// Sets one or more child properties for `child` and `container`.
    extern fn gtk_container_child_set(p_container: *Container, p_child: *gtk.Widget, p_first_prop_name: [*:0]const u8, ...) void;
    pub const childSet = gtk_container_child_set;

    /// Sets a child property for `child` and `container`.
    extern fn gtk_container_child_set_property(p_container: *Container, p_child: *gtk.Widget, p_property_name: [*:0]const u8, p_value: *const gobject.Value) void;
    pub const childSetProperty = gtk_container_child_set_property;

    /// Sets one or more child properties for `child` and `container`.
    extern fn gtk_container_child_set_valist(p_container: *Container, p_child: *gtk.Widget, p_first_property_name: [*:0]const u8, p_var_args: std.builtin.VaList) void;
    pub const childSetValist = gtk_container_child_set_valist;

    /// Returns the type of the children supported by the container.
    ///
    /// Note that this may return `G_TYPE_NONE` to indicate that no more
    /// children can be added, e.g. for a `gtk.Paned` which already has two
    /// children.
    extern fn gtk_container_child_type(p_container: *Container) usize;
    pub const childType = gtk_container_child_type;

    /// Invokes `callback` on each direct child of `container`, including
    /// children that are considered “internal” (implementation details
    /// of the container). “Internal” children generally weren’t added
    /// by the user of the container, but were added by the container
    /// implementation itself.
    ///
    /// Most applications should use `gtk.Container.foreach`, rather
    /// than `gtk.Container.forall`.
    extern fn gtk_container_forall(p_container: *Container, p_callback: gtk.Callback, p_callback_data: ?*anyopaque) void;
    pub const forall = gtk_container_forall;

    /// Invokes `callback` on each non-internal child of `container`.
    /// See `gtk.Container.forall` for details on what constitutes
    /// an “internal” child. For all practical purposes, this function
    /// should iterate over precisely those child widgets that were
    /// added to the container by the application with explicit `add`
    /// calls.
    ///
    /// It is permissible to remove the child from the `callback` handler.
    ///
    /// Most applications should use `gtk.Container.foreach`,
    /// rather than `gtk.Container.forall`.
    extern fn gtk_container_foreach(p_container: *Container, p_callback: gtk.Callback, p_callback_data: ?*anyopaque) void;
    pub const foreach = gtk_container_foreach;

    /// Retrieves the border width of the container. See
    /// `gtk.Container.setBorderWidth`.
    extern fn gtk_container_get_border_width(p_container: *Container) c_uint;
    pub const getBorderWidth = gtk_container_get_border_width;

    /// Returns the container’s non-internal children. See
    /// `gtk.Container.forall` for details on what constitutes an "internal" child.
    extern fn gtk_container_get_children(p_container: *Container) *glib.List;
    pub const getChildren = gtk_container_get_children;

    /// Retrieves the focus chain of the container, if one has been
    /// set explicitly. If no focus chain has been explicitly
    /// set, GTK+ computes the focus chain based on the positions
    /// of the children. In that case, GTK+ stores `NULL` in
    /// `focusable_widgets` and returns `FALSE`.
    extern fn gtk_container_get_focus_chain(p_container: *Container, p_focusable_widgets: **glib.List) c_int;
    pub const getFocusChain = gtk_container_get_focus_chain;

    /// Returns the current focus child widget inside `container`. This is not the
    /// currently focused widget. That can be obtained by calling
    /// `gtk.Window.getFocus`.
    extern fn gtk_container_get_focus_child(p_container: *Container) ?*gtk.Widget;
    pub const getFocusChild = gtk_container_get_focus_child;

    /// Retrieves the horizontal focus adjustment for the container. See
    /// gtk_container_set_focus_hadjustment ().
    extern fn gtk_container_get_focus_hadjustment(p_container: *Container) ?*gtk.Adjustment;
    pub const getFocusHadjustment = gtk_container_get_focus_hadjustment;

    /// Retrieves the vertical focus adjustment for the container. See
    /// `gtk.Container.setFocusVadjustment`.
    extern fn gtk_container_get_focus_vadjustment(p_container: *Container) ?*gtk.Adjustment;
    pub const getFocusVadjustment = gtk_container_get_focus_vadjustment;

    /// Returns a newly created widget path representing all the widget hierarchy
    /// from the toplevel down to and including `child`.
    extern fn gtk_container_get_path_for_child(p_container: *Container, p_child: *gtk.Widget) *gtk.WidgetPath;
    pub const getPathForChild = gtk_container_get_path_for_child;

    /// Returns the resize mode for the container. See
    /// gtk_container_set_resize_mode ().
    extern fn gtk_container_get_resize_mode(p_container: *Container) gtk.ResizeMode;
    pub const getResizeMode = gtk_container_get_resize_mode;

    /// When a container receives a call to the draw function, it must send
    /// synthetic `gtk.Widget.signals.draw` calls to all children that don’t have their
    /// own `GdkWindows`. This function provides a convenient way of doing this.
    /// A container, when it receives a call to its `gtk.Widget.signals.draw` function,
    /// calls `gtk.Container.propagateDraw` once for each child, passing in
    /// the `cr` the container received.
    ///
    /// `gtk.Container.propagateDraw` takes care of translating the origin of `cr`,
    /// and deciding whether the draw needs to be sent to the child. It is a
    /// convenient and optimized way of getting the same effect as calling
    /// `gtk.Widget.draw` on the child directly.
    ///
    /// In most cases, a container can simply either inherit the
    /// `gtk.Widget.signals.draw` implementation from `gtk.Container`, or do some drawing
    /// and then chain to the ::draw implementation from `gtk.Container`.
    extern fn gtk_container_propagate_draw(p_container: *Container, p_child: *gtk.Widget, p_cr: *cairo.Context) void;
    pub const propagateDraw = gtk_container_propagate_draw;

    /// Removes `widget` from `container`. `widget` must be inside `container`.
    /// Note that `container` will own a reference to `widget`, and that this
    /// may be the last reference held; so removing a widget from its
    /// container can destroy that widget. If you want to use `widget`
    /// again, you need to add a reference to it before removing it from
    /// a container, using `gobject.Object.ref`. If you don’t want to use `widget`
    /// again it’s usually more efficient to simply destroy it directly
    /// using `gtk.Widget.destroy` since this will remove it from the
    /// container and help break any circular reference count cycles.
    extern fn gtk_container_remove(p_container: *Container, p_widget: *gtk.Widget) void;
    pub const remove = gtk_container_remove;

    extern fn gtk_container_resize_children(p_container: *Container) void;
    pub const resizeChildren = gtk_container_resize_children;

    /// Sets the border width of the container.
    ///
    /// The border width of a container is the amount of space to leave
    /// around the outside of the container. The only exception to this is
    /// `gtk.Window`; because toplevel windows can’t leave space outside,
    /// they leave the space inside. The border is added on all sides of
    /// the container. To add space to only one side, use a specific
    /// `gtk.Widget.properties.margin` property on the child widget, for example
    /// `gtk.Widget.properties.margin`-top.
    extern fn gtk_container_set_border_width(p_container: *Container, p_border_width: c_uint) void;
    pub const setBorderWidth = gtk_container_set_border_width;

    /// Sets a focus chain, overriding the one computed automatically by GTK+.
    ///
    /// In principle each widget in the chain should be a descendant of the
    /// container, but this is not enforced by this method, since it’s allowed
    /// to set the focus chain before you pack the widgets, or have a widget
    /// in the chain that isn’t always packed. The necessary checks are done
    /// when the focus chain is actually traversed.
    extern fn gtk_container_set_focus_chain(p_container: *Container, p_focusable_widgets: *glib.List) void;
    pub const setFocusChain = gtk_container_set_focus_chain;

    /// Sets, or unsets if `child` is `NULL`, the focused child of `container`.
    ///
    /// This function emits the GtkContainer::set_focus_child signal of
    /// `container`. Implementations of `gtk.Container` can override the
    /// default behaviour by overriding the class closure of this signal.
    ///
    /// This is function is mostly meant to be used by widgets. Applications can use
    /// `gtk.Widget.grabFocus` to manually set the focus to a specific widget.
    extern fn gtk_container_set_focus_child(p_container: *Container, p_child: ?*gtk.Widget) void;
    pub const setFocusChild = gtk_container_set_focus_child;

    /// Hooks up an adjustment to focus handling in a container, so when a child
    /// of the container is focused, the adjustment is scrolled to show that
    /// widget. This function sets the horizontal alignment.
    /// See `gtk.ScrolledWindow.getHadjustment` for a typical way of obtaining
    /// the adjustment and `gtk.Container.setFocusVadjustment` for setting
    /// the vertical adjustment.
    ///
    /// The adjustments have to be in pixel units and in the same coordinate
    /// system as the allocation for immediate children of the container.
    extern fn gtk_container_set_focus_hadjustment(p_container: *Container, p_adjustment: *gtk.Adjustment) void;
    pub const setFocusHadjustment = gtk_container_set_focus_hadjustment;

    /// Hooks up an adjustment to focus handling in a container, so when a
    /// child of the container is focused, the adjustment is scrolled to
    /// show that widget. This function sets the vertical alignment. See
    /// `gtk.ScrolledWindow.getVadjustment` for a typical way of obtaining
    /// the adjustment and `gtk.Container.setFocusHadjustment` for setting
    /// the horizontal adjustment.
    ///
    /// The adjustments have to be in pixel units and in the same coordinate
    /// system as the allocation for immediate children of the container.
    extern fn gtk_container_set_focus_vadjustment(p_container: *Container, p_adjustment: *gtk.Adjustment) void;
    pub const setFocusVadjustment = gtk_container_set_focus_vadjustment;

    /// Sets the `reallocate_redraws` flag of the container to the given value.
    ///
    /// Containers requesting reallocation redraws get automatically
    /// redrawn if any of their children changed allocation.
    extern fn gtk_container_set_reallocate_redraws(p_container: *Container, p_needs_redraws: c_int) void;
    pub const setReallocateRedraws = gtk_container_set_reallocate_redraws;

    /// Sets the resize mode for the container.
    ///
    /// The resize mode of a container determines whether a resize request
    /// will be passed to the container’s parent, queued for later execution
    /// or executed immediately.
    extern fn gtk_container_set_resize_mode(p_container: *Container, p_resize_mode: gtk.ResizeMode) void;
    pub const setResizeMode = gtk_container_set_resize_mode;

    /// Removes a focus chain explicitly set with `gtk.Container.setFocusChain`.
    extern fn gtk_container_unset_focus_chain(p_container: *Container) void;
    pub const unsetFocusChain = gtk_container_unset_focus_chain;

    extern fn gtk_container_get_type() usize;
    pub const getGObjectType = gtk_container_get_type;

    extern fn g_object_ref(p_self: *gtk.Container) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Container) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Container, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ContainerAccessible = extern struct {
    pub const Parent = gtk.WidgetAccessible;
    pub const Implements = [_]type{atk.Component};
    pub const Class = gtk.ContainerAccessibleClass;
    f_parent: gtk.WidgetAccessible,
    f_priv: ?*gtk.ContainerAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_container_accessible_get_type() usize;
    pub const getGObjectType = gtk_container_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.ContainerAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ContainerAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ContainerAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ContainerCellAccessible = extern struct {
    pub const Parent = gtk.CellAccessible;
    pub const Implements = [_]type{ atk.Action, atk.Component, atk.TableCell };
    pub const Class = gtk.ContainerCellAccessibleClass;
    f_parent: gtk.CellAccessible,
    f_priv: ?*gtk.ContainerCellAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_container_cell_accessible_new() *gtk.ContainerCellAccessible;
    pub const new = gtk_container_cell_accessible_new;

    extern fn gtk_container_cell_accessible_add_child(p_container: *ContainerCellAccessible, p_child: *gtk.CellAccessible) void;
    pub const addChild = gtk_container_cell_accessible_add_child;

    /// Get a list of children.
    extern fn gtk_container_cell_accessible_get_children(p_container: *ContainerCellAccessible) *glib.List;
    pub const getChildren = gtk_container_cell_accessible_get_children;

    extern fn gtk_container_cell_accessible_remove_child(p_container: *ContainerCellAccessible, p_child: *gtk.CellAccessible) void;
    pub const removeChild = gtk_container_cell_accessible_remove_child;

    extern fn gtk_container_cell_accessible_get_type() usize;
    pub const getGObjectType = gtk_container_cell_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.ContainerCellAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ContainerCellAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ContainerCellAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// GtkCssProvider is an object implementing the `gtk.StyleProvider` interface.
/// It is able to parse [CSS-like][css-overview] input in order to style widgets.
///
/// An application can make GTK+ parse a specific CSS style sheet by calling
/// `gtk.CssProvider.loadFromFile` or `gtk.CssProvider.loadFromResource`
/// and adding the provider with `gtk.StyleContext.addProvider` or
/// `gtk.StyleContext.addProviderForScreen`.
///
/// In addition, certain files will be read when GTK+ is initialized. First, the
/// file `$XDG_CONFIG_HOME/gtk-3.0/gtk.css` is loaded if it exists. Then, GTK+
/// loads the first existing file among
/// `XDG_DATA_HOME/themes/THEME/gtk-VERSION/gtk.css`,
/// `$HOME/.themes/THEME/gtk-VERSION/gtk.css`,
/// `$XDG_DATA_DIRS/themes/THEME/gtk-VERSION/gtk.css` and
/// `DATADIR/share/themes/THEME/gtk-VERSION/gtk.css`, where `THEME` is the name of
/// the current theme (see the `gtk.Settings.properties.gtk`-theme-name setting), `DATADIR`
/// is the prefix configured when GTK+ was compiled (unless overridden by the
/// `GTK_DATA_PREFIX` environment variable), and `VERSION` is the GTK+ version number.
/// If no file is found for the current version, GTK+ tries older versions all the
/// way back to 3.0.
///
/// In the same way, GTK+ tries to load a gtk-keys.css file for the current
/// key theme, as defined by `gtk.Settings.properties.gtk`-key-theme-name.
pub const CssProvider = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gtk.StyleProvider};
    pub const Class = gtk.CssProviderClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gtk.CssProviderPrivate,

    pub const virtual_methods = struct {
        pub const parsing_error = struct {
            pub fn call(p_class: anytype, p_provider: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_section: *gtk.CssSection, p_error: *const glib.Error) void {
                return gobject.ext.as(CssProvider.Class, p_class).f_parsing_error.?(gobject.ext.as(CssProvider, p_provider), p_section, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_provider: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_section: *gtk.CssSection, p_error: *const glib.Error) callconv(.C) void) void {
                gobject.ext.as(CssProvider.Class, p_class).f_parsing_error = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// Signals that a parsing error occurred. the `path`, `line` and `position`
        /// describe the actual location of the error as accurately as possible.
        ///
        /// Parsing errors are never fatal, so the parsing will resume after
        /// the error. Errors may however cause parts of the given
        /// data or even all of it to not be parsed at all. So it is a useful idea
        /// to check that the parsing succeeds by connecting to this signal.
        ///
        /// Note that this signal may be emitted at any time as the css provider
        /// may opt to defer parsing parts or all of the input to a later time
        /// than when a loading function was called.
        pub const parsing_error = struct {
            pub const name = "parsing-error";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_section: *gtk.CssSection, p_error: *glib.Error, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(CssProvider, p_instance))),
                    gobject.signalLookup("parsing-error", CssProvider.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Returns the provider containing the style settings used as a
    /// fallback for all widgets.
    extern fn gtk_css_provider_get_default() *gtk.CssProvider;
    pub const getDefault = gtk_css_provider_get_default;

    /// Loads a theme from the usual theme paths
    extern fn gtk_css_provider_get_named(p_name: [*:0]const u8, p_variant: ?[*:0]const u8) *gtk.CssProvider;
    pub const getNamed = gtk_css_provider_get_named;

    /// Returns a newly created `gtk.CssProvider`.
    extern fn gtk_css_provider_new() *gtk.CssProvider;
    pub const new = gtk_css_provider_new;

    /// Loads `data` into `css_provider`, and by doing so clears any previously loaded
    /// information.
    extern fn gtk_css_provider_load_from_data(p_css_provider: *CssProvider, p_data: [*]const u8, p_length: isize, p_error: ?*?*glib.Error) c_int;
    pub const loadFromData = gtk_css_provider_load_from_data;

    /// Loads the data contained in `file` into `css_provider`, making it
    /// clear any previously loaded information.
    extern fn gtk_css_provider_load_from_file(p_css_provider: *CssProvider, p_file: *gio.File, p_error: ?*?*glib.Error) c_int;
    pub const loadFromFile = gtk_css_provider_load_from_file;

    /// Loads the data contained in `path` into `css_provider`, making it clear
    /// any previously loaded information.
    extern fn gtk_css_provider_load_from_path(p_css_provider: *CssProvider, p_path: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const loadFromPath = gtk_css_provider_load_from_path;

    /// Loads the data contained in the resource at `resource_path` into
    /// the `gtk.CssProvider`, clearing any previously loaded information.
    ///
    /// To track errors while loading CSS, connect to the
    /// `gtk.CssProvider.signals.parsing`-error signal.
    extern fn gtk_css_provider_load_from_resource(p_css_provider: *CssProvider, p_resource_path: [*:0]const u8) void;
    pub const loadFromResource = gtk_css_provider_load_from_resource;

    /// Converts the `provider` into a string representation in CSS
    /// format.
    ///
    /// Using `gtk.CssProvider.loadFromData` with the return value
    /// from this function on a new provider created with
    /// `gtk.CssProvider.new` will basically create a duplicate of
    /// this `provider`.
    extern fn gtk_css_provider_to_string(p_provider: *CssProvider) [*:0]u8;
    pub const toString = gtk_css_provider_to_string;

    extern fn gtk_css_provider_get_type() usize;
    pub const getGObjectType = gtk_css_provider_get_type;

    extern fn g_object_ref(p_self: *gtk.CssProvider) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.CssProvider) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CssProvider, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Dialog boxes are a convenient way to prompt the user for a small amount
/// of input, e.g. to display a message, ask a question, or anything else
/// that does not require extensive effort on the user’s part.
///
/// GTK+ treats a dialog as a window split vertically. The top section is a
/// `gtk.VBox`, and is where widgets such as a `gtk.Label` or a `gtk.Entry` should
/// be packed. The bottom area is known as the
/// “action area”. This is generally used for
/// packing buttons into the dialog which may perform functions such as
/// cancel, ok, or apply.
///
/// `gtk.Dialog` boxes are created with a call to `gtk.Dialog.new` or
/// `gtk.Dialog.newWithButtons`. `gtk.Dialog.newWithButtons` is
/// recommended; it allows you to set the dialog title, some convenient
/// flags, and add simple buttons.
///
/// If “dialog” is a newly created dialog, the two primary areas of the
/// window can be accessed through `gtk.Dialog.getContentArea` and
/// `gtk.Dialog.getActionArea`, as can be seen from the example below.
///
/// A “modal” dialog (that is, one which freezes the rest of the application
/// from user input), can be created by calling `gtk.Window.setModal` on the
/// dialog. Use the `GTK_WINDOW` macro to cast the widget returned from
/// `gtk.Dialog.new` into a `gtk.Window`. When using `gtk.Dialog.newWithButtons`
/// you can also pass the `GTK_DIALOG_MODAL` flag to make a dialog modal.
///
/// If you add buttons to `gtk.Dialog` using `gtk.Dialog.newWithButtons`,
/// `gtk.Dialog.addButton`, `gtk.Dialog.addButtons`, or
/// `gtk.Dialog.addActionWidget`, clicking the button will emit a signal
/// called `gtk.Dialog.signals.response` with a response ID that you specified. GTK+
/// will never assign a meaning to positive response IDs; these are entirely
/// user-defined. But for convenience, you can use the response IDs in the
/// `gtk.ResponseType` enumeration (these all have values less than zero). If
/// a dialog receives a delete event, the `gtk.Dialog.signals.response` signal will
/// be emitted with a response ID of `GTK_RESPONSE_DELETE_EVENT`.
///
/// If you want to block waiting for a dialog to return before returning
/// control flow to your code, you can call `gtk.Dialog.run`. This function
/// enters a recursive main loop and waits for the user to respond to the
/// dialog, returning the response ID corresponding to the button the user
/// clicked.
///
/// For the simple dialog in the following example, in reality you’d probably
/// use `gtk.MessageDialog` to save yourself some effort. But you’d need to
/// create the dialog contents manually if you had more than a simple message
/// in the dialog.
///
/// An example for simple GtkDialog usage:
/// ```
/// // Function to open a dialog box with a message
/// void
/// quick_message (GtkWindow *parent, gchar *message)
/// {
///  GtkWidget *dialog, *label, *content_area;
///  GtkDialogFlags flags;
///
///  // Create the widgets
///  flags = GTK_DIALOG_DESTROY_WITH_PARENT;
///  dialog = gtk_dialog_new_with_buttons ("Message",
///                                        parent,
///                                        flags,
///                                        _("_OK"),
///                                        GTK_RESPONSE_NONE,
///                                        NULL);
///  content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
///  label = gtk_label_new (message);
///
///  // Ensure that the dialog box is destroyed when the user responds
///
///  g_signal_connect_swapped (dialog,
///                            "response",
///                            G_CALLBACK (gtk_widget_destroy),
///                            dialog);
///
///  // Add the label, and show everything we’ve added
///
///  gtk_container_add (GTK_CONTAINER (content_area), label);
///  gtk_widget_show_all (dialog);
/// }
/// ```
///
/// # GtkDialog as GtkBuildable
///
/// The GtkDialog implementation of the `gtk.Buildable` interface exposes the
/// `vbox` and `action_area` as internal children with the names “vbox” and
/// “action_area”.
///
/// GtkDialog supports a custom `<action-widgets>` element, which can contain
/// multiple `<action-widget>` elements. The “response” attribute specifies a
/// numeric response, and the content of the element is the id of widget
/// (which should be a child of the dialogs `action_area`). To mark a response
/// as default, set the “default“ attribute of the `<action-widget>` element
/// to true.
///
/// GtkDialog supports adding action widgets by specifying “action“ as
/// the “type“ attribute of a `<child>` element. The widget will be added
/// either to the action area or the headerbar of the dialog, depending
/// on the “use-header-bar“ property. The response id has to be associated
/// with the action widget using the `<action-widgets>` element.
///
/// An example of a `gtk.Dialog` UI definition fragment:
///
/// ```
/// <object class="GtkDialog" id="dialog1">
///   <child type="action">
///     <object class="GtkButton" id="button_cancel"/>
///   </child>
///   <child type="action">
///     <object class="GtkButton" id="button_ok">
///       <property name="can-default">True</property>
///     </object>
///   </child>
///   <action-widgets>
///     <action-widget response="cancel">button_cancel</action-widget>
///     <action-widget response="ok" default="true">button_ok</action-widget>
///   </action-widgets>
/// </object>
/// ```
pub const Dialog = extern struct {
    pub const Parent = gtk.Window;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.DialogClass;
    f_window: gtk.Window,
    f_priv: ?*gtk.DialogPrivate,

    pub const virtual_methods = struct {
        /// Signal emitted when the user uses a keybinding to close the dialog.
        pub const close = struct {
            pub fn call(p_class: anytype, p_dialog: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Dialog.Class, p_class).f_close.?(gobject.ext.as(Dialog, p_dialog));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_dialog: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Dialog.Class, p_class).f_close = @ptrCast(p_implementation);
            }
        };

        /// Emits the `gtk.Dialog.signals.response` signal with the given response ID.
        /// Used to indicate that the user has responded to the dialog in some way;
        /// typically either you or `gtk.Dialog.run` will be monitoring the
        /// ::response signal and take appropriate action.
        pub const response = struct {
            pub fn call(p_class: anytype, p_dialog: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_response_id: c_int) void {
                return gobject.ext.as(Dialog.Class, p_class).f_response.?(gobject.ext.as(Dialog, p_dialog), p_response_id);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_dialog: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_response_id: c_int) callconv(.C) void) void {
                gobject.ext.as(Dialog.Class, p_class).f_response = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// `TRUE` if the dialog uses a `gtk.HeaderBar` for action buttons
        /// instead of the action-area.
        ///
        /// For technical reasons, this property is declared as an integer
        /// property, but you should only set it to `TRUE` or `FALSE`.
        pub const use_header_bar = struct {
            pub const name = "use-header-bar";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The ::close signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user uses a keybinding to close
        /// the dialog.
        ///
        /// The default binding for this signal is the Escape key.
        pub const close = struct {
            pub const name = "close";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Dialog, p_instance))),
                    gobject.signalLookup("close", Dialog.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when an action widget is clicked, the dialog receives a
        /// delete event, or the application programmer calls `gtk.Dialog.response`.
        /// On a delete event, the response ID is `GTK_RESPONSE_DELETE_EVENT`.
        /// Otherwise, it depends on which action widget was clicked.
        pub const response = struct {
            pub const name = "response";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_response_id: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Dialog, p_instance))),
                    gobject.signalLookup("response", Dialog.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new dialog box.
    ///
    /// Widgets should not be packed into this `gtk.Window`
    /// directly, but into the `vbox` and `action_area`, as described above.
    extern fn gtk_dialog_new() *gtk.Dialog;
    pub const new = gtk_dialog_new;

    /// Creates a new `gtk.Dialog` with title `title` (or `NULL` for the default
    /// title; see `gtk.Window.setTitle`) and transient parent `parent` (or
    /// `NULL` for none; see `gtk.Window.setTransientFor`). The `flags`
    /// argument can be used to make the dialog modal (`GTK_DIALOG_MODAL`)
    /// and/or to have it destroyed along with its transient parent
    /// (`GTK_DIALOG_DESTROY_WITH_PARENT`). After `flags`, button
    /// text/response ID pairs should be listed, with a `NULL` pointer ending
    /// the list. Button text can be arbitrary text. A response ID can be
    /// any positive number, or one of the values in the `gtk.ResponseType`
    /// enumeration. If the user clicks one of these dialog buttons,
    /// `gtk.Dialog` will emit the `gtk.Dialog.signals.response` signal with the corresponding
    /// response ID. If a `gtk.Dialog` receives the `gtk.Widget.signals.delete`-event signal,
    /// it will emit ::response with a response ID of `GTK_RESPONSE_DELETE_EVENT`.
    /// However, destroying a dialog does not emit the ::response signal;
    /// so be careful relying on ::response when using the
    /// `GTK_DIALOG_DESTROY_WITH_PARENT` flag. Buttons are from left to right,
    /// so the first button in the list will be the leftmost button in the dialog.
    ///
    /// Here’s a simple example:
    /// ```
    ///  GtkWidget *main_app_window; // Window the dialog should show up on
    ///  GtkWidget *dialog;
    ///  GtkDialogFlags flags = GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT;
    ///  dialog = gtk_dialog_new_with_buttons ("My dialog",
    ///                                        main_app_window,
    ///                                        flags,
    ///                                        _("_OK"),
    ///                                        GTK_RESPONSE_ACCEPT,
    ///                                        _("_Cancel"),
    ///                                        GTK_RESPONSE_REJECT,
    ///                                        NULL);
    /// ```
    extern fn gtk_dialog_new_with_buttons(p_title: ?[*:0]const u8, p_parent: ?*gtk.Window, p_flags: gtk.DialogFlags, p_first_button_text: ?[*:0]const u8, ...) *gtk.Dialog;
    pub const newWithButtons = gtk_dialog_new_with_buttons;

    /// Adds an activatable widget to the action area of a `gtk.Dialog`,
    /// connecting a signal handler that will emit the `gtk.Dialog.signals.response`
    /// signal on the dialog when the widget is activated. The widget is
    /// appended to the end of the dialog’s action area. If you want to add a
    /// non-activatable widget, simply pack it into the `action_area` field
    /// of the `gtk.Dialog` struct.
    extern fn gtk_dialog_add_action_widget(p_dialog: *Dialog, p_child: *gtk.Widget, p_response_id: c_int) void;
    pub const addActionWidget = gtk_dialog_add_action_widget;

    /// Adds a button with the given text and sets things up so that
    /// clicking the button will emit the `gtk.Dialog.signals.response` signal with
    /// the given `response_id`. The button is appended to the end of the
    /// dialog’s action area. The button widget is returned, but usually
    /// you don’t need it.
    extern fn gtk_dialog_add_button(p_dialog: *Dialog, p_button_text: [*:0]const u8, p_response_id: c_int) *gtk.Widget;
    pub const addButton = gtk_dialog_add_button;

    /// Adds more buttons, same as calling `gtk.Dialog.addButton`
    /// repeatedly.  The variable argument list should be `NULL`-terminated
    /// as with `gtk.Dialog.newWithButtons`. Each button must have both
    /// text and response ID.
    extern fn gtk_dialog_add_buttons(p_dialog: *Dialog, p_first_button_text: [*:0]const u8, ...) void;
    pub const addButtons = gtk_dialog_add_buttons;

    /// Returns the action area of `dialog`.
    extern fn gtk_dialog_get_action_area(p_dialog: *Dialog) *gtk.Box;
    pub const getActionArea = gtk_dialog_get_action_area;

    /// Returns the content area of `dialog`.
    extern fn gtk_dialog_get_content_area(p_dialog: *Dialog) *gtk.Box;
    pub const getContentArea = gtk_dialog_get_content_area;

    /// Returns the header bar of `dialog`. Note that the
    /// headerbar is only used by the dialog if the
    /// `gtk.Dialog.properties.use`-header-bar property is `TRUE`.
    extern fn gtk_dialog_get_header_bar(p_dialog: *Dialog) *gtk.HeaderBar;
    pub const getHeaderBar = gtk_dialog_get_header_bar;

    /// Gets the response id of a widget in the action area
    /// of a dialog.
    extern fn gtk_dialog_get_response_for_widget(p_dialog: *Dialog, p_widget: *gtk.Widget) c_int;
    pub const getResponseForWidget = gtk_dialog_get_response_for_widget;

    /// Gets the widget button that uses the given response ID in the action area
    /// of a dialog.
    extern fn gtk_dialog_get_widget_for_response(p_dialog: *Dialog, p_response_id: c_int) ?*gtk.Widget;
    pub const getWidgetForResponse = gtk_dialog_get_widget_for_response;

    /// Emits the `gtk.Dialog.signals.response` signal with the given response ID.
    /// Used to indicate that the user has responded to the dialog in some way;
    /// typically either you or `gtk.Dialog.run` will be monitoring the
    /// ::response signal and take appropriate action.
    extern fn gtk_dialog_response(p_dialog: *Dialog, p_response_id: c_int) void;
    pub const response = gtk_dialog_response;

    /// Blocks in a recursive main loop until the `dialog` either emits the
    /// `gtk.Dialog.signals.response` signal, or is destroyed. If the dialog is
    /// destroyed during the call to `gtk.Dialog.run`, `gtk.Dialog.run` returns
    /// `GTK_RESPONSE_NONE`. Otherwise, it returns the response ID from the
    /// ::response signal emission.
    ///
    /// Before entering the recursive main loop, `gtk.Dialog.run` calls
    /// `gtk.Widget.show` on the dialog for you. Note that you still
    /// need to show any children of the dialog yourself.
    ///
    /// During `gtk.Dialog.run`, the default behavior of `gtk.Widget.signals.delete`-event
    /// is disabled; if the dialog receives ::delete_event, it will not be
    /// destroyed as windows usually are, and `gtk.Dialog.run` will return
    /// `GTK_RESPONSE_DELETE_EVENT`. Also, during `gtk.Dialog.run` the dialog
    /// will be modal. You can force `gtk.Dialog.run` to return at any time by
    /// calling `gtk.Dialog.response` to emit the ::response signal. Destroying
    /// the dialog during `gtk.Dialog.run` is a very bad idea, because your
    /// post-run code won’t know whether the dialog was destroyed or not.
    ///
    /// After `gtk.Dialog.run` returns, you are responsible for hiding or
    /// destroying the dialog if you wish to do so.
    ///
    /// Typical usage of this function might be:
    /// ```
    ///   GtkWidget *dialog = gtk_dialog_new ();
    ///   // Set up dialog...
    ///
    ///   int result = gtk_dialog_run (GTK_DIALOG (dialog));
    ///   switch (result)
    ///     {
    ///       case GTK_RESPONSE_ACCEPT:
    ///          // do_application_specific_something ();
    ///          break;
    ///       default:
    ///          // do_nothing_since_dialog_was_cancelled ();
    ///          break;
    ///     }
    ///   gtk_widget_destroy (dialog);
    /// ```
    ///
    /// Note that even though the recursive main loop gives the effect of a
    /// modal dialog (it prevents the user from interacting with other
    /// windows in the same window group while the dialog is run), callbacks
    /// such as timeouts, IO channel watches, DND drops, etc, will
    /// be triggered during a `gtk.Dialog.run` call.
    extern fn gtk_dialog_run(p_dialog: *Dialog) c_int;
    pub const run = gtk_dialog_run;

    /// Sets an alternative button order. If the
    /// `gtk.Settings.properties.gtk`-alternative-button-order setting is set to `TRUE`,
    /// the dialog buttons are reordered according to the order of the
    /// response ids passed to this function.
    ///
    /// By default, GTK+ dialogs use the button order advocated by the
    /// [GNOME Human Interface Guidelines](http://library.gnome.org/devel/hig-book/stable/)
    /// with the affirmative button at the far
    /// right, and the cancel button left of it. But the builtin GTK+ dialogs
    /// and `GtkMessageDialogs` do provide an alternative button order,
    /// which is more suitable on some platforms, e.g. Windows.
    ///
    /// Use this function after adding all the buttons to your dialog, as the
    /// following example shows:
    ///
    /// ```
    /// cancel_button = gtk_dialog_add_button (GTK_DIALOG (dialog),
    ///                                        _("_Cancel"),
    ///                                        GTK_RESPONSE_CANCEL);
    ///
    /// ok_button = gtk_dialog_add_button (GTK_DIALOG (dialog),
    ///                                    _("_OK"),
    ///                                    GTK_RESPONSE_OK);
    ///
    /// gtk_widget_grab_default (ok_button);
    ///
    /// help_button = gtk_dialog_add_button (GTK_DIALOG (dialog),
    ///                                      _("_Help"),
    ///                                      GTK_RESPONSE_HELP);
    ///
    /// gtk_dialog_set_alternative_button_order (GTK_DIALOG (dialog),
    ///                                          GTK_RESPONSE_OK,
    ///                                          GTK_RESPONSE_CANCEL,
    ///                                          GTK_RESPONSE_HELP,
    ///                                          -1);
    /// ```
    extern fn gtk_dialog_set_alternative_button_order(p_dialog: *Dialog, p_first_response_id: c_int, ...) void;
    pub const setAlternativeButtonOrder = gtk_dialog_set_alternative_button_order;

    /// Sets an alternative button order. If the
    /// `gtk.Settings.properties.gtk`-alternative-button-order setting is set to `TRUE`,
    /// the dialog buttons are reordered according to the order of the
    /// response ids in `new_order`.
    ///
    /// See `gtk.Dialog.setAlternativeButtonOrder` for more information.
    ///
    /// This function is for use by language bindings.
    extern fn gtk_dialog_set_alternative_button_order_from_array(p_dialog: *Dialog, p_n_params: c_int, p_new_order: [*]c_int) void;
    pub const setAlternativeButtonOrderFromArray = gtk_dialog_set_alternative_button_order_from_array;

    /// Sets the last widget in the dialog’s action area with the given `response_id`
    /// as the default widget for the dialog. Pressing “Enter” normally activates
    /// the default widget.
    extern fn gtk_dialog_set_default_response(p_dialog: *Dialog, p_response_id: c_int) void;
    pub const setDefaultResponse = gtk_dialog_set_default_response;

    /// Calls `gtk_widget_set_sensitive (widget, `setting`)`
    /// for each widget in the dialog’s action area with the given `response_id`.
    /// A convenient way to sensitize/desensitize dialog buttons.
    extern fn gtk_dialog_set_response_sensitive(p_dialog: *Dialog, p_response_id: c_int, p_setting: c_int) void;
    pub const setResponseSensitive = gtk_dialog_set_response_sensitive;

    extern fn gtk_dialog_get_type() usize;
    pub const getGObjectType = gtk_dialog_get_type;

    extern fn g_object_ref(p_self: *gtk.Dialog) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Dialog) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Dialog, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.DrawingArea` widget is used for creating custom user interface
/// elements. It’s essentially a blank widget; you can draw on it. After
/// creating a drawing area, the application may want to connect to:
///
/// - Mouse and button press signals to respond to input from
///   the user. (Use `gtk.Widget.addEvents` to enable events
///   you wish to receive.)
///
/// - The `gtk.Widget.signals.realize` signal to take any necessary actions
///   when the widget is instantiated on a particular display.
///   (Create GDK resources in response to this signal.)
///
/// - The `gtk.Widget.signals.size`-allocate signal to take any necessary
///   actions when the widget changes size.
///
/// - The `gtk.Widget.signals.draw` signal to handle redrawing the
///   contents of the widget.
///
/// The following code portion demonstrates using a drawing
/// area to display a circle in the normal widget foreground
/// color.
///
/// Note that GDK automatically clears the exposed area before sending
/// the expose event, and that drawing is implicitly clipped to the exposed
/// area. If you want to have a theme-provided background, you need
/// to call `gtk.renderBackground` in your ::draw method.
///
/// ## Simple GtkDrawingArea usage
///
/// ```
/// gboolean
/// draw_callback (GtkWidget *widget, cairo_t *cr, gpointer data)
/// {
///   guint width, height;
///   GdkRGBA color;
///   GtkStyleContext *context;
///
///   context = gtk_widget_get_style_context (widget);
///
///   width = gtk_widget_get_allocated_width (widget);
///   height = gtk_widget_get_allocated_height (widget);
///
///   gtk_render_background (context, cr, 0, 0, width, height);
///
///   cairo_arc (cr,
///              width / 2.0, height / 2.0,
///              MIN (width, height) / 2.0,
///              0, 2 * G_PI);
///
///   gtk_style_context_get_color (context,
///                                gtk_style_context_get_state (context),
///                                &color);
///   gdk_cairo_set_source_rgba (cr, &color);
///
///   cairo_fill (cr);
///
///  return FALSE;
/// }
/// [...]
///   GtkWidget *drawing_area = gtk_drawing_area_new ();
///   gtk_widget_set_size_request (drawing_area, 100, 100);
///   g_signal_connect (G_OBJECT (drawing_area), "draw",
///                     G_CALLBACK (draw_callback), NULL);
/// ```
///
/// Draw signals are normally delivered when a drawing area first comes
/// onscreen, or when it’s covered by another window and then uncovered.
/// You can also force an expose event by adding to the “damage region”
/// of the drawing area’s window; `gtk.Widget.queueDrawArea` and
/// `gdk.Window.invalidateRect` are equally good ways to do this.
/// You’ll then get a draw signal for the invalid region.
///
/// The available routines for drawing are documented on the
/// [GDK Drawing Primitives][gdk3-Cairo-Interaction] page
/// and the cairo documentation.
///
/// To receive mouse events on a drawing area, you will need to enable
/// them with `gtk.Widget.addEvents`. To receive keyboard events, you
/// will need to set the “can-focus” property on the drawing area, and you
/// should probably draw some user-visible indication that the drawing
/// area is focused. Use `gtk.Widget.hasFocus` in your expose event
/// handler to decide whether to draw the focus indicator. See
/// `gtk.renderFocus` for one way to draw focus.
pub const DrawingArea = extern struct {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.DrawingAreaClass;
    f_widget: gtk.Widget,
    f_dummy: ?*anyopaque,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new drawing area.
    extern fn gtk_drawing_area_new() *gtk.DrawingArea;
    pub const new = gtk_drawing_area_new;

    extern fn gtk_drawing_area_get_type() usize;
    pub const getGObjectType = gtk_drawing_area_get_type;

    extern fn g_object_ref(p_self: *gtk.DrawingArea) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.DrawingArea) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DrawingArea, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.Entry` widget is a single line text entry
/// widget. A fairly large set of key bindings are supported
/// by default. If the entered text is longer than the allocation
/// of the widget, the widget will scroll so that the cursor
/// position is visible.
///
/// When using an entry for passwords and other sensitive information,
/// it can be put into “password mode” using `gtk.Entry.setVisibility`.
/// In this mode, entered text is displayed using a “invisible” character.
/// By default, GTK+ picks the best invisible character that is available
/// in the current font, but it can be changed with
/// `gtk.Entry.setInvisibleChar`. Since 2.16, GTK+ displays a warning
/// when Caps Lock or input methods might interfere with entering text in
/// a password entry. The warning can be turned off with the
/// `gtk.Entry.properties.caps`-lock-warning property.
///
/// Since 2.16, GtkEntry has the ability to display progress or activity
/// information behind the text. To make an entry display such information,
/// use `gtk.Entry.setProgressFraction` or `gtk.Entry.setProgressPulseStep`.
///
/// Additionally, GtkEntry can show icons at either side of the entry. These
/// icons can be activatable by clicking, can be set up as drag source and
/// can have tooltips. To add an icon, use `gtk.Entry.setIconFromGicon` or
/// one of the various other functions that set an icon from a stock id, an
/// icon name or a pixbuf. To trigger an action when the user clicks an icon,
/// connect to the `gtk.Entry.signals.icon`-press signal. To allow DND operations
/// from an icon, use `gtk.Entry.setIconDragSource`. To set a tooltip on
/// an icon, use `gtk.Entry.setIconTooltipText` or the corresponding function
/// for markup.
///
/// Note that functionality or information that is only available by clicking
/// on an icon in an entry may not be accessible at all to users which are not
/// able to use a mouse or other pointing device. It is therefore recommended
/// that any such functionality should also be available by other means, e.g.
/// via the context menu of the entry.
///
/// # CSS nodes
///
/// ```
/// entry[.read-only][.flat][.warning][.error]
/// ├── image.left
/// ├── image.right
/// ├── undershoot.left
/// ├── undershoot.right
/// ├── [selection]
/// ├── [progress[.pulse]]
/// ╰── [window.popup]
/// ```
///
/// GtkEntry has a main node with the name entry. Depending on the properties
/// of the entry, the style classes .read-only and .flat may appear. The style
/// classes .warning and .error may also be used with entries.
///
/// When the entry shows icons, it adds subnodes with the name image and the
/// style class .left or .right, depending on where the icon appears.
///
/// When the entry has a selection, it adds a subnode with the name selection.
///
/// When the entry shows progress, it adds a subnode with the name progress.
/// The node has the style class .pulse when the shown progress is pulsing.
///
/// The CSS node for a context menu is added as a subnode below entry as well.
///
/// The undershoot nodes are used to draw the underflow indication when content
/// is scrolled out of view. These nodes get the .left and .right style classes
/// added depending on where the indication is drawn.
///
/// When touch is used and touch selection handles are shown, they are using
/// CSS nodes with name cursor-handle. They get the .top or .bottom style class
/// depending on where they are shown in relation to the selection. If there is
/// just a single handle for the text cursor, it gets the style class
/// .insertion-cursor.
pub const Entry = extern struct {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.CellEditable, gtk.Editable };
    pub const Class = gtk.EntryClass;
    f_parent_instance: gtk.Widget,
    f_priv: ?*gtk.EntryPrivate,

    pub const virtual_methods = struct {
        /// Class handler for the `gtk.Entry.signals.activate` signal. The default
        ///   implementation calls `gtk.Window.activateDefault` on the entry’s top-level
        ///   window.
        pub const activate = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Entry.Class, p_class).f_activate.?(gobject.ext.as(Entry, p_entry));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_activate = @ptrCast(p_implementation);
            }
        };

        /// Class handler for the `gtk.Entry.signals.backspace` signal. The default
        ///   implementation deletes the selection or a single character or word.
        pub const backspace = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Entry.Class, p_class).f_backspace.?(gobject.ext.as(Entry, p_entry));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_backspace = @ptrCast(p_implementation);
            }
        };

        /// Class handler for the `gtk.Entry.signals.copy`-clipboard signal. The
        ///   default implementation copies the selection, if one exists.
        pub const copy_clipboard = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Entry.Class, p_class).f_copy_clipboard.?(gobject.ext.as(Entry, p_entry));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_copy_clipboard = @ptrCast(p_implementation);
            }
        };

        /// Class handler for the `gtk.Entry.signals.cut`-clipboard signal. The
        ///   default implementation cuts the selection, if one exists.
        pub const cut_clipboard = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Entry.Class, p_class).f_cut_clipboard.?(gobject.ext.as(Entry, p_entry));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_cut_clipboard = @ptrCast(p_implementation);
            }
        };

        /// Class handler for the `gtk.Entry.signals.delete`-from-cursor
        ///   signal. The default implementation deletes the selection or the specified
        ///   number of characters or words.
        pub const delete_from_cursor = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_type: gtk.DeleteType, p_count: c_int) void {
                return gobject.ext.as(Entry.Class, p_class).f_delete_from_cursor.?(gobject.ext.as(Entry, p_entry), p_type, p_count);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_type: gtk.DeleteType, p_count: c_int) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_delete_from_cursor = @ptrCast(p_implementation);
            }
        };

        /// Calculate the size of the text area frame, which is its
        ///   allocated width and requested height, minus space for margins and borders,
        ///   and taking baseline and text height into account. This virtual function
        ///   must be non-`NULL`.
        pub const get_frame_size = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_x: *c_int, p_y: *c_int, p_width: *c_int, p_height: *c_int) void {
                return gobject.ext.as(Entry.Class, p_class).f_get_frame_size.?(gobject.ext.as(Entry, p_entry), p_x, p_y, p_width, p_height);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_x: *c_int, p_y: *c_int, p_width: *c_int, p_height: *c_int) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_get_frame_size = @ptrCast(p_implementation);
            }
        };

        /// Calculate the size of the text area, which is its
        ///   allocated width and requested height, minus space for margins and borders.
        ///   This virtual function must be non-`NULL`.
        pub const get_text_area_size = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_x: *c_int, p_y: *c_int, p_width: *c_int, p_height: *c_int) void {
                return gobject.ext.as(Entry.Class, p_class).f_get_text_area_size.?(gobject.ext.as(Entry, p_entry), p_x, p_y, p_width, p_height);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_x: *c_int, p_y: *c_int, p_width: *c_int, p_height: *c_int) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_get_text_area_size = @ptrCast(p_implementation);
            }
        };

        /// Class handler for the `gtk.Entry.signals.insert`-at-cursor signal.
        ///   The default implementation inserts text at the cursor.
        pub const insert_at_cursor = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_str: [*:0]const u8) void {
                return gobject.ext.as(Entry.Class, p_class).f_insert_at_cursor.?(gobject.ext.as(Entry, p_entry), p_str);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_str: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_insert_at_cursor = @ptrCast(p_implementation);
            }
        };

        pub const insert_emoji = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Entry.Class, p_class).f_insert_emoji.?(gobject.ext.as(Entry, p_entry));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_insert_emoji = @ptrCast(p_implementation);
            }
        };

        /// Class handler for the `gtk.Entry.signals.move`-cursor signal. The
        ///   default implementation specifies the standard `gtk.Entry` cursor movement
        ///   behavior.
        pub const move_cursor = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_step: gtk.MovementStep, p_count: c_int, p_extend_selection: c_int) void {
                return gobject.ext.as(Entry.Class, p_class).f_move_cursor.?(gobject.ext.as(Entry, p_entry), p_step, p_count, p_extend_selection);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_step: gtk.MovementStep, p_count: c_int, p_extend_selection: c_int) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_move_cursor = @ptrCast(p_implementation);
            }
        };

        /// Class handler for the `gtk.Entry.signals.paste`-clipboard signal.
        ///   The default implementation pastes at the current cursor position or over
        ///   the current selection if one exists.
        pub const paste_clipboard = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Entry.Class, p_class).f_paste_clipboard.?(gobject.ext.as(Entry, p_entry));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_paste_clipboard = @ptrCast(p_implementation);
            }
        };

        /// Class handler for the `gtk.Entry.signals.populate`-popup signal. If
        ///   non-`NULL`, this will be called to add additional entries to the context
        ///   menu when it is displayed.
        pub const populate_popup = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_popup: *gtk.Widget) void {
                return gobject.ext.as(Entry.Class, p_class).f_populate_popup.?(gobject.ext.as(Entry, p_entry), p_popup);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_popup: *gtk.Widget) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_populate_popup = @ptrCast(p_implementation);
            }
        };

        /// Class handler for the `gtk.Entry.signals.toggle`-overwrite signal.
        ///   The default implementation toggles overwrite mode and blinks the cursor.
        pub const toggle_overwrite = struct {
            pub fn call(p_class: anytype, p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Entry.Class, p_class).f_toggle_overwrite.?(gobject.ext.as(Entry, p_entry));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_entry: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Entry.Class, p_class).f_toggle_overwrite = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const activates_default = struct {
            pub const name = "activates-default";

            pub const Type = c_int;
        };

        /// A list of Pango attributes to apply to the text of the entry.
        ///
        /// This is mainly useful to change the size or weight of the text.
        ///
        /// The `pango.Attribute`'s `start_index` and `end_index` must refer to the
        /// `gtk.EntryBuffer` text, i.e. without the preedit string.
        pub const attributes = struct {
            pub const name = "attributes";

            pub const Type = ?*pango.AttrList;
        };

        pub const buffer = struct {
            pub const name = "buffer";

            pub const Type = ?*gtk.EntryBuffer;
        };

        /// Whether password entries will show a warning when Caps Lock is on.
        ///
        /// Note that the warning is shown using a secondary icon, and thus
        /// does not work if you are using the secondary icon position for some
        /// other purpose.
        pub const caps_lock_warning = struct {
            pub const name = "caps-lock-warning";

            pub const Type = c_int;
        };

        /// The auxiliary completion object to use with the entry.
        pub const completion = struct {
            pub const name = "completion";

            pub const Type = ?*gtk.EntryCompletion;
        };

        pub const cursor_position = struct {
            pub const name = "cursor-position";

            pub const Type = c_int;
        };

        pub const editable = struct {
            pub const name = "editable";

            pub const Type = c_int;
        };

        pub const enable_emoji_completion = struct {
            pub const name = "enable-emoji-completion";

            pub const Type = c_int;
        };

        pub const has_frame = struct {
            pub const name = "has-frame";

            pub const Type = c_int;
        };

        /// Which IM (input method) module should be used for this entry.
        /// See `gtk.IMContext`.
        ///
        /// Setting this to a non-`NULL` value overrides the
        /// system-wide IM module setting. See the GtkSettings
        /// `gtk.Settings.properties.gtk`-im-module property.
        pub const im_module = struct {
            pub const name = "im-module";

            pub const Type = ?[*:0]u8;
        };

        /// Sets the text area's border between the text and the frame.
        pub const inner_border = struct {
            pub const name = "inner-border";

            pub const Type = ?*gtk.Border;
        };

        /// Additional hints (beyond `gtk.Entry.properties.input`-purpose) that
        /// allow input methods to fine-tune their behaviour.
        pub const input_hints = struct {
            pub const name = "input-hints";

            pub const Type = gtk.InputHints;
        };

        /// The purpose of this text field.
        ///
        /// This property can be used by on-screen keyboards and other input
        /// methods to adjust their behaviour.
        ///
        /// Note that setting the purpose to `GTK_INPUT_PURPOSE_PASSWORD` or
        /// `GTK_INPUT_PURPOSE_PIN` is independent from setting
        /// `gtk.Entry.properties.visibility`.
        pub const input_purpose = struct {
            pub const name = "input-purpose";

            pub const Type = gtk.InputPurpose;
        };

        /// The invisible character is used when masking entry contents (in
        /// \"password mode\")"). When it is not explicitly set with the
        /// `gtk.Entry.properties.invisible`-char property, GTK+ determines the character
        /// to use from a list of possible candidates, depending on availability
        /// in the current font.
        ///
        /// This style property allows the theme to prepend a character
        /// to the list of candidates.
        pub const invisible_char = struct {
            pub const name = "invisible-char";

            pub const Type = c_uint;
        };

        /// Whether the invisible char has been set for the `gtk.Entry`.
        pub const invisible_char_set = struct {
            pub const name = "invisible-char-set";

            pub const Type = c_int;
        };

        pub const max_length = struct {
            pub const name = "max-length";

            pub const Type = c_int;
        };

        /// The desired maximum width of the entry, in characters.
        /// If this property is set to -1, the width will be calculated
        /// automatically.
        pub const max_width_chars = struct {
            pub const name = "max-width-chars";

            pub const Type = c_int;
        };

        /// If text is overwritten when typing in the `gtk.Entry`.
        pub const overwrite_mode = struct {
            pub const name = "overwrite-mode";

            pub const Type = c_int;
        };

        /// The text that will be displayed in the `gtk.Entry` when it is empty
        /// and unfocused.
        pub const placeholder_text = struct {
            pub const name = "placeholder-text";

            pub const Type = ?[*:0]u8;
        };

        /// If :populate-all is `TRUE`, the `gtk.Entry.signals.populate`-popup
        /// signal is also emitted for touch popups.
        pub const populate_all = struct {
            pub const name = "populate-all";

            pub const Type = c_int;
        };

        /// Whether the primary icon is activatable.
        ///
        /// GTK+ emits the `gtk.Entry.signals.icon`-press and `gtk.Entry.signals.icon`-release
        /// signals only on sensitive, activatable icons.
        ///
        /// Sensitive, but non-activatable icons can be used for purely
        /// informational purposes.
        pub const primary_icon_activatable = struct {
            pub const name = "primary-icon-activatable";

            pub const Type = c_int;
        };

        /// The `gio.Icon` to use for the primary icon for the entry.
        pub const primary_icon_gicon = struct {
            pub const name = "primary-icon-gicon";

            pub const Type = ?*gio.Icon;
        };

        /// The icon name to use for the primary icon for the entry.
        pub const primary_icon_name = struct {
            pub const name = "primary-icon-name";

            pub const Type = ?[*:0]u8;
        };

        /// A pixbuf to use as the primary icon for the entry.
        pub const primary_icon_pixbuf = struct {
            pub const name = "primary-icon-pixbuf";

            pub const Type = ?*gdkpixbuf.Pixbuf;
        };

        /// Whether the primary icon is sensitive.
        ///
        /// An insensitive icon appears grayed out. GTK+ does not emit the
        /// `gtk.Entry.signals.icon`-press and `gtk.Entry.signals.icon`-release signals and
        /// does not allow DND from insensitive icons.
        ///
        /// An icon should be set insensitive if the action that would trigger
        /// when clicked is currently not available.
        pub const primary_icon_sensitive = struct {
            pub const name = "primary-icon-sensitive";

            pub const Type = c_int;
        };

        /// The stock id to use for the primary icon for the entry.
        pub const primary_icon_stock = struct {
            pub const name = "primary-icon-stock";

            pub const Type = ?[*:0]u8;
        };

        /// The representation which is used for the primary icon of the entry.
        pub const primary_icon_storage_type = struct {
            pub const name = "primary-icon-storage-type";

            pub const Type = gtk.ImageType;
        };

        /// The contents of the tooltip on the primary icon, which is marked up
        /// with the [Pango text markup language][PangoMarkupFormat].
        ///
        /// Also see `gtk.Entry.setIconTooltipMarkup`.
        pub const primary_icon_tooltip_markup = struct {
            pub const name = "primary-icon-tooltip-markup";

            pub const Type = ?[*:0]u8;
        };

        /// The contents of the tooltip on the primary icon.
        ///
        /// Also see `gtk.Entry.setIconTooltipText`.
        pub const primary_icon_tooltip_text = struct {
            pub const name = "primary-icon-tooltip-text";

            pub const Type = ?[*:0]u8;
        };

        /// The current fraction of the task that's been completed.
        pub const progress_fraction = struct {
            pub const name = "progress-fraction";

            pub const Type = f64;
        };

        /// The fraction of total entry width to move the progress
        /// bouncing block for each call to `gtk.Entry.progressPulse`.
        pub const progress_pulse_step = struct {
            pub const name = "progress-pulse-step";

            pub const Type = f64;
        };

        pub const scroll_offset = struct {
            pub const name = "scroll-offset";

            pub const Type = c_int;
        };

        /// Whether the secondary icon is activatable.
        ///
        /// GTK+ emits the `gtk.Entry.signals.icon`-press and `gtk.Entry.signals.icon`-release
        /// signals only on sensitive, activatable icons.
        ///
        /// Sensitive, but non-activatable icons can be used for purely
        /// informational purposes.
        pub const secondary_icon_activatable = struct {
            pub const name = "secondary-icon-activatable";

            pub const Type = c_int;
        };

        /// The `gio.Icon` to use for the secondary icon for the entry.
        pub const secondary_icon_gicon = struct {
            pub const name = "secondary-icon-gicon";

            pub const Type = ?*gio.Icon;
        };

        /// The icon name to use for the secondary icon for the entry.
        pub const secondary_icon_name = struct {
            pub const name = "secondary-icon-name";

            pub const Type = ?[*:0]u8;
        };

        /// An pixbuf to use as the secondary icon for the entry.
        pub const secondary_icon_pixbuf = struct {
            pub const name = "secondary-icon-pixbuf";

            pub const Type = ?*gdkpixbuf.Pixbuf;
        };

        /// Whether the secondary icon is sensitive.
        ///
        /// An insensitive icon appears grayed out. GTK+ does not emit the
        /// `gtk.Entry.signals.icon`-press and `gtk.Entry.signals.icon`-release signals and
        /// does not allow DND from insensitive icons.
        ///
        /// An icon should be set insensitive if the action that would trigger
        /// when clicked is currently not available.
        pub const secondary_icon_sensitive = struct {
            pub const name = "secondary-icon-sensitive";

            pub const Type = c_int;
        };

        /// The stock id to use for the secondary icon for the entry.
        pub const secondary_icon_stock = struct {
            pub const name = "secondary-icon-stock";

            pub const Type = ?[*:0]u8;
        };

        /// The representation which is used for the secondary icon of the entry.
        pub const secondary_icon_storage_type = struct {
            pub const name = "secondary-icon-storage-type";

            pub const Type = gtk.ImageType;
        };

        /// The contents of the tooltip on the secondary icon, which is marked up
        /// with the [Pango text markup language][PangoMarkupFormat].
        ///
        /// Also see `gtk.Entry.setIconTooltipMarkup`.
        pub const secondary_icon_tooltip_markup = struct {
            pub const name = "secondary-icon-tooltip-markup";

            pub const Type = ?[*:0]u8;
        };

        /// The contents of the tooltip on the secondary icon.
        ///
        /// Also see `gtk.Entry.setIconTooltipText`.
        pub const secondary_icon_tooltip_text = struct {
            pub const name = "secondary-icon-tooltip-text";

            pub const Type = ?[*:0]u8;
        };

        pub const selection_bound = struct {
            pub const name = "selection-bound";

            pub const Type = c_int;
        };

        /// Which kind of shadow to draw around the entry when
        /// `gtk.Entry.properties.has`-frame is set to `TRUE`.
        pub const shadow_type = struct {
            pub const name = "shadow-type";

            pub const Type = gtk.ShadowType;
        };

        pub const show_emoji_icon = struct {
            pub const name = "show-emoji-icon";

            pub const Type = c_int;
        };

        pub const tabs = struct {
            pub const name = "tabs";

            pub const Type = ?*pango.TabArray;
        };

        pub const text = struct {
            pub const name = "text";

            pub const Type = ?[*:0]u8;
        };

        /// The length of the text in the `gtk.Entry`.
        pub const text_length = struct {
            pub const name = "text-length";

            pub const Type = c_uint;
        };

        /// When `TRUE`, pasted multi-line text is truncated to the first line.
        pub const truncate_multiline = struct {
            pub const name = "truncate-multiline";

            pub const Type = c_int;
        };

        pub const visibility = struct {
            pub const name = "visibility";

            pub const Type = c_int;
        };

        pub const width_chars = struct {
            pub const name = "width-chars";

            pub const Type = c_int;
        };

        /// The horizontal alignment, from 0 (left) to 1 (right).
        /// Reversed for RTL layouts.
        pub const xalign = struct {
            pub const name = "xalign";

            pub const Type = f32;
        };
    };

    pub const signals = struct {
        /// The ::activate signal is emitted when the user hits
        /// the Enter key.
        ///
        /// While this signal is used as a
        /// [keybinding signal][GtkBindingSignal],
        /// it is also commonly used by applications to intercept
        /// activation of entries.
        ///
        /// The default bindings for this signal are all forms of the Enter key.
        pub const activate = struct {
            pub const name = "activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("activate", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::backspace signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// The default bindings for this signal are
        /// Backspace and Shift-Backspace.
        pub const backspace = struct {
            pub const name = "backspace";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("backspace", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::copy-clipboard signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted to copy the selection to the clipboard.
        ///
        /// The default bindings for this signal are
        /// Ctrl-c and Ctrl-Insert.
        pub const copy_clipboard = struct {
            pub const name = "copy-clipboard";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("copy-clipboard", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::cut-clipboard signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted to cut the selection to the clipboard.
        ///
        /// The default bindings for this signal are
        /// Ctrl-x and Shift-Delete.
        pub const cut_clipboard = struct {
            pub const name = "cut-clipboard";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("cut-clipboard", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::delete-from-cursor signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user initiates a text deletion.
        ///
        /// If the `type` is `GTK_DELETE_CHARS`, GTK+ deletes the selection
        /// if there is one, otherwise it deletes the requested number
        /// of characters.
        ///
        /// The default bindings for this signal are
        /// Delete for deleting a character and Ctrl-Delete for
        /// deleting a word.
        pub const delete_from_cursor = struct {
            pub const name = "delete-from-cursor";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_type: gtk.DeleteType, p_count: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("delete-from-cursor", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::icon-press signal is emitted when an activatable icon
        /// is clicked.
        pub const icon_press = struct {
            pub const name = "icon-press";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_icon_pos: gtk.EntryIconPosition, p_event: *gdk.Event, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("icon-press", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::icon-release signal is emitted on the button release from a
        /// mouse click over an activatable icon.
        pub const icon_release = struct {
            pub const name = "icon-release";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_icon_pos: gtk.EntryIconPosition, p_event: *gdk.Event, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("icon-release", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::insert-at-cursor signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user initiates the insertion of a
        /// fixed string at the cursor.
        ///
        /// This signal has no default bindings.
        pub const insert_at_cursor = struct {
            pub const name = "insert-at-cursor";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_string: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("insert-at-cursor", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::insert-emoji signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted to present the Emoji chooser for the `entry`.
        ///
        /// The default bindings for this signal are Ctrl-. and Ctrl-;
        pub const insert_emoji = struct {
            pub const name = "insert-emoji";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("insert-emoji", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::move-cursor signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user initiates a cursor movement.
        /// If the cursor is not visible in `entry`, this signal causes
        /// the viewport to be moved instead.
        ///
        /// Applications should not connect to it, but may emit it with
        /// `gobject.signalEmitByName` if they need to control the cursor
        /// programmatically.
        ///
        /// The default bindings for this signal come in two variants,
        /// the variant with the Shift modifier extends the selection,
        /// the variant without the Shift modifer does not.
        /// There are too many key combinations to list them all here.
        /// - Arrow keys move by individual characters/lines
        /// - Ctrl-arrow key combinations move by words/paragraphs
        /// - Home/End keys move to the ends of the buffer
        pub const move_cursor = struct {
            pub const name = "move-cursor";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_step: gtk.MovementStep, p_count: c_int, p_extend_selection: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("move-cursor", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::paste-clipboard signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted to paste the contents of the clipboard
        /// into the text view.
        ///
        /// The default bindings for this signal are
        /// Ctrl-v and Shift-Insert.
        pub const paste_clipboard = struct {
            pub const name = "paste-clipboard";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("paste-clipboard", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::populate-popup signal gets emitted before showing the
        /// context menu of the entry.
        ///
        /// If you need to add items to the context menu, connect
        /// to this signal and append your items to the `widget`, which
        /// will be a `gtk.Menu` in this case.
        ///
        /// If `gtk.Entry.properties.populate`-all is `TRUE`, this signal will
        /// also be emitted to populate touch popups. In this case,
        /// `widget` will be a different container, e.g. a `gtk.Toolbar`.
        /// The signal handler should not make assumptions about the
        /// type of `widget`.
        pub const populate_popup = struct {
            pub const name = "populate-popup";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_widget: *gtk.Widget, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("populate-popup", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// If an input method is used, the typed text will not immediately
        /// be committed to the buffer. So if you are interested in the text,
        /// connect to this signal.
        pub const preedit_changed = struct {
            pub const name = "preedit-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_preedit: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("preedit-changed", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::toggle-overwrite signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted to toggle the overwrite mode of the entry.
        ///
        /// The default bindings for this signal is Insert.
        pub const toggle_overwrite = struct {
            pub const name = "toggle-overwrite";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Entry, p_instance))),
                    gobject.signalLookup("toggle-overwrite", Entry.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new entry.
    extern fn gtk_entry_new() *gtk.Entry;
    pub const new = gtk_entry_new;

    /// Creates a new entry with the specified text buffer.
    extern fn gtk_entry_new_with_buffer(p_buffer: *gtk.EntryBuffer) *gtk.Entry;
    pub const newWithBuffer = gtk_entry_new_with_buffer;

    /// Retrieves the value set by `gtk.Entry.setActivatesDefault`.
    extern fn gtk_entry_get_activates_default(p_entry: *Entry) c_int;
    pub const getActivatesDefault = gtk_entry_get_activates_default;

    /// Gets the value set by `gtk.Entry.setAlignment`.
    extern fn gtk_entry_get_alignment(p_entry: *Entry) f32;
    pub const getAlignment = gtk_entry_get_alignment;

    /// Gets the attribute list that was set on the entry using
    /// `gtk.Entry.setAttributes`, if any.
    extern fn gtk_entry_get_attributes(p_entry: *Entry) ?*pango.AttrList;
    pub const getAttributes = gtk_entry_get_attributes;

    /// Get the `gtk.EntryBuffer` object which holds the text for
    /// this widget.
    extern fn gtk_entry_get_buffer(p_entry: *Entry) *gtk.EntryBuffer;
    pub const getBuffer = gtk_entry_get_buffer;

    /// Returns the auxiliary completion object currently in use by `entry`.
    extern fn gtk_entry_get_completion(p_entry: *Entry) *gtk.EntryCompletion;
    pub const getCompletion = gtk_entry_get_completion;

    /// Returns the index of the icon which is the source of the current
    /// DND operation, or -1.
    ///
    /// This function is meant to be used in a `gtk.Widget.signals.drag`-data-get
    /// callback.
    extern fn gtk_entry_get_current_icon_drag_source(p_entry: *Entry) c_int;
    pub const getCurrentIconDragSource = gtk_entry_get_current_icon_drag_source;

    /// Retrieves the horizontal cursor adjustment for the entry.
    /// See `gtk.Entry.setCursorHadjustment`.
    extern fn gtk_entry_get_cursor_hadjustment(p_entry: *Entry) ?*gtk.Adjustment;
    pub const getCursorHadjustment = gtk_entry_get_cursor_hadjustment;

    /// Gets the value set by `gtk.Entry.setHasFrame`.
    extern fn gtk_entry_get_has_frame(p_entry: *Entry) c_int;
    pub const getHasFrame = gtk_entry_get_has_frame;

    /// Returns whether the icon is activatable.
    extern fn gtk_entry_get_icon_activatable(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition) c_int;
    pub const getIconActivatable = gtk_entry_get_icon_activatable;

    /// Gets the area where entry’s icon at `icon_pos` is drawn.
    /// This function is useful when drawing something to the
    /// entry in a draw callback.
    ///
    /// If the entry is not realized or has no icon at the given position,
    /// `icon_area` is filled with zeros. Otherwise, `icon_area` will be filled
    /// with the icon’s allocation, relative to `entry`’s allocation.
    ///
    /// See also `gtk.Entry.getTextArea`
    extern fn gtk_entry_get_icon_area(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition, p_icon_area: *gdk.Rectangle) void;
    pub const getIconArea = gtk_entry_get_icon_area;

    /// Finds the icon at the given position and return its index. The
    /// position’s coordinates are relative to the `entry`’s top left corner.
    /// If `x`, `y` doesn’t lie inside an icon, -1 is returned.
    /// This function is intended for use in a `gtk.Widget.signals.query`-tooltip
    /// signal handler.
    extern fn gtk_entry_get_icon_at_pos(p_entry: *Entry, p_x: c_int, p_y: c_int) c_int;
    pub const getIconAtPos = gtk_entry_get_icon_at_pos;

    /// Retrieves the `gio.Icon` used for the icon, or `NULL` if there is
    /// no icon or if the icon was set by some other method (e.g., by
    /// stock, pixbuf, or icon name).
    extern fn gtk_entry_get_icon_gicon(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition) ?*gio.Icon;
    pub const getIconGicon = gtk_entry_get_icon_gicon;

    /// Retrieves the icon name used for the icon, or `NULL` if there is
    /// no icon or if the icon was set by some other method (e.g., by
    /// pixbuf, stock or gicon).
    extern fn gtk_entry_get_icon_name(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition) ?[*:0]const u8;
    pub const getIconName = gtk_entry_get_icon_name;

    /// Retrieves the image used for the icon.
    ///
    /// Unlike the other methods of setting and getting icon data, this
    /// method will work regardless of whether the icon was set using a
    /// `gdkpixbuf.Pixbuf`, a `gio.Icon`, a stock item, or an icon name.
    extern fn gtk_entry_get_icon_pixbuf(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition) ?*gdkpixbuf.Pixbuf;
    pub const getIconPixbuf = gtk_entry_get_icon_pixbuf;

    /// Returns whether the icon appears sensitive or insensitive.
    extern fn gtk_entry_get_icon_sensitive(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition) c_int;
    pub const getIconSensitive = gtk_entry_get_icon_sensitive;

    /// Retrieves the stock id used for the icon, or `NULL` if there is
    /// no icon or if the icon was set by some other method (e.g., by
    /// pixbuf, icon name or gicon).
    extern fn gtk_entry_get_icon_stock(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition) [*:0]const u8;
    pub const getIconStock = gtk_entry_get_icon_stock;

    /// Gets the type of representation being used by the icon
    /// to store image data. If the icon has no image data,
    /// the return value will be `GTK_IMAGE_EMPTY`.
    extern fn gtk_entry_get_icon_storage_type(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition) gtk.ImageType;
    pub const getIconStorageType = gtk_entry_get_icon_storage_type;

    /// Gets the contents of the tooltip on the icon at the specified
    /// position in `entry`.
    extern fn gtk_entry_get_icon_tooltip_markup(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition) ?[*:0]u8;
    pub const getIconTooltipMarkup = gtk_entry_get_icon_tooltip_markup;

    /// Gets the contents of the tooltip on the icon at the specified
    /// position in `entry`.
    extern fn gtk_entry_get_icon_tooltip_text(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition) ?[*:0]u8;
    pub const getIconTooltipText = gtk_entry_get_icon_tooltip_text;

    /// This function returns the entry’s `gtk.Entry.properties.inner`-border property. See
    /// `gtk.Entry.setInnerBorder` for more information.
    extern fn gtk_entry_get_inner_border(p_entry: *Entry) ?*const gtk.Border;
    pub const getInnerBorder = gtk_entry_get_inner_border;

    /// Gets the value of the `gtk.Entry.properties.input`-hints property.
    extern fn gtk_entry_get_input_hints(p_entry: *Entry) gtk.InputHints;
    pub const getInputHints = gtk_entry_get_input_hints;

    /// Gets the value of the `gtk.Entry.properties.input`-purpose property.
    extern fn gtk_entry_get_input_purpose(p_entry: *Entry) gtk.InputPurpose;
    pub const getInputPurpose = gtk_entry_get_input_purpose;

    /// Retrieves the character displayed in place of the real characters
    /// for entries with visibility set to false. See `gtk.Entry.setInvisibleChar`.
    extern fn gtk_entry_get_invisible_char(p_entry: *Entry) u32;
    pub const getInvisibleChar = gtk_entry_get_invisible_char;

    /// Gets the `pango.Layout` used to display the entry.
    /// The layout is useful to e.g. convert text positions to
    /// pixel positions, in combination with `gtk.Entry.getLayoutOffsets`.
    /// The returned layout is owned by the entry and must not be
    /// modified or freed by the caller.
    ///
    /// Keep in mind that the layout text may contain a preedit string, so
    /// `gtk.Entry.layoutIndexToTextIndex` and
    /// `gtk.Entry.textIndexToLayoutIndex` are needed to convert byte
    /// indices in the layout to byte indices in the entry contents.
    extern fn gtk_entry_get_layout(p_entry: *Entry) *pango.Layout;
    pub const getLayout = gtk_entry_get_layout;

    /// Obtains the position of the `pango.Layout` used to render text
    /// in the entry, in widget coordinates. Useful if you want to line
    /// up the text in an entry with some other text, e.g. when using the
    /// entry to implement editable cells in a sheet widget.
    ///
    /// Also useful to convert mouse events into coordinates inside the
    /// `pango.Layout`, e.g. to take some action if some part of the entry text
    /// is clicked.
    ///
    /// Note that as the user scrolls around in the entry the offsets will
    /// change; you’ll need to connect to the “notify::scroll-offset”
    /// signal to track this. Remember when using the `pango.Layout`
    /// functions you need to convert to and from pixels using
    /// `PANGO_PIXELS` or `PANGO_SCALE`.
    ///
    /// Keep in mind that the layout text may contain a preedit string, so
    /// `gtk.Entry.layoutIndexToTextIndex` and
    /// `gtk.Entry.textIndexToLayoutIndex` are needed to convert byte
    /// indices in the layout to byte indices in the entry contents.
    extern fn gtk_entry_get_layout_offsets(p_entry: *Entry, p_x: ?*c_int, p_y: ?*c_int) void;
    pub const getLayoutOffsets = gtk_entry_get_layout_offsets;

    /// Retrieves the maximum allowed length of the text in
    /// `entry`. See `gtk.Entry.setMaxLength`.
    ///
    /// This is equivalent to getting `entry`'s `gtk.EntryBuffer` and
    /// calling `gtk.EntryBuffer.getMaxLength` on it.
    extern fn gtk_entry_get_max_length(p_entry: *Entry) c_int;
    pub const getMaxLength = gtk_entry_get_max_length;

    /// Retrieves the desired maximum width of `entry`, in characters.
    /// See `gtk.Entry.setMaxWidthChars`.
    extern fn gtk_entry_get_max_width_chars(p_entry: *Entry) c_int;
    pub const getMaxWidthChars = gtk_entry_get_max_width_chars;

    /// Gets the value set by `gtk.Entry.setOverwriteMode`.
    extern fn gtk_entry_get_overwrite_mode(p_entry: *Entry) c_int;
    pub const getOverwriteMode = gtk_entry_get_overwrite_mode;

    /// Retrieves the text that will be displayed when `entry` is empty and unfocused
    extern fn gtk_entry_get_placeholder_text(p_entry: *Entry) [*:0]const u8;
    pub const getPlaceholderText = gtk_entry_get_placeholder_text;

    /// Returns the current fraction of the task that’s been completed.
    /// See `gtk.Entry.setProgressFraction`.
    extern fn gtk_entry_get_progress_fraction(p_entry: *Entry) f64;
    pub const getProgressFraction = gtk_entry_get_progress_fraction;

    /// Retrieves the pulse step set with `gtk.Entry.setProgressPulseStep`.
    extern fn gtk_entry_get_progress_pulse_step(p_entry: *Entry) f64;
    pub const getProgressPulseStep = gtk_entry_get_progress_pulse_step;

    /// Gets the tabstops that were set on the entry using `gtk.Entry.setTabs`, if
    /// any.
    extern fn gtk_entry_get_tabs(p_entry: *Entry) ?*pango.TabArray;
    pub const getTabs = gtk_entry_get_tabs;

    /// Retrieves the contents of the entry widget.
    /// See also `gtk.Editable.getChars`.
    ///
    /// This is equivalent to getting `entry`'s `gtk.EntryBuffer` and calling
    /// `gtk.EntryBuffer.getText` on it.
    extern fn gtk_entry_get_text(p_entry: *Entry) [*:0]const u8;
    pub const getText = gtk_entry_get_text;

    /// Gets the area where the entry’s text is drawn. This function is
    /// useful when drawing something to the entry in a draw callback.
    ///
    /// If the entry is not realized, `text_area` is filled with zeros.
    ///
    /// See also `gtk.Entry.getIconArea`.
    extern fn gtk_entry_get_text_area(p_entry: *Entry, p_text_area: *gdk.Rectangle) void;
    pub const getTextArea = gtk_entry_get_text_area;

    /// Retrieves the current length of the text in
    /// `entry`.
    ///
    /// This is equivalent to getting `entry`'s `gtk.EntryBuffer` and
    /// calling `gtk.EntryBuffer.getLength` on it.
    extern fn gtk_entry_get_text_length(p_entry: *Entry) u16;
    pub const getTextLength = gtk_entry_get_text_length;

    /// Retrieves whether the text in `entry` is visible. See
    /// `gtk.Entry.setVisibility`.
    extern fn gtk_entry_get_visibility(p_entry: *Entry) c_int;
    pub const getVisibility = gtk_entry_get_visibility;

    /// Gets the value set by `gtk.Entry.setWidthChars`.
    extern fn gtk_entry_get_width_chars(p_entry: *Entry) c_int;
    pub const getWidthChars = gtk_entry_get_width_chars;

    /// Causes `entry` to have keyboard focus.
    ///
    /// It behaves like `gtk.Widget.grabFocus`,
    /// except that it doesn't select the contents of the entry.
    /// You only want to call this on some special entries
    /// which the user usually doesn't want to replace all text in,
    /// such as search-as-you-type entries.
    extern fn gtk_entry_grab_focus_without_selecting(p_entry: *Entry) void;
    pub const grabFocusWithoutSelecting = gtk_entry_grab_focus_without_selecting;

    /// Allow the `gtk.Entry` input method to internally handle key press
    /// and release events. If this function returns `TRUE`, then no further
    /// processing should be done for this key event. See
    /// `gtk.IMContext.filterKeypress`.
    ///
    /// Note that you are expected to call this function from your handler
    /// when overriding key event handling. This is needed in the case when
    /// you need to insert your own key handling between the input method
    /// and the default key event handling of the `gtk.Entry`.
    /// See `gtk.TextView.resetImContext` for an example of use.
    extern fn gtk_entry_im_context_filter_keypress(p_entry: *Entry, p_event: *gdk.EventKey) c_int;
    pub const imContextFilterKeypress = gtk_entry_im_context_filter_keypress;

    /// Converts from a position in the entry’s `pango.Layout` (returned by
    /// `gtk.Entry.getLayout`) to a position in the entry contents
    /// (returned by `gtk.Entry.getText`).
    extern fn gtk_entry_layout_index_to_text_index(p_entry: *Entry, p_layout_index: c_int) c_int;
    pub const layoutIndexToTextIndex = gtk_entry_layout_index_to_text_index;

    /// Indicates that some progress is made, but you don’t know how much.
    /// Causes the entry’s progress indicator to enter “activity mode,”
    /// where a block bounces back and forth. Each call to
    /// `gtk.Entry.progressPulse` causes the block to move by a little bit
    /// (the amount of movement per pulse is determined by
    /// `gtk.Entry.setProgressPulseStep`).
    extern fn gtk_entry_progress_pulse(p_entry: *Entry) void;
    pub const progressPulse = gtk_entry_progress_pulse;

    /// Reset the input method context of the entry if needed.
    ///
    /// This can be necessary in the case where modifying the buffer
    /// would confuse on-going input method behavior.
    extern fn gtk_entry_reset_im_context(p_entry: *Entry) void;
    pub const resetImContext = gtk_entry_reset_im_context;

    /// If `setting` is `TRUE`, pressing Enter in the `entry` will activate the default
    /// widget for the window containing the entry. This usually means that
    /// the dialog box containing the entry will be closed, since the default
    /// widget is usually one of the dialog buttons.
    ///
    /// (For experts: if `setting` is `TRUE`, the entry calls
    /// `gtk.Window.activateDefault` on the window containing the entry, in
    /// the default handler for the `gtk.Entry.signals.activate` signal.)
    extern fn gtk_entry_set_activates_default(p_entry: *Entry, p_setting: c_int) void;
    pub const setActivatesDefault = gtk_entry_set_activates_default;

    /// Sets the alignment for the contents of the entry. This controls
    /// the horizontal positioning of the contents when the displayed
    /// text is shorter than the width of the entry.
    extern fn gtk_entry_set_alignment(p_entry: *Entry, p_xalign: f32) void;
    pub const setAlignment = gtk_entry_set_alignment;

    /// Sets a `pango.AttrList`; the attributes in the list are applied to the
    /// entry text.
    extern fn gtk_entry_set_attributes(p_entry: *Entry, p_attrs: *pango.AttrList) void;
    pub const setAttributes = gtk_entry_set_attributes;

    /// Set the `gtk.EntryBuffer` object which holds the text for
    /// this widget.
    extern fn gtk_entry_set_buffer(p_entry: *Entry, p_buffer: *gtk.EntryBuffer) void;
    pub const setBuffer = gtk_entry_set_buffer;

    /// Sets `completion` to be the auxiliary completion object to use with `entry`.
    /// All further configuration of the completion mechanism is done on
    /// `completion` using the `gtk.EntryCompletion` API. Completion is disabled if
    /// `completion` is set to `NULL`.
    extern fn gtk_entry_set_completion(p_entry: *Entry, p_completion: ?*gtk.EntryCompletion) void;
    pub const setCompletion = gtk_entry_set_completion;

    /// Hooks up an adjustment to the cursor position in an entry, so that when
    /// the cursor is moved, the adjustment is scrolled to show that position.
    /// See `gtk.ScrolledWindow.getHadjustment` for a typical way of obtaining
    /// the adjustment.
    ///
    /// The adjustment has to be in pixel units and in the same coordinate system
    /// as the entry.
    extern fn gtk_entry_set_cursor_hadjustment(p_entry: *Entry, p_adjustment: ?*gtk.Adjustment) void;
    pub const setCursorHadjustment = gtk_entry_set_cursor_hadjustment;

    /// Sets whether the entry has a beveled frame around it.
    extern fn gtk_entry_set_has_frame(p_entry: *Entry, p_setting: c_int) void;
    pub const setHasFrame = gtk_entry_set_has_frame;

    /// Sets whether the icon is activatable.
    extern fn gtk_entry_set_icon_activatable(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition, p_activatable: c_int) void;
    pub const setIconActivatable = gtk_entry_set_icon_activatable;

    /// Sets up the icon at the given position so that GTK+ will start a drag
    /// operation when the user clicks and drags the icon.
    ///
    /// To handle the drag operation, you need to connect to the usual
    /// `gtk.Widget.signals.drag`-data-get (or possibly `gtk.Widget.signals.drag`-data-delete)
    /// signal, and use `gtk.Entry.getCurrentIconDragSource` in
    /// your signal handler to find out if the drag was started from
    /// an icon.
    ///
    /// By default, GTK+ uses the icon as the drag icon. You can use the
    /// `gtk.Widget.signals.drag`-begin signal to set a different icon. Note that you
    /// have to use `g_signal_connect_after` to ensure that your signal handler
    /// gets executed after the default handler.
    extern fn gtk_entry_set_icon_drag_source(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition, p_target_list: *gtk.TargetList, p_actions: gdk.DragAction) void;
    pub const setIconDragSource = gtk_entry_set_icon_drag_source;

    /// Sets the icon shown in the entry at the specified position
    /// from the current icon theme.
    /// If the icon isn’t known, a “broken image” icon will be displayed
    /// instead.
    ///
    /// If `icon` is `NULL`, no icon will be shown in the specified position.
    extern fn gtk_entry_set_icon_from_gicon(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition, p_icon: ?*gio.Icon) void;
    pub const setIconFromGicon = gtk_entry_set_icon_from_gicon;

    /// Sets the icon shown in the entry at the specified position
    /// from the current icon theme.
    ///
    /// If the icon name isn’t known, a “broken image” icon will be displayed
    /// instead.
    ///
    /// If `icon_name` is `NULL`, no icon will be shown in the specified position.
    extern fn gtk_entry_set_icon_from_icon_name(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition, p_icon_name: ?[*:0]const u8) void;
    pub const setIconFromIconName = gtk_entry_set_icon_from_icon_name;

    /// Sets the icon shown in the specified position using a pixbuf.
    ///
    /// If `pixbuf` is `NULL`, no icon will be shown in the specified position.
    extern fn gtk_entry_set_icon_from_pixbuf(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition, p_pixbuf: ?*gdkpixbuf.Pixbuf) void;
    pub const setIconFromPixbuf = gtk_entry_set_icon_from_pixbuf;

    /// Sets the icon shown in the entry at the specified position from
    /// a stock image.
    ///
    /// If `stock_id` is `NULL`, no icon will be shown in the specified position.
    extern fn gtk_entry_set_icon_from_stock(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition, p_stock_id: ?[*:0]const u8) void;
    pub const setIconFromStock = gtk_entry_set_icon_from_stock;

    /// Sets the sensitivity for the specified icon.
    extern fn gtk_entry_set_icon_sensitive(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition, p_sensitive: c_int) void;
    pub const setIconSensitive = gtk_entry_set_icon_sensitive;

    /// Sets `tooltip` as the contents of the tooltip for the icon at
    /// the specified position. `tooltip` is assumed to be marked up with
    /// the [Pango text markup language][PangoMarkupFormat].
    ///
    /// Use `NULL` for `tooltip` to remove an existing tooltip.
    ///
    /// See also `gtk.Widget.setTooltipMarkup` and
    /// `gtk.Entry.setIconTooltipText`.
    extern fn gtk_entry_set_icon_tooltip_markup(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition, p_tooltip: ?[*:0]const u8) void;
    pub const setIconTooltipMarkup = gtk_entry_set_icon_tooltip_markup;

    /// Sets `tooltip` as the contents of the tooltip for the icon
    /// at the specified position.
    ///
    /// Use `NULL` for `tooltip` to remove an existing tooltip.
    ///
    /// See also `gtk.Widget.setTooltipText` and
    /// `gtk.Entry.setIconTooltipMarkup`.
    ///
    /// If you unset the widget tooltip via `gtk.Widget.setTooltipText` or
    /// `gtk.Widget.setTooltipMarkup`, this sets GtkWidget:has-tooltip to `FALSE`,
    /// which suppresses icon tooltips too. You can resolve this by then calling
    /// `gtk.Widget.setHasTooltip` to set GtkWidget:has-tooltip back to `TRUE`, or
    /// setting at least one non-empty tooltip on any icon achieves the same result.
    extern fn gtk_entry_set_icon_tooltip_text(p_entry: *Entry, p_icon_pos: gtk.EntryIconPosition, p_tooltip: ?[*:0]const u8) void;
    pub const setIconTooltipText = gtk_entry_set_icon_tooltip_text;

    /// Sets `entry`’s inner-border property to `border`, or clears it if `NULL`
    /// is passed. The inner-border is the area around the entry’s text, but
    /// inside its frame.
    ///
    /// If set, this property overrides the inner-border style property.
    /// Overriding the style-provided border is useful when you want to do
    /// in-place editing of some text in a canvas or list widget, where
    /// pixel-exact positioning of the entry is important.
    extern fn gtk_entry_set_inner_border(p_entry: *Entry, p_border: ?*const gtk.Border) void;
    pub const setInnerBorder = gtk_entry_set_inner_border;

    /// Sets the `gtk.Entry.properties.input`-hints property, which
    /// allows input methods to fine-tune their behaviour.
    extern fn gtk_entry_set_input_hints(p_entry: *Entry, p_hints: gtk.InputHints) void;
    pub const setInputHints = gtk_entry_set_input_hints;

    /// Sets the `gtk.Entry.properties.input`-purpose property which
    /// can be used by on-screen keyboards and other input
    /// methods to adjust their behaviour.
    extern fn gtk_entry_set_input_purpose(p_entry: *Entry, p_purpose: gtk.InputPurpose) void;
    pub const setInputPurpose = gtk_entry_set_input_purpose;

    /// Sets the character to use in place of the actual text when
    /// `gtk.Entry.setVisibility` has been called to set text visibility
    /// to `FALSE`. i.e. this is the character used in “password mode” to
    /// show the user how many characters have been typed. By default, GTK+
    /// picks the best invisible char available in the current font. If you
    /// set the invisible char to 0, then the user will get no feedback
    /// at all; there will be no text on the screen as they type.
    extern fn gtk_entry_set_invisible_char(p_entry: *Entry, p_ch: u32) void;
    pub const setInvisibleChar = gtk_entry_set_invisible_char;

    /// Sets the maximum allowed length of the contents of the widget. If
    /// the current contents are longer than the given length, then they
    /// will be truncated to fit.
    ///
    /// This is equivalent to getting `entry`'s `gtk.EntryBuffer` and
    /// calling `gtk.EntryBuffer.setMaxLength` on it.
    /// ```
    extern fn gtk_entry_set_max_length(p_entry: *Entry, p_max: c_int) void;
    pub const setMaxLength = gtk_entry_set_max_length;

    /// Sets the desired maximum width in characters of `entry`.
    extern fn gtk_entry_set_max_width_chars(p_entry: *Entry, p_n_chars: c_int) void;
    pub const setMaxWidthChars = gtk_entry_set_max_width_chars;

    /// Sets whether the text is overwritten when typing in the `gtk.Entry`.
    extern fn gtk_entry_set_overwrite_mode(p_entry: *Entry, p_overwrite: c_int) void;
    pub const setOverwriteMode = gtk_entry_set_overwrite_mode;

    /// Sets text to be displayed in `entry` when it is empty and unfocused.
    /// This can be used to give a visual hint of the expected contents of
    /// the `gtk.Entry`.
    ///
    /// Note that since the placeholder text gets removed when the entry
    /// received focus, using this feature is a bit problematic if the entry
    /// is given the initial focus in a window. Sometimes this can be
    /// worked around by delaying the initial focus setting until the
    /// first key event arrives.
    extern fn gtk_entry_set_placeholder_text(p_entry: *Entry, p_text: ?[*:0]const u8) void;
    pub const setPlaceholderText = gtk_entry_set_placeholder_text;

    /// Causes the entry’s progress indicator to “fill in” the given
    /// fraction of the bar. The fraction should be between 0.0 and 1.0,
    /// inclusive.
    extern fn gtk_entry_set_progress_fraction(p_entry: *Entry, p_fraction: f64) void;
    pub const setProgressFraction = gtk_entry_set_progress_fraction;

    /// Sets the fraction of total entry width to move the progress
    /// bouncing block for each call to `gtk.Entry.progressPulse`.
    extern fn gtk_entry_set_progress_pulse_step(p_entry: *Entry, p_fraction: f64) void;
    pub const setProgressPulseStep = gtk_entry_set_progress_pulse_step;

    /// Sets a `pango.TabArray`; the tabstops in the array are applied to the entry
    /// text.
    extern fn gtk_entry_set_tabs(p_entry: *Entry, p_tabs: *pango.TabArray) void;
    pub const setTabs = gtk_entry_set_tabs;

    /// Sets the text in the widget to the given
    /// value, replacing the current contents.
    ///
    /// See `gtk.EntryBuffer.setText`.
    extern fn gtk_entry_set_text(p_entry: *Entry, p_text: [*:0]const u8) void;
    pub const setText = gtk_entry_set_text;

    /// Sets whether the contents of the entry are visible or not.
    /// When visibility is set to `FALSE`, characters are displayed
    /// as the invisible char, and will also appear that way when
    /// the text in the entry widget is copied elsewhere.
    ///
    /// By default, GTK+ picks the best invisible character available
    /// in the current font, but it can be changed with
    /// `gtk.Entry.setInvisibleChar`.
    ///
    /// Note that you probably want to set `gtk.Entry.properties.input`-purpose
    /// to `GTK_INPUT_PURPOSE_PASSWORD` or `GTK_INPUT_PURPOSE_PIN` to
    /// inform input methods about the purpose of this entry,
    /// in addition to setting visibility to `FALSE`.
    extern fn gtk_entry_set_visibility(p_entry: *Entry, p_visible: c_int) void;
    pub const setVisibility = gtk_entry_set_visibility;

    /// Changes the size request of the entry to be about the right size
    /// for `n_chars` characters. Note that it changes the size
    /// request, the size can still be affected by
    /// how you pack the widget into containers. If `n_chars` is -1, the
    /// size reverts to the default entry size.
    extern fn gtk_entry_set_width_chars(p_entry: *Entry, p_n_chars: c_int) void;
    pub const setWidthChars = gtk_entry_set_width_chars;

    /// Converts from a position in the entry contents (returned
    /// by `gtk.Entry.getText`) to a position in the
    /// entry’s `pango.Layout` (returned by `gtk.Entry.getLayout`,
    /// with text retrieved via `pango.Layout.getText`).
    extern fn gtk_entry_text_index_to_layout_index(p_entry: *Entry, p_text_index: c_int) c_int;
    pub const textIndexToLayoutIndex = gtk_entry_text_index_to_layout_index;

    /// Unsets the invisible char previously set with
    /// `gtk.Entry.setInvisibleChar`. So that the
    /// default invisible char is used again.
    extern fn gtk_entry_unset_invisible_char(p_entry: *Entry) void;
    pub const unsetInvisibleChar = gtk_entry_unset_invisible_char;

    extern fn gtk_entry_get_type() usize;
    pub const getGObjectType = gtk_entry_get_type;

    extern fn g_object_ref(p_self: *gtk.Entry) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Entry) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Entry, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const EntryAccessible = extern struct {
    pub const Parent = gtk.WidgetAccessible;
    pub const Implements = [_]type{ atk.Action, atk.Component, atk.EditableText, atk.Text };
    pub const Class = gtk.EntryAccessibleClass;
    f_parent: gtk.WidgetAccessible,
    f_priv: ?*gtk.EntryAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_entry_accessible_get_type() usize;
    pub const getGObjectType = gtk_entry_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.EntryAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.EntryAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EntryAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.EntryBuffer` class contains the actual text displayed in a
/// `gtk.Entry` widget.
///
/// A single `gtk.EntryBuffer` object can be shared by multiple `gtk.Entry`
/// widgets which will then share the same text content, but not the cursor
/// position, visibility attributes, icon etc.
///
/// `gtk.EntryBuffer` may be derived from. Such a derived class might allow
/// text to be stored in an alternate location, such as non-pageable memory,
/// useful in the case of important passwords. Or a derived class could
/// integrate with an application’s concept of undo/redo.
pub const EntryBuffer = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtk.EntryBufferClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gtk.EntryBufferPrivate,

    pub const virtual_methods = struct {
        /// Deletes a sequence of characters from the buffer. `n_chars` characters are
        /// deleted starting at `position`. If `n_chars` is negative, then all characters
        /// until the end of the text are deleted.
        ///
        /// If `position` or `n_chars` are out of bounds, then they are coerced to sane
        /// values.
        ///
        /// Note that the positions are specified in characters, not bytes.
        pub const delete_text = struct {
            pub fn call(p_class: anytype, p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_position: c_uint, p_n_chars: c_uint) c_uint {
                return gobject.ext.as(EntryBuffer.Class, p_class).f_delete_text.?(gobject.ext.as(EntryBuffer, p_buffer), p_position, p_n_chars);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_position: c_uint, p_n_chars: c_uint) callconv(.C) c_uint) void {
                gobject.ext.as(EntryBuffer.Class, p_class).f_delete_text = @ptrCast(p_implementation);
            }
        };

        pub const deleted_text = struct {
            pub fn call(p_class: anytype, p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_position: c_uint, p_n_chars: c_uint) void {
                return gobject.ext.as(EntryBuffer.Class, p_class).f_deleted_text.?(gobject.ext.as(EntryBuffer, p_buffer), p_position, p_n_chars);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_position: c_uint, p_n_chars: c_uint) callconv(.C) void) void {
                gobject.ext.as(EntryBuffer.Class, p_class).f_deleted_text = @ptrCast(p_implementation);
            }
        };

        /// Retrieves the length in characters of the buffer.
        pub const get_length = struct {
            pub fn call(p_class: anytype, p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_uint {
                return gobject.ext.as(EntryBuffer.Class, p_class).f_get_length.?(gobject.ext.as(EntryBuffer, p_buffer));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_uint) void {
                gobject.ext.as(EntryBuffer.Class, p_class).f_get_length = @ptrCast(p_implementation);
            }
        };

        pub const get_text = struct {
            pub fn call(p_class: anytype, p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_n_bytes: *usize) [*:0]const u8 {
                return gobject.ext.as(EntryBuffer.Class, p_class).f_get_text.?(gobject.ext.as(EntryBuffer, p_buffer), p_n_bytes);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_n_bytes: *usize) callconv(.C) [*:0]const u8) void {
                gobject.ext.as(EntryBuffer.Class, p_class).f_get_text = @ptrCast(p_implementation);
            }
        };

        /// Inserts `n_chars` characters of `chars` into the contents of the
        /// buffer, at position `position`.
        ///
        /// If `n_chars` is negative, then characters from chars will be inserted
        /// until a null-terminator is found. If `position` or `n_chars` are out of
        /// bounds, or the maximum buffer text length is exceeded, then they are
        /// coerced to sane values.
        ///
        /// Note that the position and length are in characters, not in bytes.
        pub const insert_text = struct {
            pub fn call(p_class: anytype, p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_position: c_uint, p_chars: [*:0]const u8, p_n_chars: c_uint) c_uint {
                return gobject.ext.as(EntryBuffer.Class, p_class).f_insert_text.?(gobject.ext.as(EntryBuffer, p_buffer), p_position, p_chars, p_n_chars);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_position: c_uint, p_chars: [*:0]const u8, p_n_chars: c_uint) callconv(.C) c_uint) void {
                gobject.ext.as(EntryBuffer.Class, p_class).f_insert_text = @ptrCast(p_implementation);
            }
        };

        pub const inserted_text = struct {
            pub fn call(p_class: anytype, p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_position: c_uint, p_chars: [*:0]const u8, p_n_chars: c_uint) void {
                return gobject.ext.as(EntryBuffer.Class, p_class).f_inserted_text.?(gobject.ext.as(EntryBuffer, p_buffer), p_position, p_chars, p_n_chars);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_buffer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_position: c_uint, p_chars: [*:0]const u8, p_n_chars: c_uint) callconv(.C) void) void {
                gobject.ext.as(EntryBuffer.Class, p_class).f_inserted_text = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The length (in characters) of the text in buffer.
        pub const length = struct {
            pub const name = "length";

            pub const Type = c_uint;
        };

        /// The maximum length (in characters) of the text in the buffer.
        pub const max_length = struct {
            pub const name = "max-length";

            pub const Type = c_int;
        };

        /// The contents of the buffer.
        pub const text = struct {
            pub const name = "text";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// This signal is emitted after text is deleted from the buffer.
        pub const deleted_text = struct {
            pub const name = "deleted-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_position: c_uint, p_n_chars: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EntryBuffer, p_instance))),
                    gobject.signalLookup("deleted-text", EntryBuffer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted after text is inserted into the buffer.
        pub const inserted_text = struct {
            pub const name = "inserted-text";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_position: c_uint, p_chars: [*:0]u8, p_n_chars: c_uint, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EntryBuffer, p_instance))),
                    gobject.signalLookup("inserted-text", EntryBuffer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Create a new GtkEntryBuffer object.
    ///
    /// Optionally, specify initial text to set in the buffer.
    extern fn gtk_entry_buffer_new(p_initial_chars: ?[*:0]const u8, p_n_initial_chars: c_int) *gtk.EntryBuffer;
    pub const new = gtk_entry_buffer_new;

    /// Deletes a sequence of characters from the buffer. `n_chars` characters are
    /// deleted starting at `position`. If `n_chars` is negative, then all characters
    /// until the end of the text are deleted.
    ///
    /// If `position` or `n_chars` are out of bounds, then they are coerced to sane
    /// values.
    ///
    /// Note that the positions are specified in characters, not bytes.
    extern fn gtk_entry_buffer_delete_text(p_buffer: *EntryBuffer, p_position: c_uint, p_n_chars: c_int) c_uint;
    pub const deleteText = gtk_entry_buffer_delete_text;

    /// Used when subclassing `gtk.EntryBuffer`
    extern fn gtk_entry_buffer_emit_deleted_text(p_buffer: *EntryBuffer, p_position: c_uint, p_n_chars: c_uint) void;
    pub const emitDeletedText = gtk_entry_buffer_emit_deleted_text;

    /// Used when subclassing `gtk.EntryBuffer`
    extern fn gtk_entry_buffer_emit_inserted_text(p_buffer: *EntryBuffer, p_position: c_uint, p_chars: [*:0]const u8, p_n_chars: c_uint) void;
    pub const emitInsertedText = gtk_entry_buffer_emit_inserted_text;

    /// Retrieves the length in bytes of the buffer.
    /// See `gtk.EntryBuffer.getLength`.
    extern fn gtk_entry_buffer_get_bytes(p_buffer: *EntryBuffer) usize;
    pub const getBytes = gtk_entry_buffer_get_bytes;

    /// Retrieves the length in characters of the buffer.
    extern fn gtk_entry_buffer_get_length(p_buffer: *EntryBuffer) c_uint;
    pub const getLength = gtk_entry_buffer_get_length;

    /// Retrieves the maximum allowed length of the text in
    /// `buffer`. See `gtk.EntryBuffer.setMaxLength`.
    extern fn gtk_entry_buffer_get_max_length(p_buffer: *EntryBuffer) c_int;
    pub const getMaxLength = gtk_entry_buffer_get_max_length;

    /// Retrieves the contents of the buffer.
    ///
    /// The memory pointer returned by this call will not change
    /// unless this object emits a signal, or is finalized.
    extern fn gtk_entry_buffer_get_text(p_buffer: *EntryBuffer) [*:0]const u8;
    pub const getText = gtk_entry_buffer_get_text;

    /// Inserts `n_chars` characters of `chars` into the contents of the
    /// buffer, at position `position`.
    ///
    /// If `n_chars` is negative, then characters from chars will be inserted
    /// until a null-terminator is found. If `position` or `n_chars` are out of
    /// bounds, or the maximum buffer text length is exceeded, then they are
    /// coerced to sane values.
    ///
    /// Note that the position and length are in characters, not in bytes.
    extern fn gtk_entry_buffer_insert_text(p_buffer: *EntryBuffer, p_position: c_uint, p_chars: [*:0]const u8, p_n_chars: c_int) c_uint;
    pub const insertText = gtk_entry_buffer_insert_text;

    /// Sets the maximum allowed length of the contents of the buffer. If
    /// the current contents are longer than the given length, then they
    /// will be truncated to fit.
    extern fn gtk_entry_buffer_set_max_length(p_buffer: *EntryBuffer, p_max_length: c_int) void;
    pub const setMaxLength = gtk_entry_buffer_set_max_length;

    /// Sets the text in the buffer.
    ///
    /// This is roughly equivalent to calling `gtk.EntryBuffer.deleteText`
    /// and `gtk.EntryBuffer.insertText`.
    ///
    /// Note that `n_chars` is in characters, not in bytes.
    extern fn gtk_entry_buffer_set_text(p_buffer: *EntryBuffer, p_chars: [*:0]const u8, p_n_chars: c_int) void;
    pub const setText = gtk_entry_buffer_set_text;

    extern fn gtk_entry_buffer_get_type() usize;
    pub const getGObjectType = gtk_entry_buffer_get_type;

    extern fn g_object_ref(p_self: *gtk.EntryBuffer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.EntryBuffer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EntryBuffer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.EntryCompletion` is an auxiliary object to be used in conjunction with
/// `gtk.Entry` to provide the completion functionality. It implements the
/// `gtk.CellLayout` interface, to allow the user to add extra cells to the
/// `gtk.TreeView` with completion matches.
///
/// “Completion functionality” means that when the user modifies the text
/// in the entry, `gtk.EntryCompletion` checks which rows in the model match
/// the current content of the entry, and displays a list of matches.
/// By default, the matching is done by comparing the entry text
/// case-insensitively against the text column of the model (see
/// `gtk.EntryCompletion.setTextColumn`), but this can be overridden
/// with a custom match function (see `gtk.EntryCompletion.setMatchFunc`).
///
/// When the user selects a completion, the content of the entry is
/// updated. By default, the content of the entry is replaced by the
/// text column of the model, but this can be overridden by connecting
/// to the `gtk.EntryCompletion.signals.match`-selected signal and updating the
/// entry in the signal handler. Note that you should return `TRUE` from
/// the signal handler to suppress the default behaviour.
///
/// To add completion functionality to an entry, use `gtk.Entry.setCompletion`.
///
/// In addition to regular completion matches, which will be inserted into the
/// entry when they are selected, `gtk.EntryCompletion` also allows to display
/// “actions” in the popup window. Their appearance is similar to menuitems,
/// to differentiate them clearly from completion strings. When an action is
/// selected, the `gtk.EntryCompletion.signals.action`-activated signal is emitted.
///
/// GtkEntryCompletion uses a `gtk.TreeModelFilter` model to represent the
/// subset of the entire model that is currently matching. While the
/// GtkEntryCompletion signals `gtk.EntryCompletion.signals.match`-selected and
/// `gtk.EntryCompletion.signals.cursor`-on-match take the original model and an
/// iter pointing to that model as arguments, other callbacks and signals
/// (such as `GtkCellLayoutDataFuncs` or `gtk.CellArea.signals.apply`-attributes)
/// will generally take the filter model as argument. As long as you are
/// only calling `gtk.TreeModel.get`, this will make no difference to
/// you. If for some reason, you need the original model, use
/// `gtk.TreeModelFilter.getModel`. Don’t forget to use
/// `gtk.TreeModelFilter.convertIterToChildIter` to obtain a
/// matching iter.
pub const EntryCompletion = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gtk.Buildable, gtk.CellLayout };
    pub const Class = gtk.EntryCompletionClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gtk.EntryCompletionPrivate,

    pub const virtual_methods = struct {
        pub const action_activated = struct {
            pub fn call(p_class: anytype, p_completion: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_index_: c_int) void {
                return gobject.ext.as(EntryCompletion.Class, p_class).f_action_activated.?(gobject.ext.as(EntryCompletion, p_completion), p_index_);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_completion: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_index_: c_int) callconv(.C) void) void {
                gobject.ext.as(EntryCompletion.Class, p_class).f_action_activated = @ptrCast(p_implementation);
            }
        };

        pub const cursor_on_match = struct {
            pub fn call(p_class: anytype, p_completion: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_model: *gtk.TreeModel, p_iter: *gtk.TreeIter) c_int {
                return gobject.ext.as(EntryCompletion.Class, p_class).f_cursor_on_match.?(gobject.ext.as(EntryCompletion, p_completion), p_model, p_iter);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_completion: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_model: *gtk.TreeModel, p_iter: *gtk.TreeIter) callconv(.C) c_int) void {
                gobject.ext.as(EntryCompletion.Class, p_class).f_cursor_on_match = @ptrCast(p_implementation);
            }
        };

        pub const insert_prefix = struct {
            pub fn call(p_class: anytype, p_completion: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prefix: [*:0]const u8) c_int {
                return gobject.ext.as(EntryCompletion.Class, p_class).f_insert_prefix.?(gobject.ext.as(EntryCompletion, p_completion), p_prefix);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_completion: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prefix: [*:0]const u8) callconv(.C) c_int) void {
                gobject.ext.as(EntryCompletion.Class, p_class).f_insert_prefix = @ptrCast(p_implementation);
            }
        };

        pub const match_selected = struct {
            pub fn call(p_class: anytype, p_completion: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_model: *gtk.TreeModel, p_iter: *gtk.TreeIter) c_int {
                return gobject.ext.as(EntryCompletion.Class, p_class).f_match_selected.?(gobject.ext.as(EntryCompletion, p_completion), p_model, p_iter);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_completion: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_model: *gtk.TreeModel, p_iter: *gtk.TreeIter) callconv(.C) c_int) void {
                gobject.ext.as(EntryCompletion.Class, p_class).f_match_selected = @ptrCast(p_implementation);
            }
        };

        pub const no_matches = struct {
            pub fn call(p_class: anytype, p_completion: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(EntryCompletion.Class, p_class).f_no_matches.?(gobject.ext.as(EntryCompletion, p_completion));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_completion: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(EntryCompletion.Class, p_class).f_no_matches = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The `gtk.CellArea` used to layout cell renderers in the treeview column.
        ///
        /// If no area is specified when creating the entry completion with
        /// `gtk.EntryCompletion.newWithArea` a horizontally oriented
        /// `gtk.CellAreaBox` will be used.
        pub const cell_area = struct {
            pub const name = "cell-area";

            pub const Type = ?*gtk.CellArea;
        };

        /// Determines whether the common prefix of the possible completions
        /// should be inserted automatically in the entry. Note that this
        /// requires text-column to be set, even if you are using a custom
        /// match function.
        pub const inline_completion = struct {
            pub const name = "inline-completion";

            pub const Type = c_int;
        };

        /// Determines whether the possible completions on the popup
        /// will appear in the entry as you navigate through them.
        pub const inline_selection = struct {
            pub const name = "inline-selection";

            pub const Type = c_int;
        };

        pub const minimum_key_length = struct {
            pub const name = "minimum-key-length";

            pub const Type = c_int;
        };

        pub const model = struct {
            pub const name = "model";

            pub const Type = ?*gtk.TreeModel;
        };

        /// Determines whether the possible completions should be
        /// shown in a popup window.
        pub const popup_completion = struct {
            pub const name = "popup-completion";

            pub const Type = c_int;
        };

        /// Determines whether the completions popup window will be
        /// resized to the width of the entry.
        pub const popup_set_width = struct {
            pub const name = "popup-set-width";

            pub const Type = c_int;
        };

        /// Determines whether the completions popup window will shown
        /// for a single possible completion. You probably want to set
        /// this to `FALSE` if you are using
        /// [inline completion][GtkEntryCompletion--inline-completion].
        pub const popup_single_match = struct {
            pub const name = "popup-single-match";

            pub const Type = c_int;
        };

        /// The column of the model containing the strings.
        /// Note that the strings must be UTF-8.
        pub const text_column = struct {
            pub const name = "text-column";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// Gets emitted when an action is activated.
        pub const action_activated = struct {
            pub const name = "action-activated";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_index: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EntryCompletion, p_instance))),
                    gobject.signalLookup("action-activated", EntryCompletion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Gets emitted when a match from the cursor is on a match
        /// of the list. The default behaviour is to replace the contents
        /// of the entry with the contents of the text column in the row
        /// pointed to by `iter`.
        ///
        /// Note that `model` is the model that was passed to
        /// `gtk.EntryCompletion.setModel`.
        pub const cursor_on_match = struct {
            pub const name = "cursor-on-match";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_model: *gtk.TreeModel, p_iter: *gtk.TreeIter, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EntryCompletion, p_instance))),
                    gobject.signalLookup("cursor-on-match", EntryCompletion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Gets emitted when the inline autocompletion is triggered.
        /// The default behaviour is to make the entry display the
        /// whole prefix and select the newly inserted part.
        ///
        /// Applications may connect to this signal in order to insert only a
        /// smaller part of the `prefix` into the entry - e.g. the entry used in
        /// the `gtk.FileChooser` inserts only the part of the prefix up to the
        /// next '/'.
        pub const insert_prefix = struct {
            pub const name = "insert-prefix";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_prefix: [*:0]u8, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EntryCompletion, p_instance))),
                    gobject.signalLookup("insert-prefix", EntryCompletion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Gets emitted when a match from the list is selected.
        /// The default behaviour is to replace the contents of the
        /// entry with the contents of the text column in the row
        /// pointed to by `iter`.
        ///
        /// Note that `model` is the model that was passed to
        /// `gtk.EntryCompletion.setModel`.
        pub const match_selected = struct {
            pub const name = "match-selected";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_model: *gtk.TreeModel, p_iter: *gtk.TreeIter, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EntryCompletion, p_instance))),
                    gobject.signalLookup("match-selected", EntryCompletion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Gets emitted when the filter model has zero
        /// number of rows in completion_complete method.
        /// (In other words when GtkEntryCompletion is out of
        ///  suggestions)
        pub const no_matches = struct {
            pub const name = "no-matches";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EntryCompletion, p_instance))),
                    gobject.signalLookup("no-matches", EntryCompletion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.EntryCompletion` object.
    extern fn gtk_entry_completion_new() *gtk.EntryCompletion;
    pub const new = gtk_entry_completion_new;

    /// Creates a new `gtk.EntryCompletion` object using the
    /// specified `area` to layout cells in the underlying
    /// `gtk.TreeViewColumn` for the drop-down menu.
    extern fn gtk_entry_completion_new_with_area(p_area: *gtk.CellArea) *gtk.EntryCompletion;
    pub const newWithArea = gtk_entry_completion_new_with_area;

    /// Requests a completion operation, or in other words a refiltering of the
    /// current list with completions, using the current key. The completion list
    /// view will be updated accordingly.
    extern fn gtk_entry_completion_complete(p_completion: *EntryCompletion) void;
    pub const complete = gtk_entry_completion_complete;

    /// Computes the common prefix that is shared by all rows in `completion`
    /// that start with `key`. If no row matches `key`, `NULL` will be returned.
    /// Note that a text column must have been set for this function to work,
    /// see `gtk.EntryCompletion.setTextColumn` for details.
    extern fn gtk_entry_completion_compute_prefix(p_completion: *EntryCompletion, p_key: [*:0]const u8) ?[*:0]u8;
    pub const computePrefix = gtk_entry_completion_compute_prefix;

    /// Deletes the action at `index_` from `completion`’s action list.
    ///
    /// Note that `index_` is a relative position and the position of an
    /// action may have changed since it was inserted.
    extern fn gtk_entry_completion_delete_action(p_completion: *EntryCompletion, p_index_: c_int) void;
    pub const deleteAction = gtk_entry_completion_delete_action;

    /// Get the original text entered by the user that triggered
    /// the completion or `NULL` if there’s no completion ongoing.
    extern fn gtk_entry_completion_get_completion_prefix(p_completion: *EntryCompletion) [*:0]const u8;
    pub const getCompletionPrefix = gtk_entry_completion_get_completion_prefix;

    /// Gets the entry `completion` has been attached to.
    extern fn gtk_entry_completion_get_entry(p_completion: *EntryCompletion) *gtk.Widget;
    pub const getEntry = gtk_entry_completion_get_entry;

    /// Returns whether the common prefix of the possible completions should
    /// be automatically inserted in the entry.
    extern fn gtk_entry_completion_get_inline_completion(p_completion: *EntryCompletion) c_int;
    pub const getInlineCompletion = gtk_entry_completion_get_inline_completion;

    /// Returns `TRUE` if inline-selection mode is turned on.
    extern fn gtk_entry_completion_get_inline_selection(p_completion: *EntryCompletion) c_int;
    pub const getInlineSelection = gtk_entry_completion_get_inline_selection;

    /// Returns the minimum key length as set for `completion`.
    extern fn gtk_entry_completion_get_minimum_key_length(p_completion: *EntryCompletion) c_int;
    pub const getMinimumKeyLength = gtk_entry_completion_get_minimum_key_length;

    /// Returns the model the `gtk.EntryCompletion` is using as data source.
    /// Returns `NULL` if the model is unset.
    extern fn gtk_entry_completion_get_model(p_completion: *EntryCompletion) ?*gtk.TreeModel;
    pub const getModel = gtk_entry_completion_get_model;

    /// Returns whether the completions should be presented in a popup window.
    extern fn gtk_entry_completion_get_popup_completion(p_completion: *EntryCompletion) c_int;
    pub const getPopupCompletion = gtk_entry_completion_get_popup_completion;

    /// Returns whether the  completion popup window will be resized to the
    /// width of the entry.
    extern fn gtk_entry_completion_get_popup_set_width(p_completion: *EntryCompletion) c_int;
    pub const getPopupSetWidth = gtk_entry_completion_get_popup_set_width;

    /// Returns whether the completion popup window will appear even if there is
    /// only a single match.
    extern fn gtk_entry_completion_get_popup_single_match(p_completion: *EntryCompletion) c_int;
    pub const getPopupSingleMatch = gtk_entry_completion_get_popup_single_match;

    /// Returns the column in the model of `completion` to get strings from.
    extern fn gtk_entry_completion_get_text_column(p_completion: *EntryCompletion) c_int;
    pub const getTextColumn = gtk_entry_completion_get_text_column;

    /// Inserts an action in `completion`’s action item list at position `index_`
    /// with markup `markup`.
    extern fn gtk_entry_completion_insert_action_markup(p_completion: *EntryCompletion, p_index_: c_int, p_markup: [*:0]const u8) void;
    pub const insertActionMarkup = gtk_entry_completion_insert_action_markup;

    /// Inserts an action in `completion`’s action item list at position `index_`
    /// with text `text`. If you want the action item to have markup, use
    /// `gtk.EntryCompletion.insertActionMarkup`.
    ///
    /// Note that `index_` is a relative position in the list of actions and
    /// the position of an action can change when deleting a different action.
    extern fn gtk_entry_completion_insert_action_text(p_completion: *EntryCompletion, p_index_: c_int, p_text: [*:0]const u8) void;
    pub const insertActionText = gtk_entry_completion_insert_action_text;

    /// Requests a prefix insertion.
    extern fn gtk_entry_completion_insert_prefix(p_completion: *EntryCompletion) void;
    pub const insertPrefix = gtk_entry_completion_insert_prefix;

    /// Sets whether the common prefix of the possible completions should
    /// be automatically inserted in the entry.
    extern fn gtk_entry_completion_set_inline_completion(p_completion: *EntryCompletion, p_inline_completion: c_int) void;
    pub const setInlineCompletion = gtk_entry_completion_set_inline_completion;

    /// Sets whether it is possible to cycle through the possible completions
    /// inside the entry.
    extern fn gtk_entry_completion_set_inline_selection(p_completion: *EntryCompletion, p_inline_selection: c_int) void;
    pub const setInlineSelection = gtk_entry_completion_set_inline_selection;

    /// Sets the match function for `completion` to be `func`. The match function
    /// is used to determine if a row should or should not be in the completion
    /// list.
    extern fn gtk_entry_completion_set_match_func(p_completion: *EntryCompletion, p_func: gtk.EntryCompletionMatchFunc, p_func_data: ?*anyopaque, p_func_notify: ?glib.DestroyNotify) void;
    pub const setMatchFunc = gtk_entry_completion_set_match_func;

    /// Requires the length of the search key for `completion` to be at least
    /// `length`. This is useful for long lists, where completing using a small
    /// key takes a lot of time and will come up with meaningless results anyway
    /// (ie, a too large dataset).
    extern fn gtk_entry_completion_set_minimum_key_length(p_completion: *EntryCompletion, p_length: c_int) void;
    pub const setMinimumKeyLength = gtk_entry_completion_set_minimum_key_length;

    /// Sets the model for a `gtk.EntryCompletion`. If `completion` already has
    /// a model set, it will remove it before setting the new model.
    /// If model is `NULL`, then it will unset the model.
    extern fn gtk_entry_completion_set_model(p_completion: *EntryCompletion, p_model: ?*gtk.TreeModel) void;
    pub const setModel = gtk_entry_completion_set_model;

    /// Sets whether the completions should be presented in a popup window.
    extern fn gtk_entry_completion_set_popup_completion(p_completion: *EntryCompletion, p_popup_completion: c_int) void;
    pub const setPopupCompletion = gtk_entry_completion_set_popup_completion;

    /// Sets whether the completion popup window will be resized to be the same
    /// width as the entry.
    extern fn gtk_entry_completion_set_popup_set_width(p_completion: *EntryCompletion, p_popup_set_width: c_int) void;
    pub const setPopupSetWidth = gtk_entry_completion_set_popup_set_width;

    /// Sets whether the completion popup window will appear even if there is
    /// only a single match. You may want to set this to `FALSE` if you
    /// are using [inline completion][GtkEntryCompletion--inline-completion].
    extern fn gtk_entry_completion_set_popup_single_match(p_completion: *EntryCompletion, p_popup_single_match: c_int) void;
    pub const setPopupSingleMatch = gtk_entry_completion_set_popup_single_match;

    /// Convenience function for setting up the most used case of this code: a
    /// completion list with just strings. This function will set up `completion`
    /// to have a list displaying all (and just) strings in the completion list,
    /// and to get those strings from `column` in the model of `completion`.
    ///
    /// This functions creates and adds a `gtk.CellRendererText` for the selected
    /// column. If you need to set the text column, but don't want the cell
    /// renderer, use `gobject.Object.set` to set the `gtk.EntryCompletion.properties.text`-column
    /// property directly.
    extern fn gtk_entry_completion_set_text_column(p_completion: *EntryCompletion, p_column: c_int) void;
    pub const setTextColumn = gtk_entry_completion_set_text_column;

    extern fn gtk_entry_completion_get_type() usize;
    pub const getGObjectType = gtk_entry_completion_get_type;

    extern fn g_object_ref(p_self: *gtk.EntryCompletion) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.EntryCompletion) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EntryCompletion, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const EntryIconAccessible = opaque {
    pub const Parent = atk.Object;
    pub const Implements = [_]type{ atk.Action, atk.Component };
    pub const Class = opaque {
        pub const Instance = EntryIconAccessible;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_entry_icon_accessible_get_type() usize;
    pub const getGObjectType = gtk_entry_icon_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.EntryIconAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.EntryIconAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EntryIconAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.EventBox` widget is a subclass of `gtk.Bin` which also has its
/// own window. It is useful since it allows you to catch events for widgets
/// which do not have their own window.
pub const EventBox = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.EventBoxClass;
    f_bin: gtk.Bin,
    f_priv: ?*gtk.EventBoxPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const above_child = struct {
            pub const name = "above-child";

            pub const Type = c_int;
        };

        pub const visible_window = struct {
            pub const name = "visible-window";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.EventBox`.
    extern fn gtk_event_box_new() *gtk.EventBox;
    pub const new = gtk_event_box_new;

    /// Returns whether the event box window is above or below the
    /// windows of its child. See `gtk.EventBox.setAboveChild`
    /// for details.
    extern fn gtk_event_box_get_above_child(p_event_box: *EventBox) c_int;
    pub const getAboveChild = gtk_event_box_get_above_child;

    /// Returns whether the event box has a visible window.
    /// See `gtk.EventBox.setVisibleWindow` for details.
    extern fn gtk_event_box_get_visible_window(p_event_box: *EventBox) c_int;
    pub const getVisibleWindow = gtk_event_box_get_visible_window;

    /// Set whether the event box window is positioned above the windows
    /// of its child, as opposed to below it. If the window is above, all
    /// events inside the event box will go to the event box. If the window
    /// is below, events in windows of child widgets will first got to that
    /// widget, and then to its parents.
    ///
    /// The default is to keep the window below the child.
    extern fn gtk_event_box_set_above_child(p_event_box: *EventBox, p_above_child: c_int) void;
    pub const setAboveChild = gtk_event_box_set_above_child;

    /// Set whether the event box uses a visible or invisible child
    /// window. The default is to use visible windows.
    ///
    /// In an invisible window event box, the window that the
    /// event box creates is a `GDK_INPUT_ONLY` window, which
    /// means that it is invisible and only serves to receive
    /// events.
    ///
    /// A visible window event box creates a visible (`GDK_INPUT_OUTPUT`)
    /// window that acts as the parent window for all the widgets
    /// contained in the event box.
    ///
    /// You should generally make your event box invisible if
    /// you just want to trap events. Creating a visible window
    /// may cause artifacts that are visible to the user, especially
    /// if the user is using a theme with gradients or pixmaps.
    ///
    /// The main reason to create a non input-only event box is if
    /// you want to set the background to a different color or
    /// draw on it.
    ///
    /// There is one unexpected issue for an invisible event box that has its
    /// window below the child. (See `gtk.EventBox.setAboveChild`.)
    /// Since the input-only window is not an ancestor window of any windows
    /// that descendent widgets of the event box create, events on these
    /// windows aren’t propagated up by the windowing system, but only by GTK+.
    /// The practical effect of this is if an event isn’t in the event
    /// mask for the descendant window (see `gtk.Widget.addEvents`),
    /// it won’t be received by the event box.
    ///
    /// This problem doesn’t occur for visible event boxes, because in
    /// that case, the event box window is actually the ancestor of the
    /// descendant windows, not just at the same place on the screen.
    extern fn gtk_event_box_set_visible_window(p_event_box: *EventBox, p_visible_window: c_int) void;
    pub const setVisibleWindow = gtk_event_box_set_visible_window;

    extern fn gtk_event_box_get_type() usize;
    pub const getGObjectType = gtk_event_box_get_type;

    extern fn g_object_ref(p_self: *gtk.EventBox) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.EventBox) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EventBox, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.EventController` is a base, low-level implementation for event
/// controllers. Those react to a series of `GdkEvents`, and possibly trigger
/// actions as a consequence of those.
pub const EventController = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtk.EventControllerClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The propagation phase at which this controller will handle events.
        pub const propagation_phase = struct {
            pub const name = "propagation-phase";

            pub const Type = gtk.PropagationPhase;
        };

        /// The widget receiving the `GdkEvents` that the controller will handle.
        pub const widget = struct {
            pub const name = "widget";

            pub const Type = ?*gtk.Widget;
        };
    };

    pub const signals = struct {};

    /// Gets the propagation phase at which `controller` handles events.
    extern fn gtk_event_controller_get_propagation_phase(p_controller: *EventController) gtk.PropagationPhase;
    pub const getPropagationPhase = gtk_event_controller_get_propagation_phase;

    /// Returns the `gtk.Widget` this controller relates to.
    extern fn gtk_event_controller_get_widget(p_controller: *EventController) *gtk.Widget;
    pub const getWidget = gtk_event_controller_get_widget;

    /// Feeds an events into `controller`, so it can be interpreted
    /// and the controller actions triggered.
    extern fn gtk_event_controller_handle_event(p_controller: *EventController, p_event: *const gdk.Event) c_int;
    pub const handleEvent = gtk_event_controller_handle_event;

    /// Resets the `controller` to a clean state. Every interaction
    /// the controller did through `gtk.EventController.signals.handle`-event
    /// will be dropped at this point.
    extern fn gtk_event_controller_reset(p_controller: *EventController) void;
    pub const reset = gtk_event_controller_reset;

    /// Sets the propagation phase at which a controller handles events.
    ///
    /// If `phase` is `GTK_PHASE_NONE`, no automatic event handling will be
    /// performed, but other additional gesture maintenance will. In that phase,
    /// the events can be managed by calling `gtk.EventController.handleEvent`.
    extern fn gtk_event_controller_set_propagation_phase(p_controller: *EventController, p_phase: gtk.PropagationPhase) void;
    pub const setPropagationPhase = gtk_event_controller_set_propagation_phase;

    extern fn gtk_event_controller_get_type() usize;
    pub const getGObjectType = gtk_event_controller_get_type;

    extern fn g_object_ref(p_self: *gtk.EventController) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.EventController) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EventController, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.EventControllerKey` is an event controller meant for situations
/// where you need access to key events.
///
/// This object was added in 3.24.
pub const EventControllerKey = opaque {
    pub const Parent = gtk.EventController;
    pub const Implements = [_]type{};
    pub const Class = gtk.EventControllerKeyClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        pub const focus_in = struct {
            pub const name = "focus-in";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerKey, p_instance))),
                    gobject.signalLookup("focus-in", EventControllerKey.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const focus_out = struct {
            pub const name = "focus-out";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerKey, p_instance))),
                    gobject.signalLookup("focus-out", EventControllerKey.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const im_update = struct {
            pub const name = "im-update";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerKey, p_instance))),
                    gobject.signalLookup("im-update", EventControllerKey.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted whenever a key is pressed.
        pub const key_pressed = struct {
            pub const name = "key-pressed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_keyval: c_uint, p_keycode: c_uint, p_state: gdk.ModifierType, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerKey, p_instance))),
                    gobject.signalLookup("key-pressed", EventControllerKey.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted whenever a key is released.
        pub const key_released = struct {
            pub const name = "key-released";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_keyval: c_uint, p_keycode: c_uint, p_state: gdk.ModifierType, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerKey, p_instance))),
                    gobject.signalLookup("key-released", EventControllerKey.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const modifiers = struct {
            pub const name = "modifiers";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: gdk.ModifierType, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerKey, p_instance))),
                    gobject.signalLookup("modifiers", EventControllerKey.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    extern fn gtk_event_controller_key_new(p_widget: *gtk.Widget) *gtk.EventControllerKey;
    pub const new = gtk_event_controller_key_new;

    extern fn gtk_event_controller_key_forward(p_controller: *EventControllerKey, p_widget: *gtk.Widget) c_int;
    pub const forward = gtk_event_controller_key_forward;

    extern fn gtk_event_controller_key_get_group(p_controller: *EventControllerKey) c_uint;
    pub const getGroup = gtk_event_controller_key_get_group;

    /// Gets the IM context of a key controller.
    extern fn gtk_event_controller_key_get_im_context(p_controller: *EventControllerKey) *gtk.IMContext;
    pub const getImContext = gtk_event_controller_key_get_im_context;

    extern fn gtk_event_controller_key_set_im_context(p_controller: *EventControllerKey, p_im_context: *gtk.IMContext) void;
    pub const setImContext = gtk_event_controller_key_set_im_context;

    extern fn gtk_event_controller_key_get_type() usize;
    pub const getGObjectType = gtk_event_controller_key_get_type;

    extern fn g_object_ref(p_self: *gtk.EventControllerKey) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.EventControllerKey) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EventControllerKey, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.EventControllerMotion` is an event controller meant for situations
/// where you need to track the position of the pointer.
///
/// This object was added in 3.24.
pub const EventControllerMotion = opaque {
    pub const Parent = gtk.EventController;
    pub const Implements = [_]type{};
    pub const Class = gtk.EventControllerMotionClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// Signals that the pointer has entered the widget.
        pub const enter = struct {
            pub const name = "enter";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_x: f64, p_y: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerMotion, p_instance))),
                    gobject.signalLookup("enter", EventControllerMotion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Signals that pointer has left the widget.
        pub const leave = struct {
            pub const name = "leave";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerMotion, p_instance))),
                    gobject.signalLookup("leave", EventControllerMotion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the pointer moves inside the widget.
        pub const motion = struct {
            pub const name = "motion";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_x: f64, p_y: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerMotion, p_instance))),
                    gobject.signalLookup("motion", EventControllerMotion.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new event controller that will handle motion events
    /// for the given `widget`.
    extern fn gtk_event_controller_motion_new(p_widget: *gtk.Widget) *gtk.EventControllerMotion;
    pub const new = gtk_event_controller_motion_new;

    extern fn gtk_event_controller_motion_get_type() usize;
    pub const getGObjectType = gtk_event_controller_motion_get_type;

    extern fn g_object_ref(p_self: *gtk.EventControllerMotion) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.EventControllerMotion) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EventControllerMotion, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.EventControllerScroll` is an event controller meant to handle
/// scroll events from mice and touchpads. It is capable of handling
/// both discrete and continuous scroll events, abstracting them both
/// on the `gtk.EventControllerScroll.signals.scroll` signal (deltas in the
/// discrete case are multiples of 1).
///
/// In the case of continuous scroll events, `gtk.EventControllerScroll`
/// encloses all `gtk.EventControllerScroll.signals.scroll` events between two
/// `gtk.EventControllerScroll.signals.scroll`-begin and `gtk.EventControllerScroll.signals.scroll`-end
/// signals.
///
/// The behavior of the event controller can be modified by the
/// flags given at creation time, or modified at a later point through
/// `gtk.EventControllerScroll.setFlags` (e.g. because the scrolling
/// conditions of the widget changed).
///
/// The controller can be set up to emit motion for either/both vertical
/// and horizontal scroll events through `GTK_EVENT_CONTROLLER_SCROLL_VERTICAL`,
/// `GTK_EVENT_CONTROLLER_SCROLL_HORIZONTAL` and `GTK_EVENT_CONTROLLER_SCROLL_BOTH`.
/// If any axis is disabled, the respective `gtk.EventControllerScroll.signals.scroll`
/// delta will be 0. Vertical scroll events will be translated to horizontal
/// motion for the devices incapable of horizontal scrolling.
///
/// The event controller can also be forced to emit discrete events on all devices
/// through `GTK_EVENT_CONTROLLER_SCROLL_DISCRETE`. This can be used to implement
/// discrete actions triggered through scroll events (e.g. switching across
/// combobox options).
///
/// The `GTK_EVENT_CONTROLLER_SCROLL_KINETIC` flag toggles the emission of the
/// `gtk.EventControllerScroll.signals.decelerate` signal, emitted at the end of scrolling
/// with two X/Y velocity arguments that are consistent with the motion that
/// was received.
///
/// This object was added in 3.24.
pub const EventControllerScroll = opaque {
    pub const Parent = gtk.EventController;
    pub const Implements = [_]type{};
    pub const Class = gtk.EventControllerScrollClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The flags affecting event controller behavior
        pub const flags = struct {
            pub const name = "flags";

            pub const Type = gtk.EventControllerScrollFlags;
        };
    };

    pub const signals = struct {
        /// Emitted after scroll is finished if the `GTK_EVENT_CONTROLLER_SCROLL_KINETIC`
        /// flag is set. `vel_x` and `vel_y` express the initial velocity that was
        /// imprinted by the scroll events. `vel_x` and `vel_y` are expressed in
        /// pixels/ms.
        pub const decelerate = struct {
            pub const name = "decelerate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_vel_x: f64, p_vel_y: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerScroll, p_instance))),
                    gobject.signalLookup("decelerate", EventControllerScroll.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Signals that the widget should scroll by the
        /// amount specified by `dx` and `dy`.
        pub const scroll = struct {
            pub const name = "scroll";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_dx: f64, p_dy: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerScroll, p_instance))),
                    gobject.signalLookup("scroll", EventControllerScroll.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Signals that a new scrolling operation has begun. It will
        /// only be emitted on devices capable of it.
        pub const scroll_begin = struct {
            pub const name = "scroll-begin";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerScroll, p_instance))),
                    gobject.signalLookup("scroll-begin", EventControllerScroll.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Signals that a new scrolling operation has finished. It will
        /// only be emitted on devices capable of it.
        pub const scroll_end = struct {
            pub const name = "scroll-end";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EventControllerScroll, p_instance))),
                    gobject.signalLookup("scroll-end", EventControllerScroll.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new event controller that will handle scroll events
    /// for the given `widget`.
    extern fn gtk_event_controller_scroll_new(p_widget: *gtk.Widget, p_flags: gtk.EventControllerScrollFlags) *gtk.EventControllerScroll;
    pub const new = gtk_event_controller_scroll_new;

    /// Gets the flags conditioning the scroll controller behavior.
    extern fn gtk_event_controller_scroll_get_flags(p_scroll: *EventControllerScroll) gtk.EventControllerScrollFlags;
    pub const getFlags = gtk_event_controller_scroll_get_flags;

    /// Sets the flags conditioning scroll controller behavior.
    extern fn gtk_event_controller_scroll_set_flags(p_scroll: *EventControllerScroll, p_flags: gtk.EventControllerScrollFlags) void;
    pub const setFlags = gtk_event_controller_scroll_set_flags;

    extern fn gtk_event_controller_scroll_get_type() usize;
    pub const getGObjectType = gtk_event_controller_scroll_get_type;

    extern fn g_object_ref(p_self: *gtk.EventControllerScroll) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.EventControllerScroll) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EventControllerScroll, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gtk.Expander` allows the user to hide or show its child by clicking
/// on an expander triangle similar to the triangles used in a `gtk.TreeView`.
///
/// Normally you use an expander as you would use any other descendant
/// of `gtk.Bin`; you create the child widget and use `gtk.Container.add`
/// to add it to the expander. When the expander is toggled, it will take
/// care of showing and hiding the child automatically.
///
/// # Special Usage
///
/// There are situations in which you may prefer to show and hide the
/// expanded widget yourself, such as when you want to actually create
/// the widget at expansion time. In this case, create a `gtk.Expander`
/// but do not add a child to it. The expander widget has an
/// `gtk.Expander.properties.expanded` property which can be used to monitor
/// its expansion state. You should watch this property with a signal
/// connection as follows:
///
/// ```
/// static void
/// expander_callback (GObject    *object,
///                    GParamSpec *param_spec,
///                    gpointer    user_data)
/// {
///   GtkExpander *expander;
///
///   expander = GTK_EXPANDER (object);
///
///   if (gtk_expander_get_expanded (expander))
///     {
///       // Show or create widgets
///     }
///   else
///     {
///       // Hide or destroy widgets
///     }
/// }
///
/// static void
/// create_expander (void)
/// {
///   GtkWidget *expander = gtk_expander_new_with_mnemonic ("_More Options");
///   g_signal_connect (expander, "notify::expanded",
///                     G_CALLBACK (expander_callback), NULL);
///
///   // ...
/// }
/// ```
///
/// # GtkExpander as GtkBuildable
///
/// The GtkExpander implementation of the GtkBuildable interface supports
/// placing a child in the label position by specifying “label” as the
/// “type” attribute of a `<child>` element. A normal content child can be
/// specified without specifying a `<child>` type attribute.
///
/// An example of a UI definition fragment with GtkExpander:
///
/// ```
/// <object class="GtkExpander">
///   <child type="label">
///     <object class="GtkLabel" id="expander-label"/>
///   </child>
///   <child>
///     <object class="GtkEntry" id="expander-content"/>
///   </child>
/// </object>
/// ```
///
/// # CSS nodes
///
/// ```
/// expander
/// ├── title
/// │   ├── arrow
/// │   ╰── <label widget>
/// ╰── <child>
/// ```
///
/// GtkExpander has three CSS nodes, the main node with the name expander,
/// a subnode with name title and node below it with name arrow. The arrow of an
/// expander that is showing its child gets the :checked pseudoclass added to it.
pub const Expander = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.ExpanderClass;
    f_bin: gtk.Bin,
    f_priv: ?*gtk.ExpanderPrivate,

    pub const virtual_methods = struct {
        /// Keybinding signal is emitted when the user hits the Enter key.
        pub const activate = struct {
            pub fn call(p_class: anytype, p_expander: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Expander.Class, p_class).f_activate.?(gobject.ext.as(Expander, p_expander));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_expander: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Expander.Class, p_class).f_activate = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const expanded = struct {
            pub const name = "expanded";

            pub const Type = c_int;
        };

        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };

        /// Whether the label widget should fill all available horizontal space.
        ///
        /// Note that this property is ignored since 3.20.
        pub const label_fill = struct {
            pub const name = "label-fill";

            pub const Type = c_int;
        };

        pub const label_widget = struct {
            pub const name = "label-widget";

            pub const Type = ?*gtk.Widget;
        };

        /// When this property is `TRUE`, the expander will resize the toplevel
        /// widget containing the expander upon expanding and collapsing.
        pub const resize_toplevel = struct {
            pub const name = "resize-toplevel";

            pub const Type = c_int;
        };

        /// Space to put between the label and the child when the
        /// expander is expanded.
        pub const spacing = struct {
            pub const name = "spacing";

            pub const Type = c_int;
        };

        pub const use_markup = struct {
            pub const name = "use-markup";

            pub const Type = c_int;
        };

        pub const use_underline = struct {
            pub const name = "use-underline";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        pub const activate = struct {
            pub const name = "activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Expander, p_instance))),
                    gobject.signalLookup("activate", Expander.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new expander using `label` as the text of the label.
    extern fn gtk_expander_new(p_label: ?[*:0]const u8) *gtk.Expander;
    pub const new = gtk_expander_new;

    /// Creates a new expander using `label` as the text of the label.
    /// If characters in `label` are preceded by an underscore, they are underlined.
    /// If you need a literal underscore character in a label, use “__” (two
    /// underscores). The first underlined character represents a keyboard
    /// accelerator called a mnemonic.
    /// Pressing Alt and that key activates the button.
    extern fn gtk_expander_new_with_mnemonic(p_label: ?[*:0]const u8) *gtk.Expander;
    pub const newWithMnemonic = gtk_expander_new_with_mnemonic;

    /// Queries a `gtk.Expander` and returns its current state. Returns `TRUE`
    /// if the child widget is revealed.
    ///
    /// See `gtk.Expander.setExpanded`.
    extern fn gtk_expander_get_expanded(p_expander: *Expander) c_int;
    pub const getExpanded = gtk_expander_get_expanded;

    /// Fetches the text from a label widget including any embedded
    /// underlines indicating mnemonics and Pango markup, as set by
    /// `gtk.Expander.setLabel`. If the label text has not been set the
    /// return value will be `NULL`. This will be the case if you create an
    /// empty button with `gtk.Button.new` to use as a container.
    ///
    /// Note that this function behaved differently in versions prior to
    /// 2.14 and used to return the label text stripped of embedded
    /// underlines indicating mnemonics and Pango markup. This problem can
    /// be avoided by fetching the label text directly from the label
    /// widget.
    extern fn gtk_expander_get_label(p_expander: *Expander) ?[*:0]const u8;
    pub const getLabel = gtk_expander_get_label;

    /// Returns whether the label widget will fill all available
    /// horizontal space allocated to `expander`.
    extern fn gtk_expander_get_label_fill(p_expander: *Expander) c_int;
    pub const getLabelFill = gtk_expander_get_label_fill;

    /// Retrieves the label widget for the frame. See
    /// `gtk.Expander.setLabelWidget`.
    extern fn gtk_expander_get_label_widget(p_expander: *Expander) ?*gtk.Widget;
    pub const getLabelWidget = gtk_expander_get_label_widget;

    /// Returns whether the expander will resize the toplevel widget
    /// containing the expander upon resizing and collpasing.
    extern fn gtk_expander_get_resize_toplevel(p_expander: *Expander) c_int;
    pub const getResizeToplevel = gtk_expander_get_resize_toplevel;

    /// Gets the value set by `gtk.Expander.setSpacing`.
    extern fn gtk_expander_get_spacing(p_expander: *Expander) c_int;
    pub const getSpacing = gtk_expander_get_spacing;

    /// Returns whether the label’s text is interpreted as marked up with
    /// the [Pango text markup language][PangoMarkupFormat].
    /// See `gtk.Expander.setUseMarkup`.
    extern fn gtk_expander_get_use_markup(p_expander: *Expander) c_int;
    pub const getUseMarkup = gtk_expander_get_use_markup;

    /// Returns whether an embedded underline in the expander label
    /// indicates a mnemonic. See `gtk.Expander.setUseUnderline`.
    extern fn gtk_expander_get_use_underline(p_expander: *Expander) c_int;
    pub const getUseUnderline = gtk_expander_get_use_underline;

    /// Sets the state of the expander. Set to `TRUE`, if you want
    /// the child widget to be revealed, and `FALSE` if you want the
    /// child widget to be hidden.
    extern fn gtk_expander_set_expanded(p_expander: *Expander, p_expanded: c_int) void;
    pub const setExpanded = gtk_expander_set_expanded;

    /// Sets the text of the label of the expander to `label`.
    ///
    /// This will also clear any previously set labels.
    extern fn gtk_expander_set_label(p_expander: *Expander, p_label: ?[*:0]const u8) void;
    pub const setLabel = gtk_expander_set_label;

    /// Sets whether the label widget should fill all available
    /// horizontal space allocated to `expander`.
    ///
    /// Note that this function has no effect since 3.20.
    extern fn gtk_expander_set_label_fill(p_expander: *Expander, p_label_fill: c_int) void;
    pub const setLabelFill = gtk_expander_set_label_fill;

    /// Set the label widget for the expander. This is the widget
    /// that will appear embedded alongside the expander arrow.
    extern fn gtk_expander_set_label_widget(p_expander: *Expander, p_label_widget: ?*gtk.Widget) void;
    pub const setLabelWidget = gtk_expander_set_label_widget;

    /// Sets whether the expander will resize the toplevel widget
    /// containing the expander upon resizing and collpasing.
    extern fn gtk_expander_set_resize_toplevel(p_expander: *Expander, p_resize_toplevel: c_int) void;
    pub const setResizeToplevel = gtk_expander_set_resize_toplevel;

    /// Sets the spacing field of `expander`, which is the number of
    /// pixels to place between expander and the child.
    extern fn gtk_expander_set_spacing(p_expander: *Expander, p_spacing: c_int) void;
    pub const setSpacing = gtk_expander_set_spacing;

    /// Sets whether the text of the label contains markup in
    /// [Pango’s text markup language][PangoMarkupFormat].
    /// See `gtk.Label.setMarkup`.
    extern fn gtk_expander_set_use_markup(p_expander: *Expander, p_use_markup: c_int) void;
    pub const setUseMarkup = gtk_expander_set_use_markup;

    /// If true, an underline in the text of the expander label indicates
    /// the next character should be used for the mnemonic accelerator key.
    extern fn gtk_expander_set_use_underline(p_expander: *Expander, p_use_underline: c_int) void;
    pub const setUseUnderline = gtk_expander_set_use_underline;

    extern fn gtk_expander_get_type() usize;
    pub const getGObjectType = gtk_expander_get_type;

    extern fn g_object_ref(p_self: *gtk.Expander) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Expander) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Expander, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ExpanderAccessible = extern struct {
    pub const Parent = gtk.ContainerAccessible;
    pub const Implements = [_]type{ atk.Action, atk.Component };
    pub const Class = gtk.ExpanderAccessibleClass;
    f_parent: gtk.ContainerAccessible,
    f_priv: ?*gtk.ExpanderAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_expander_accessible_get_type() usize;
    pub const getGObjectType = gtk_expander_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.ExpanderAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.ExpanderAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ExpanderAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.FileChooserButton` is a widget that lets the user select a
/// file.  It implements the `gtk.FileChooser` interface.  Visually, it is a
/// file name with a button to bring up a `gtk.FileChooserDialog`.
/// The user can then use that dialog to change the file associated with
/// that button.  This widget does not support setting the
/// `gtk.FileChooser.properties.select`-multiple property to `TRUE`.
///
/// ## Create a button to let the user select a file in /etc
///
/// ```
/// {
///   GtkWidget *button;
///
///   button = gtk_file_chooser_button_new (_("Select a file"),
///                                         GTK_FILE_CHOOSER_ACTION_OPEN);
///   gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (button),
///                                        "/etc");
/// }
/// ```
///
/// The `gtk.FileChooserButton` supports the `GtkFileChooserActions`
/// `GTK_FILE_CHOOSER_ACTION_OPEN` and `GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER`.
///
/// > The `gtk.FileChooserButton` will ellipsize the label, and will thus
/// > request little horizontal space.  To give the button more space,
/// > you should call `gtk.Widget.getPreferredSize`,
/// > `gtk.FileChooserButton.setWidthChars`, or pack the button in
/// > such a way that other interface elements give space to the
/// > widget.
///
/// # CSS nodes
///
/// GtkFileChooserButton has a CSS node with name “filechooserbutton”, containing
/// a subnode for the internal button with name “button” and style class “.file”.
pub const FileChooserButton = extern struct {
    pub const Parent = gtk.Box;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.FileChooser, gtk.Orientable };
    pub const Class = gtk.FileChooserButtonClass;
    f_parent: gtk.Box,
    f_priv: ?*gtk.FileChooserButtonPrivate,

    pub const virtual_methods = struct {
        /// Signal emitted when the user selects a file.
        pub const file_set = struct {
            pub fn call(p_class: anytype, p_fc: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(FileChooserButton.Class, p_class).f_file_set.?(gobject.ext.as(FileChooserButton, p_fc));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_fc: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(FileChooserButton.Class, p_class).f_file_set = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// Instance of the `gtk.FileChooserDialog` associated with the button.
        pub const dialog = struct {
            pub const name = "dialog";

            pub const Type = ?*gtk.FileChooser;
        };

        /// Title to put on the `gtk.FileChooserDialog` associated with the button.
        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };

        /// The width of the entry and label inside the button, in characters.
        pub const width_chars = struct {
            pub const name = "width-chars";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The ::file-set signal is emitted when the user selects a file.
        ///
        /// Note that this signal is only emitted when the user
        /// changes the file.
        pub const file_set = struct {
            pub const name = "file-set";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserButton, p_instance))),
                    gobject.signalLookup("file-set", FileChooserButton.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new file-selecting button widget.
    extern fn gtk_file_chooser_button_new(p_title: [*:0]const u8, p_action: gtk.FileChooserAction) *gtk.FileChooserButton;
    pub const new = gtk_file_chooser_button_new;

    /// Creates a `gtk.FileChooserButton` widget which uses `dialog` as its
    /// file-picking window.
    ///
    /// Note that `dialog` must be a `gtk.Dialog` (or subclass) which
    /// implements the `gtk.FileChooser` interface and must not have
    /// `GTK_DIALOG_DESTROY_WITH_PARENT` set.
    ///
    /// Also note that the dialog needs to have its confirmative button
    /// added with response `GTK_RESPONSE_ACCEPT` or `GTK_RESPONSE_OK` in
    /// order for the button to take over the file selected in the dialog.
    extern fn gtk_file_chooser_button_new_with_dialog(p_dialog: *gtk.Dialog) *gtk.FileChooserButton;
    pub const newWithDialog = gtk_file_chooser_button_new_with_dialog;

    /// Returns whether the button grabs focus when it is clicked with the mouse.
    /// See `gtk.FileChooserButton.setFocusOnClick`.
    extern fn gtk_file_chooser_button_get_focus_on_click(p_button: *FileChooserButton) c_int;
    pub const getFocusOnClick = gtk_file_chooser_button_get_focus_on_click;

    /// Retrieves the title of the browse dialog used by `button`. The returned value
    /// should not be modified or freed.
    extern fn gtk_file_chooser_button_get_title(p_button: *FileChooserButton) [*:0]const u8;
    pub const getTitle = gtk_file_chooser_button_get_title;

    /// Retrieves the width in characters of the `button` widget’s entry and/or label.
    extern fn gtk_file_chooser_button_get_width_chars(p_button: *FileChooserButton) c_int;
    pub const getWidthChars = gtk_file_chooser_button_get_width_chars;

    /// Sets whether the button will grab focus when it is clicked with the mouse.
    /// Making mouse clicks not grab focus is useful in places like toolbars where
    /// you don’t want the keyboard focus removed from the main area of the
    /// application.
    extern fn gtk_file_chooser_button_set_focus_on_click(p_button: *FileChooserButton, p_focus_on_click: c_int) void;
    pub const setFocusOnClick = gtk_file_chooser_button_set_focus_on_click;

    /// Modifies the `title` of the browse dialog used by `button`.
    extern fn gtk_file_chooser_button_set_title(p_button: *FileChooserButton, p_title: [*:0]const u8) void;
    pub const setTitle = gtk_file_chooser_button_set_title;

    /// Sets the width (in characters) that `button` will use to `n_chars`.
    extern fn gtk_file_chooser_button_set_width_chars(p_button: *FileChooserButton, p_n_chars: c_int) void;
    pub const setWidthChars = gtk_file_chooser_button_set_width_chars;

    extern fn gtk_file_chooser_button_get_type() usize;
    pub const getGObjectType = gtk_file_chooser_button_get_type;

    extern fn g_object_ref(p_self: *gtk.FileChooserButton) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FileChooserButton) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileChooserButton, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.FileChooserDialog` is a dialog box suitable for use with
/// “File/Open” or “File/Save as” commands.  This widget works by
/// putting a `gtk.FileChooserWidget` inside a `gtk.Dialog`.  It exposes
/// the `gtk.FileChooser` interface, so you can use all of the
/// `gtk.FileChooser` functions on the file chooser dialog as well as
/// those for `gtk.Dialog`.
///
/// Note that `gtk.FileChooserDialog` does not have any methods of its
/// own.  Instead, you should use the functions that work on a
/// `gtk.FileChooser`.
///
/// If you want to integrate well with the platform you should use the
/// `gtk.FileChooserNative` API, which will use a platform-specific
/// dialog if available and fall back to GtkFileChooserDialog
/// otherwise.
///
/// ## Typical usage
///
/// In the simplest of cases, you can the following code to use
/// `gtk.FileChooserDialog` to select a file for opening:
///
/// ```
/// GtkWidget *dialog;
/// GtkFileChooserAction action = GTK_FILE_CHOOSER_ACTION_OPEN;
/// gint res;
///
/// dialog = gtk_file_chooser_dialog_new ("Open File",
///                                       parent_window,
///                                       action,
///                                       _("_Cancel"),
///                                       GTK_RESPONSE_CANCEL,
///                                       _("_Open"),
///                                       GTK_RESPONSE_ACCEPT,
///                                       NULL);
///
/// res = gtk_dialog_run (GTK_DIALOG (dialog));
/// if (res == GTK_RESPONSE_ACCEPT)
///   {
///     char *filename;
///     GtkFileChooser *chooser = GTK_FILE_CHOOSER (dialog);
///     filename = gtk_file_chooser_get_filename (chooser);
///     open_file (filename);
///     g_free (filename);
///   }
///
/// gtk_widget_destroy (dialog);
/// ```
///
/// To use a dialog for saving, you can use this:
///
/// ```
/// GtkWidget *dialog;
/// GtkFileChooser *chooser;
/// GtkFileChooserAction action = GTK_FILE_CHOOSER_ACTION_SAVE;
/// gint res;
///
/// dialog = gtk_file_chooser_dialog_new ("Save File",
///                                       parent_window,
///                                       action,
///                                       _("_Cancel"),
///                                       GTK_RESPONSE_CANCEL,
///                                       _("_Save"),
///                                       GTK_RESPONSE_ACCEPT,
///                                       NULL);
/// chooser = GTK_FILE_CHOOSER (dialog);
///
/// gtk_file_chooser_set_do_overwrite_confirmation (chooser, TRUE);
///
/// if (user_edited_a_new_document)
///   gtk_file_chooser_set_current_name (chooser,
///                                      _("Untitled document"));
/// else
///   gtk_file_chooser_set_filename (chooser,
///                                  existing_filename);
///
/// res = gtk_dialog_run (GTK_DIALOG (dialog));
/// if (res == GTK_RESPONSE_ACCEPT)
///   {
///     char *filename;
///
///     filename = gtk_file_chooser_get_filename (chooser);
///     save_to_file (filename);
///     g_free (filename);
///   }
///
/// gtk_widget_destroy (dialog);
/// ```
///
/// ## Setting up a file chooser dialog
///
/// There are various cases in which you may need to use a `gtk.FileChooserDialog`:
///
/// - To select a file for opening. Use `GTK_FILE_CHOOSER_ACTION_OPEN`.
///
/// - To save a file for the first time. Use `GTK_FILE_CHOOSER_ACTION_SAVE`,
///   and suggest a name such as “Untitled” with `gtk.FileChooser.setCurrentName`.
///
/// - To save a file under a different name. Use `GTK_FILE_CHOOSER_ACTION_SAVE`,
///   and set the existing filename with `gtk.FileChooser.setFilename`.
///
/// - To choose a folder instead of a file. Use `GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER`.
///
/// Note that old versions of the file chooser’s documentation suggested
/// using `gtk.FileChooser.setCurrentFolder` in various
/// situations, with the intention of letting the application
/// suggest a reasonable default folder.  This is no longer
/// considered to be a good policy, as now the file chooser is
/// able to make good suggestions on its own.  In general, you
/// should only cause the file chooser to show a specific folder
/// when it is appropriate to use `gtk.FileChooser.setFilename`,
/// i.e. when you are doing a Save As command and you already
/// have a file saved somewhere.
///
/// ## Response Codes
///
/// `gtk.FileChooserDialog` inherits from `gtk.Dialog`, so buttons that
/// go in its action area have response codes such as
/// `GTK_RESPONSE_ACCEPT` and `GTK_RESPONSE_CANCEL`.  For example, you
/// could call `gtk.FileChooserDialog.new` as follows:
///
/// ```
/// GtkWidget *dialog;
/// GtkFileChooserAction action = GTK_FILE_CHOOSER_ACTION_OPEN;
///
/// dialog = gtk_file_chooser_dialog_new ("Open File",
///                                       parent_window,
///                                       action,
///                                       _("_Cancel"),
///                                       GTK_RESPONSE_CANCEL,
///                                       _("_Open"),
///                                       GTK_RESPONSE_ACCEPT,
///                                       NULL);
/// ```
///
/// This will create buttons for “Cancel” and “Open” that use stock
/// response identifiers from `gtk.ResponseType`.  For most dialog
/// boxes you can use your own custom response codes rather than the
/// ones in `gtk.ResponseType`, but `gtk.FileChooserDialog` assumes that
/// its “accept”-type action, e.g. an “Open” or “Save” button,
/// will have one of the following response codes:
///
/// - `GTK_RESPONSE_ACCEPT`
/// - `GTK_RESPONSE_OK`
/// - `GTK_RESPONSE_YES`
/// - `GTK_RESPONSE_APPLY`
///
/// This is because `gtk.FileChooserDialog` must intercept responses
/// and switch to folders if appropriate, rather than letting the
/// dialog terminate — the implementation uses these known
/// response codes to know which responses can be blocked if
/// appropriate.
///
/// To summarize, make sure you use a
/// [stock response code][gtkfilechooserdialog-responses]
/// when you use `gtk.FileChooserDialog` to ensure proper operation.
pub const FileChooserDialog = extern struct {
    pub const Parent = gtk.Dialog;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.FileChooser };
    pub const Class = gtk.FileChooserDialogClass;
    f_parent_instance: gtk.Dialog,
    f_priv: ?*gtk.FileChooserDialogPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtk.FileChooserDialog`.  This function is analogous to
    /// `gtk.Dialog.newWithButtons`.
    extern fn gtk_file_chooser_dialog_new(p_title: ?[*:0]const u8, p_parent: ?*gtk.Window, p_action: gtk.FileChooserAction, p_first_button_text: ?[*:0]const u8, ...) *gtk.FileChooserDialog;
    pub const new = gtk_file_chooser_dialog_new;

    extern fn gtk_file_chooser_dialog_get_type() usize;
    pub const getGObjectType = gtk_file_chooser_dialog_get_type;

    extern fn g_object_ref(p_self: *gtk.FileChooserDialog) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FileChooserDialog) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileChooserDialog, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.FileChooserNative` is an abstraction of a dialog box suitable
/// for use with “File/Open” or “File/Save as” commands. By default, this
/// just uses a `gtk.FileChooserDialog` to implement the actual dialog.
/// However, on certain platforms, such as Windows and macOS, the native platform
/// file chooser is used instead. When the application is running in a
/// sandboxed environment without direct filesystem access (such as Flatpak),
/// `gtk.FileChooserNative` may call the proper APIs (portals) to let the user
/// choose a file and make it available to the application.
///
/// While the API of `gtk.FileChooserNative` closely mirrors `gtk.FileChooserDialog`, the main
/// difference is that there is no access to any `gtk.Window` or `gtk.Widget` for the dialog.
/// This is required, as there may not be one in the case of a platform native dialog.
/// Showing, hiding and running the dialog is handled by the `gtk.NativeDialog` functions.
///
/// ## Typical usage
///
/// In the simplest of cases, you can the following code to use
/// `gtk.FileChooserDialog` to select a file for opening:
///
/// ```
/// GtkFileChooserNative *native;
/// GtkFileChooserAction action = GTK_FILE_CHOOSER_ACTION_OPEN;
/// gint res;
///
/// native = gtk_file_chooser_native_new ("Open File",
///                                       parent_window,
///                                       action,
///                                       "_Open",
///                                       "_Cancel");
///
/// res = gtk_native_dialog_run (GTK_NATIVE_DIALOG (native));
/// if (res == GTK_RESPONSE_ACCEPT)
///   {
///     char *filename;
///     GtkFileChooser *chooser = GTK_FILE_CHOOSER (native);
///     filename = gtk_file_chooser_get_filename (chooser);
///     open_file (filename);
///     g_free (filename);
///   }
///
/// g_object_unref (native);
/// ```
///
/// To use a dialog for saving, you can use this:
///
/// ```
/// GtkFileChooserNative *native;
/// GtkFileChooser *chooser;
/// GtkFileChooserAction action = GTK_FILE_CHOOSER_ACTION_SAVE;
/// gint res;
///
/// native = gtk_file_chooser_native_new ("Save File",
///                                       parent_window,
///                                       action,
///                                       "_Save",
///                                       "_Cancel");
/// chooser = GTK_FILE_CHOOSER (native);
///
/// gtk_file_chooser_set_do_overwrite_confirmation (chooser, TRUE);
///
/// if (user_edited_a_new_document)
///   gtk_file_chooser_set_current_name (chooser,
///                                      _("Untitled document"));
/// else
///   gtk_file_chooser_set_filename (chooser,
///                                  existing_filename);
///
/// res = gtk_native_dialog_run (GTK_NATIVE_DIALOG (native));
/// if (res == GTK_RESPONSE_ACCEPT)
///   {
///     char *filename;
///
///     filename = gtk_file_chooser_get_filename (chooser);
///     save_to_file (filename);
///     g_free (filename);
///   }
///
/// g_object_unref (native);
/// ```
///
/// For more information on how to best set up a file dialog, see `gtk.FileChooserDialog`.
///
/// ## Response Codes
///
/// `gtk.FileChooserNative` inherits from `gtk.NativeDialog`, which means it
/// will return `GTK_RESPONSE_ACCEPT` if the user accepted, and
/// `GTK_RESPONSE_CANCEL` if he pressed cancel. It can also return
/// `GTK_RESPONSE_DELETE_EVENT` if the window was unexpectedly closed.
///
/// ## Differences from
///
/// There are a few things in the GtkFileChooser API that are not
/// possible to use with `gtk.FileChooserNative`, as such use would
/// prohibit the use of a native dialog.
///
/// There is no support for the signals that are emitted when the user
/// navigates in the dialog, including:
/// * `gtk.FileChooser.signals.current`-folder-changed
/// * `gtk.FileChooser.signals.selection`-changed
/// * `gtk.FileChooser.signals.file`-activated
/// * `gtk.FileChooser.signals.confirm`-overwrite
///
/// You can also not use the methods that directly control user navigation:
/// * `gtk.FileChooser.unselectFilename`
/// * `gtk.FileChooser.selectAll`
/// * `gtk.FileChooser.unselectAll`
///
/// If you need any of the above you will have to use `gtk.FileChooserDialog` directly.
///
/// No operations that change the the dialog work while the dialog is
/// visible. Set all the properties that are required before showing the dialog.
///
/// ## Win32 details
///
/// On windows the IFileDialog implementation (added in Windows Vista) is
/// used. It supports many of the features that `gtk.FileChooserDialog`
/// does, but there are some things it does not handle:
///
/// * Extra widgets added with `gtk.FileChooser.setExtraWidget`.
///
/// * Use of custom previews by connecting to `gtk.FileChooser.signals.update`-preview.
///
/// * Any `gtk.FileFilter` added using a mimetype or custom filter.
///
/// If any of these features are used the regular `gtk.FileChooserDialog`
/// will be used in place of the native one.
///
/// ## Portal details
///
/// When the org.freedesktop.portal.FileChooser portal is available on the
/// session bus, it is used to bring up an out-of-process file chooser. Depending
/// on the kind of session the application is running in, this may or may not
/// be a GTK+ file chooser. In this situation, the following things are not
/// supported and will be silently ignored:
///
/// * Extra widgets added with `gtk.FileChooser.setExtraWidget`.
///
/// * Use of custom previews by connecting to `gtk.FileChooser.signals.update`-preview.
///
/// * Any `gtk.FileFilter` added with a custom filter.
///
/// ## macOS details
///
/// On macOS the NSSavePanel and NSOpenPanel classes are used to provide native
/// file chooser dialogs. Some features provided by `gtk.FileChooserDialog` are
/// not supported:
///
/// * Extra widgets added with `gtk.FileChooser.setExtraWidget`, unless the
///   widget is an instance of GtkLabel, in which case the label text will be used
///   to set the NSSavePanel message instance property.
///
/// * Use of custom previews by connecting to `gtk.FileChooser.signals.update`-preview.
///
/// * Any `gtk.FileFilter` added with a custom filter.
///
/// * Shortcut folders.
pub const FileChooserNative = opaque {
    pub const Parent = gtk.NativeDialog;
    pub const Implements = [_]type{gtk.FileChooser};
    pub const Class = gtk.FileChooserNativeClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The text used for the label on the accept button in the dialog, or
        /// `NULL` to use the default text.
        pub const accept_label = struct {
            pub const name = "accept-label";

            pub const Type = ?[*:0]u8;
        };

        /// The text used for the label on the cancel button in the dialog, or
        /// `NULL` to use the default text.
        pub const cancel_label = struct {
            pub const name = "cancel-label";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.FileChooserNative`.
    extern fn gtk_file_chooser_native_new(p_title: ?[*:0]const u8, p_parent: ?*gtk.Window, p_action: gtk.FileChooserAction, p_accept_label: ?[*:0]const u8, p_cancel_label: ?[*:0]const u8) *gtk.FileChooserNative;
    pub const new = gtk_file_chooser_native_new;

    /// Retrieves the custom label text for the accept button.
    extern fn gtk_file_chooser_native_get_accept_label(p_self: *FileChooserNative) ?[*:0]const u8;
    pub const getAcceptLabel = gtk_file_chooser_native_get_accept_label;

    /// Retrieves the custom label text for the cancel button.
    extern fn gtk_file_chooser_native_get_cancel_label(p_self: *FileChooserNative) ?[*:0]const u8;
    pub const getCancelLabel = gtk_file_chooser_native_get_cancel_label;

    /// Sets the custom label text for the accept button.
    ///
    /// If characters in `label` are preceded by an underscore, they are underlined.
    /// If you need a literal underscore character in a label, use “__” (two
    /// underscores). The first underlined character represents a keyboard
    /// accelerator called a mnemonic.
    /// Pressing Alt and that key activates the button.
    extern fn gtk_file_chooser_native_set_accept_label(p_self: *FileChooserNative, p_accept_label: ?[*:0]const u8) void;
    pub const setAcceptLabel = gtk_file_chooser_native_set_accept_label;

    /// Sets the custom label text for the cancel button.
    ///
    /// If characters in `label` are preceded by an underscore, they are underlined.
    /// If you need a literal underscore character in a label, use “__” (two
    /// underscores). The first underlined character represents a keyboard
    /// accelerator called a mnemonic.
    /// Pressing Alt and that key activates the button.
    extern fn gtk_file_chooser_native_set_cancel_label(p_self: *FileChooserNative, p_cancel_label: ?[*:0]const u8) void;
    pub const setCancelLabel = gtk_file_chooser_native_set_cancel_label;

    extern fn gtk_file_chooser_native_get_type() usize;
    pub const getGObjectType = gtk_file_chooser_native_get_type;

    extern fn g_object_ref(p_self: *gtk.FileChooserNative) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FileChooserNative) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileChooserNative, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.FileChooserWidget` is a widget for choosing files.
/// It exposes the `gtk.FileChooser` interface, and you should
/// use the methods of this interface to interact with the
/// widget.
///
/// # CSS nodes
///
/// GtkFileChooserWidget has a single CSS node with name filechooser.
pub const FileChooserWidget = extern struct {
    pub const Parent = gtk.Box;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.FileChooser, gtk.Orientable };
    pub const Class = gtk.FileChooserWidgetClass;
    f_parent_instance: gtk.Box,
    f_priv: ?*gtk.FileChooserWidgetPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const search_mode = struct {
            pub const name = "search-mode";

            pub const Type = c_int;
        };

        pub const subtitle = struct {
            pub const name = "subtitle";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// The ::desktop-folder signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to make the file chooser show the user's Desktop
        /// folder in the file list.
        ///
        /// The default binding for this signal is `Alt + D`.
        pub const desktop_folder = struct {
            pub const name = "desktop-folder";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("desktop-folder", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::down-folder signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to make the file chooser go to a child of the current folder
        /// in the file hierarchy. The subfolder that will be used is displayed in the
        /// path bar widget of the file chooser. For example, if the path bar is showing
        /// "/foo/bar/baz", with bar currently displayed, then this will cause the file
        /// chooser to switch to the "baz" subfolder.
        ///
        /// The default binding for this signal is `Alt + Down`.
        pub const down_folder = struct {
            pub const name = "down-folder";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("down-folder", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::home-folder signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to make the file chooser show the user's home
        /// folder in the file list.
        ///
        /// The default binding for this signal is `Alt + Home`.
        pub const home_folder = struct {
            pub const name = "home-folder";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("home-folder", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::location-popup signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to make the file chooser show a "Location" prompt which
        /// the user can use to manually type the name of the file he wishes to select.
        ///
        /// The default bindings for this signal are `Control + L` with a `path` string
        /// of "" (the empty string).  It is also bound to `/` with a `path` string of
        /// "`/`" (a slash):  this lets you type `/` and immediately type a path name.
        /// On Unix systems, this is bound to `~` (tilde) with a `path` string of "~"
        /// itself for access to home directories.
        pub const location_popup = struct {
            pub const name = "location-popup";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_path: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("location-popup", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::location-popup-on-paste signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to make the file chooser show a "Location" prompt when the user
        /// pastes into a `gtk.FileChooserWidget`.
        ///
        /// The default binding for this signal is `Control + V`.
        pub const location_popup_on_paste = struct {
            pub const name = "location-popup-on-paste";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("location-popup-on-paste", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::location-toggle-popup signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to toggle the visibility of a "Location" prompt which the user
        /// can use to manually type the name of the file he wishes to select.
        ///
        /// The default binding for this signal is `Control + L`.
        pub const location_toggle_popup = struct {
            pub const name = "location-toggle-popup";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("location-toggle-popup", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::places-shortcut signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to move the focus to the places sidebar.
        ///
        /// The default binding for this signal is `Alt + P`.
        pub const places_shortcut = struct {
            pub const name = "places-shortcut";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("places-shortcut", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::quick-bookmark signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to make the file chooser switch to the bookmark specified
        /// in the `bookmark_index` parameter. For example, if you have three bookmarks,
        /// you can pass 0, 1, 2 to this signal to switch to each of them, respectively.
        ///
        /// The default binding for this signal is `Alt + 1`, `Alt + 2`,
        /// etc. until `Alt + 0`.  Note that in the default binding, that
        /// `Alt + 1` is actually defined to switch to the bookmark at index
        /// 0, and so on successively; `Alt + 0` is defined to switch to the
        /// bookmark at index 10.
        pub const quick_bookmark = struct {
            pub const name = "quick-bookmark";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_bookmark_index: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("quick-bookmark", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::recent-shortcut signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to make the file chooser show the Recent location.
        ///
        /// The default binding for this signal is `Alt + R`.
        pub const recent_shortcut = struct {
            pub const name = "recent-shortcut";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("recent-shortcut", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::search-shortcut signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to make the file chooser show the search entry.
        ///
        /// The default binding for this signal is `Alt + S`.
        pub const search_shortcut = struct {
            pub const name = "search-shortcut";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("search-shortcut", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::show-hidden signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to make the file chooser display hidden files.
        ///
        /// The default binding for this signal is `Control + H`.
        pub const show_hidden = struct {
            pub const name = "show-hidden";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("show-hidden", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::up-folder signal is a [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user asks for it.
        ///
        /// This is used to make the file chooser go to the parent of the current folder
        /// in the file hierarchy.
        ///
        /// The default binding for this signal is `Alt + Up`.
        pub const up_folder = struct {
            pub const name = "up-folder";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileChooserWidget, p_instance))),
                    gobject.signalLookup("up-folder", FileChooserWidget.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.FileChooserWidget`. This is a file chooser widget that can
    /// be embedded in custom windows, and it is the same widget that is used by
    /// `gtk.FileChooserDialog`.
    extern fn gtk_file_chooser_widget_new(p_action: gtk.FileChooserAction) *gtk.FileChooserWidget;
    pub const new = gtk_file_chooser_widget_new;

    extern fn gtk_file_chooser_widget_get_type() usize;
    pub const getGObjectType = gtk_file_chooser_widget_get_type;

    extern fn g_object_ref(p_self: *gtk.FileChooserWidget) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FileChooserWidget) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileChooserWidget, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FileChooserWidgetAccessible = extern struct {
    pub const Parent = gtk.ContainerAccessible;
    pub const Implements = [_]type{ atk.Action, atk.Component };
    pub const Class = gtk.FileChooserWidgetAccessibleClass;
    f_parent: gtk.ContainerAccessible,
    f_priv: ?*gtk.FileChooserWidgetAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_file_chooser_widget_accessible_get_type() usize;
    pub const getGObjectType = gtk_file_chooser_widget_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.FileChooserWidgetAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FileChooserWidgetAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileChooserWidgetAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A GtkFileFilter can be used to restrict the files being shown in a
/// `gtk.FileChooser`. Files can be filtered based on their name (with
/// `gtk.FileFilter.addPattern`), on their mime type (with
/// `gtk.FileFilter.addMimeType`), or by a custom filter function
/// (with `gtk.FileFilter.addCustom`).
///
/// Filtering by mime types handles aliasing and subclassing of mime
/// types; e.g. a filter for text/plain also matches a file with mime
/// type application/rtf, since application/rtf is a subclass of
/// text/plain. Note that `gtk.FileFilter` allows wildcards for the
/// subtype of a mime type, so you can e.g. filter for image/\*.
///
/// Normally, filters are used by adding them to a `gtk.FileChooser`,
/// see `gtk.FileChooser.addFilter`, but it is also possible
/// to manually use a filter on a file with `gtk.FileFilter.filter`.
///
/// # GtkFileFilter as GtkBuildable
///
/// The GtkFileFilter implementation of the GtkBuildable interface
/// supports adding rules using the `<mime-types>`, `<patterns>` and
/// `<applications>` elements and listing the rules within. Specifying
/// a `<mime-type>` or `<pattern>` has the same effect as as calling
/// `gtk.FileFilter.addMimeType` or `gtk.FileFilter.addPattern`.
///
/// An example of a UI definition fragment specifying GtkFileFilter
/// rules:
///
/// ```
/// <object class="GtkFileFilter">
///   <mime-types>
///     <mime-type>text/plain</mime-type>
///     <mime-type>image/ *</mime-type>
///   </mime-types>
///   <patterns>
///     <pattern>*.txt</pattern>
///     <pattern>*.png</pattern>
///   </patterns>
/// </object>
/// ```
pub const FileFilter = opaque {
    pub const Parent = gobject.InitiallyUnowned;
    pub const Implements = [_]type{gtk.Buildable};
    pub const Class = opaque {
        pub const Instance = FileFilter;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtk.FileFilter` with no rules added to it.
    /// Such a filter doesn’t accept any files, so is not
    /// particularly useful until you add rules with
    /// `gtk.FileFilter.addMimeType`, `gtk.FileFilter.addPattern`,
    /// or `gtk.FileFilter.addCustom`. To create a filter
    /// that accepts any file, use:
    /// ```
    /// GtkFileFilter *filter = gtk_file_filter_new ();
    /// gtk_file_filter_add_pattern (filter, "*");
    /// ```
    extern fn gtk_file_filter_new() *gtk.FileFilter;
    pub const new = gtk_file_filter_new;

    /// Deserialize a file filter from an a{sv} variant in
    /// the format produced by `gtk.FileFilter.toGvariant`.
    extern fn gtk_file_filter_new_from_gvariant(p_variant: *glib.Variant) *gtk.FileFilter;
    pub const newFromGvariant = gtk_file_filter_new_from_gvariant;

    /// Adds rule to a filter that allows files based on a custom callback
    /// function. The bitfield `needed` which is passed in provides information
    /// about what sorts of information that the filter function needs;
    /// this allows GTK+ to avoid retrieving expensive information when
    /// it isn’t needed by the filter.
    extern fn gtk_file_filter_add_custom(p_filter: *FileFilter, p_needed: gtk.FileFilterFlags, p_func: gtk.FileFilterFunc, p_data: ?*anyopaque, p_notify: ?glib.DestroyNotify) void;
    pub const addCustom = gtk_file_filter_add_custom;

    /// Adds a rule allowing a given mime type to `filter`.
    extern fn gtk_file_filter_add_mime_type(p_filter: *FileFilter, p_mime_type: [*:0]const u8) void;
    pub const addMimeType = gtk_file_filter_add_mime_type;

    /// Adds a rule allowing a shell style glob to a filter.
    extern fn gtk_file_filter_add_pattern(p_filter: *FileFilter, p_pattern: [*:0]const u8) void;
    pub const addPattern = gtk_file_filter_add_pattern;

    /// Adds a rule allowing image files in the formats supported
    /// by GdkPixbuf.
    extern fn gtk_file_filter_add_pixbuf_formats(p_filter: *FileFilter) void;
    pub const addPixbufFormats = gtk_file_filter_add_pixbuf_formats;

    /// Tests whether a file should be displayed according to `filter`.
    /// The `gtk.FileFilterInfo` `filter_info` should include
    /// the fields returned from `gtk.FileFilter.getNeeded`.
    ///
    /// This function will not typically be used by applications; it
    /// is intended principally for use in the implementation of
    /// `gtk.FileChooser`.
    extern fn gtk_file_filter_filter(p_filter: *FileFilter, p_filter_info: *const gtk.FileFilterInfo) c_int;
    pub const filter = gtk_file_filter_filter;

    /// Gets the human-readable name for the filter. See `gtk.FileFilter.setName`.
    extern fn gtk_file_filter_get_name(p_filter: *FileFilter) ?[*:0]const u8;
    pub const getName = gtk_file_filter_get_name;

    /// Gets the fields that need to be filled in for the `gtk.FileFilterInfo`
    /// passed to `gtk.FileFilter.filter`
    ///
    /// This function will not typically be used by applications; it
    /// is intended principally for use in the implementation of
    /// `gtk.FileChooser`.
    extern fn gtk_file_filter_get_needed(p_filter: *FileFilter) gtk.FileFilterFlags;
    pub const getNeeded = gtk_file_filter_get_needed;

    /// Sets the human-readable name of the filter; this is the string
    /// that will be displayed in the file selector user interface if
    /// there is a selectable list of filters.
    extern fn gtk_file_filter_set_name(p_filter: *FileFilter, p_name: ?[*:0]const u8) void;
    pub const setName = gtk_file_filter_set_name;

    /// Serialize a file filter to an a{sv} variant.
    extern fn gtk_file_filter_to_gvariant(p_filter: *FileFilter) *glib.Variant;
    pub const toGvariant = gtk_file_filter_to_gvariant;

    extern fn gtk_file_filter_get_type() usize;
    pub const getGObjectType = gtk_file_filter_get_type;

    extern fn g_object_ref(p_self: *gtk.FileFilter) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FileFilter) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileFilter, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.Fixed` widget is a container which can place child widgets
/// at fixed positions and with fixed sizes, given in pixels. `gtk.Fixed`
/// performs no automatic layout management.
///
/// For most applications, you should not use this container! It keeps
/// you from having to learn about the other GTK+ containers, but it
/// results in broken applications.  With `gtk.Fixed`, the following
/// things will result in truncated text, overlapping widgets, and
/// other display bugs:
///
/// - Themes, which may change widget sizes.
///
/// - Fonts other than the one you used to write the app will of course
///   change the size of widgets containing text; keep in mind that
///   users may use a larger font because of difficulty reading the
///   default, or they may be using a different OS that provides different fonts.
///
/// - Translation of text into other languages changes its size. Also,
///   display of non-English text will use a different font in many
///   cases.
///
/// In addition, `gtk.Fixed` does not pay attention to text direction and thus may
/// produce unwanted results if your app is run under right-to-left languages
/// such as Hebrew or Arabic. That is: normally GTK+ will order containers
/// appropriately for the text direction, e.g. to put labels to the right of the
/// thing they label when using an RTL language, but it can’t do that with
/// `gtk.Fixed`. So if you need to reorder widgets depending on the text direction,
/// you would need to manually detect it and adjust child positions accordingly.
///
/// Finally, fixed positioning makes it kind of annoying to add/remove
/// GUI elements, since you have to reposition all the other
/// elements. This is a long-term maintenance problem for your
/// application.
///
/// If you know none of these things are an issue for your application,
/// and prefer the simplicity of `gtk.Fixed`, by all means use the
/// widget. But you should be aware of the tradeoffs.
///
/// See also `gtk.Layout`, which shares the ability to perform fixed positioning
/// of child widgets and additionally adds custom drawing and scrollability.
pub const Fixed = extern struct {
    pub const Parent = gtk.Container;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.FixedClass;
    f_container: gtk.Container,
    f_priv: ?*gtk.FixedPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtk.Fixed`.
    extern fn gtk_fixed_new() *gtk.Fixed;
    pub const new = gtk_fixed_new;

    /// Moves a child of a `gtk.Fixed` container to the given position.
    extern fn gtk_fixed_move(p_fixed: *Fixed, p_widget: *gtk.Widget, p_x: c_int, p_y: c_int) void;
    pub const move = gtk_fixed_move;

    /// Adds a widget to a `gtk.Fixed` container at the given position.
    extern fn gtk_fixed_put(p_fixed: *Fixed, p_widget: *gtk.Widget, p_x: c_int, p_y: c_int) void;
    pub const put = gtk_fixed_put;

    extern fn gtk_fixed_get_type() usize;
    pub const getGObjectType = gtk_fixed_get_type;

    extern fn g_object_ref(p_self: *gtk.Fixed) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Fixed) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Fixed, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A GtkFlowBox positions child widgets in sequence according to its
/// orientation.
///
/// For instance, with the horizontal orientation, the widgets will be
/// arranged from left to right, starting a new row under the previous
/// row when necessary. Reducing the width in this case will require more
/// rows, so a larger height will be requested.
///
/// Likewise, with the vertical orientation, the widgets will be arranged
/// from top to bottom, starting a new column to the right when necessary.
/// Reducing the height will require more columns, so a larger width will
/// be requested.
///
/// The size request of a GtkFlowBox alone may not be what you expect; if you
/// need to be able to shrink it along both axes and dynamically reflow its
/// children, you may have to wrap it in a `gtk.ScrolledWindow` to enable that.
///
/// The children of a GtkFlowBox can be dynamically sorted and filtered.
///
/// Although a GtkFlowBox must have only `gtk.FlowBoxChild` children,
/// you can add any kind of widget to it via `gtk.Container.add`, and
/// a GtkFlowBoxChild widget will automatically be inserted between
/// the box and the widget.
///
/// Also see `gtk.ListBox`.
///
/// GtkFlowBox was added in GTK+ 3.12.
///
/// # CSS nodes
///
/// ```
/// flowbox
/// ├── flowboxchild
/// │   ╰── <child>
/// ├── flowboxchild
/// │   ╰── <child>
/// ┊
/// ╰── [rubberband]
/// ```
///
/// GtkFlowBox uses a single CSS node with name flowbox. GtkFlowBoxChild
/// uses a single CSS node with name flowboxchild.
/// For rubberband selection, a subnode with name rubberband is used.
pub const FlowBox = extern struct {
    pub const Parent = gtk.Container;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.FlowBoxClass;
    f_container: gtk.Container,

    pub const virtual_methods = struct {
        pub const activate_cursor_child = struct {
            pub fn call(p_class: anytype, p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(FlowBox.Class, p_class).f_activate_cursor_child.?(gobject.ext.as(FlowBox, p_box));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(FlowBox.Class, p_class).f_activate_cursor_child = @ptrCast(p_implementation);
            }
        };

        pub const child_activated = struct {
            pub fn call(p_class: anytype, p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.FlowBoxChild) void {
                return gobject.ext.as(FlowBox.Class, p_class).f_child_activated.?(gobject.ext.as(FlowBox, p_box), p_child);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.FlowBoxChild) callconv(.C) void) void {
                gobject.ext.as(FlowBox.Class, p_class).f_child_activated = @ptrCast(p_implementation);
            }
        };

        pub const move_cursor = struct {
            pub fn call(p_class: anytype, p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_step: gtk.MovementStep, p_count: c_int) c_int {
                return gobject.ext.as(FlowBox.Class, p_class).f_move_cursor.?(gobject.ext.as(FlowBox, p_box), p_step, p_count);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_step: gtk.MovementStep, p_count: c_int) callconv(.C) c_int) void {
                gobject.ext.as(FlowBox.Class, p_class).f_move_cursor = @ptrCast(p_implementation);
            }
        };

        /// Select all children of `box`, if the selection
        /// mode allows it.
        pub const select_all = struct {
            pub fn call(p_class: anytype, p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(FlowBox.Class, p_class).f_select_all.?(gobject.ext.as(FlowBox, p_box));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(FlowBox.Class, p_class).f_select_all = @ptrCast(p_implementation);
            }
        };

        pub const selected_children_changed = struct {
            pub fn call(p_class: anytype, p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(FlowBox.Class, p_class).f_selected_children_changed.?(gobject.ext.as(FlowBox, p_box));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(FlowBox.Class, p_class).f_selected_children_changed = @ptrCast(p_implementation);
            }
        };

        pub const toggle_cursor_child = struct {
            pub fn call(p_class: anytype, p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(FlowBox.Class, p_class).f_toggle_cursor_child.?(gobject.ext.as(FlowBox, p_box));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(FlowBox.Class, p_class).f_toggle_cursor_child = @ptrCast(p_implementation);
            }
        };

        /// Unselect all children of `box`, if the selection
        /// mode allows it.
        pub const unselect_all = struct {
            pub fn call(p_class: anytype, p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(FlowBox.Class, p_class).f_unselect_all.?(gobject.ext.as(FlowBox, p_box));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(FlowBox.Class, p_class).f_unselect_all = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// Determines whether children can be activated with a single
        /// click, or require a double-click.
        pub const activate_on_single_click = struct {
            pub const name = "activate-on-single-click";

            pub const Type = c_int;
        };

        /// The amount of horizontal space between two children.
        pub const column_spacing = struct {
            pub const name = "column-spacing";

            pub const Type = c_uint;
        };

        /// Determines whether all children should be allocated the
        /// same size.
        pub const homogeneous = struct {
            pub const name = "homogeneous";

            pub const Type = c_int;
        };

        /// The maximum amount of children to request space for consecutively
        /// in the given orientation.
        pub const max_children_per_line = struct {
            pub const name = "max-children-per-line";

            pub const Type = c_uint;
        };

        /// The minimum number of children to allocate consecutively
        /// in the given orientation.
        ///
        /// Setting the minimum children per line ensures
        /// that a reasonably small height will be requested
        /// for the overall minimum width of the box.
        pub const min_children_per_line = struct {
            pub const name = "min-children-per-line";

            pub const Type = c_uint;
        };

        /// The amount of vertical space between two children.
        pub const row_spacing = struct {
            pub const name = "row-spacing";

            pub const Type = c_uint;
        };

        /// The selection mode used by the flow  box.
        pub const selection_mode = struct {
            pub const name = "selection-mode";

            pub const Type = gtk.SelectionMode;
        };
    };

    pub const signals = struct {
        /// The ::activate-cursor-child signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user activates the `box`.
        pub const activate_cursor_child = struct {
            pub const name = "activate-cursor-child";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FlowBox, p_instance))),
                    gobject.signalLookup("activate-cursor-child", FlowBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::child-activated signal is emitted when a child has been
        /// activated by the user.
        pub const child_activated = struct {
            pub const name = "child-activated";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_child: *gtk.FlowBoxChild, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FlowBox, p_instance))),
                    gobject.signalLookup("child-activated", FlowBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::move-cursor signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted when the user initiates a cursor movement.
        ///
        /// Applications should not connect to it, but may emit it with
        /// `gobject.signalEmitByName` if they need to control the cursor
        /// programmatically.
        ///
        /// The default bindings for this signal come in two variants,
        /// the variant with the Shift modifier extends the selection,
        /// the variant without the Shift modifer does not.
        /// There are too many key combinations to list them all here.
        /// - Arrow keys move by individual children
        /// - Home/End keys move to the ends of the box
        /// - PageUp/PageDown keys move vertically by pages
        pub const move_cursor = struct {
            pub const name = "move-cursor";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_step: gtk.MovementStep, p_count: c_int, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FlowBox, p_instance))),
                    gobject.signalLookup("move-cursor", FlowBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::select-all signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted to select all children of the box, if
        /// the selection mode permits it.
        ///
        /// The default bindings for this signal is Ctrl-a.
        pub const select_all = struct {
            pub const name = "select-all";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FlowBox, p_instance))),
                    gobject.signalLookup("select-all", FlowBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::selected-children-changed signal is emitted when the
        /// set of selected children changes.
        ///
        /// Use `gtk.FlowBox.selectedForeach` or
        /// `gtk.FlowBox.getSelectedChildren` to obtain the
        /// selected children.
        pub const selected_children_changed = struct {
            pub const name = "selected-children-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FlowBox, p_instance))),
                    gobject.signalLookup("selected-children-changed", FlowBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::toggle-cursor-child signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which toggles the selection of the child that has the focus.
        ///
        /// The default binding for this signal is Ctrl-Space.
        pub const toggle_cursor_child = struct {
            pub const name = "toggle-cursor-child";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FlowBox, p_instance))),
                    gobject.signalLookup("toggle-cursor-child", FlowBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::unselect-all signal is a
        /// [keybinding signal][GtkBindingSignal]
        /// which gets emitted to unselect all children of the box, if
        /// the selection mode permits it.
        ///
        /// The default bindings for this signal is Ctrl-Shift-a.
        pub const unselect_all = struct {
            pub const name = "unselect-all";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FlowBox, p_instance))),
                    gobject.signalLookup("unselect-all", FlowBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a GtkFlowBox.
    extern fn gtk_flow_box_new() *gtk.FlowBox;
    pub const new = gtk_flow_box_new;

    /// Binds `model` to `box`.
    ///
    /// If `box` was already bound to a model, that previous binding is
    /// destroyed.
    ///
    /// The contents of `box` are cleared and then filled with widgets that
    /// represent items from `model`. `box` is updated whenever `model` changes.
    /// If `model` is `NULL`, `box` is left empty.
    ///
    /// It is undefined to add or remove widgets directly (for example, with
    /// `gtk.FlowBox.insert` or `gtk.Container.add`) while `box` is bound to a
    /// model.
    ///
    /// Note that using a model is incompatible with the filtering and sorting
    /// functionality in GtkFlowBox. When using a model, filtering and sorting
    /// should be implemented by the model.
    extern fn gtk_flow_box_bind_model(p_box: *FlowBox, p_model: ?*gio.ListModel, p_create_widget_func: gtk.FlowBoxCreateWidgetFunc, p_user_data: ?*anyopaque, p_user_data_free_func: ?glib.DestroyNotify) void;
    pub const bindModel = gtk_flow_box_bind_model;

    /// Returns whether children activate on single clicks.
    extern fn gtk_flow_box_get_activate_on_single_click(p_box: *FlowBox) c_int;
    pub const getActivateOnSingleClick = gtk_flow_box_get_activate_on_single_click;

    /// Gets the nth child in the `box`.
    extern fn gtk_flow_box_get_child_at_index(p_box: *FlowBox, p_idx: c_int) ?*gtk.FlowBoxChild;
    pub const getChildAtIndex = gtk_flow_box_get_child_at_index;

    /// Gets the child in the (`x`, `y`) position.
    extern fn gtk_flow_box_get_child_at_pos(p_box: *FlowBox, p_x: c_int, p_y: c_int) ?*gtk.FlowBoxChild;
    pub const getChildAtPos = gtk_flow_box_get_child_at_pos;

    /// Gets the horizontal spacing.
    extern fn gtk_flow_box_get_column_spacing(p_box: *FlowBox) c_uint;
    pub const getColumnSpacing = gtk_flow_box_get_column_spacing;

    /// Returns whether the box is homogeneous (all children are the
    /// same size). See `gtk.Box.setHomogeneous`.
    extern fn gtk_flow_box_get_homogeneous(p_box: *FlowBox) c_int;
    pub const getHomogeneous = gtk_flow_box_get_homogeneous;

    /// Gets the maximum number of children per line.
    extern fn gtk_flow_box_get_max_children_per_line(p_box: *FlowBox) c_uint;
    pub const getMaxChildrenPerLine = gtk_flow_box_get_max_children_per_line;

    /// Gets the minimum number of children per line.
    extern fn gtk_flow_box_get_min_children_per_line(p_box: *FlowBox) c_uint;
    pub const getMinChildrenPerLine = gtk_flow_box_get_min_children_per_line;

    /// Gets the vertical spacing.
    extern fn gtk_flow_box_get_row_spacing(p_box: *FlowBox) c_uint;
    pub const getRowSpacing = gtk_flow_box_get_row_spacing;

    /// Creates a list of all selected children.
    extern fn gtk_flow_box_get_selected_children(p_box: *FlowBox) *glib.List;
    pub const getSelectedChildren = gtk_flow_box_get_selected_children;

    /// Gets the selection mode of `box`.
    extern fn gtk_flow_box_get_selection_mode(p_box: *FlowBox) gtk.SelectionMode;
    pub const getSelectionMode = gtk_flow_box_get_selection_mode;

    /// Inserts the `widget` into `box` at `position`.
    ///
    /// If a sort function is set, the widget will actually be inserted
    /// at the calculated position and this function has the same effect
    /// as `gtk.Container.add`.
    ///
    /// If `position` is -1, or larger than the total number of children
    /// in the `box`, then the `widget` will be appended to the end.
    extern fn gtk_flow_box_insert(p_box: *FlowBox, p_widget: *gtk.Widget, p_position: c_int) void;
    pub const insert = gtk_flow_box_insert;

    /// Updates the filtering for all children.
    ///
    /// Call this function when the result of the filter
    /// function on the `box` is changed due ot an external
    /// factor. For instance, this would be used if the
    /// filter function just looked for a specific search
    /// term, and the entry with the string has changed.
    extern fn gtk_flow_box_invalidate_filter(p_box: *FlowBox) void;
    pub const invalidateFilter = gtk_flow_box_invalidate_filter;

    /// Updates the sorting for all children.
    ///
    /// Call this when the result of the sort function on
    /// `box` is changed due to an external factor.
    extern fn gtk_flow_box_invalidate_sort(p_box: *FlowBox) void;
    pub const invalidateSort = gtk_flow_box_invalidate_sort;

    /// Select all children of `box`, if the selection
    /// mode allows it.
    extern fn gtk_flow_box_select_all(p_box: *FlowBox) void;
    pub const selectAll = gtk_flow_box_select_all;

    /// Selects a single child of `box`, if the selection
    /// mode allows it.
    extern fn gtk_flow_box_select_child(p_box: *FlowBox, p_child: *gtk.FlowBoxChild) void;
    pub const selectChild = gtk_flow_box_select_child;

    /// Calls a function for each selected child.
    ///
    /// Note that the selection cannot be modified from within
    /// this function.
    extern fn gtk_flow_box_selected_foreach(p_box: *FlowBox, p_func: gtk.FlowBoxForeachFunc, p_data: ?*anyopaque) void;
    pub const selectedForeach = gtk_flow_box_selected_foreach;

    /// If `single` is `TRUE`, children will be activated when you click
    /// on them, otherwise you need to double-click.
    extern fn gtk_flow_box_set_activate_on_single_click(p_box: *FlowBox, p_single: c_int) void;
    pub const setActivateOnSingleClick = gtk_flow_box_set_activate_on_single_click;

    /// Sets the horizontal space to add between children.
    /// See the `gtk.FlowBox.properties.column`-spacing property.
    extern fn gtk_flow_box_set_column_spacing(p_box: *FlowBox, p_spacing: c_uint) void;
    pub const setColumnSpacing = gtk_flow_box_set_column_spacing;

    /// By setting a filter function on the `box` one can decide dynamically
    /// which of the children to show. For instance, to implement a search
    /// function that only shows the children matching the search terms.
    ///
    /// The `filter_func` will be called for each child after the call, and
    /// it will continue to be called each time a child changes (via
    /// `gtk.FlowBoxChild.changed`) or when `gtk.FlowBox.invalidateFilter`
    /// is called.
    ///
    /// Note that using a filter function is incompatible with using a model
    /// (see `gtk.FlowBox.bindModel`).
    extern fn gtk_flow_box_set_filter_func(p_box: *FlowBox, p_filter_func: ?gtk.FlowBoxFilterFunc, p_user_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) void;
    pub const setFilterFunc = gtk_flow_box_set_filter_func;

    /// Hooks up an adjustment to focus handling in `box`.
    /// The adjustment is also used for autoscrolling during
    /// rubberband selection. See `gtk.ScrolledWindow.getHadjustment`
    /// for a typical way of obtaining the adjustment, and
    /// `gtk.FlowBox.setVadjustment`for setting the vertical
    /// adjustment.
    ///
    /// The adjustments have to be in pixel units and in the same
    /// coordinate system as the allocation for immediate children
    /// of the box.
    extern fn gtk_flow_box_set_hadjustment(p_box: *FlowBox, p_adjustment: *gtk.Adjustment) void;
    pub const setHadjustment = gtk_flow_box_set_hadjustment;

    /// Sets the `gtk.FlowBox.properties.homogeneous` property of `box`, controlling
    /// whether or not all children of `box` are given equal space
    /// in the box.
    extern fn gtk_flow_box_set_homogeneous(p_box: *FlowBox, p_homogeneous: c_int) void;
    pub const setHomogeneous = gtk_flow_box_set_homogeneous;

    /// Sets the maximum number of children to request and
    /// allocate space for in `box`’s orientation.
    ///
    /// Setting the maximum number of children per line
    /// limits the overall natural size request to be no more
    /// than `n_children` children long in the given orientation.
    extern fn gtk_flow_box_set_max_children_per_line(p_box: *FlowBox, p_n_children: c_uint) void;
    pub const setMaxChildrenPerLine = gtk_flow_box_set_max_children_per_line;

    /// Sets the minimum number of children to line up
    /// in `box`’s orientation before flowing.
    extern fn gtk_flow_box_set_min_children_per_line(p_box: *FlowBox, p_n_children: c_uint) void;
    pub const setMinChildrenPerLine = gtk_flow_box_set_min_children_per_line;

    /// Sets the vertical space to add between children.
    /// See the `gtk.FlowBox.properties.row`-spacing property.
    extern fn gtk_flow_box_set_row_spacing(p_box: *FlowBox, p_spacing: c_uint) void;
    pub const setRowSpacing = gtk_flow_box_set_row_spacing;

    /// Sets how selection works in `box`.
    /// See `gtk.SelectionMode` for details.
    extern fn gtk_flow_box_set_selection_mode(p_box: *FlowBox, p_mode: gtk.SelectionMode) void;
    pub const setSelectionMode = gtk_flow_box_set_selection_mode;

    /// By setting a sort function on the `box`, one can dynamically
    /// reorder the children of the box, based on the contents of
    /// the children.
    ///
    /// The `sort_func` will be called for each child after the call,
    /// and will continue to be called each time a child changes (via
    /// `gtk.FlowBoxChild.changed`) and when `gtk.FlowBox.invalidateSort`
    /// is called.
    ///
    /// Note that using a sort function is incompatible with using a model
    /// (see `gtk.FlowBox.bindModel`).
    extern fn gtk_flow_box_set_sort_func(p_box: *FlowBox, p_sort_func: ?gtk.FlowBoxSortFunc, p_user_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) void;
    pub const setSortFunc = gtk_flow_box_set_sort_func;

    /// Hooks up an adjustment to focus handling in `box`.
    /// The adjustment is also used for autoscrolling during
    /// rubberband selection. See `gtk.ScrolledWindow.getVadjustment`
    /// for a typical way of obtaining the adjustment, and
    /// `gtk.FlowBox.setHadjustment`for setting the horizontal
    /// adjustment.
    ///
    /// The adjustments have to be in pixel units and in the same
    /// coordinate system as the allocation for immediate children
    /// of the box.
    extern fn gtk_flow_box_set_vadjustment(p_box: *FlowBox, p_adjustment: *gtk.Adjustment) void;
    pub const setVadjustment = gtk_flow_box_set_vadjustment;

    /// Unselect all children of `box`, if the selection
    /// mode allows it.
    extern fn gtk_flow_box_unselect_all(p_box: *FlowBox) void;
    pub const unselectAll = gtk_flow_box_unselect_all;

    /// Unselects a single child of `box`, if the selection
    /// mode allows it.
    extern fn gtk_flow_box_unselect_child(p_box: *FlowBox, p_child: *gtk.FlowBoxChild) void;
    pub const unselectChild = gtk_flow_box_unselect_child;

    extern fn gtk_flow_box_get_type() usize;
    pub const getGObjectType = gtk_flow_box_get_type;

    extern fn g_object_ref(p_self: *gtk.FlowBox) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FlowBox) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FlowBox, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FlowBoxAccessible = extern struct {
    pub const Parent = gtk.ContainerAccessible;
    pub const Implements = [_]type{ atk.Component, atk.Selection };
    pub const Class = gtk.FlowBoxAccessibleClass;
    f_parent: gtk.ContainerAccessible,
    f_priv: ?*gtk.FlowBoxAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_flow_box_accessible_get_type() usize;
    pub const getGObjectType = gtk_flow_box_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.FlowBoxAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FlowBoxAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FlowBoxAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FlowBoxChild = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.FlowBoxChildClass;
    f_parent_instance: gtk.Bin,

    pub const virtual_methods = struct {
        pub const activate = struct {
            pub fn call(p_class: anytype, p_child: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(FlowBoxChild.Class, p_class).f_activate.?(gobject.ext.as(FlowBoxChild, p_child));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_child: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(FlowBoxChild.Class, p_class).f_activate = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// The ::activate signal is emitted when the user activates
        /// a child widget in a `gtk.FlowBox`, either by clicking or
        /// double-clicking, or by using the Space or Enter key.
        ///
        /// While this signal is used as a
        /// [keybinding signal][GtkBindingSignal],
        /// it can be used by applications for their own purposes.
        pub const activate = struct {
            pub const name = "activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FlowBoxChild, p_instance))),
                    gobject.signalLookup("activate", FlowBoxChild.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.FlowBoxChild`, to be used as a child
    /// of a `gtk.FlowBox`.
    extern fn gtk_flow_box_child_new() *gtk.FlowBoxChild;
    pub const new = gtk_flow_box_child_new;

    /// Marks `child` as changed, causing any state that depends on this
    /// to be updated. This affects sorting and filtering.
    ///
    /// Note that calls to this method must be in sync with the data
    /// used for the sorting and filtering functions. For instance, if
    /// the list is mirroring some external data set, and *two* children
    /// changed in the external data set when you call
    /// `gtk.FlowBoxChild.changed` on the first child, the sort function
    /// must only read the new data for the first of the two changed
    /// children, otherwise the resorting of the children will be wrong.
    ///
    /// This generally means that if you don’t fully control the data
    /// model, you have to duplicate the data that affects the sorting
    /// and filtering functions into the widgets themselves. Another
    /// alternative is to call `gtk.FlowBox.invalidateSort` on any
    /// model change, but that is more expensive.
    extern fn gtk_flow_box_child_changed(p_child: *FlowBoxChild) void;
    pub const changed = gtk_flow_box_child_changed;

    /// Gets the current index of the `child` in its `gtk.FlowBox` container.
    extern fn gtk_flow_box_child_get_index(p_child: *FlowBoxChild) c_int;
    pub const getIndex = gtk_flow_box_child_get_index;

    /// Returns whether the `child` is currently selected in its
    /// `gtk.FlowBox` container.
    extern fn gtk_flow_box_child_is_selected(p_child: *FlowBoxChild) c_int;
    pub const isSelected = gtk_flow_box_child_is_selected;

    extern fn gtk_flow_box_child_get_type() usize;
    pub const getGObjectType = gtk_flow_box_child_get_type;

    extern fn g_object_ref(p_self: *gtk.FlowBoxChild) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FlowBoxChild) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FlowBoxChild, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FlowBoxChildAccessible = extern struct {
    pub const Parent = gtk.ContainerAccessible;
    pub const Implements = [_]type{atk.Component};
    pub const Class = gtk.FlowBoxChildAccessibleClass;
    f_parent: gtk.ContainerAccessible,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_flow_box_child_accessible_get_type() usize;
    pub const getGObjectType = gtk_flow_box_child_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.FlowBoxChildAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FlowBoxChildAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FlowBoxChildAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.FontButton` is a button which displays the currently selected
/// font an allows to open a font chooser dialog to change the font.
/// It is suitable widget for selecting a font in a preference dialog.
///
/// # CSS nodes
///
/// GtkFontButton has a single CSS node with name button and style class .font.
pub const FontButton = extern struct {
    pub const Parent = gtk.Button;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Actionable, gtk.Activatable, gtk.Buildable, gtk.FontChooser };
    pub const Class = gtk.FontButtonClass;
    f_button: gtk.Button,
    f_priv: ?*gtk.FontButtonPrivate,

    pub const virtual_methods = struct {
        pub const font_set = struct {
            pub fn call(p_class: anytype, p_gfp: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(FontButton.Class, p_class).f_font_set.?(gobject.ext.as(FontButton, p_gfp));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_gfp: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(FontButton.Class, p_class).f_font_set = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The name of the currently selected font.
        pub const font_name = struct {
            pub const name = "font-name";

            pub const Type = ?[*:0]u8;
        };

        /// If this property is set to `TRUE`, the selected font size will be shown
        /// in the label. For a more WYSIWYG way to show the selected size, see the
        /// ::use-size property.
        pub const show_size = struct {
            pub const name = "show-size";

            pub const Type = c_int;
        };

        /// If this property is set to `TRUE`, the name of the selected font style
        /// will be shown in the label. For a more WYSIWYG way to show the selected
        /// style, see the ::use-font property.
        pub const show_style = struct {
            pub const name = "show-style";

            pub const Type = c_int;
        };

        /// The title of the font chooser dialog.
        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };

        /// If this property is set to `TRUE`, the label will be drawn
        /// in the selected font.
        pub const use_font = struct {
            pub const name = "use-font";

            pub const Type = c_int;
        };

        /// If this property is set to `TRUE`, the label will be drawn
        /// with the selected font size.
        pub const use_size = struct {
            pub const name = "use-size";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The ::font-set signal is emitted when the user selects a font.
        /// When handling this signal, use `gtk.FontChooser.getFont`
        /// to find out which font was just selected.
        ///
        /// Note that this signal is only emitted when the user
        /// changes the font. If you need to react to programmatic font changes
        /// as well, use the notify::font signal.
        pub const font_set = struct {
            pub const name = "font-set";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FontButton, p_instance))),
                    gobject.signalLookup("font-set", FontButton.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new font picker widget.
    extern fn gtk_font_button_new() *gtk.FontButton;
    pub const new = gtk_font_button_new;

    /// Creates a new font picker widget.
    extern fn gtk_font_button_new_with_font(p_fontname: [*:0]const u8) *gtk.FontButton;
    pub const newWithFont = gtk_font_button_new_with_font;

    /// Retrieves the name of the currently selected font. This name includes
    /// style and size information as well. If you want to render something
    /// with the font, use this string with `pango.fontDescriptionFromString` .
    /// If you’re interested in peeking certain values (family name,
    /// style, size, weight) just query these properties from the
    /// `pango.FontDescription` object.
    extern fn gtk_font_button_get_font_name(p_font_button: *FontButton) [*:0]const u8;
    pub const getFontName = gtk_font_button_get_font_name;

    /// Returns whether the font size will be shown in the label.
    extern fn gtk_font_button_get_show_size(p_font_button: *FontButton) c_int;
    pub const getShowSize = gtk_font_button_get_show_size;

    /// Returns whether the name of the font style will be shown in the label.
    extern fn gtk_font_button_get_show_style(p_font_button: *FontButton) c_int;
    pub const getShowStyle = gtk_font_button_get_show_style;

    /// Retrieves the title of the font chooser dialog.
    extern fn gtk_font_button_get_title(p_font_button: *FontButton) [*:0]const u8;
    pub const getTitle = gtk_font_button_get_title;

    /// Returns whether the selected font is used in the label.
    extern fn gtk_font_button_get_use_font(p_font_button: *FontButton) c_int;
    pub const getUseFont = gtk_font_button_get_use_font;

    /// Returns whether the selected size is used in the label.
    extern fn gtk_font_button_get_use_size(p_font_button: *FontButton) c_int;
    pub const getUseSize = gtk_font_button_get_use_size;

    /// Sets or updates the currently-displayed font in font picker dialog.
    extern fn gtk_font_button_set_font_name(p_font_button: *FontButton, p_fontname: [*:0]const u8) c_int;
    pub const setFontName = gtk_font_button_set_font_name;

    /// If `show_size` is `TRUE`, the font size will be displayed along with the name of the selected font.
    extern fn gtk_font_button_set_show_size(p_font_button: *FontButton, p_show_size: c_int) void;
    pub const setShowSize = gtk_font_button_set_show_size;

    /// If `show_style` is `TRUE`, the font style will be displayed along with name of the selected font.
    extern fn gtk_font_button_set_show_style(p_font_button: *FontButton, p_show_style: c_int) void;
    pub const setShowStyle = gtk_font_button_set_show_style;

    /// Sets the title for the font chooser dialog.
    extern fn gtk_font_button_set_title(p_font_button: *FontButton, p_title: [*:0]const u8) void;
    pub const setTitle = gtk_font_button_set_title;

    /// If `use_font` is `TRUE`, the font name will be written using the selected font.
    extern fn gtk_font_button_set_use_font(p_font_button: *FontButton, p_use_font: c_int) void;
    pub const setUseFont = gtk_font_button_set_use_font;

    /// If `use_size` is `TRUE`, the font name will be written using the selected size.
    extern fn gtk_font_button_set_use_size(p_font_button: *FontButton, p_use_size: c_int) void;
    pub const setUseSize = gtk_font_button_set_use_size;

    extern fn gtk_font_button_get_type() usize;
    pub const getGObjectType = gtk_font_button_get_type;

    extern fn g_object_ref(p_self: *gtk.FontButton) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FontButton) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FontButton, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.FontChooserDialog` widget is a dialog for selecting a font.
/// It implements the `gtk.FontChooser` interface.
///
/// # GtkFontChooserDialog as GtkBuildable
///
/// The GtkFontChooserDialog implementation of the `gtk.Buildable`
/// interface exposes the buttons with the names “select_button”
/// and “cancel_button”.
pub const FontChooserDialog = extern struct {
    pub const Parent = gtk.Dialog;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.FontChooser };
    pub const Class = gtk.FontChooserDialogClass;
    f_parent_instance: gtk.Dialog,
    f_priv: ?*gtk.FontChooserDialogPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtk.FontChooserDialog`.
    extern fn gtk_font_chooser_dialog_new(p_title: ?[*:0]const u8, p_parent: ?*gtk.Window) *gtk.FontChooserDialog;
    pub const new = gtk_font_chooser_dialog_new;

    extern fn gtk_font_chooser_dialog_get_type() usize;
    pub const getGObjectType = gtk_font_chooser_dialog_get_type;

    extern fn g_object_ref(p_self: *gtk.FontChooserDialog) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FontChooserDialog) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FontChooserDialog, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.FontChooserWidget` widget lists the available fonts,
/// styles and sizes, allowing the user to select a font. It is
/// used in the `gtk.FontChooserDialog` widget to provide a
/// dialog box for selecting fonts.
///
/// To set the font which is initially selected, use
/// `gtk.FontChooser.setFont` or `gtk.FontChooser.setFontDesc`.
///
/// To get the selected font use `gtk.FontChooser.getFont` or
/// `gtk.FontChooser.getFontDesc`.
///
/// To change the text which is shown in the preview area, use
/// `gtk.FontChooser.setPreviewText`.
///
/// # CSS nodes
///
/// GtkFontChooserWidget has a single CSS node with name fontchooser.
pub const FontChooserWidget = extern struct {
    pub const Parent = gtk.Box;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.FontChooser, gtk.Orientable };
    pub const Class = gtk.FontChooserWidgetClass;
    f_parent_instance: gtk.Box,
    f_priv: ?*gtk.FontChooserWidgetPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// A toggle action that can be used to switch to the tweak page
        /// of the font chooser widget, which lets the user tweak the
        /// OpenType features and variation axes of the selected font.
        ///
        /// The action will be enabled or disabled depending on whether
        /// the selected font has any features or axes.
        pub const tweak_action = struct {
            pub const name = "tweak-action";

            pub const Type = ?*gio.Action;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.FontChooserWidget`.
    extern fn gtk_font_chooser_widget_new() *gtk.FontChooserWidget;
    pub const new = gtk_font_chooser_widget_new;

    extern fn gtk_font_chooser_widget_get_type() usize;
    pub const getGObjectType = gtk_font_chooser_widget_get_type;

    extern fn g_object_ref(p_self: *gtk.FontChooserWidget) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FontChooserWidget) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FontChooserWidget, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FontSelection = extern struct {
    pub const Parent = gtk.Box;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.FontSelectionClass;
    f_parent_instance: gtk.Box,
    f_priv: ?*gtk.FontSelectionPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const font_name = struct {
            pub const name = "font-name";

            pub const Type = ?[*:0]u8;
        };

        pub const preview_text = struct {
            pub const name = "preview-text";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.FontSelection`.
    extern fn gtk_font_selection_new() *gtk.FontSelection;
    pub const new = gtk_font_selection_new;

    /// Gets the `pango.FontFace` representing the selected font group
    /// details (i.e. family, slant, weight, width, etc).
    extern fn gtk_font_selection_get_face(p_fontsel: *FontSelection) *pango.FontFace;
    pub const getFace = gtk_font_selection_get_face;

    /// This returns the `gtk.TreeView` which lists all styles available for
    /// the selected font. For example, “Regular”, “Bold”, etc.
    extern fn gtk_font_selection_get_face_list(p_fontsel: *FontSelection) *gtk.Widget;
    pub const getFaceList = gtk_font_selection_get_face_list;

    /// Gets the `pango.FontFamily` representing the selected font family.
    extern fn gtk_font_selection_get_family(p_fontsel: *FontSelection) *pango.FontFamily;
    pub const getFamily = gtk_font_selection_get_family;

    /// This returns the `gtk.TreeView` that lists font families, for
    /// example, “Sans”, “Serif”, etc.
    extern fn gtk_font_selection_get_family_list(p_fontsel: *FontSelection) *gtk.Widget;
    pub const getFamilyList = gtk_font_selection_get_family_list;

    /// Gets the currently-selected font name.
    ///
    /// Note that this can be a different string than what you set with
    /// `gtk.FontSelection.setFontName`, as the font selection widget may
    /// normalize font names and thus return a string with a different structure.
    /// For example, “Helvetica Italic Bold 12” could be normalized to
    /// “Helvetica Bold Italic 12”. Use `pango.FontDescription.equal`
    /// if you want to compare two font descriptions.
    extern fn gtk_font_selection_get_font_name(p_fontsel: *FontSelection) [*:0]u8;
    pub const getFontName = gtk_font_selection_get_font_name;

    /// This returns the `gtk.Entry` used to display the font as a preview.
    extern fn gtk_font_selection_get_preview_entry(p_fontsel: *FontSelection) *gtk.Widget;
    pub const getPreviewEntry = gtk_font_selection_get_preview_entry;

    /// Gets the text displayed in the preview area.
    extern fn gtk_font_selection_get_preview_text(p_fontsel: *FontSelection) [*:0]const u8;
    pub const getPreviewText = gtk_font_selection_get_preview_text;

    /// The selected font size.
    extern fn gtk_font_selection_get_size(p_fontsel: *FontSelection) c_int;
    pub const getSize = gtk_font_selection_get_size;

    /// This returns the `gtk.Entry` used to allow the user to edit the font
    /// number manually instead of selecting it from the list of font sizes.
    extern fn gtk_font_selection_get_size_entry(p_fontsel: *FontSelection) *gtk.Widget;
    pub const getSizeEntry = gtk_font_selection_get_size_entry;

    /// This returns the `gtk.TreeView` used to list font sizes.
    extern fn gtk_font_selection_get_size_list(p_fontsel: *FontSelection) *gtk.Widget;
    pub const getSizeList = gtk_font_selection_get_size_list;

    /// Sets the currently-selected font.
    ///
    /// Note that the `fontsel` needs to know the screen in which it will appear
    /// for this to work; this can be guaranteed by simply making sure that the
    /// `fontsel` is inserted in a toplevel window before you call this function.
    extern fn gtk_font_selection_set_font_name(p_fontsel: *FontSelection, p_fontname: [*:0]const u8) c_int;
    pub const setFontName = gtk_font_selection_set_font_name;

    /// Sets the text displayed in the preview area.
    /// The `text` is used to show how the selected font looks.
    extern fn gtk_font_selection_set_preview_text(p_fontsel: *FontSelection, p_text: [*:0]const u8) void;
    pub const setPreviewText = gtk_font_selection_set_preview_text;

    extern fn gtk_font_selection_get_type() usize;
    pub const getGObjectType = gtk_font_selection_get_type;

    extern fn g_object_ref(p_self: *gtk.FontSelection) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FontSelection) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FontSelection, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FontSelectionDialog = extern struct {
    pub const Parent = gtk.Dialog;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.FontSelectionDialogClass;
    f_parent_instance: gtk.Dialog,
    f_priv: ?*gtk.FontSelectionDialogPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtk.FontSelectionDialog`.
    extern fn gtk_font_selection_dialog_new(p_title: [*:0]const u8) *gtk.FontSelectionDialog;
    pub const new = gtk_font_selection_dialog_new;

    /// Gets the “Cancel” button.
    extern fn gtk_font_selection_dialog_get_cancel_button(p_fsd: *FontSelectionDialog) *gtk.Widget;
    pub const getCancelButton = gtk_font_selection_dialog_get_cancel_button;

    /// Gets the currently-selected font name.
    ///
    /// Note that this can be a different string than what you set with
    /// `gtk.FontSelectionDialog.setFontName`, as the font selection widget
    /// may normalize font names and thus return a string with a different
    /// structure. For example, “Helvetica Italic Bold 12” could be normalized
    /// to “Helvetica Bold Italic 12”.  Use `pango.FontDescription.equal`
    /// if you want to compare two font descriptions.
    extern fn gtk_font_selection_dialog_get_font_name(p_fsd: *FontSelectionDialog) [*:0]u8;
    pub const getFontName = gtk_font_selection_dialog_get_font_name;

    /// Retrieves the `gtk.FontSelection` widget embedded in the dialog.
    extern fn gtk_font_selection_dialog_get_font_selection(p_fsd: *FontSelectionDialog) *gtk.Widget;
    pub const getFontSelection = gtk_font_selection_dialog_get_font_selection;

    /// Gets the “OK” button.
    extern fn gtk_font_selection_dialog_get_ok_button(p_fsd: *FontSelectionDialog) *gtk.Widget;
    pub const getOkButton = gtk_font_selection_dialog_get_ok_button;

    /// Gets the text displayed in the preview area.
    extern fn gtk_font_selection_dialog_get_preview_text(p_fsd: *FontSelectionDialog) [*:0]const u8;
    pub const getPreviewText = gtk_font_selection_dialog_get_preview_text;

    /// Sets the currently selected font.
    extern fn gtk_font_selection_dialog_set_font_name(p_fsd: *FontSelectionDialog, p_fontname: [*:0]const u8) c_int;
    pub const setFontName = gtk_font_selection_dialog_set_font_name;

    /// Sets the text displayed in the preview area.
    extern fn gtk_font_selection_dialog_set_preview_text(p_fsd: *FontSelectionDialog, p_text: [*:0]const u8) void;
    pub const setPreviewText = gtk_font_selection_dialog_set_preview_text;

    extern fn gtk_font_selection_dialog_get_type() usize;
    pub const getGObjectType = gtk_font_selection_dialog_get_type;

    extern fn g_object_ref(p_self: *gtk.FontSelectionDialog) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FontSelectionDialog) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FontSelectionDialog, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The frame widget is a bin that surrounds its child with a decorative
/// frame and an optional label. If present, the label is drawn in a gap
/// in the top side of the frame. The position of the label can be
/// controlled with `gtk.Frame.setLabelAlign`.
///
/// # GtkFrame as GtkBuildable
///
/// The GtkFrame implementation of the `gtk.Buildable` interface supports
/// placing a child in the label position by specifying “label” as the
/// “type” attribute of a `<child>` element. A normal content child can
/// be specified without specifying a `<child>` type attribute.
///
/// An example of a UI definition fragment with `GtkFrame`:
///
/// ```
/// <object class="GtkFrame">
///   <child type="label">
///     <object class="GtkLabel" id="frame-label"/>
///   </child>
///   <child>
///     <object class="GtkEntry" id="frame-content"/>
///   </child>
/// </object>
/// ```
///
/// # CSS nodes
///
/// ```
/// frame
/// ├── border[.flat]
/// ├── <label widget>
/// ╰── <child>
/// ```
///
/// GtkFrame has a main CSS node named “frame” and a subnode named “border”. The
/// “border” node is used to draw the visible border. You can set the appearance
/// of the border using CSS properties like “border-style” on the “border” node.
///
/// The border node can be given the style class “.flat”, which is used by themes
/// to disable drawing of the border. To do this from code, call
/// `gtk.Frame.setShadowType` with `GTK_SHADOW_NONE` to add the “.flat” class or
/// any other shadow type to remove it.
pub const Frame = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.FrameClass;
    f_bin: gtk.Bin,
    f_priv: ?*gtk.FramePrivate,

    pub const virtual_methods = struct {
        pub const compute_child_allocation = struct {
            pub fn call(p_class: anytype, p_frame: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_allocation: *gtk.Allocation) void {
                return gobject.ext.as(Frame.Class, p_class).f_compute_child_allocation.?(gobject.ext.as(Frame, p_frame), p_allocation);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_frame: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_allocation: *gtk.Allocation) callconv(.C) void) void {
                gobject.ext.as(Frame.Class, p_class).f_compute_child_allocation = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };

        pub const label_widget = struct {
            pub const name = "label-widget";

            pub const Type = ?*gtk.Widget;
        };

        pub const label_xalign = struct {
            pub const name = "label-xalign";

            pub const Type = f32;
        };

        pub const label_yalign = struct {
            pub const name = "label-yalign";

            pub const Type = f32;
        };

        pub const shadow_type = struct {
            pub const name = "shadow-type";

            pub const Type = gtk.ShadowType;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.Frame`, with optional label `label`.
    /// If `label` is `NULL`, the label is omitted.
    extern fn gtk_frame_new(p_label: ?[*:0]const u8) *gtk.Frame;
    pub const new = gtk_frame_new;

    /// If the frame’s label widget is a `gtk.Label`, returns the
    /// text in the label widget. (The frame will have a `gtk.Label`
    /// for the label widget if a non-`NULL` argument was passed
    /// to `gtk.Frame.new`.)
    extern fn gtk_frame_get_label(p_frame: *Frame) ?[*:0]const u8;
    pub const getLabel = gtk_frame_get_label;

    /// Retrieves the X and Y alignment of the frame’s label. See
    /// `gtk.Frame.setLabelAlign`.
    extern fn gtk_frame_get_label_align(p_frame: *Frame, p_xalign: ?*f32, p_yalign: ?*f32) void;
    pub const getLabelAlign = gtk_frame_get_label_align;

    /// Retrieves the label widget for the frame. See
    /// `gtk.Frame.setLabelWidget`.
    extern fn gtk_frame_get_label_widget(p_frame: *Frame) ?*gtk.Widget;
    pub const getLabelWidget = gtk_frame_get_label_widget;

    /// Retrieves the shadow type of the frame. See
    /// `gtk.Frame.setShadowType`.
    extern fn gtk_frame_get_shadow_type(p_frame: *Frame) gtk.ShadowType;
    pub const getShadowType = gtk_frame_get_shadow_type;

    /// Removes the current `gtk.Frame.properties.label`-widget. If `label` is not `NULL`, creates a
    /// new `gtk.Label` with that text and adds it as the `gtk.Frame.properties.label`-widget.
    extern fn gtk_frame_set_label(p_frame: *Frame, p_label: ?[*:0]const u8) void;
    pub const setLabel = gtk_frame_set_label;

    /// Sets the alignment of the frame widget’s label. The
    /// default values for a newly created frame are 0.0 and 0.5.
    extern fn gtk_frame_set_label_align(p_frame: *Frame, p_xalign: f32, p_yalign: f32) void;
    pub const setLabelAlign = gtk_frame_set_label_align;

    /// Sets the `gtk.Frame.properties.label`-widget for the frame. This is the widget that
    /// will appear embedded in the top edge of the frame as a title.
    extern fn gtk_frame_set_label_widget(p_frame: *Frame, p_label_widget: ?*gtk.Widget) void;
    pub const setLabelWidget = gtk_frame_set_label_widget;

    /// Sets the `gtk.Frame.properties.shadow`-type for `frame`, i.e. whether it is drawn without
    /// (`GTK_SHADOW_NONE`) or with (other values) a visible border. Values other than
    /// `GTK_SHADOW_NONE` are treated identically by GtkFrame. The chosen type is
    /// applied by removing or adding the .flat class to the CSS node named border.
    extern fn gtk_frame_set_shadow_type(p_frame: *Frame, p_type: gtk.ShadowType) void;
    pub const setShadowType = gtk_frame_set_shadow_type;

    extern fn gtk_frame_get_type() usize;
    pub const getGObjectType = gtk_frame_get_type;

    extern fn g_object_ref(p_self: *gtk.Frame) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Frame) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Frame, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const FrameAccessible = extern struct {
    pub const Parent = gtk.ContainerAccessible;
    pub const Implements = [_]type{atk.Component};
    pub const Class = gtk.FrameAccessibleClass;
    f_parent: gtk.ContainerAccessible,
    f_priv: ?*gtk.FrameAccessiblePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_frame_accessible_get_type() usize;
    pub const getGObjectType = gtk_frame_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.FrameAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.FrameAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FrameAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.GLArea` is a widget that allows drawing with OpenGL.
///
/// `gtk.GLArea` sets up its own `gdk.GLContext` for the window it creates, and
/// creates a custom GL framebuffer that the widget will do GL rendering onto.
/// It also ensures that this framebuffer is the default GL rendering target
/// when rendering.
///
/// In order to draw, you have to connect to the `gtk.GLArea.signals.render` signal,
/// or subclass `gtk.GLArea` and override the `GtkGLAreaClass`.`render` virtual
/// function.
///
/// The `gtk.GLArea` widget ensures that the `gdk.GLContext` is associated with
/// the widget's drawing area, and it is kept updated when the size and
/// position of the drawing area changes.
///
/// ## Drawing with GtkGLArea
///
/// The simplest way to draw using OpenGL commands in a `gtk.GLArea` is to
/// create a widget instance and connect to the `gtk.GLArea.signals.render` signal:
///
/// ```
///   // create a GtkGLArea instance
///   GtkWidget *gl_area = gtk_gl_area_new ();
///
///   // connect to the "render" signal
///   g_signal_connect (gl_area, "render", G_CALLBACK (render), NULL);
/// ```
///
/// The ``render`` function will be called when the `gtk.GLArea` is ready
/// for you to draw its content:
///
/// ```
///   static gboolean
///   render (GtkGLArea *area, GdkGLContext *context)
///   {
///     // inside this function it's safe to use GL; the given
///     // `gdk.GLContext` has been made current to the drawable
///     // surface used by the `gtk.GLArea` and the viewport has
///     // already been set to be the size of the allocation
///
///     // we can start by clearing the buffer
///     glClearColor (0, 0, 0, 0);
///     glClear (GL_COLOR_BUFFER_BIT);
///
///     // draw your object
///     draw_an_object ();
///
///     // we completed our drawing; the draw commands will be
///     // flushed at the end of the signal emission chain, and
///     // the buffers will be drawn on the window
///     return TRUE;
///   }
/// ```
///
/// If you need to initialize OpenGL state, e.g. buffer objects or
/// shaders, you should use the `gtk.Widget.signals.realize` signal; you
/// can use the `gtk.Widget.signals.unrealize` signal to clean up. Since the
/// `gdk.GLContext` creation and initialization may fail, you will
/// need to check for errors, using `gtk.GLArea.getError`. An example
/// of how to safely initialize the GL state is:
///
/// ```
///   static void
///   on_realize (GtkGLarea *area)
///   {
///     // We need to make the context current if we want to
///     // call GL API
///     gtk_gl_area_make_current (area);
///
///     // If there were errors during the initialization or
///     // when trying to make the context current, this
///     // function will return a `glib.Error` for you to catch
///     if (gtk_gl_area_get_error (area) != NULL)
///       return;
///
///     // You can also use `gtk.GLArea.setError` in order
///     // to show eventual initialization errors on the
///     // GtkGLArea widget itself
///     GError *internal_error = NULL;
///     init_buffer_objects (&error);
///     if (error != NULL)
///       {
///         gtk_gl_area_set_error (area, error);
///         g_error_free (error);
///         return;
///       }
///
///     init_shaders (&error);
///     if (error != NULL)
///       {
///         gtk_gl_area_set_error (area, error);
///         g_error_free (error);
///         return;
///       }
///   }
/// ```
///
/// If you need to change the options for creating the `gdk.GLContext`
/// you should use the `gtk.GLArea.signals.create`-context signal.
pub const GLArea = extern struct {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.GLAreaClass;
    f_parent_instance: gtk.Widget,

    pub const virtual_methods = struct {
        /// class closure for the `gtk.GLArea.signals.create`-context signal
        pub const create_context = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *gdk.GLContext {
                return gobject.ext.as(GLArea.Class, p_class).f_create_context.?(gobject.ext.as(GLArea, p_area));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *gdk.GLContext) void {
                gobject.ext.as(GLArea.Class, p_class).f_create_context = @ptrCast(p_implementation);
            }
        };

        /// class closure for the `gtk.GLArea.signals.render` signal
        pub const render = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gdk.GLContext) c_int {
                return gobject.ext.as(GLArea.Class, p_class).f_render.?(gobject.ext.as(GLArea, p_area), p_context);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_context: *gdk.GLContext) callconv(.C) c_int) void {
                gobject.ext.as(GLArea.Class, p_class).f_render = @ptrCast(p_implementation);
            }
        };

        /// class closeure for the `gtk.GLArea.signals.resize` signal
        pub const resize = struct {
            pub fn call(p_class: anytype, p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_width: c_int, p_height: c_int) void {
                return gobject.ext.as(GLArea.Class, p_class).f_resize.?(gobject.ext.as(GLArea, p_area), p_width, p_height);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_area: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_width: c_int, p_height: c_int) callconv(.C) void) void {
                gobject.ext.as(GLArea.Class, p_class).f_resize = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// If set to `TRUE` the `gtk.GLArea.signals.render` signal will be emitted every time
        /// the widget draws. This is the default and is useful if drawing the widget
        /// is faster.
        ///
        /// If set to `FALSE` the data from previous rendering is kept around and will
        /// be used for drawing the widget the next time, unless the window is resized.
        /// In order to force a rendering `gtk.GLArea.queueRender` must be called.
        /// This mode is useful when the scene changes seldomly, but takes a long time
        /// to redraw.
        pub const auto_render = struct {
            pub const name = "auto-render";

            pub const Type = c_int;
        };

        /// The `gdk.GLContext` used by the `gtk.GLArea` widget.
        ///
        /// The `gtk.GLArea` widget is responsible for creating the `gdk.GLContext`
        /// instance. If you need to render with other kinds of buffers (stencil,
        /// depth, etc), use render buffers.
        pub const context = struct {
            pub const name = "context";

            pub const Type = ?*gdk.GLContext;
        };

        /// If set to `TRUE` the buffer allocated by the widget will have an alpha channel
        /// component, and when rendering to the window the result will be composited over
        /// whatever is below the widget.
        ///
        /// If set to `FALSE` there will be no alpha channel, and the buffer will fully
        /// replace anything below the widget.
        pub const has_alpha = struct {
            pub const name = "has-alpha";

            pub const Type = c_int;
        };

        /// If set to `TRUE` the widget will allocate and enable a depth buffer for the
        /// target framebuffer.
        pub const has_depth_buffer = struct {
            pub const name = "has-depth-buffer";

            pub const Type = c_int;
        };

        /// If set to `TRUE` the widget will allocate and enable a stencil buffer for the
        /// target framebuffer.
        pub const has_stencil_buffer = struct {
            pub const name = "has-stencil-buffer";

            pub const Type = c_int;
        };

        /// If set to `TRUE` the widget will try to create a `gdk.GLContext` using
        /// OpenGL ES instead of OpenGL.
        ///
        /// See also: `gdk.GLContext.setUseEs`
        pub const use_es = struct {
            pub const name = "use-es";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The ::create-context signal is emitted when the widget is being
        /// realized, and allows you to override how the GL context is
        /// created. This is useful when you want to reuse an existing GL
        /// context, or if you want to try creating different kinds of GL
        /// options.
        ///
        /// If context creation fails then the signal handler can use
        /// `gtk.GLArea.setError` to register a more detailed error
        /// of how the construction failed.
        pub const create_context = struct {
            pub const name = "create-context";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) *gdk.GLContext, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GLArea, p_instance))),
                    gobject.signalLookup("create-context", GLArea.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::render signal is emitted every time the contents
        /// of the `gtk.GLArea` should be redrawn.
        ///
        /// The `context` is bound to the `area` prior to emitting this function,
        /// and the buffers are painted to the window once the emission terminates.
        pub const render = struct {
            pub const name = "render";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_context: *gdk.GLContext, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GLArea, p_instance))),
                    gobject.signalLookup("render", GLArea.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::resize signal is emitted once when the widget is realized, and
        /// then each time the widget is changed while realized. This is useful
        /// in order to keep GL state up to date with the widget size, like for
        /// instance camera properties which may depend on the width/height ratio.
        ///
        /// The GL context for the area is guaranteed to be current when this signal
        /// is emitted.
        ///
        /// The default handler sets up the GL viewport.
        pub const resize = struct {
            pub const name = "resize";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_width: c_int, p_height: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GLArea, p_instance))),
                    gobject.signalLookup("resize", GLArea.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.GLArea` widget.
    extern fn gtk_gl_area_new() *gtk.GLArea;
    pub const new = gtk_gl_area_new;

    /// Ensures that the `area` framebuffer object is made the current draw
    /// and read target, and that all the required buffers for the `area`
    /// are created and bound to the frambuffer.
    ///
    /// This function is automatically called before emitting the
    /// `gtk.GLArea.signals.render` signal, and doesn't normally need to be called
    /// by application code.
    extern fn gtk_gl_area_attach_buffers(p_area: *GLArea) void;
    pub const attachBuffers = gtk_gl_area_attach_buffers;

    /// Returns whether the area is in auto render mode or not.
    extern fn gtk_gl_area_get_auto_render(p_area: *GLArea) c_int;
    pub const getAutoRender = gtk_gl_area_get_auto_render;

    /// Retrieves the `gdk.GLContext` used by `area`.
    extern fn gtk_gl_area_get_context(p_area: *GLArea) *gdk.GLContext;
    pub const getContext = gtk_gl_area_get_context;

    /// Gets the current error set on the `area`.
    extern fn gtk_gl_area_get_error(p_area: *GLArea) ?*glib.Error;
    pub const getError = gtk_gl_area_get_error;

    /// Returns whether the area has an alpha component.
    extern fn gtk_gl_area_get_has_alpha(p_area: *GLArea) c_int;
    pub const getHasAlpha = gtk_gl_area_get_has_alpha;

    /// Returns whether the area has a depth buffer.
    extern fn gtk_gl_area_get_has_depth_buffer(p_area: *GLArea) c_int;
    pub const getHasDepthBuffer = gtk_gl_area_get_has_depth_buffer;

    /// Returns whether the area has a stencil buffer.
    extern fn gtk_gl_area_get_has_stencil_buffer(p_area: *GLArea) c_int;
    pub const getHasStencilBuffer = gtk_gl_area_get_has_stencil_buffer;

    /// Retrieves the required version of OpenGL set
    /// using `gtk.GLArea.setRequiredVersion`.
    extern fn gtk_gl_area_get_required_version(p_area: *GLArea, p_major: *c_int, p_minor: *c_int) void;
    pub const getRequiredVersion = gtk_gl_area_get_required_version;

    /// Retrieves the value set by `gtk.GLArea.setUseEs`.
    extern fn gtk_gl_area_get_use_es(p_area: *GLArea) c_int;
    pub const getUseEs = gtk_gl_area_get_use_es;

    /// Ensures that the `gdk.GLContext` used by `area` is associated with
    /// the `gtk.GLArea`.
    ///
    /// This function is automatically called before emitting the
    /// `gtk.GLArea.signals.render` signal, and doesn't normally need to be called
    /// by application code.
    extern fn gtk_gl_area_make_current(p_area: *GLArea) void;
    pub const makeCurrent = gtk_gl_area_make_current;

    /// Marks the currently rendered data (if any) as invalid, and queues
    /// a redraw of the widget, ensuring that the `gtk.GLArea.signals.render` signal
    /// is emitted during the draw.
    ///
    /// This is only needed when the `gtk.GLArea.setAutoRender` has
    /// been called with a `FALSE` value. The default behaviour is to
    /// emit `gtk.GLArea.signals.render` on each draw.
    extern fn gtk_gl_area_queue_render(p_area: *GLArea) void;
    pub const queueRender = gtk_gl_area_queue_render;

    /// If `auto_render` is `TRUE` the `gtk.GLArea.signals.render` signal will be
    /// emitted every time the widget draws. This is the default and is
    /// useful if drawing the widget is faster.
    ///
    /// If `auto_render` is `FALSE` the data from previous rendering is kept
    /// around and will be used for drawing the widget the next time,
    /// unless the window is resized. In order to force a rendering
    /// `gtk.GLArea.queueRender` must be called. This mode is useful when
    /// the scene changes seldomly, but takes a long time to redraw.
    extern fn gtk_gl_area_set_auto_render(p_area: *GLArea, p_auto_render: c_int) void;
    pub const setAutoRender = gtk_gl_area_set_auto_render;

    /// Sets an error on the area which will be shown instead of the
    /// GL rendering. This is useful in the `gtk.GLArea.signals.create`-context
    /// signal if GL context creation fails.
    extern fn gtk_gl_area_set_error(p_area: *GLArea, p_error: ?*const glib.Error) void;
    pub const setError = gtk_gl_area_set_error;

    /// If `has_alpha` is `TRUE` the buffer allocated by the widget will have
    /// an alpha channel component, and when rendering to the window the
    /// result will be composited over whatever is below the widget.
    ///
    /// If `has_alpha` is `FALSE` there will be no alpha channel, and the
    /// buffer will fully replace anything below the widget.
    extern fn gtk_gl_area_set_has_alpha(p_area: *GLArea, p_has_alpha: c_int) void;
    pub const setHasAlpha = gtk_gl_area_set_has_alpha;

    /// If `has_depth_buffer` is `TRUE` the widget will allocate and
    /// enable a depth buffer for the target framebuffer. Otherwise
    /// there will be none.
    extern fn gtk_gl_area_set_has_depth_buffer(p_area: *GLArea, p_has_depth_buffer: c_int) void;
    pub const setHasDepthBuffer = gtk_gl_area_set_has_depth_buffer;

    /// If `has_stencil_buffer` is `TRUE` the widget will allocate and
    /// enable a stencil buffer for the target framebuffer. Otherwise
    /// there will be none.
    extern fn gtk_gl_area_set_has_stencil_buffer(p_area: *GLArea, p_has_stencil_buffer: c_int) void;
    pub const setHasStencilBuffer = gtk_gl_area_set_has_stencil_buffer;

    /// Sets the required version of OpenGL to be used when creating the context
    /// for the widget.
    ///
    /// This function must be called before the area has been realized.
    extern fn gtk_gl_area_set_required_version(p_area: *GLArea, p_major: c_int, p_minor: c_int) void;
    pub const setRequiredVersion = gtk_gl_area_set_required_version;

    /// Sets whether the `area` should create an OpenGL or an OpenGL ES context.
    ///
    /// You should check the capabilities of the `gdk.GLContext` before drawing
    /// with either API.
    extern fn gtk_gl_area_set_use_es(p_area: *GLArea, p_use_es: c_int) void;
    pub const setUseEs = gtk_gl_area_set_use_es;

    extern fn gtk_gl_area_get_type() usize;
    pub const getGObjectType = gtk_gl_area_get_type;

    extern fn g_object_ref(p_self: *gtk.GLArea) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.GLArea) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GLArea, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.Gesture` is the base object for gesture recognition, although this
/// object is quite generalized to serve as a base for multi-touch gestures,
/// it is suitable to implement single-touch and pointer-based gestures (using
/// the special `NULL` `gdk.EventSequence` value for these).
///
/// The number of touches that a `gtk.Gesture` need to be recognized is controlled
/// by the `gtk.Gesture.properties.n`-points property, if a gesture is keeping track of less
/// or more than that number of sequences, it won't check wether the gesture
/// is recognized.
///
/// As soon as the gesture has the expected number of touches, the gesture will
/// run the `gtk.Gesture.signals.check` signal regularly on input events until the gesture
/// is recognized, the criteria to consider a gesture as "recognized" is left to
/// `gtk.Gesture` subclasses.
///
/// A recognized gesture will then emit the following signals:
/// - `gtk.Gesture.signals.begin` when the gesture is recognized.
/// - A number of `gtk.Gesture.signals.update`, whenever an input event is processed.
/// - `gtk.Gesture.signals.end` when the gesture is no longer recognized.
///
/// ## Event propagation
///
/// In order to receive events, a gesture needs to either set a propagation phase
/// through `gtk.EventController.setPropagationPhase`, or feed those manually
/// through `gtk.EventController.handleEvent`.
///
/// In the capture phase, events are propagated from the toplevel down to the
/// target widget, and gestures that are attached to containers above the widget
/// get a chance to interact with the event before it reaches the target.
///
/// After the capture phase, GTK+ emits the traditional `gtk.Widget.signals.button`-press-event,
/// `gtk.Widget.signals.button`-release-event, `gtk.Widget.signals.touch`-event, etc signals. Gestures
/// with the `GTK_PHASE_TARGET` phase are fed events from the default `gtk.Widget.signals.event`
/// handlers.
///
/// In the bubble phase, events are propagated up from the target widget to the
/// toplevel, and gestures that are attached to containers above the widget get
/// a chance to interact with events that have not been handled yet.
///
/// ## States of a sequence
///
/// Whenever input interaction happens, a single event may trigger a cascade of
/// `GtkGestures`, both across the parents of the widget receiving the event and
/// in parallel within an individual widget. It is a responsibility of the
/// widgets using those gestures to set the state of touch sequences accordingly
/// in order to enable cooperation of gestures around the `GdkEventSequences`
/// triggering those.
///
/// Within a widget, gestures can be grouped through `gtk.Gesture.group`,
/// grouped gestures synchronize the state of sequences, so calling
/// `gtk.Gesture.setSequenceState` on one will effectively propagate
/// the state throughout the group.
///
/// By default, all sequences start out in the `GTK_EVENT_SEQUENCE_NONE` state,
/// sequences in this state trigger the gesture event handler, but event
/// propagation will continue unstopped by gestures.
///
/// If a sequence enters into the `GTK_EVENT_SEQUENCE_DENIED` state, the gesture
/// group will effectively ignore the sequence, letting events go unstopped
/// through the gesture, but the "slot" will still remain occupied while
/// the touch is active.
///
/// If a sequence enters in the `GTK_EVENT_SEQUENCE_CLAIMED` state, the gesture
/// group will grab all interaction on the sequence, by:
/// - Setting the same sequence to `GTK_EVENT_SEQUENCE_DENIED` on every other gesture
///   group within the widget, and every gesture on parent widgets in the propagation
///   chain.
/// - calling `gtk.Gesture.signals.cancel` on every gesture in widgets underneath in the
///   propagation chain.
/// - Stopping event propagation after the gesture group handles the event.
///
/// Note: if a sequence is set early to `GTK_EVENT_SEQUENCE_CLAIMED` on
/// `GDK_TOUCH_BEGIN`/`GDK_BUTTON_PRESS` (so those events are captured before
/// reaching the event widget, this implies `GTK_PHASE_CAPTURE`), one similar
/// event will emulated if the sequence changes to `GTK_EVENT_SEQUENCE_DENIED`.
/// This way event coherence is preserved before event propagation is unstopped
/// again.
///
/// Sequence states can't be changed freely, see `gtk.Gesture.setSequenceState`
/// to know about the possible lifetimes of a `gdk.EventSequence`.
///
/// ## Touchpad gestures
///
/// On the platforms that support it, `gtk.Gesture` will handle transparently
/// touchpad gesture events. The only precautions users of `gtk.Gesture` should do
/// to enable this support are:
/// - Enabling `GDK_TOUCHPAD_GESTURE_MASK` on their `GdkWindows`
/// - If the gesture has `GTK_PHASE_NONE`, ensuring events of type
///   `GDK_TOUCHPAD_SWIPE` and `GDK_TOUCHPAD_PINCH` are handled by the `gtk.Gesture`
pub const Gesture = opaque {
    pub const Parent = gtk.EventController;
    pub const Implements = [_]type{};
    pub const Class = gtk.GestureClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The number of touch points that trigger recognition on this gesture,
        pub const n_points = struct {
            pub const name = "n-points";

            pub const Type = c_uint;
        };

        /// If non-`NULL`, the gesture will only listen for events that happen on
        /// this `gdk.Window`, or a child of it.
        pub const window = struct {
            pub const name = "window";

            pub const Type = ?*gdk.Window;
        };
    };

    pub const signals = struct {
        /// This signal is emitted when the gesture is recognized. This means the
        /// number of touch sequences matches `gtk.Gesture.properties.n`-points, and the `gtk.Gesture.signals.check`
        /// handler(s) returned `TRUE`.
        ///
        /// Note: These conditions may also happen when an extra touch (eg. a third touch
        /// on a 2-touches gesture) is lifted, in that situation `sequence` won't pertain
        /// to the current set of active touches, so don't rely on this being true.
        pub const begin = struct {
            pub const name = "begin";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_sequence: ?*gdk.EventSequence, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Gesture, p_instance))),
                    gobject.signalLookup("begin", Gesture.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted whenever a sequence is cancelled. This usually
        /// happens on active touches when `gtk.EventController.reset` is called
        /// on `gesture` (manually, due to grabs...), or the individual `sequence`
        /// was claimed by parent widgets' controllers (see `gtk.Gesture.setSequenceState`).
        ///
        /// `gesture` must forget everything about `sequence` as a reaction to this signal.
        pub const cancel = struct {
            pub const name = "cancel";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_sequence: ?*gdk.EventSequence, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Gesture, p_instance))),
                    gobject.signalLookup("cancel", Gesture.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted when `gesture` either stopped recognizing the event
        /// sequences as something to be handled (the `gtk.Gesture.signals.check` handler returned
        /// `FALSE`), or the number of touch sequences became higher or lower than
        /// `gtk.Gesture.properties.n`-points.
        ///
        /// Note: `sequence` might not pertain to the group of sequences that were
        /// previously triggering recognition on `gesture` (ie. a just pressed touch
        /// sequence that exceeds `gtk.Gesture.properties.n`-points). This situation may be detected
        /// by checking through `gtk.Gesture.handlesSequence`.
        pub const end = struct {
            pub const name = "end";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_sequence: ?*gdk.EventSequence, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Gesture, p_instance))),
                    gobject.signalLookup("end", Gesture.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted whenever a sequence state changes. See
        /// `gtk.Gesture.setSequenceState` to know more about the expectable
        /// sequence lifetimes.
        pub const sequence_state_changed = struct {
            pub const name = "sequence-state-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_sequence: ?*gdk.EventSequence, p_state: gtk.EventSequenceState, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Gesture, p_instance))),
                    gobject.signalLookup("sequence-state-changed", Gesture.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted whenever an event is handled while the gesture is
        /// recognized. `sequence` is guaranteed to pertain to the set of active touches.
        pub const update = struct {
            pub const name = "update";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_sequence: ?*gdk.EventSequence, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Gesture, p_instance))),
                    gobject.signalLookup("update", Gesture.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// If there are touch sequences being currently handled by `gesture`,
    /// this function returns `TRUE` and fills in `rect` with the bounding
    /// box containing all active touches. Otherwise, `FALSE` will be
    /// returned.
    ///
    /// Note: This function will yield unexpected results on touchpad
    /// gestures. Since there is no correlation between physical and
    /// pixel distances, these will look as if constrained in an
    /// infinitely small area, `rect` width and height will thus be 0
    /// regardless of the number of touchpoints.
    extern fn gtk_gesture_get_bounding_box(p_gesture: *Gesture, p_rect: *gdk.Rectangle) c_int;
    pub const getBoundingBox = gtk_gesture_get_bounding_box;

    /// If there are touch sequences being currently handled by `gesture`,
    /// this function returns `TRUE` and fills in `x` and `y` with the center
    /// of the bounding box containing all active touches. Otherwise, `FALSE`
    /// will be returned.
    extern fn gtk_gesture_get_bounding_box_center(p_gesture: *Gesture, p_x: *f64, p_y: *f64) c_int;
    pub const getBoundingBoxCenter = gtk_gesture_get_bounding_box_center;

    /// Returns the master `gdk.Device` that is currently operating
    /// on `gesture`, or `NULL` if the gesture is not being interacted.
    extern fn gtk_gesture_get_device(p_gesture: *Gesture) ?*gdk.Device;
    pub const getDevice = gtk_gesture_get_device;

    /// Returns all gestures in the group of `gesture`
    extern fn gtk_gesture_get_group(p_gesture: *Gesture) *glib.List;
    pub const getGroup = gtk_gesture_get_group;

    /// Returns the last event that was processed for `sequence`.
    ///
    /// Note that the returned pointer is only valid as long as the `sequence`
    /// is still interpreted by the `gesture`. If in doubt, you should make
    /// a copy of the event.
    extern fn gtk_gesture_get_last_event(p_gesture: *Gesture, p_sequence: ?*gdk.EventSequence) ?*const gdk.Event;
    pub const getLastEvent = gtk_gesture_get_last_event;

    /// Returns the `gdk.EventSequence` that was last updated on `gesture`.
    extern fn gtk_gesture_get_last_updated_sequence(p_gesture: *Gesture) ?*gdk.EventSequence;
    pub const getLastUpdatedSequence = gtk_gesture_get_last_updated_sequence;

    /// If `sequence` is currently being interpreted by `gesture`, this
    /// function returns `TRUE` and fills in `x` and `y` with the last coordinates
    /// stored for that event sequence. The coordinates are always relative to the
    /// widget allocation.
    extern fn gtk_gesture_get_point(p_gesture: *Gesture, p_sequence: ?*gdk.EventSequence, p_x: ?*f64, p_y: ?*f64) c_int;
    pub const getPoint = gtk_gesture_get_point;

    /// Returns the `sequence` state, as seen by `gesture`.
    extern fn gtk_gesture_get_sequence_state(p_gesture: *Gesture, p_sequence: *gdk.EventSequence) gtk.EventSequenceState;
    pub const getSequenceState = gtk_gesture_get_sequence_state;

    /// Returns the list of `GdkEventSequences` currently being interpreted
    /// by `gesture`.
    extern fn gtk_gesture_get_sequences(p_gesture: *Gesture) *glib.List;
    pub const getSequences = gtk_gesture_get_sequences;

    /// Returns the user-defined window that receives the events
    /// handled by `gesture`. See `gtk.Gesture.setWindow` for more
    /// information.
    extern fn gtk_gesture_get_window(p_gesture: *Gesture) ?*gdk.Window;
    pub const getWindow = gtk_gesture_get_window;

    /// Adds `gesture` to the same group than `group_gesture`. Gestures
    /// are by default isolated in their own groups.
    ///
    /// When gestures are grouped, the state of `GdkEventSequences`
    /// is kept in sync for all of those, so calling `gtk.Gesture.setSequenceState`,
    /// on one will transfer the same value to the others.
    ///
    /// Groups also perform an "implicit grabbing" of sequences, if a
    /// `gdk.EventSequence` state is set to `GTK_EVENT_SEQUENCE_CLAIMED` on one group,
    /// every other gesture group attached to the same `gtk.Widget` will switch the
    /// state for that sequence to `GTK_EVENT_SEQUENCE_DENIED`.
    extern fn gtk_gesture_group(p_group_gesture: *Gesture, p_gesture: *gtk.Gesture) void;
    pub const group = gtk_gesture_group;

    /// Returns `TRUE` if `gesture` is currently handling events corresponding to
    /// `sequence`.
    extern fn gtk_gesture_handles_sequence(p_gesture: *Gesture, p_sequence: ?*gdk.EventSequence) c_int;
    pub const handlesSequence = gtk_gesture_handles_sequence;

    /// Returns `TRUE` if the gesture is currently active.
    /// A gesture is active meanwhile there are touch sequences
    /// interacting with it.
    extern fn gtk_gesture_is_active(p_gesture: *Gesture) c_int;
    pub const isActive = gtk_gesture_is_active;

    /// Returns `TRUE` if both gestures pertain to the same group.
    extern fn gtk_gesture_is_grouped_with(p_gesture: *Gesture, p_other: *gtk.Gesture) c_int;
    pub const isGroupedWith = gtk_gesture_is_grouped_with;

    /// Returns `TRUE` if the gesture is currently recognized.
    /// A gesture is recognized if there are as many interacting
    /// touch sequences as required by `gesture`, and `gtk.Gesture.signals.check`
    /// returned `TRUE` for the sequences being currently interpreted.
    extern fn gtk_gesture_is_recognized(p_gesture: *Gesture) c_int;
    pub const isRecognized = gtk_gesture_is_recognized;

    /// Sets the state of `sequence` in `gesture`. Sequences start
    /// in state `GTK_EVENT_SEQUENCE_NONE`, and whenever they change
    /// state, they can never go back to that state. Likewise,
    /// sequences in state `GTK_EVENT_SEQUENCE_DENIED` cannot turn
    /// back to a not denied state. With these rules, the lifetime
    /// of an event sequence is constrained to the next four:
    ///
    /// * None
    /// * None → Denied
    /// * None → Claimed
    /// * None → Claimed → Denied
    ///
    /// Note: Due to event handling ordering, it may be unsafe to
    /// set the state on another gesture within a `gtk.Gesture.signals.begin`
    /// signal handler, as the callback might be executed before
    /// the other gesture knows about the sequence. A safe way to
    /// perform this could be:
    ///
    /// ```
    /// static void
    /// first_gesture_begin_cb (GtkGesture       *first_gesture,
    ///                         GdkEventSequence *sequence,
    ///                         gpointer          user_data)
    /// {
    ///   gtk_gesture_set_sequence_state (first_gesture, sequence, GTK_EVENT_SEQUENCE_CLAIMED);
    ///   gtk_gesture_set_sequence_state (second_gesture, sequence, GTK_EVENT_SEQUENCE_DENIED);
    /// }
    ///
    /// static void
    /// second_gesture_begin_cb (GtkGesture       *second_gesture,
    ///                          GdkEventSequence *sequence,
    ///                          gpointer          user_data)
    /// {
    ///   if (gtk_gesture_get_sequence_state (first_gesture, sequence) == GTK_EVENT_SEQUENCE_CLAIMED)
    ///     gtk_gesture_set_sequence_state (second_gesture, sequence, GTK_EVENT_SEQUENCE_DENIED);
    /// }
    /// ```
    ///
    /// If both gestures are in the same group, just set the state on
    /// the gesture emitting the event, the sequence will be already
    /// be initialized to the group's global state when the second
    /// gesture processes the event.
    extern fn gtk_gesture_set_sequence_state(p_gesture: *Gesture, p_sequence: *gdk.EventSequence, p_state: gtk.EventSequenceState) c_int;
    pub const setSequenceState = gtk_gesture_set_sequence_state;

    /// Sets the state of all sequences that `gesture` is currently
    /// interacting with. See `gtk.Gesture.setSequenceState`
    /// for more details on sequence states.
    extern fn gtk_gesture_set_state(p_gesture: *Gesture, p_state: gtk.EventSequenceState) c_int;
    pub const setState = gtk_gesture_set_state;

    /// Sets a specific window to receive events about, so `gesture`
    /// will effectively handle only events targeting `window`, or
    /// a child of it. `window` must pertain to `gtk.EventController.getWidget`.
    extern fn gtk_gesture_set_window(p_gesture: *Gesture, p_window: ?*gdk.Window) void;
    pub const setWindow = gtk_gesture_set_window;

    /// Separates `gesture` into an isolated group.
    extern fn gtk_gesture_ungroup(p_gesture: *Gesture) void;
    pub const ungroup = gtk_gesture_ungroup;

    extern fn gtk_gesture_get_type() usize;
    pub const getGObjectType = gtk_gesture_get_type;

    extern fn g_object_ref(p_self: *gtk.Gesture) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Gesture) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Gesture, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.GestureDrag` is a `gtk.Gesture` implementation that recognizes drag
/// operations. The drag operation itself can be tracked throught the
/// `gtk.GestureDrag.signals.drag`-begin, `gtk.GestureDrag.signals.drag`-update and
/// `gtk.GestureDrag.signals.drag`-end signals, or the relevant coordinates be
/// extracted through `gtk.GestureDrag.getOffset` and
/// `gtk.GestureDrag.getStartPoint`.
pub const GestureDrag = opaque {
    pub const Parent = gtk.GestureSingle;
    pub const Implements = [_]type{};
    pub const Class = gtk.GestureDragClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// This signal is emitted whenever dragging starts.
        pub const drag_begin = struct {
            pub const name = "drag-begin";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_start_x: f64, p_start_y: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureDrag, p_instance))),
                    gobject.signalLookup("drag-begin", GestureDrag.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted whenever the dragging is finished.
        pub const drag_end = struct {
            pub const name = "drag-end";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_offset_x: f64, p_offset_y: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureDrag, p_instance))),
                    gobject.signalLookup("drag-end", GestureDrag.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted whenever the dragging point moves.
        pub const drag_update = struct {
            pub const name = "drag-update";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_offset_x: f64, p_offset_y: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureDrag, p_instance))),
                    gobject.signalLookup("drag-update", GestureDrag.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Returns a newly created `gtk.Gesture` that recognizes drags.
    extern fn gtk_gesture_drag_new(p_widget: *gtk.Widget) *gtk.GestureDrag;
    pub const new = gtk_gesture_drag_new;

    /// If the `gesture` is active, this function returns `TRUE` and
    /// fills in `x` and `y` with the coordinates of the current point,
    /// as an offset to the starting drag point.
    extern fn gtk_gesture_drag_get_offset(p_gesture: *GestureDrag, p_x: ?*f64, p_y: ?*f64) c_int;
    pub const getOffset = gtk_gesture_drag_get_offset;

    /// If the `gesture` is active, this function returns `TRUE`
    /// and fills in `x` and `y` with the drag start coordinates,
    /// in window-relative coordinates.
    extern fn gtk_gesture_drag_get_start_point(p_gesture: *GestureDrag, p_x: ?*f64, p_y: ?*f64) c_int;
    pub const getStartPoint = gtk_gesture_drag_get_start_point;

    extern fn gtk_gesture_drag_get_type() usize;
    pub const getGObjectType = gtk_gesture_drag_get_type;

    extern fn g_object_ref(p_self: *gtk.GestureDrag) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.GestureDrag) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GestureDrag, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.GestureLongPress` is a `gtk.Gesture` implementation able to recognize
/// long presses, triggering the `gtk.GestureLongPress.signals.pressed` after the
/// timeout is exceeded.
///
/// If the touchpoint is lifted before the timeout passes, or if it drifts
/// too far of the initial press point, the `gtk.GestureLongPress.signals.cancelled`
/// signal will be emitted.
pub const GestureLongPress = opaque {
    pub const Parent = gtk.GestureSingle;
    pub const Implements = [_]type{};
    pub const Class = gtk.GestureLongPressClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const delay_factor = struct {
            pub const name = "delay-factor";

            pub const Type = f64;
        };
    };

    pub const signals = struct {
        /// This signal is emitted whenever a press moved too far, or was released
        /// before `gtk.GestureLongPress.signals.pressed` happened.
        pub const cancelled = struct {
            pub const name = "cancelled";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureLongPress, p_instance))),
                    gobject.signalLookup("cancelled", GestureLongPress.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted whenever a press goes unmoved/unreleased longer than
        /// what the GTK+ defaults tell.
        pub const pressed = struct {
            pub const name = "pressed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_x: f64, p_y: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureLongPress, p_instance))),
                    gobject.signalLookup("pressed", GestureLongPress.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Returns a newly created `gtk.Gesture` that recognizes long presses.
    extern fn gtk_gesture_long_press_new(p_widget: *gtk.Widget) *gtk.GestureLongPress;
    pub const new = gtk_gesture_long_press_new;

    extern fn gtk_gesture_long_press_get_type() usize;
    pub const getGObjectType = gtk_gesture_long_press_get_type;

    extern fn g_object_ref(p_self: *gtk.GestureLongPress) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.GestureLongPress) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GestureLongPress, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.GestureMultiPress` is a `gtk.Gesture` implementation able to recognize
/// multiple clicks on a nearby zone, which can be listened for through the
/// `gtk.GestureMultiPress.signals.pressed` signal. Whenever time or distance between
/// clicks exceed the GTK+ defaults, `gtk.GestureMultiPress.signals.stopped` is emitted,
/// and the click counter is reset.
///
/// Callers may also restrict the area that is considered valid for a >1
/// touch/button press through `gtk.GestureMultiPress.setArea`, so any
/// click happening outside that area is considered to be a first click of
/// its own.
pub const GestureMultiPress = opaque {
    pub const Parent = gtk.GestureSingle;
    pub const Implements = [_]type{};
    pub const Class = gtk.GestureMultiPressClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// This signal is emitted whenever a button or touch press happens.
        pub const pressed = struct {
            pub const name = "pressed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_n_press: c_int, p_x: f64, p_y: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureMultiPress, p_instance))),
                    gobject.signalLookup("pressed", GestureMultiPress.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted when a button or touch is released. `n_press`
        /// will report the number of press that is paired to this event, note
        /// that `gtk.GestureMultiPress.signals.stopped` may have been emitted between the
        /// press and its release, `n_press` will only start over at the next press.
        pub const released = struct {
            pub const name = "released";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_n_press: c_int, p_x: f64, p_y: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureMultiPress, p_instance))),
                    gobject.signalLookup("released", GestureMultiPress.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted whenever any time/distance threshold has
        /// been exceeded.
        pub const stopped = struct {
            pub const name = "stopped";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureMultiPress, p_instance))),
                    gobject.signalLookup("stopped", GestureMultiPress.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Returns a newly created `gtk.Gesture` that recognizes single and multiple
    /// presses.
    extern fn gtk_gesture_multi_press_new(p_widget: *gtk.Widget) *gtk.GestureMultiPress;
    pub const new = gtk_gesture_multi_press_new;

    /// If an area was set through `gtk.GestureMultiPress.setArea`,
    /// this function will return `TRUE` and fill in `rect` with the
    /// press area. See `gtk.GestureMultiPress.setArea` for more
    /// details on what the press area represents.
    extern fn gtk_gesture_multi_press_get_area(p_gesture: *GestureMultiPress, p_rect: *gdk.Rectangle) c_int;
    pub const getArea = gtk_gesture_multi_press_get_area;

    /// If `rect` is non-`NULL`, the press area will be checked to be
    /// confined within the rectangle, otherwise the button count
    /// will be reset so the press is seen as being the first one.
    /// If `rect` is `NULL`, the area will be reset to an unrestricted
    /// state.
    ///
    /// Note: The rectangle is only used to determine whether any
    /// non-first click falls within the expected area. This is not
    /// akin to an input shape.
    extern fn gtk_gesture_multi_press_set_area(p_gesture: *GestureMultiPress, p_rect: ?*const gdk.Rectangle) void;
    pub const setArea = gtk_gesture_multi_press_set_area;

    extern fn gtk_gesture_multi_press_get_type() usize;
    pub const getGObjectType = gtk_gesture_multi_press_get_type;

    extern fn g_object_ref(p_self: *gtk.GestureMultiPress) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.GestureMultiPress) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GestureMultiPress, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.GesturePan` is a `gtk.Gesture` implementation able to recognize
/// pan gestures, those are drags that are locked to happen along one
/// axis. The axis that a `gtk.GesturePan` handles is defined at
/// construct time, and can be changed through
/// `gtk.GesturePan.setOrientation`.
///
/// When the gesture starts to be recognized, `gtk.GesturePan` will
/// attempt to determine as early as possible whether the sequence
/// is moving in the expected direction, and denying the sequence if
/// this does not happen.
///
/// Once a panning gesture along the expected axis is recognized,
/// the `gtk.GesturePan.signals.pan` signal will be emitted as input events
/// are received, containing the offset in the given axis.
pub const GesturePan = opaque {
    pub const Parent = gtk.GestureDrag;
    pub const Implements = [_]type{};
    pub const Class = gtk.GesturePanClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The expected orientation of pan gestures.
        pub const orientation = struct {
            pub const name = "orientation";

            pub const Type = gtk.Orientation;
        };
    };

    pub const signals = struct {
        /// This signal is emitted once a panning gesture along the
        /// expected axis is detected.
        pub const pan = struct {
            pub const name = "pan";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_direction: gtk.PanDirection, p_offset: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GesturePan, p_instance))),
                    gobject.signalLookup("pan", GesturePan.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Returns a newly created `gtk.Gesture` that recognizes pan gestures.
    extern fn gtk_gesture_pan_new(p_widget: *gtk.Widget, p_orientation: gtk.Orientation) *gtk.GesturePan;
    pub const new = gtk_gesture_pan_new;

    /// Returns the orientation of the pan gestures that this `gesture` expects.
    extern fn gtk_gesture_pan_get_orientation(p_gesture: *GesturePan) gtk.Orientation;
    pub const getOrientation = gtk_gesture_pan_get_orientation;

    /// Sets the orientation to be expected on pan gestures.
    extern fn gtk_gesture_pan_set_orientation(p_gesture: *GesturePan, p_orientation: gtk.Orientation) void;
    pub const setOrientation = gtk_gesture_pan_set_orientation;

    extern fn gtk_gesture_pan_get_type() usize;
    pub const getGObjectType = gtk_gesture_pan_get_type;

    extern fn g_object_ref(p_self: *gtk.GesturePan) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.GesturePan) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GesturePan, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.GestureRotate` is a `gtk.Gesture` implementation able to recognize
/// 2-finger rotations, whenever the angle between both handled sequences
/// changes, the `gtk.GestureRotate.signals.angle`-changed signal is emitted.
pub const GestureRotate = opaque {
    pub const Parent = gtk.Gesture;
    pub const Implements = [_]type{};
    pub const Class = gtk.GestureRotateClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// This signal is emitted when the angle between both tracked points
        /// changes.
        pub const angle_changed = struct {
            pub const name = "angle-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_angle: f64, p_angle_delta: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureRotate, p_instance))),
                    gobject.signalLookup("angle-changed", GestureRotate.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Returns a newly created `gtk.Gesture` that recognizes 2-touch
    /// rotation gestures.
    extern fn gtk_gesture_rotate_new(p_widget: *gtk.Widget) *gtk.GestureRotate;
    pub const new = gtk_gesture_rotate_new;

    /// If `gesture` is active, this function returns the angle difference
    /// in radians since the gesture was first recognized. If `gesture` is
    /// not active, 0 is returned.
    extern fn gtk_gesture_rotate_get_angle_delta(p_gesture: *GestureRotate) f64;
    pub const getAngleDelta = gtk_gesture_rotate_get_angle_delta;

    extern fn gtk_gesture_rotate_get_type() usize;
    pub const getGObjectType = gtk_gesture_rotate_get_type;

    extern fn g_object_ref(p_self: *gtk.GestureRotate) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.GestureRotate) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GestureRotate, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.GestureSingle` is a subclass of `gtk.Gesture`, optimized (although
/// not restricted) for dealing with mouse and single-touch gestures. Under
/// interaction, these gestures stick to the first interacting sequence, which
/// is accessible through `gtk.GestureSingle.getCurrentSequence` while the
/// gesture is being interacted with.
///
/// By default gestures react to both `GDK_BUTTON_PRIMARY` and touch
/// events, `gtk.GestureSingle.setTouchOnly` can be used to change the
/// touch behavior. Callers may also specify a different mouse button number
/// to interact with through `gtk.GestureSingle.setButton`, or react to any
/// mouse button by setting 0. While the gesture is active, the button being
/// currently pressed can be known through `gtk.GestureSingle.getCurrentButton`.
pub const GestureSingle = opaque {
    pub const Parent = gtk.Gesture;
    pub const Implements = [_]type{};
    pub const Class = gtk.GestureSingleClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Mouse button number to listen to, or 0 to listen for any button.
        pub const button = struct {
            pub const name = "button";

            pub const Type = c_uint;
        };

        /// Whether the gesture is exclusive. Exclusive gestures only listen to pointer
        /// and pointer emulated events.
        pub const exclusive = struct {
            pub const name = "exclusive";

            pub const Type = c_int;
        };

        /// Whether the gesture handles only touch events.
        pub const touch_only = struct {
            pub const name = "touch-only";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Returns the button number `gesture` listens for, or 0 if `gesture`
    /// reacts to any button press.
    extern fn gtk_gesture_single_get_button(p_gesture: *GestureSingle) c_uint;
    pub const getButton = gtk_gesture_single_get_button;

    /// Returns the button number currently interacting with `gesture`, or 0 if there
    /// is none.
    extern fn gtk_gesture_single_get_current_button(p_gesture: *GestureSingle) c_uint;
    pub const getCurrentButton = gtk_gesture_single_get_current_button;

    /// Returns the event sequence currently interacting with `gesture`.
    /// This is only meaningful if `gtk.Gesture.isActive` returns `TRUE`.
    extern fn gtk_gesture_single_get_current_sequence(p_gesture: *GestureSingle) ?*gdk.EventSequence;
    pub const getCurrentSequence = gtk_gesture_single_get_current_sequence;

    /// Gets whether a gesture is exclusive. For more information, see
    /// `gtk.GestureSingle.setExclusive`.
    extern fn gtk_gesture_single_get_exclusive(p_gesture: *GestureSingle) c_int;
    pub const getExclusive = gtk_gesture_single_get_exclusive;

    /// Returns `TRUE` if the gesture is only triggered by touch events.
    extern fn gtk_gesture_single_get_touch_only(p_gesture: *GestureSingle) c_int;
    pub const getTouchOnly = gtk_gesture_single_get_touch_only;

    /// Sets the button number `gesture` listens to. If non-0, every
    /// button press from a different button number will be ignored.
    /// Touch events implicitly match with button 1.
    extern fn gtk_gesture_single_set_button(p_gesture: *GestureSingle, p_button: c_uint) void;
    pub const setButton = gtk_gesture_single_set_button;

    /// Sets whether `gesture` is exclusive. An exclusive gesture will
    /// only handle pointer and "pointer emulated" touch events, so at
    /// any given time, there is only one sequence able to interact with
    /// those.
    extern fn gtk_gesture_single_set_exclusive(p_gesture: *GestureSingle, p_exclusive: c_int) void;
    pub const setExclusive = gtk_gesture_single_set_exclusive;

    /// If `touch_only` is `TRUE`, `gesture` will only handle events of type
    /// `GDK_TOUCH_BEGIN`, `GDK_TOUCH_UPDATE` or `GDK_TOUCH_END`. If `FALSE`,
    /// mouse events will be handled too.
    extern fn gtk_gesture_single_set_touch_only(p_gesture: *GestureSingle, p_touch_only: c_int) void;
    pub const setTouchOnly = gtk_gesture_single_set_touch_only;

    extern fn gtk_gesture_single_get_type() usize;
    pub const getGObjectType = gtk_gesture_single_get_type;

    extern fn g_object_ref(p_self: *gtk.GestureSingle) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.GestureSingle) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GestureSingle, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.GestureStylus` is a `gtk.Gesture` implementation specific to stylus
/// input. The provided signals just provide the basic information
pub const GestureStylus = opaque {
    pub const Parent = gtk.GestureSingle;
    pub const Implements = [_]type{};
    pub const Class = gtk.GestureStylusClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        pub const down = struct {
            pub const name = "down";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: f64, p_p0: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureStylus, p_instance))),
                    gobject.signalLookup("down", GestureStylus.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const motion = struct {
            pub const name = "motion";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: f64, p_p0: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureStylus, p_instance))),
                    gobject.signalLookup("motion", GestureStylus.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const proximity = struct {
            pub const name = "proximity";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: f64, p_p0: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureStylus, p_instance))),
                    gobject.signalLookup("proximity", GestureStylus.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const up = struct {
            pub const name = "up";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: f64, p_p0: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureStylus, p_instance))),
                    gobject.signalLookup("up", GestureStylus.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gtk.GestureStylus`.
    extern fn gtk_gesture_stylus_new(p_widget: *gtk.Widget) *gtk.GestureStylus;
    pub const new = gtk_gesture_stylus_new;

    /// Returns the current values for the requested `axes`. This function
    /// must be called from either the `gtk.GestureStylus.properties.down`,
    /// `gtk.GestureStylus.properties.motion`, `gtk.GestureStylus.properties.up` or `gtk.GestureStylus.properties.proximity`
    /// signals.
    extern fn gtk_gesture_stylus_get_axes(p_gesture: *GestureStylus, p_axes: [*]gdk.AxisUse, p_values: [*]*f64) c_int;
    pub const getAxes = gtk_gesture_stylus_get_axes;

    /// Returns the current value for the requested `axis`. This function
    /// must be called from either the `gtk.GestureStylus.properties.down`,
    /// `gtk.GestureStylus.properties.motion`, `gtk.GestureStylus.properties.up` or `gtk.GestureStylus.properties.proximity`
    /// signals.
    extern fn gtk_gesture_stylus_get_axis(p_gesture: *GestureStylus, p_axis: gdk.AxisUse, p_value: *f64) c_int;
    pub const getAxis = gtk_gesture_stylus_get_axis;

    /// Returns the `gdk.DeviceTool` currently driving input through this gesture.
    /// This function must be called from either the `gtk.GestureStylus.signals.down`,
    /// `gtk.GestureStylus.signals.motion`, `gtk.GestureStylus.signals.up` or `gtk.GestureStylus.signals.proximity`
    /// signal handlers.
    extern fn gtk_gesture_stylus_get_device_tool(p_gesture: *GestureStylus) ?*gdk.DeviceTool;
    pub const getDeviceTool = gtk_gesture_stylus_get_device_tool;

    extern fn gtk_gesture_stylus_get_type() usize;
    pub const getGObjectType = gtk_gesture_stylus_get_type;

    extern fn g_object_ref(p_self: *gtk.GestureStylus) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.GestureStylus) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GestureStylus, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.GestureSwipe` is a `gtk.Gesture` implementation able to recognize
/// swipes, after a press/move/.../move/release sequence happens, the
/// `gtk.GestureSwipe.signals.swipe` signal will be emitted, providing the velocity
/// and directionality of the sequence at the time it was lifted.
///
/// If the velocity is desired in intermediate points,
/// `gtk.GestureSwipe.getVelocity` can be called on eg. a
/// `gtk.Gesture.signals.update` handler.
///
/// All velocities are reported in pixels/sec units.
pub const GestureSwipe = opaque {
    pub const Parent = gtk.GestureSingle;
    pub const Implements = [_]type{};
    pub const Class = gtk.GestureSwipeClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// This signal is emitted when the recognized gesture is finished, velocity
        /// and direction are a product of previously recorded events.
        pub const swipe = struct {
            pub const name = "swipe";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_velocity_x: f64, p_velocity_y: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureSwipe, p_instance))),
                    gobject.signalLookup("swipe", GestureSwipe.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Returns a newly created `gtk.Gesture` that recognizes swipes.
    extern fn gtk_gesture_swipe_new(p_widget: *gtk.Widget) *gtk.GestureSwipe;
    pub const new = gtk_gesture_swipe_new;

    /// If the gesture is recognized, this function returns `TRUE` and fill in
    /// `velocity_x` and `velocity_y` with the recorded velocity, as per the
    /// last event(s) processed.
    extern fn gtk_gesture_swipe_get_velocity(p_gesture: *GestureSwipe, p_velocity_x: *f64, p_velocity_y: *f64) c_int;
    pub const getVelocity = gtk_gesture_swipe_get_velocity;

    extern fn gtk_gesture_swipe_get_type() usize;
    pub const getGObjectType = gtk_gesture_swipe_get_type;

    extern fn g_object_ref(p_self: *gtk.GestureSwipe) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.GestureSwipe) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GestureSwipe, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.GestureZoom` is a `gtk.Gesture` implementation able to recognize
/// pinch/zoom gestures, whenever the distance between both tracked
/// sequences changes, the `gtk.GestureZoom.signals.scale`-changed signal is
/// emitted to report the scale factor.
pub const GestureZoom = opaque {
    pub const Parent = gtk.Gesture;
    pub const Implements = [_]type{};
    pub const Class = gtk.GestureZoomClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// This signal is emitted whenever the distance between both tracked
        /// sequences changes.
        pub const scale_changed = struct {
            pub const name = "scale-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_scale: f64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(GestureZoom, p_instance))),
                    gobject.signalLookup("scale-changed", GestureZoom.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Returns a newly created `gtk.Gesture` that recognizes zoom
    /// in/out gestures (usually known as pinch/zoom).
    extern fn gtk_gesture_zoom_new(p_widget: *gtk.Widget) *gtk.GestureZoom;
    pub const new = gtk_gesture_zoom_new;

    /// If `gesture` is active, this function returns the zooming difference
    /// since the gesture was recognized (hence the starting point is
    /// considered 1:1). If `gesture` is not active, 1 is returned.
    extern fn gtk_gesture_zoom_get_scale_delta(p_gesture: *GestureZoom) f64;
    pub const getScaleDelta = gtk_gesture_zoom_get_scale_delta;

    extern fn gtk_gesture_zoom_get_type() usize;
    pub const getGObjectType = gtk_gesture_zoom_get_type;

    extern fn g_object_ref(p_self: *gtk.GestureZoom) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.GestureZoom) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *GestureZoom, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// GtkGrid is a container which arranges its child widgets in
/// rows and columns, with arbitrary positions and horizontal/vertical spans.
///
/// Children are added using `gtk.Grid.attach`. They can span multiple
/// rows or columns. It is also possible to add a child next to an
/// existing child, using `gtk.Grid.attachNextTo`. The behaviour of
/// GtkGrid when several children occupy the same grid cell is undefined.
///
/// GtkGrid can be used like a `gtk.Box` by just using `gtk.Container.add`,
/// which will place children next to each other in the direction determined
/// by the `gtk.Orientable.properties.orientation` property. However, if all you want is a
/// single row or column, then `gtk.Box` is the preferred widget.
///
/// # CSS nodes
///
/// GtkGrid uses a single CSS node with name grid.
pub const Grid = extern struct {
    pub const Parent = gtk.Container;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.GridClass;
    f_container: gtk.Container,
    f_priv: ?*gtk.GridPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const baseline_row = struct {
            pub const name = "baseline-row";

            pub const Type = c_int;
        };

        pub const column_homogeneous = struct {
            pub const name = "column-homogeneous";

            pub const Type = c_int;
        };

        pub const column_spacing = struct {
            pub const name = "column-spacing";

            pub const Type = c_int;
        };

        pub const row_homogeneous = struct {
            pub const name = "row-homogeneous";

            pub const Type = c_int;
        };

        pub const row_spacing = struct {
            pub const name = "row-spacing";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new grid widget.
    extern fn gtk_grid_new() *gtk.Grid;
    pub const new = gtk_grid_new;

    /// Adds a widget to the grid.
    ///
    /// The position of `child` is determined by `left` and `top`. The
    /// number of “cells” that `child` will occupy is determined by
    /// `width` and `height`.
    extern fn gtk_grid_attach(p_grid: *Grid, p_child: *gtk.Widget, p_left: c_int, p_top: c_int, p_width: c_int, p_height: c_int) void;
    pub const attach = gtk_grid_attach;

    /// Adds a widget to the grid.
    ///
    /// The widget is placed next to `sibling`, on the side determined by
    /// `side`. When `sibling` is `NULL`, the widget is placed in row (for
    /// left or right placement) or column 0 (for top or bottom placement),
    /// at the end indicated by `side`.
    ///
    /// Attaching widgets labeled [1], [2], [3] with `sibling` == `NULL` and
    /// `side` == `GTK_POS_LEFT` yields a layout of [3][2][1].
    extern fn gtk_grid_attach_next_to(p_grid: *Grid, p_child: *gtk.Widget, p_sibling: ?*gtk.Widget, p_side: gtk.PositionType, p_width: c_int, p_height: c_int) void;
    pub const attachNextTo = gtk_grid_attach_next_to;

    /// Returns which row defines the global baseline of `grid`.
    extern fn gtk_grid_get_baseline_row(p_grid: *Grid) c_int;
    pub const getBaselineRow = gtk_grid_get_baseline_row;

    /// Gets the child of `grid` whose area covers the grid
    /// cell whose upper left corner is at `left`, `top`.
    extern fn gtk_grid_get_child_at(p_grid: *Grid, p_left: c_int, p_top: c_int) ?*gtk.Widget;
    pub const getChildAt = gtk_grid_get_child_at;

    /// Returns whether all columns of `grid` have the same width.
    extern fn gtk_grid_get_column_homogeneous(p_grid: *Grid) c_int;
    pub const getColumnHomogeneous = gtk_grid_get_column_homogeneous;

    /// Returns the amount of space between the columns of `grid`.
    extern fn gtk_grid_get_column_spacing(p_grid: *Grid) c_uint;
    pub const getColumnSpacing = gtk_grid_get_column_spacing;

    /// Returns the baseline position of `row` as set
    /// by `gtk.Grid.setRowBaselinePosition` or the default value
    /// `GTK_BASELINE_POSITION_CENTER`.
    extern fn gtk_grid_get_row_baseline_position(p_grid: *Grid, p_row: c_int) gtk.BaselinePosition;
    pub const getRowBaselinePosition = gtk_grid_get_row_baseline_position;

    /// Returns whether all rows of `grid` have the same height.
    extern fn gtk_grid_get_row_homogeneous(p_grid: *Grid) c_int;
    pub const getRowHomogeneous = gtk_grid_get_row_homogeneous;

    /// Returns the amount of space between the rows of `grid`.
    extern fn gtk_grid_get_row_spacing(p_grid: *Grid) c_uint;
    pub const getRowSpacing = gtk_grid_get_row_spacing;

    /// Inserts a column at the specified position.
    ///
    /// Children which are attached at or to the right of this position
    /// are moved one column to the right. Children which span across this
    /// position are grown to span the new column.
    extern fn gtk_grid_insert_column(p_grid: *Grid, p_position: c_int) void;
    pub const insertColumn = gtk_grid_insert_column;

    /// Inserts a row or column at the specified position.
    ///
    /// The new row or column is placed next to `sibling`, on the side
    /// determined by `side`. If `side` is `GTK_POS_TOP` or `GTK_POS_BOTTOM`,
    /// a row is inserted. If `side` is `GTK_POS_LEFT` of `GTK_POS_RIGHT`,
    /// a column is inserted.
    extern fn gtk_grid_insert_next_to(p_grid: *Grid, p_sibling: *gtk.Widget, p_side: gtk.PositionType) void;
    pub const insertNextTo = gtk_grid_insert_next_to;

    /// Inserts a row at the specified position.
    ///
    /// Children which are attached at or below this position
    /// are moved one row down. Children which span across this
    /// position are grown to span the new row.
    extern fn gtk_grid_insert_row(p_grid: *Grid, p_position: c_int) void;
    pub const insertRow = gtk_grid_insert_row;

    /// Removes a column from the grid.
    ///
    /// Children that are placed in this column are removed,
    /// spanning children that overlap this column have their
    /// width reduced by one, and children after the column
    /// are moved to the left.
    extern fn gtk_grid_remove_column(p_grid: *Grid, p_position: c_int) void;
    pub const removeColumn = gtk_grid_remove_column;

    /// Removes a row from the grid.
    ///
    /// Children that are placed in this row are removed,
    /// spanning children that overlap this row have their
    /// height reduced by one, and children below the row
    /// are moved up.
    extern fn gtk_grid_remove_row(p_grid: *Grid, p_position: c_int) void;
    pub const removeRow = gtk_grid_remove_row;

    /// Sets which row defines the global baseline for the entire grid.
    /// Each row in the grid can have its own local baseline, but only
    /// one of those is global, meaning it will be the baseline in the
    /// parent of the `grid`.
    extern fn gtk_grid_set_baseline_row(p_grid: *Grid, p_row: c_int) void;
    pub const setBaselineRow = gtk_grid_set_baseline_row;

    /// Sets whether all columns of `grid` will have the same width.
    extern fn gtk_grid_set_column_homogeneous(p_grid: *Grid, p_homogeneous: c_int) void;
    pub const setColumnHomogeneous = gtk_grid_set_column_homogeneous;

    /// Sets the amount of space between columns of `grid`.
    extern fn gtk_grid_set_column_spacing(p_grid: *Grid, p_spacing: c_uint) void;
    pub const setColumnSpacing = gtk_grid_set_column_spacing;

    /// Sets how the baseline should be positioned on `row` of the
    /// grid, in case that row is assigned more space than is requested.
    extern fn gtk_grid_set_row_baseline_position(p_grid: *Grid, p_row: c_int, p_pos: gtk.BaselinePosition) void;
    pub const setRowBaselinePosition = gtk_grid_set_row_baseline_position;

    /// Sets whether all rows of `grid` will have the same height.
    extern fn gtk_grid_set_row_homogeneous(p_grid: *Grid, p_homogeneous: c_int) void;
    pub const setRowHomogeneous = gtk_grid_set_row_homogeneous;

    /// Sets the amount of space between rows of `grid`.
    extern fn gtk_grid_set_row_spacing(p_grid: *Grid, p_spacing: c_uint) void;
    pub const setRowSpacing = gtk_grid_set_row_spacing;

    extern fn gtk_grid_get_type() usize;
    pub const getGObjectType = gtk_grid_get_type;

    extern fn g_object_ref(p_self: *gtk.Grid) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.Grid) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Grid, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.HBox` is a container that organizes child widgets into a single row.
///
/// Use the `gtk.Box` packing interface to determine the arrangement,
/// spacing, width, and alignment of `gtk.HBox` children.
///
/// All children are allocated the same height.
///
/// GtkHBox has been deprecated. You can use `gtk.Box` instead, which is a
/// very quick and easy change. If you have derived your own classes from
/// GtkHBox, you can simply change the inheritance to derive directly
/// from `gtk.Box`. No further changes are needed, since the default
/// value of the `gtk.Orientable.properties.orientation` property is
/// `GTK_ORIENTATION_HORIZONTAL`.
///
/// If you have a grid-like layout composed of nested boxes, and you don’t
/// need first-child or last-child styling, the recommendation is to switch
/// to `gtk.Grid`. For more information about migrating to `gtk.Grid`, see
/// [Migrating from other containers to GtkGrid][gtk-migrating-GtkGrid].
pub const HBox = extern struct {
    pub const Parent = gtk.Box;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.HBoxClass;
    f_box: gtk.Box,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtk.HBox`.
    extern fn gtk_hbox_new(p_homogeneous: c_int, p_spacing: c_int) *gtk.HBox;
    pub const new = gtk_hbox_new;

    extern fn gtk_hbox_get_type() usize;
    pub const getGObjectType = gtk_hbox_get_type;

    extern fn g_object_ref(p_self: *gtk.HBox) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.HBox) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HBox, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const HButtonBox = extern struct {
    pub const Parent = gtk.ButtonBox;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.HButtonBoxClass;
    f_button_box: gtk.ButtonBox,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new horizontal button box.
    extern fn gtk_hbutton_box_new() *gtk.HButtonBox;
    pub const new = gtk_hbutton_box_new;

    extern fn gtk_hbutton_box_get_type() usize;
    pub const getGObjectType = gtk_hbutton_box_get_type;

    extern fn g_object_ref(p_self: *gtk.HButtonBox) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.HButtonBox) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HButtonBox, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The HPaned widget is a container widget with two
/// children arranged horizontally. The division between
/// the two panes is adjustable by the user by dragging
/// a handle. See `gtk.Paned` for details.
///
/// GtkHPaned has been deprecated, use `gtk.Paned` instead.
pub const HPaned = extern struct {
    pub const Parent = gtk.Paned;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.HPanedClass;
    f_paned: gtk.Paned,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Create a new `gtk.HPaned`
    extern fn gtk_hpaned_new() *gtk.HPaned;
    pub const new = gtk_hpaned_new;

    extern fn gtk_hpaned_get_type() usize;
    pub const getGObjectType = gtk_hpaned_get_type;

    extern fn g_object_ref(p_self: *gtk.HPaned) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.HPaned) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HPaned, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.HSV` is the “color wheel” part of a complete color selector widget.
/// It allows to select a color by determining its HSV components in an
/// intuitive way. Moving the selection around the outer ring changes the hue,
/// and moving the selection point inside the inner triangle changes value and
/// saturation.
///
/// `gtk.HSV` has been deprecated together with `gtk.ColorSelection`, where
/// it was used.
pub const HSV = extern struct {
    pub const Parent = gtk.Widget;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.HSVClass;
    f_parent_instance: gtk.Widget,
    f_priv: ?*gtk.HSVPrivate,

    pub const virtual_methods = struct {
        pub const changed = struct {
            pub fn call(p_class: anytype, p_hsv: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(HSV.Class, p_class).f_changed.?(gobject.ext.as(HSV, p_hsv));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_hsv: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(HSV.Class, p_class).f_changed = @ptrCast(p_implementation);
            }
        };

        pub const move = struct {
            pub fn call(p_class: anytype, p_hsv: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_type: gtk.DirectionType) void {
                return gobject.ext.as(HSV.Class, p_class).f_move.?(gobject.ext.as(HSV, p_hsv), p_type);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_hsv: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_type: gtk.DirectionType) callconv(.C) void) void {
                gobject.ext.as(HSV.Class, p_class).f_move = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        pub const changed = struct {
            pub const name = "changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(HSV, p_instance))),
                    gobject.signalLookup("changed", HSV.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        pub const move = struct {
            pub const name = "move";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: gtk.DirectionType, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(HSV, p_instance))),
                    gobject.signalLookup("move", HSV.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Converts a color from HSV space to RGB.
    ///
    /// Input values must be in the [0.0, 1.0] range;
    /// output values will be in the same range.
    extern fn gtk_hsv_to_rgb(p_h: f64, p_s: f64, p_v: f64, p_r: *f64, p_g: *f64, p_b: *f64) void;
    pub const toRgb = gtk_hsv_to_rgb;

    /// Creates a new HSV color selector.
    extern fn gtk_hsv_new() *gtk.HSV;
    pub const new = gtk_hsv_new;

    /// Queries the current color in an HSV color selector.
    /// Returned values will be in the [0.0, 1.0] range.
    extern fn gtk_hsv_get_color(p_hsv: *HSV, p_h: *f64, p_s: *f64, p_v: *f64) void;
    pub const getColor = gtk_hsv_get_color;

    /// Queries the size and ring width of an HSV color selector.
    extern fn gtk_hsv_get_metrics(p_hsv: *HSV, p_size: *c_int, p_ring_width: *c_int) void;
    pub const getMetrics = gtk_hsv_get_metrics;

    /// An HSV color selector can be said to be adjusting if multiple rapid
    /// changes are being made to its value, for example, when the user is
    /// adjusting the value with the mouse. This function queries whether
    /// the HSV color selector is being adjusted or not.
    extern fn gtk_hsv_is_adjusting(p_hsv: *HSV) c_int;
    pub const isAdjusting = gtk_hsv_is_adjusting;

    /// Sets the current color in an HSV color selector.
    /// Color component values must be in the [0.0, 1.0] range.
    extern fn gtk_hsv_set_color(p_hsv: *HSV, p_h: f64, p_s: f64, p_v: f64) void;
    pub const setColor = gtk_hsv_set_color;

    /// Sets the size and ring width of an HSV color selector.
    extern fn gtk_hsv_set_metrics(p_hsv: *HSV, p_size: c_int, p_ring_width: c_int) void;
    pub const setMetrics = gtk_hsv_set_metrics;

    extern fn gtk_hsv_get_type() usize;
    pub const getGObjectType = gtk_hsv_get_type;

    extern fn g_object_ref(p_self: *gtk.HSV) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.HSV) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HSV, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.HScale` widget is used to allow the user to select a value using
/// a horizontal slider. To create one, use `gtk.HScale.newWithRange`.
///
/// The position to show the current value, and the number of decimal places
/// shown can be set using the parent `gtk.Scale` class’s functions.
///
/// GtkHScale has been deprecated, use `gtk.Scale` instead.
pub const HScale = extern struct {
    pub const Parent = gtk.Scale;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.HScaleClass;
    f_scale: gtk.Scale,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtk.HScale`.
    extern fn gtk_hscale_new(p_adjustment: ?*gtk.Adjustment) *gtk.HScale;
    pub const new = gtk_hscale_new;

    /// Creates a new horizontal scale widget that lets the user input a
    /// number between `min` and `max` (including `min` and `max`) with the
    /// increment `step`.  `step` must be nonzero; it’s the distance the
    /// slider moves when using the arrow keys to adjust the scale value.
    ///
    /// Note that the way in which the precision is derived works best if `step`
    /// is a power of ten. If the resulting precision is not suitable for your
    /// needs, use `gtk.Scale.setDigits` to correct it.
    extern fn gtk_hscale_new_with_range(p_min: f64, p_max: f64, p_step: f64) *gtk.HScale;
    pub const newWithRange = gtk_hscale_new_with_range;

    extern fn gtk_hscale_get_type() usize;
    pub const getGObjectType = gtk_hscale_get_type;

    extern fn g_object_ref(p_self: *gtk.HScale) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.HScale) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HScale, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.HScrollbar` widget is a widget arranged horizontally creating a
/// scrollbar. See `gtk.Scrollbar` for details on
/// scrollbars. `gtk.Adjustment` pointers may be added to handle the
/// adjustment of the scrollbar or it may be left `NULL` in which case one
/// will be created for you. See `gtk.Scrollbar` for a description of what the
/// fields in an adjustment represent for a scrollbar.
///
/// GtkHScrollbar has been deprecated, use `gtk.Scrollbar` instead.
pub const HScrollbar = extern struct {
    pub const Parent = gtk.Scrollbar;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.HScrollbarClass;
    f_scrollbar: gtk.Scrollbar,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new horizontal scrollbar.
    extern fn gtk_hscrollbar_new(p_adjustment: ?*gtk.Adjustment) *gtk.HScrollbar;
    pub const new = gtk_hscrollbar_new;

    extern fn gtk_hscrollbar_get_type() usize;
    pub const getGObjectType = gtk_hscrollbar_get_type;

    extern fn g_object_ref(p_self: *gtk.HScrollbar) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.HScrollbar) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HScrollbar, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.HSeparator` widget is a horizontal separator, used to group the
/// widgets within a window. It displays a horizontal line with a shadow to
/// make it appear sunken into the interface.
///
/// > The `gtk.HSeparator` widget is not used as a separator within menus.
/// > To create a separator in a menu create an empty `gtk.SeparatorMenuItem`
/// > widget using `gtk.SeparatorMenuItem.new` and add it to the menu with
/// > `gtk.MenuShell.append`.
///
/// GtkHSeparator has been deprecated, use `gtk.Separator` instead.
pub const HSeparator = extern struct {
    pub const Parent = gtk.Separator;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable, gtk.Orientable };
    pub const Class = gtk.HSeparatorClass;
    f_separator: gtk.Separator,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gtk.HSeparator`.
    extern fn gtk_hseparator_new() *gtk.HSeparator;
    pub const new = gtk_hseparator_new;

    extern fn gtk_hseparator_get_type() usize;
    pub const getGObjectType = gtk_hseparator_get_type;

    extern fn g_object_ref(p_self: *gtk.HSeparator) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.HSeparator) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HSeparator, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `gtk.HandleBox` widget allows a portion of a window to be "torn
/// off". It is a bin widget which displays its child and a handle that
/// the user can drag to tear off a separate window (the “float
/// window”) containing the child widget. A thin
/// “ghost” is drawn in the original location of the
/// handlebox. By dragging the separate window back to its original
/// location, it can be reattached.
///
/// When reattaching, the ghost and float window, must be aligned
/// along one of the edges, the “snap edge”.
/// This either can be specified by the application programmer
/// explicitly, or GTK+ will pick a reasonable default based
/// on the handle position.
///
/// To make detaching and reattaching the handlebox as minimally confusing
/// as possible to the user, it is important to set the snap edge so that
/// the snap edge does not move when the handlebox is deattached. For
/// instance, if the handlebox is packed at the bottom of a VBox, then
/// when the handlebox is detached, the bottom edge of the handlebox's
/// allocation will remain fixed as the height of the handlebox shrinks,
/// so the snap edge should be set to `GTK_POS_BOTTOM`.
///
/// > `gtk.HandleBox` has been deprecated. It is very specialized, lacks features
/// > to make it useful and most importantly does not fit well into modern
/// > application design. Do not use it. There is no replacement.
pub const HandleBox = extern struct {
    pub const Parent = gtk.Bin;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.HandleBoxClass;
    f_bin: gtk.Bin,
    f_priv: ?*gtk.HandleBoxPrivate,

    pub const virtual_methods = struct {
        /// Signal emitted when the contents of the handlebox
        ///    are reattached to the main window. Deprecated: 3.4.
        pub const child_attached = struct {
            pub fn call(p_class: anytype, p_handle_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget) void {
                return gobject.ext.as(HandleBox.Class, p_class).f_child_attached.?(gobject.ext.as(HandleBox, p_handle_box), p_child);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_handle_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget) callconv(.C) void) void {
                gobject.ext.as(HandleBox.Class, p_class).f_child_attached = @ptrCast(p_implementation);
            }
        };

        /// Signal emitted when the contents of the handlebox
        ///    are detached from the main window. Deprecated: 3.4.
        pub const child_detached = struct {
            pub fn call(p_class: anytype, p_handle_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget) void {
                return gobject.ext.as(HandleBox.Class, p_class).f_child_detached.?(gobject.ext.as(HandleBox, p_handle_box), p_child);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_handle_box: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_child: *gtk.Widget) callconv(.C) void) void {
                gobject.ext.as(HandleBox.Class, p_class).f_child_detached = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const child_detached = struct {
            pub const name = "child-detached";

            pub const Type = c_int;
        };

        pub const handle_position = struct {
            pub const name = "handle-position";

            pub const Type = gtk.PositionType;
        };

        pub const shadow_type = struct {
            pub const name = "shadow-type";

            pub const Type = gtk.ShadowType;
        };

        pub const snap_edge = struct {
            pub const name = "snap-edge";

            pub const Type = gtk.PositionType;
        };

        pub const snap_edge_set = struct {
            pub const name = "snap-edge-set";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// This signal is emitted when the contents of the
        /// handlebox are reattached to the main window.
        pub const child_attached = struct {
            pub const name = "child-attached";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_widget: *gtk.Widget, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(HandleBox, p_instance))),
                    gobject.signalLookup("child-attached", HandleBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted when the contents of the
        /// handlebox are detached from the main window.
        pub const child_detached = struct {
            pub const name = "child-detached";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_widget: *gtk.Widget, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(HandleBox, p_instance))),
                    gobject.signalLookup("child-detached", HandleBox.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Create a new handle box.
    extern fn gtk_handle_box_new() *gtk.HandleBox;
    pub const new = gtk_handle_box_new;

    /// Whether the handlebox’s child is currently detached.
    extern fn gtk_handle_box_get_child_detached(p_handle_box: *HandleBox) c_int;
    pub const getChildDetached = gtk_handle_box_get_child_detached;

    /// Gets the handle position of the handle box. See
    /// `gtk.HandleBox.setHandlePosition`.
    extern fn gtk_handle_box_get_handle_position(p_handle_box: *HandleBox) gtk.PositionType;
    pub const getHandlePosition = gtk_handle_box_get_handle_position;

    /// Gets the type of shadow drawn around the handle box. See
    /// `gtk.HandleBox.setShadowType`.
    extern fn gtk_handle_box_get_shadow_type(p_handle_box: *HandleBox) gtk.ShadowType;
    pub const getShadowType = gtk_handle_box_get_shadow_type;

    /// Gets the edge used for determining reattachment of the handle box.
    /// See `gtk.HandleBox.setSnapEdge`.
    extern fn gtk_handle_box_get_snap_edge(p_handle_box: *HandleBox) gtk.PositionType;
    pub const getSnapEdge = gtk_handle_box_get_snap_edge;

    /// Sets the side of the handlebox where the handle is drawn.
    extern fn gtk_handle_box_set_handle_position(p_handle_box: *HandleBox, p_position: gtk.PositionType) void;
    pub const setHandlePosition = gtk_handle_box_set_handle_position;

    /// Sets the type of shadow to be drawn around the border
    /// of the handle box.
    extern fn gtk_handle_box_set_shadow_type(p_handle_box: *HandleBox, p_type: gtk.ShadowType) void;
    pub const setShadowType = gtk_handle_box_set_shadow_type;

    /// Sets the snap edge of a handlebox. The snap edge is
    /// the edge of the detached child that must be aligned
    /// with the corresponding edge of the “ghost” left
    /// behind when the child was detached to reattach
    /// the torn-off window. Usually, the snap edge should
    /// be chosen so that it stays in the same place on
    /// the screen when the handlebox is torn off.
    ///
    /// If the snap edge is not set, then an appropriate value
    /// will be guessed from the handle position. If the
    /// handle position is `GTK_POS_RIGHT` or `GTK_POS_LEFT`,
    /// then the snap edge will be `GTK_POS_TOP`, otherwise
    /// it will be `GTK_POS_LEFT`.
    extern fn gtk_handle_box_set_snap_edge(p_handle_box: *HandleBox, p_edge: gtk.PositionType) void;
    pub const setSnapEdge = gtk_handle_box_set_snap_edge;

    extern fn gtk_handle_box_get_type() usize;
    pub const getGObjectType = gtk_handle_box_get_type;

    extern fn g_object_ref(p_self: *gtk.HandleBox) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.HandleBox) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HandleBox, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// GtkHeaderBar is similar to a horizontal `gtk.Box`. It allows children to
/// be placed at the start or the end. In addition, it allows a title and
/// subtitle to be displayed. The title will be centered with respect to
/// the width of the box, even if the children at either side take up
/// different amounts of space. The height of the titlebar will be
/// set to provide sufficient space for the subtitle, even if none is
/// currently set. If a subtitle is not needed, the space reservation
/// can be turned off with `gtk.HeaderBar.setHasSubtitle`.
///
/// GtkHeaderBar can add typical window frame controls, such as minimize,
/// maximize and close buttons, or the window icon.
///
/// For these reasons, GtkHeaderBar is the natural choice for use as the custom
/// titlebar widget of a `gtk.Window` (see `gtk.Window.setTitlebar`), as it gives
/// features typical of titlebars while allowing the addition of child widgets.
pub const HeaderBar = extern struct {
    pub const Parent = gtk.Container;
    pub const Implements = [_]type{ atk.ImplementorIface, gtk.Buildable };
    pub const Class = gtk.HeaderBarClass;
    f_container: gtk.Container,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const custom_title = struct {
            pub const name = "custom-title";

            pub const Type = ?*gtk.Widget;
        };

        /// The decoration layout for buttons. If this property is
        /// not set, the `gtk.Settings.properties.gtk`-decoration-layout setting
        /// is used.
        ///
        /// See `gtk.HeaderBar.setDecorationLayout` for information
        /// about the format of this string.
        pub const decoration_layout = struct {
            pub const name = "decoration-layout";

            pub const Type = ?[*:0]u8;
        };

        /// Set to `TRUE` if `gtk.HeaderBar.properties.decoration`-layout is set.
        pub const decoration_layout_set = struct {
            pub const name = "decoration-layout-set";

            pub const Type = c_int;
        };

        /// If `TRUE`, reserve space for a subtitle, even if none
        /// is currently set.
        pub const has_subtitle = struct {
            pub const name = "has-subtitle";

            pub const Type = c_int;
        };

        /// Whether to show window decorations.
        ///
        /// Which buttons are actually shown and where is determined
        /// by the `gtk.HeaderBar.properties.decoration`-layout property, and by
        /// the state of the window (e.g. a close button will not be
        /// shown if the window can't be closed).
        pub const show_close_button = struct {
            pub const name = "show-close-button";

            pub const Type = c_int;
        };

        pub const spacing = struct {
            pub const name = "spacing";

            pub const Type = c_int;
        };

        pub const subtitle = struct {
            pub const name = "subtitle";

            pub const Type = ?[*:0]u8;
        };

        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gtk.HeaderBar` widget.
    extern fn gtk_header_bar_new() *gtk.HeaderBar;
    pub const new = gtk_header_bar_new;

    /// Retrieves the custom title widget of the header. See
    /// `gtk.HeaderBar.setCustomTitle`.
    extern fn gtk_header_bar_get_custom_title(p_bar: *HeaderBar) ?*gtk.Widget;
    pub const getCustomTitle = gtk_header_bar_get_custom_title;

    /// Gets the decoration layout set with
    /// `gtk.HeaderBar.setDecorationLayout`.
    extern fn gtk_header_bar_get_decoration_layout(p_bar: *HeaderBar) [*:0]const u8;
    pub const getDecorationLayout = gtk_header_bar_get_decoration_layout;

    /// Retrieves whether the header bar reserves space for
    /// a subtitle, regardless if one is currently set or not.
    extern fn gtk_header_bar_get_has_subtitle(p_bar: *HeaderBar) c_int;
    pub const getHasSubtitle = gtk_header_bar_get_has_subtitle;

    /// Returns whether this header bar shows the standard window
    /// decorations.
    extern fn gtk_header_bar_get_show_close_button(p_bar: *HeaderBar) c_int;
    pub const getShowCloseButton = gtk_header_bar_get_show_close_button;

    /// Retrieves the subtitle of the header. See `gtk.HeaderBar.setSubtitle`.
    extern fn gtk_header_bar_get_subtitle(p_bar: *HeaderBar) ?[*:0]const u8;
    pub const getSubtitle = gtk_header_bar_get_subtitle;

    /// Retrieves the title of the header. See `gtk.HeaderBar.setTitle`.
    extern fn gtk_header_bar_get_title(p_bar: *HeaderBar) ?[*:0]const u8;
    pub const getTitle = gtk_header_bar_get_title;

    /// Adds `child` to `bar`, packed with reference to the
    /// end of the `bar`.
    extern fn gtk_header_bar_pack_end(p_bar: *HeaderBar, p_child: *gtk.Widget) void;
    pub const packEnd = gtk_header_bar_pack_end;

    /// Adds `child` to `bar`, packed with reference to the
    /// start of the `bar`.
    extern fn gtk_header_bar_pack_start(p_bar: *HeaderBar, p_child: *gtk.Widget) void;
    pub const packStart = gtk_header_bar_pack_start;

    /// Sets a custom title for the `gtk.HeaderBar`.
    ///
    /// The title should help a user identify the current view. This
    /// supersedes any title set by `gtk.HeaderBar.setTitle` or
    /// `gtk.HeaderBar.setSubtitle`. To achieve the same style as
    /// the builtin title and subtitle, use the “title” and “subtitle”
    /// style classes.
    ///
    /// You should set the custom title to `NULL`, for the header title
    /// label to be visible again.
    extern fn gtk_header_bar_set_custom_title(p_bar: *HeaderBar, p_title_widget: ?*gtk.Widget) void;
    pub const setCustomTitle = gtk_header_bar_set_custom_title;

    /// Sets the decoration layout for this header bar, overriding
    /// the `gtk.Settings.properties.gtk`-decoration-layout setting.
    ///
    /// There can be valid reasons for overriding the setting, such
    /// as a header bar design that does not allow for buttons to take
    /// room on the right, or only offers room for a single close button.
    /// Split header bars are another example for overriding the
    /// setting.
    ///
    /// The format of the string is button names, separated by commas.
    /// A colon separates the buttons that should appear on the left
    /// from those on the right. Recognized button names are minimize,
    /// maximize, close, icon (the window icon) and menu (a menu button
    /// for the fallback app menu).
    ///
    /// For example, “menu:minimize,maximize,close” specifies a menu
    /// on the left, and minimize, maximize and close buttons on the right.
    extern fn gtk_header_bar_set_decoration_layout(p_bar: *HeaderBar, p_layout: ?[*:0]const u8) void;
    pub const setDecorationLayout = gtk_header_bar_set_decoration_layout;

    /// Sets whether the header bar should reserve space
    /// for a subtitle, even if none is currently set.
    extern fn gtk_header_bar_set_has_subtitle(p_bar: *HeaderBar, p_setting: c_int) void;
    pub const setHasSubtitle = gtk_header_bar_set_has_subtitle;

    /// Sets whether this header bar shows the standard window decorations,
    /// including close, maximize, and minimize.
    extern fn gtk_header_bar_set_show_close_button(p_bar: *HeaderBar, p_setting: c_int) void;
    pub const setShowCloseButton = gtk_header_bar_set_show_close_button;

    /// Sets the subtitle of the `gtk.HeaderBar`. The title should give a user
    /// an additional detail to help him identify the current view.
    ///
    /// Note that GtkHeaderBar by default reserves room for the subtitle,
    /// even if none is currently set. If this is not desired, set the
    /// `gtk.HeaderBar.properties.has`-subtitle property to `FALSE`.
    extern fn gtk_header_bar_set_subtitle(p_bar: *HeaderBar, p_subtitle: ?[*:0]const u8) void;
    pub const setSubtitle = gtk_header_bar_set_subtitle;

    /// Sets the title of the `gtk.HeaderBar`. The title should help a user
    /// identify the current view. A good title should not include the
    /// application name.
    extern fn gtk_header_bar_set_title(p_bar: *HeaderBar, p_title: ?[*:0]const u8) void;
    pub const setTitle = gtk_header_bar_set_title;

    extern fn gtk_header_bar_get_type() usize;
    pub const getGObjectType = gtk_header_bar_get_type;

    extern fn g_object_ref(p_self: *gtk.HeaderBar) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.HeaderBar) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HeaderBar, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const HeaderBarAccessible = extern struct {
    pub const Parent = gtk.ContainerAccessible;
    pub const Implements = [_]type{atk.Component};
    pub const Class = gtk.HeaderBarAccessibleClass;
    f_parent: gtk.ContainerAccessible,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn gtk_header_bar_accessible_get_type() usize;
    pub const getGObjectType = gtk_header_bar_accessible_get_type;

    extern fn g_object_ref(p_self: *gtk.HeaderBarAccessible) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gtk.HeaderBarAccessible) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *HeaderBarAccessible, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gtk.IMContext` defines the interface for GTK+ input methods. An input method
/// is used by GTK+ text input widgets like `gtk.Entry` to map from key events to
/// Unicode character strings.
///
/// The default input method can be set programmatically via the
/// `gtk.Settings.properties.gtk`-im-module GtkSettings property. Alternatively, you may set
/// the GTK_IM_MODULE environment variable as documented in
/// [Running GTK+ Applications][gtk-running].
///
/// The `gtk.Entry` `gtk.Entry.properties.im`-module and `gtk.TextView` `gtk.TextView.properties.im`-module
/// properties may also be used to set input methods for specific widget
/// instances. For instance, a certain entry widget might be expected to contain
/// certain characters which would be easier to input with a certain input
/// method.
///
/// An input method may consume multiple key events in sequence and finally
/// output the composed result. This is called preediting, and an input method
/// may provide feedback about this process by displaying the intermediate
/// composition states as preedit text. For instance, the default GTK+ input
/// method implements the input of arbitrary Unicode code points by holding down
/// the Control and Shift keys and then typing “U” followed by the hexadecimal
/// digits of the code point.  When releasing the Control and Shift keys,
/// preediting ends and the character is inserted as text. Ctrl+Shift+u20AC for
/// example results in the € sign.
///
/// Additional input methods can be made available for use by GTK+ widgets as
/// loadable modules. An input method module is a small shared library which
/// implements a subclass of `gtk.IMContext` or `gtk.IMContextSimple` and exports
/// these four functions:
///
/// ```
/// void im_module_init(GTypeModule *module);
/// ```
/// This function should register the `gobject.Type` of the `gtk.IMContext` subclass which
/// implements the input method by means of `gobject.TypeModule.registerType`. Note
/// that `gobject.typeRegisterStatic` cannot be used as the type needs to be
/// registered dynamically.
///
/// ```
/// void im_module_exit(void);
/// ```
/// Here goes any cleanup code your input method might require on module unload.
///
/// ```
/// void im_module_list(const GtkIMContextInfo ***contexts, int *n_contexts)
/// {
///   *contexts = info_list;
///   *n_contexts = G_N_ELEMENTS (info_list);
/// }
/// ```
/// This function returns the list of input methods provided by the module. The
/// example implementation above shows a common solution and simply returns a
/// pointer to statically defined array of `gtk.IMContextInfo` items for each
/// provided input method.
///
/// ```
/// GtkIMContext * im_module_create(const gchar *context_id);
/// ```
/// This function should return a pointer to a newly created instance of the
/// `gtk.IMContext` subclass identified by `context_id`. The context ID is the same
/// as specified in the `gtk.IMContextInfo` array returned by `im_module_list`.
///
/// After a new loadable input method module has been installed on the system,
/// the configuration file `gtk.immodules` needs to be
/// regenerated by [gtk-query-immodules-3.0][gtk-query-immodules-3.0],
/// in order for the new input method to become available to GTK+ applications.
pub const IMContext = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gtk.IMContextClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {
        /// Default handler of the `gtk.IMContext.signals.commit` signal.
        pub const commit = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_str: [*:0]const u8) void {
                return gobject.ext.as(IMContext.Class, p_class).f_commit.?(gobject.ext.as(IMContext, p_context), p_str);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_str: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_commit = @ptrCast(p_implementation);
            }
        };

        /// Asks the widget that the input context is attached to to delete
        /// characters around the cursor position by emitting the
        /// GtkIMContext::delete_surrounding signal. Note that `offset` and `n_chars`
        /// are in characters not in bytes which differs from the usage other
        /// places in `gtk.IMContext`.
        ///
        /// In order to use this function, you should first call
        /// `gtk.IMContext.getSurrounding` to get the current context, and
        /// call this function immediately afterwards to make sure that you
        /// know what you are deleting. You should also account for the fact
        /// that even if the signal was handled, the input context might not
        /// have deleted all the characters that were requested to be deleted.
        ///
        /// This function is used by an input method that wants to make
        /// subsitutions in the existing text in response to new input. It is
        /// not useful for applications.
        pub const delete_surrounding = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_offset: c_int, p_n_chars: c_int) c_int {
                return gobject.ext.as(IMContext.Class, p_class).f_delete_surrounding.?(gobject.ext.as(IMContext, p_context), p_offset, p_n_chars);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_offset: c_int, p_n_chars: c_int) callconv(.C) c_int) void {
                gobject.ext.as(IMContext.Class, p_class).f_delete_surrounding = @ptrCast(p_implementation);
            }
        };

        /// Allow an input method to internally handle key press and release
        /// events. If this function returns `TRUE`, then no further processing
        /// should be done for this key event.
        pub const filter_keypress = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: *gdk.EventKey) c_int {
                return gobject.ext.as(IMContext.Class, p_class).f_filter_keypress.?(gobject.ext.as(IMContext, p_context), p_event);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: *gdk.EventKey) callconv(.C) c_int) void {
                gobject.ext.as(IMContext.Class, p_class).f_filter_keypress = @ptrCast(p_implementation);
            }
        };

        /// Notify the input method that the widget to which this
        /// input context corresponds has gained focus. The input method
        /// may, for example, change the displayed feedback to reflect
        /// this change.
        pub const focus_in = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(IMContext.Class, p_class).f_focus_in.?(gobject.ext.as(IMContext, p_context));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_focus_in = @ptrCast(p_implementation);
            }
        };

        /// Notify the input method that the widget to which this
        /// input context corresponds has lost focus. The input method
        /// may, for example, change the displayed feedback or reset the contexts
        /// state to reflect this change.
        pub const focus_out = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(IMContext.Class, p_class).f_focus_out.?(gobject.ext.as(IMContext, p_context));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_focus_out = @ptrCast(p_implementation);
            }
        };

        /// Retrieve the current preedit string for the input context,
        /// and a list of attributes to apply to the string.
        /// This string should be displayed inserted at the insertion
        /// point.
        pub const get_preedit_string = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_str: *[*:0]u8, p_attrs: **pango.AttrList, p_cursor_pos: *c_int) void {
                return gobject.ext.as(IMContext.Class, p_class).f_get_preedit_string.?(gobject.ext.as(IMContext, p_context), p_str, p_attrs, p_cursor_pos);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_str: *[*:0]u8, p_attrs: **pango.AttrList, p_cursor_pos: *c_int) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_get_preedit_string = @ptrCast(p_implementation);
            }
        };

        /// Retrieves context around the insertion point. Input methods
        /// typically want context in order to constrain input text based on
        /// existing text; this is important for languages such as Thai where
        /// only some sequences of characters are allowed.
        ///
        /// This function is implemented by emitting the
        /// GtkIMContext::retrieve_surrounding signal on the input method; in
        /// response to this signal, a widget should provide as much context as
        /// is available, up to an entire paragraph, by calling
        /// `gtk.IMContext.setSurrounding`. Note that there is no obligation
        /// for a widget to respond to the ::retrieve_surrounding signal, so input
        /// methods must be prepared to function without context.
        pub const get_surrounding = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: *[*:0]u8, p_cursor_index: *c_int) c_int {
                return gobject.ext.as(IMContext.Class, p_class).f_get_surrounding.?(gobject.ext.as(IMContext, p_context), p_text, p_cursor_index);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: *[*:0]u8, p_cursor_index: *c_int) callconv(.C) c_int) void {
                gobject.ext.as(IMContext.Class, p_class).f_get_surrounding = @ptrCast(p_implementation);
            }
        };

        /// Default handler of the `gtk.IMContext.signals.preedit`-changed
        ///   signal.
        pub const preedit_changed = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(IMContext.Class, p_class).f_preedit_changed.?(gobject.ext.as(IMContext, p_context));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_preedit_changed = @ptrCast(p_implementation);
            }
        };

        /// Default handler of the `gtk.IMContext.signals.preedit`-end signal.
        pub const preedit_end = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(IMContext.Class, p_class).f_preedit_end.?(gobject.ext.as(IMContext, p_context));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_preedit_end = @ptrCast(p_implementation);
            }
        };

        /// Default handler of the `gtk.IMContext.signals.preedit`-start signal.
        pub const preedit_start = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(IMContext.Class, p_class).f_preedit_start.?(gobject.ext.as(IMContext, p_context));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_preedit_start = @ptrCast(p_implementation);
            }
        };

        /// Notify the input method that a change such as a change in cursor
        /// position has been made. This will typically cause the input
        /// method to clear the preedit state.
        pub const reset = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(IMContext.Class, p_class).f_reset.?(gobject.ext.as(IMContext, p_context));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_reset = @ptrCast(p_implementation);
            }
        };

        /// Default handler of the
        ///   `gtk.IMContext.signals.retrieve`-surrounding signal.
        pub const retrieve_surrounding = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(IMContext.Class, p_class).f_retrieve_surrounding.?(gobject.ext.as(IMContext, p_context));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(IMContext.Class, p_class).f_retrieve_surrounding = @ptrCast(p_implementation);
            }
        };

        /// Set the client window for the input context; this is the
        /// `gdk.Window` in which the input appears. This window is
        /// used in order to correctly position status windows, and may
        /// also be used for purposes internal to the input method.
        pub const set_client_window = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_window: ?*gdk.Window) void {
                return gobject.ext.as(IMContext.Class, p_class).f_set_client_window.?(gobject.ext.as(IMContext, p_context), p_window);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_window: ?*gdk.Window) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_set_client_window = @ptrCast(p_implementation);
            }
        };

        /// Notify the input method that a change in cursor
        /// position has been made. The location is relative to the client
        /// window.
        pub const set_cursor_location = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_area: *gdk.Rectangle) void {
                return gobject.ext.as(IMContext.Class, p_class).f_set_cursor_location.?(gobject.ext.as(IMContext, p_context), p_area);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_area: *gdk.Rectangle) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_set_cursor_location = @ptrCast(p_implementation);
            }
        };

        /// Sets surrounding context around the insertion point and preedit
        /// string. This function is expected to be called in response to the
        /// GtkIMContext::retrieve_surrounding signal, and will likely have no
        /// effect if called at other times.
        pub const set_surrounding = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: [*:0]const u8, p_len: c_int, p_cursor_index: c_int) void {
                return gobject.ext.as(IMContext.Class, p_class).f_set_surrounding.?(gobject.ext.as(IMContext, p_context), p_text, p_len, p_cursor_index);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_text: [*:0]const u8, p_len: c_int, p_cursor_index: c_int) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_set_surrounding = @ptrCast(p_implementation);
            }
        };

        /// Sets whether the IM context should use the preedit string
        /// to display feedback. If `use_preedit` is FALSE (default
        /// is TRUE), then the IM context may use some other method to display
        /// feedback, such as displaying it in a child of the root window.
        pub const set_use_preedit = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_use_preedit: c_int) void {
                return gobject.ext.as(IMContext.Class, p_class).f_set_use_preedit.?(gobject.ext.as(IMContext, p_context), p_use_preedit);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_use_preedit: c_int) callconv(.C) void) void {
                gobject.ext.as(IMContext.Class, p_class).f_set_use_preedit = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        pub const input_hints = struct {
            pub const name = "input-hints";

            pub const Type = gtk.InputHints;
        };

        pub c