﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/directconnect/DirectConnectRequest.h>
#include <aws/directconnect/DirectConnect_EXPORTS.h>
#include <aws/directconnect/model/Tag.h>

#include <utility>

namespace Aws {
namespace DirectConnect {
namespace Model {

/**
 */
class AllocateHostedConnectionRequest : public DirectConnectRequest {
 public:
  AWS_DIRECTCONNECT_API AllocateHostedConnectionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AllocateHostedConnection"; }

  AWS_DIRECTCONNECT_API Aws::String SerializePayload() const override;

  AWS_DIRECTCONNECT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the interconnect or LAG.</p>
   */
  inline const Aws::String& GetConnectionId() const { return m_connectionId; }
  inline bool ConnectionIdHasBeenSet() const { return m_connectionIdHasBeenSet; }
  template <typename ConnectionIdT = Aws::String>
  void SetConnectionId(ConnectionIdT&& value) {
    m_connectionIdHasBeenSet = true;
    m_connectionId = std::forward<ConnectionIdT>(value);
  }
  template <typename ConnectionIdT = Aws::String>
  AllocateHostedConnectionRequest& WithConnectionId(ConnectionIdT&& value) {
    SetConnectionId(std::forward<ConnectionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Amazon Web Services account ID of the customer for the
   * connection.</p>
   */
  inline const Aws::String& GetOwnerAccount() const { return m_ownerAccount; }
  inline bool OwnerAccountHasBeenSet() const { return m_ownerAccountHasBeenSet; }
  template <typename OwnerAccountT = Aws::String>
  void SetOwnerAccount(OwnerAccountT&& value) {
    m_ownerAccountHasBeenSet = true;
    m_ownerAccount = std::forward<OwnerAccountT>(value);
  }
  template <typename OwnerAccountT = Aws::String>
  AllocateHostedConnectionRequest& WithOwnerAccount(OwnerAccountT&& value) {
    SetOwnerAccount(std::forward<OwnerAccountT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The bandwidth of the connection. The possible values are 50Mbps, 100Mbps,
   * 200Mbps, 300Mbps, 400Mbps, 500Mbps, 1Gbps, 2Gbps, 5Gbps, 10Gbps, and 25Gbps.
   * Note that only those Direct Connect Partners who have met specific requirements
   * are allowed to create a 1Gbps, 2Gbps, 5Gbps, 10Gbps, or 25Gbps hosted
   * connection. </p>
   */
  inline const Aws::String& GetBandwidth() const { return m_bandwidth; }
  inline bool BandwidthHasBeenSet() const { return m_bandwidthHasBeenSet; }
  template <typename BandwidthT = Aws::String>
  void SetBandwidth(BandwidthT&& value) {
    m_bandwidthHasBeenSet = true;
    m_bandwidth = std::forward<BandwidthT>(value);
  }
  template <typename BandwidthT = Aws::String>
  AllocateHostedConnectionRequest& WithBandwidth(BandwidthT&& value) {
    SetBandwidth(std::forward<BandwidthT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the hosted connection.</p>
   */
  inline const Aws::String& GetConnectionName() const { return m_connectionName; }
  inline bool ConnectionNameHasBeenSet() const { return m_connectionNameHasBeenSet; }
  template <typename ConnectionNameT = Aws::String>
  void SetConnectionName(ConnectionNameT&& value) {
    m_connectionNameHasBeenSet = true;
    m_connectionName = std::forward<ConnectionNameT>(value);
  }
  template <typename ConnectionNameT = Aws::String>
  AllocateHostedConnectionRequest& WithConnectionName(ConnectionNameT&& value) {
    SetConnectionName(std::forward<ConnectionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The dedicated VLAN provisioned to the hosted connection.</p>
   */
  inline int GetVlan() const { return m_vlan; }
  inline bool VlanHasBeenSet() const { return m_vlanHasBeenSet; }
  inline void SetVlan(int value) {
    m_vlanHasBeenSet = true;
    m_vlan = value;
  }
  inline AllocateHostedConnectionRequest& WithVlan(int value) {
    SetVlan(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags associated with the connection.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  AllocateHostedConnectionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  AllocateHostedConnectionRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_connectionId;

  Aws::String m_ownerAccount;

  Aws::String m_bandwidth;

  Aws::String m_connectionName;

  int m_vlan{0};

  Aws::Vector<Tag> m_tags;
  bool m_connectionIdHasBeenSet = false;
  bool m_ownerAccountHasBeenSet = false;
  bool m_bandwidthHasBeenSet = false;
  bool m_connectionNameHasBeenSet = false;
  bool m_vlanHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace DirectConnect
}  // namespace Aws
