﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/pinpoint/Pinpoint_EXPORTS.h>
#include <aws/pinpoint/model/Action.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Pinpoint {
namespace Model {

/**
 * <p>Specifies the default settings and content for a push notification that's
 * sent directly to an endpoint.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/pinpoint-2016-12-01/DefaultPushNotificationMessage">AWS
 * API Reference</a></p>
 */
class DefaultPushNotificationMessage {
 public:
  AWS_PINPOINT_API DefaultPushNotificationMessage() = default;
  AWS_PINPOINT_API DefaultPushNotificationMessage(Aws::Utils::Json::JsonView jsonValue);
  AWS_PINPOINT_API DefaultPushNotificationMessage& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PINPOINT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The default action to occur if a recipient taps the push notification. Valid
   * values are:</p> <ul><li><p>OPEN_APP - Your app opens or it becomes the
   * foreground app if it was sent to the background. This is the default
   * action.</p></li> <li><p>DEEP_LINK - Your app opens and displays a designated
   * user interface in the app. This setting uses the deep-linking features of the
   * iOS and Android platforms.</p></li> <li><p>URL - The default mobile browser on
   * the recipient's device opens and loads the web page at a URL that you
   * specify.</p></li></ul>
   */
  inline Action GetAction() const { return m_action; }
  inline bool ActionHasBeenSet() const { return m_actionHasBeenSet; }
  inline void SetAction(Action value) {
    m_actionHasBeenSet = true;
    m_action = value;
  }
  inline DefaultPushNotificationMessage& WithAction(Action value) {
    SetAction(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The default body of the notification message.</p>
   */
  inline const Aws::String& GetBody() const { return m_body; }
  inline bool BodyHasBeenSet() const { return m_bodyHasBeenSet; }
  template <typename BodyT = Aws::String>
  void SetBody(BodyT&& value) {
    m_bodyHasBeenSet = true;
    m_body = std::forward<BodyT>(value);
  }
  template <typename BodyT = Aws::String>
  DefaultPushNotificationMessage& WithBody(BodyT&& value) {
    SetBody(std::forward<BodyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The JSON data payload to use for the default push notification, if the
   * notification is a silent push notification. This payload is added to the
   * data.pinpoint.jsonBody object of the notification.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetData() const { return m_data; }
  inline bool DataHasBeenSet() const { return m_dataHasBeenSet; }
  template <typename DataT = Aws::Map<Aws::String, Aws::String>>
  void SetData(DataT&& value) {
    m_dataHasBeenSet = true;
    m_data = std::forward<DataT>(value);
  }
  template <typename DataT = Aws::Map<Aws::String, Aws::String>>
  DefaultPushNotificationMessage& WithData(DataT&& value) {
    SetData(std::forward<DataT>(value));
    return *this;
  }
  template <typename DataKeyT = Aws::String, typename DataValueT = Aws::String>
  DefaultPushNotificationMessage& AddData(DataKeyT&& key, DataValueT&& value) {
    m_dataHasBeenSet = true;
    m_data.emplace(std::forward<DataKeyT>(key), std::forward<DataValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether the default notification is a silent push notification,
   * which is a push notification that doesn't display on a recipient's device.
   * Silent push notifications can be used for cases such as updating an app's
   * configuration or delivering messages to an in-app notification center.</p>
   */
  inline bool GetSilentPush() const { return m_silentPush; }
  inline bool SilentPushHasBeenSet() const { return m_silentPushHasBeenSet; }
  inline void SetSilentPush(bool value) {
    m_silentPushHasBeenSet = true;
    m_silentPush = value;
  }
  inline DefaultPushNotificationMessage& WithSilentPush(bool value) {
    SetSilentPush(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The default message variables to use in the notification message. You can
   * override the default variables with individual address variables.</p>
   */
  inline const Aws::Map<Aws::String, Aws::Vector<Aws::String>>& GetSubstitutions() const { return m_substitutions; }
  inline bool SubstitutionsHasBeenSet() const { return m_substitutionsHasBeenSet; }
  template <typename SubstitutionsT = Aws::Map<Aws::String, Aws::Vector<Aws::String>>>
  void SetSubstitutions(SubstitutionsT&& value) {
    m_substitutionsHasBeenSet = true;
    m_substitutions = std::forward<SubstitutionsT>(value);
  }
  template <typename SubstitutionsT = Aws::Map<Aws::String, Aws::Vector<Aws::String>>>
  DefaultPushNotificationMessage& WithSubstitutions(SubstitutionsT&& value) {
    SetSubstitutions(std::forward<SubstitutionsT>(value));
    return *this;
  }
  template <typename SubstitutionsKeyT = Aws::String, typename SubstitutionsValueT = Aws::Vector<Aws::String>>
  DefaultPushNotificationMessage& AddSubstitutions(SubstitutionsKeyT&& key, SubstitutionsValueT&& value) {
    m_substitutionsHasBeenSet = true;
    m_substitutions.emplace(std::forward<SubstitutionsKeyT>(key), std::forward<SubstitutionsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The default title to display above the notification message on a recipient's
   * device.</p>
   */
  inline const Aws::String& GetTitle() const { return m_title; }
  inline bool TitleHasBeenSet() const { return m_titleHasBeenSet; }
  template <typename TitleT = Aws::String>
  void SetTitle(TitleT&& value) {
    m_titleHasBeenSet = true;
    m_title = std::forward<TitleT>(value);
  }
  template <typename TitleT = Aws::String>
  DefaultPushNotificationMessage& WithTitle(TitleT&& value) {
    SetTitle(std::forward<TitleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The default URL to open in a recipient's default mobile browser, if a
   * recipient taps the push notification and the value of the Action property is
   * URL.</p>
   */
  inline const Aws::String& GetUrl() const { return m_url; }
  inline bool UrlHasBeenSet() const { return m_urlHasBeenSet; }
  template <typename UrlT = Aws::String>
  void SetUrl(UrlT&& value) {
    m_urlHasBeenSet = true;
    m_url = std::forward<UrlT>(value);
  }
  template <typename UrlT = Aws::String>
  DefaultPushNotificationMessage& WithUrl(UrlT&& value) {
    SetUrl(std::forward<UrlT>(value));
    return *this;
  }
  ///@}
 private:
  Action m_action{Action::NOT_SET};

  Aws::String m_body;

  Aws::Map<Aws::String, Aws::String> m_data;

  bool m_silentPush{false};

  Aws::Map<Aws::String, Aws::Vector<Aws::String>> m_substitutions;

  Aws::String m_title;

  Aws::String m_url;
  bool m_actionHasBeenSet = false;
  bool m_bodyHasBeenSet = false;
  bool m_dataHasBeenSet = false;
  bool m_silentPushHasBeenSet = false;
  bool m_substitutionsHasBeenSet = false;
  bool m_titleHasBeenSet = false;
  bool m_urlHasBeenSet = false;
};

}  // namespace Model
}  // namespace Pinpoint
}  // namespace Aws
