﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/pinpoint/PinpointRequest.h>
#include <aws/pinpoint/Pinpoint_EXPORTS.h>
#include <aws/pinpoint/model/EndpointRequest.h>

#include <utility>

namespace Aws {
namespace Pinpoint {
namespace Model {

/**
 */
class UpdateEndpointRequest : public PinpointRequest {
 public:
  AWS_PINPOINT_API UpdateEndpointRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateEndpoint"; }

  AWS_PINPOINT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique identifier for the application. This identifier is displayed as
   * the <b>Project ID</b> on the Amazon Pinpoint console.</p>
   */
  inline const Aws::String& GetApplicationId() const { return m_applicationId; }
  inline bool ApplicationIdHasBeenSet() const { return m_applicationIdHasBeenSet; }
  template <typename ApplicationIdT = Aws::String>
  void SetApplicationId(ApplicationIdT&& value) {
    m_applicationIdHasBeenSet = true;
    m_applicationId = std::forward<ApplicationIdT>(value);
  }
  template <typename ApplicationIdT = Aws::String>
  UpdateEndpointRequest& WithApplicationId(ApplicationIdT&& value) {
    SetApplicationId(std::forward<ApplicationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The case insensitive unique identifier for the endpoint. The identifier can't
   * contain <code>$</code>, <code>{</code> or <code>}</code>.</p>
   */
  inline const Aws::String& GetEndpointId() const { return m_endpointId; }
  inline bool EndpointIdHasBeenSet() const { return m_endpointIdHasBeenSet; }
  template <typename EndpointIdT = Aws::String>
  void SetEndpointId(EndpointIdT&& value) {
    m_endpointIdHasBeenSet = true;
    m_endpointId = std::forward<EndpointIdT>(value);
  }
  template <typename EndpointIdT = Aws::String>
  UpdateEndpointRequest& WithEndpointId(EndpointIdT&& value) {
    SetEndpointId(std::forward<EndpointIdT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const EndpointRequest& GetEndpointRequest() const { return m_endpointRequest; }
  inline bool EndpointRequestHasBeenSet() const { return m_endpointRequestHasBeenSet; }
  template <typename EndpointRequestT = EndpointRequest>
  void SetEndpointRequest(EndpointRequestT&& value) {
    m_endpointRequestHasBeenSet = true;
    m_endpointRequest = std::forward<EndpointRequestT>(value);
  }
  template <typename EndpointRequestT = EndpointRequest>
  UpdateEndpointRequest& WithEndpointRequest(EndpointRequestT&& value) {
    SetEndpointRequest(std::forward<EndpointRequestT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationId;

  Aws::String m_endpointId;

  EndpointRequest m_endpointRequest;
  bool m_applicationIdHasBeenSet = false;
  bool m_endpointIdHasBeenSet = false;
  bool m_endpointRequestHasBeenSet = false;
};

}  // namespace Model
}  // namespace Pinpoint
}  // namespace Aws
