"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SamlAuthentication = void 0;
var _querystring = require("querystring");
var _security_cookie = require("../../../session/security_cookie");
var _routes = require("./routes");
var _authentication_type = require("../authentication_type");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
var _http = require("../../../../../../src/core/server/http");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          *   Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          *   Licensed under the Apache License, Version 2.0 (the "License").
                                                                                                                                                                                                                                                                                                                          *   You may not use this file except in compliance with the License.
                                                                                                                                                                                                                                                                                                                          *   A copy of the License is located at
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          *       http://www.apache.org/licenses/LICENSE-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          *   or in the "license" file accompanying this file. This file is distributed
                                                                                                                                                                                                                                                                                                                          *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
                                                                                                                                                                                                                                                                                                                          *   express or implied. See the License for the specific language governing
                                                                                                                                                                                                                                                                                                                          *   permissions and limitations under the License.
                                                                                                                                                                                                                                                                                                                          */
class SamlAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);
    _defineProperty(this, "type", _common.AuthType.SAML);
    // Check if we can get the previous tenant information from the expired cookie.
    _defineProperty(this, "redirectSAMlCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/saml/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });
  }
  generateNextUrl(request) {
    let path = (0, _http.getRedirectUrl)({
      request,
      basePath: this.coreSetup.http.basePath.serverBasePath,
      nextUrl: request.url.pathname || '/app/opensearch-dashboards'
    });
    if (request.url.search) {
      path += request.url.search;
    }
    return (0, _querystring.escape)(path);
  }
  async init() {
    this.createExtraStorage();
    const samlAuthRoutes = new _routes.SamlAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.securityClient, this.coreSetup);
    samlAuthRoutes.setupRoutes();
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const extraCookiePrefix = this.config.saml.extra_storage.cookie_prefix;
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= this.config.saml.extra_storage.additional_cookies; i++) {
      hapiServer.states.add(extraCookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions(logger) {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.saml.extra_storage.cookie_prefix,
      additionalCookies: this.config.saml.extra_storage.additional_cookies,
      logger
    };
  }
  requestIncludesAuthInfo(request) {
    return request.headers[SamlAuthentication.AUTH_HEADER_NAME] ? true : false;
  }
  async getAdditionalAuthHeader(request) {
    return {};
  }

  // SAML expiry time is set by the IDP and returned via the security backend. Keep alive should not modify this value.
  getKeepAliveExpiry(cookie, request) {
    return cookie.expiryTime;
  }
  getCookie(request, authInfo) {
    const authorizationHeaderValue = request.headers[SamlAuthentication.AUTH_HEADER_NAME];
    (0, _cookie_splitter.setExtraAuthStorage)(request, authorizationHeaderValue, this.getExtraAuthStorageOptions(this.logger));
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: _common.AuthType.SAML,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }

  // Can be improved to check if the token is expiring.
  async isValidCookie(cookie, request) {
    var _cookie$credentials;
    return cookie.authType === _common.AuthType.SAML && cookie.username && cookie.expiryTime && (((_cookie$credentials = cookie.credentials) === null || _cookie$credentials === void 0 ? void 0 : _cookie$credentials.authHeaderValue) || this.getExtraAuthStorageValue(request, cookie));
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectSAMlCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;
    let extraValue = '';
    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions(this.logger));
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;
    const headers = {};
    if ((_cookie$credentials3 = cookie.credentials) !== null && _cookie$credentials3 !== void 0 && _cookie$credentials3.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        headers[SamlAuthentication.AUTH_HEADER_NAME] = extraAuthStorageValue;
      } catch (error) {
        this.logger.error(error);
        // @todo Re-throw?
        // throw error;
      }
    } else {
      var _cookie$credentials4;
      headers[SamlAuthentication.AUTH_HEADER_NAME] = (_cookie$credentials4 = cookie.credentials) === null || _cookie$credentials4 === void 0 ? void 0 : _cookie$credentials4.authHeaderValue;
    }
    return headers;
  }
}
exports.SamlAuthentication = SamlAuthentication;
_defineProperty(SamlAuthentication, "AUTH_HEADER_NAME", 'authorization');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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