/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.security.ssl;

import com.google.common.collect.ImmutableMap;
import io.netty.handler.ssl.ClientAuth;
import java.io.IOException;
import java.nio.file.Path;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.crypto.Cipher;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.OpenSearchException;
import org.opensearch.common.collect.Tuple;
import org.opensearch.common.settings.Setting;
import org.opensearch.common.settings.Settings;
import org.opensearch.env.Environment;
import org.opensearch.security.ssl.SslConfiguration;
import org.opensearch.security.ssl.SslContextHandler;
import org.opensearch.security.ssl.config.CertType;
import org.opensearch.security.ssl.config.KeyStoreConfiguration;
import org.opensearch.security.ssl.config.SslCertificatesLoader;
import org.opensearch.security.ssl.config.SslParameters;
import org.opensearch.security.ssl.config.TrustStoreConfiguration;
import org.opensearch.security.ssl.util.SSLConfigConstants;
import org.opensearch.transport.AuxTransport;
import org.opensearch.watcher.FileChangesListener;
import org.opensearch.watcher.FileWatcher;
import org.opensearch.watcher.ResourceWatcher;
import org.opensearch.watcher.ResourceWatcherService;

public class SslSettingsManager {
    private static final Logger LOGGER = LogManager.getLogger(SslSettingsManager.class);
    private final Map<CertType, SslContextHandler> sslSettingsContexts;

    public SslSettingsManager(Environment environment) {
        this.sslSettingsContexts = this.buildSslContexts(environment);
    }

    public Optional<SslConfiguration> sslConfiguration(CertType certType) {
        return Optional.ofNullable(this.sslSettingsContexts.get(certType)).map(SslContextHandler::sslConfiguration);
    }

    public Optional<SslContextHandler> sslContextHandler(CertType certType) {
        return Optional.ofNullable(this.sslSettingsContexts.get(certType));
    }

    private Map<CertType, SslContextHandler> buildSslContexts(Environment environment) {
        ImmutableMap.Builder contexts = new ImmutableMap.Builder();
        Map<CertType, SslConfiguration> configurations = this.loadConfigurations(environment);
        configurations.forEach((cert, sslConfig) -> {
            if (cert == CertType.TRANSPORT) {
                Optional.ofNullable((SslConfiguration)configurations.get(CertType.TRANSPORT)).ifPresentOrElse(sslConfiguration -> {
                    contexts.put((Object)CertType.TRANSPORT, (Object)new SslContextHandler((SslConfiguration)sslConfiguration));
                    SslConfiguration transportClientConfiguration = Optional.ofNullable((SslConfiguration)configurations.get(CertType.TRANSPORT_CLIENT)).orElse((SslConfiguration)sslConfiguration);
                    contexts.put((Object)CertType.TRANSPORT_CLIENT, (Object)new SslContextHandler(transportClientConfiguration, true));
                }, () -> LOGGER.warn("SSL Configuration for Transport Layer hasn't been set"));
                return;
            }
            if (cert == CertType.TRANSPORT_CLIENT) {
                return;
            }
            Optional.ofNullable((SslConfiguration)configurations.get(cert)).ifPresentOrElse(sslConfiguration -> contexts.put(cert, (Object)new SslContextHandler((SslConfiguration)sslConfiguration)), () -> LOGGER.warn("SSL Configuration for {} Layer hasn't been set", (Object)cert.id()));
        });
        return contexts.build();
    }

    public synchronized void reloadSslContext(CertType certType) {
        this.sslContextHandler(certType).ifPresentOrElse(sscContextHandler -> {
            try {
                if (sscContextHandler.reloadSslContext()) {
                    LOGGER.info("{} SSL context reloaded", (Object)certType.id());
                }
            }
            catch (CertificateException e) {
                throw new OpenSearchException((Throwable)e);
            }
        }, () -> LOGGER.error("Missing SSL Context for {}", (Object)certType.id()));
    }

    private Map<CertType, SslConfiguration> loadConfigurations(Environment environment) {
        Settings settings = environment.settings();
        ImmutableMap.Builder configurationBuilder = ImmutableMap.builder();
        if (settings.getByPrefix(CertType.HTTP.sslSettingPrefix()).isEmpty() && settings.getByPrefix(CertType.TRANSPORT.sslSettingPrefix()).isEmpty()) {
            throw new OpenSearchException("No SSL configuration found", new Object[0]);
        }
        this.jceWarnings();
        for (String auxType : (List)AuxTransport.AUX_TRANSPORT_TYPES_SETTING.get(environment.settings())) {
            CertType auxCert = new CertType("plugins.security.ssl.aux." + auxType + ".");
            Setting auxEnabled = SSLConfigConstants.SECURITY_SSL_AUX_ENABLED.getConcreteSettingForNamespace(auxType);
            CertType.CERT_TYPE_REGISTRY.register(auxCert);
            if (!((Boolean)auxEnabled.get(settings)).booleanValue() || this.clientNode(settings)) continue;
            this.validateSettings(auxCert, settings, false);
            SslParameters auxSslParameters = SslParameters.loader(auxCert, settings).load();
            Tuple<TrustStoreConfiguration, KeyStoreConfiguration> auxTrustAndKeyStore = new SslCertificatesLoader(auxCert.sslSettingPrefix()).loadConfiguration(environment);
            configurationBuilder.put((Object)auxCert, (Object)new SslConfiguration(auxSslParameters, (TrustStoreConfiguration)auxTrustAndKeyStore.v1(), (KeyStoreConfiguration)auxTrustAndKeyStore.v2()));
            LOGGER.info("TLS {} Provider                    : {}", (Object)auxCert.id(), (Object)auxSslParameters.provider());
            LOGGER.info("Enabled TLS protocols for {} layer : {}", (Object)auxCert.id(), auxSslParameters.allowedProtocols());
        }
        boolean httpEnabled = settings.getAsBoolean(CertType.HTTP.sslSettingPrefix() + "enabled", Boolean.valueOf(false));
        if (httpEnabled && !this.clientNode(settings)) {
            this.validateSettings(CertType.HTTP, settings, false);
            SslParameters httpSslParameters = SslParameters.loader(CertType.HTTP, settings).load();
            Tuple<TrustStoreConfiguration, KeyStoreConfiguration> httpTrustAndKeyStore = new SslCertificatesLoader(CertType.HTTP.sslSettingPrefix()).loadConfiguration(environment);
            configurationBuilder.put((Object)CertType.HTTP, (Object)new SslConfiguration(httpSslParameters, (TrustStoreConfiguration)httpTrustAndKeyStore.v1(), (KeyStoreConfiguration)httpTrustAndKeyStore.v2()));
            LOGGER.info("TLS HTTP Provider                    : {}", (Object)httpSslParameters.provider());
            LOGGER.info("Enabled TLS protocols for HTTP layer : {}", httpSslParameters.allowedProtocols());
        }
        Settings transportSettings = settings.getByPrefix(CertType.TRANSPORT.sslSettingPrefix());
        SslParameters transportSslParameters = SslParameters.loader(CertType.TRANSPORT, settings).load();
        if (transportSettings.getAsBoolean("enabled", Boolean.valueOf(true)).booleanValue()) {
            if (this.hasExtendedKeyUsageEnabled(transportSettings)) {
                this.validateTransportSettings(transportSettings);
                Tuple<TrustStoreConfiguration, KeyStoreConfiguration> transportServerTrustAndKeyStore = new SslCertificatesLoader(CertType.TRANSPORT.sslSettingPrefix(), "server.").loadConfiguration(environment);
                configurationBuilder.put((Object)CertType.TRANSPORT, (Object)new SslConfiguration(transportSslParameters, (TrustStoreConfiguration)transportServerTrustAndKeyStore.v1(), (KeyStoreConfiguration)transportServerTrustAndKeyStore.v2()));
                Tuple<TrustStoreConfiguration, KeyStoreConfiguration> transportClientTrustAndKeyStore = new SslCertificatesLoader(CertType.TRANSPORT.sslSettingPrefix(), "client.").loadConfiguration(environment);
                configurationBuilder.put((Object)CertType.TRANSPORT_CLIENT, (Object)new SslConfiguration(transportSslParameters, (TrustStoreConfiguration)transportClientTrustAndKeyStore.v1(), (KeyStoreConfiguration)transportClientTrustAndKeyStore.v2()));
            } else {
                this.validateTransportSettings(transportSettings);
                Tuple<TrustStoreConfiguration, KeyStoreConfiguration> transportTrustAndKeyStore = new SslCertificatesLoader(CertType.TRANSPORT.sslSettingPrefix()).loadConfiguration(environment);
                configurationBuilder.put((Object)CertType.TRANSPORT, (Object)new SslConfiguration(transportSslParameters, (TrustStoreConfiguration)transportTrustAndKeyStore.v1(), (KeyStoreConfiguration)transportTrustAndKeyStore.v2()));
            }
            LOGGER.info("TLS Transport Client Provider             : {}", (Object)transportSslParameters.provider());
            LOGGER.info("TLS Transport Server Provider             : {}", (Object)transportSslParameters.provider());
            LOGGER.info("Enabled TLS protocols for Transport layer : {}", transportSslParameters.allowedProtocols());
        }
        return configurationBuilder.build();
    }

    public void addSslConfigurationsChangeListener(ResourceWatcherService resourceWatcherService) {
        for (Path directoryToMonitor : this.directoriesToMonitor()) {
            FileWatcher fileWatcher = new FileWatcher(directoryToMonitor);
            fileWatcher.addListener((Object)new FileChangesListener(){

                public void onFileCreated(Path file) {
                    this.onFileChanged(file);
                }

                public void onFileDeleted(Path file) {
                    this.onFileChanged(file);
                }

                public void onFileChanged(Path file) {
                    for (Map.Entry<CertType, SslContextHandler> e : SslSettingsManager.this.sslSettingsContexts.entrySet()) {
                        CertType certType = e.getKey();
                        SslConfiguration sslConfiguration = e.getValue().sslConfiguration();
                        if (!sslConfiguration.dependentFiles().contains(file)) continue;
                        SslSettingsManager.this.reloadSslContext(certType);
                    }
                }
            });
            try {
                resourceWatcherService.add((ResourceWatcher)fileWatcher, ResourceWatcherService.Frequency.HIGH);
                LOGGER.info("Added SSL configuration change listener for: {}", (Object)directoryToMonitor);
            }
            catch (IOException e) {
                throw new OpenSearchException("Couldn't add SSL configurations change listener", (Throwable)e, new Object[0]);
            }
        }
    }

    private Set<Path> directoriesToMonitor() {
        return this.sslSettingsContexts.values().stream().map(SslContextHandler::sslConfiguration).flatMap(c -> c.dependentFiles().stream()).map(Path::getParent).collect(Collectors.toSet());
    }

    private boolean clientNode(Settings settings) {
        return !"node".equals(settings.get("client.type"));
    }

    private void validateSettings(CertType certType, Settings settings, boolean enabled_default) {
        Settings certSettings = settings.getByPrefix(certType.sslSettingPrefix());
        if (certSettings.isEmpty()) {
            return;
        }
        if (!certSettings.getAsBoolean("enabled", Boolean.valueOf(enabled_default)).booleanValue()) {
            return;
        }
        if (this.hasPemStoreSettings(certSettings)) {
            this.validatePemStoreSettings(certType, settings);
        } else if (this.hasKeyOrTrustStoreSettings(certSettings)) {
            this.validateKeyStoreSettings(certType, settings);
        } else {
            throw new OpenSearchException("Wrong " + certType.id() + " SSL configuration. One of Keystore and Truststore files or X.509 PEM certificates and PKCS#8 keys groups should be set to configure " + certType.id() + " layer", new Object[0]);
        }
    }

    private void validatePemStoreSettings(CertType transportType, Settings settings) throws OpenSearchException {
        Settings transportSettings = settings.getByPrefix(transportType.sslSettingPrefix());
        ClientAuth clientAuth = ClientAuth.valueOf((String)transportSettings.get("clientauth_mode", ClientAuth.OPTIONAL.name()).toUpperCase(Locale.ROOT));
        if (!transportSettings.hasValue("pemcert_filepath") || !transportSettings.hasValue("pemkey_filepath")) {
            throw new OpenSearchException("Wrong " + transportType.id().toLowerCase(Locale.ROOT) + " SSL configuration. " + String.join((CharSequence)", ", transportSettings.get("pemcert_filepath"), transportSettings.get("pemkey_filepath")) + " must be set", new Object[0]);
        }
        if (clientAuth == ClientAuth.REQUIRE && !transportSettings.hasValue("pemtrustedcas_filepath")) {
            throw new OpenSearchException("Wrong " + transportType.id().toLowerCase(Locale.ROOT) + " SSL configuration. pemtrustedcas_filepath must be set if client auth is required", new Object[0]);
        }
    }

    private void validateKeyStoreSettings(CertType transportType, Settings settings) throws OpenSearchException {
        Settings transportSettings = settings.getByPrefix(transportType.sslSettingPrefix());
        ClientAuth clientAuth = ClientAuth.valueOf((String)transportSettings.get("clientauth_mode", ClientAuth.OPTIONAL.name()).toUpperCase(Locale.ROOT));
        if (!transportSettings.hasValue("keystore_filepath")) {
            throw new OpenSearchException("Wrong " + transportType.id().toLowerCase(Locale.ROOT) + " SSL configuration. " + transportSettings.get("keystore_filepath") + " must be set", new Object[0]);
        }
        if (clientAuth == ClientAuth.REQUIRE && !transportSettings.hasValue("truststore_filepath")) {
            throw new OpenSearchException("Wrong " + transportType.id().toLowerCase(Locale.ROOT) + " SSL configuration. truststore_filepath must be set if client auth is required", new Object[0]);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void validateTransportSettings(Settings transportSettings) {
        if (!this.hasExtendedKeyUsageEnabled(transportSettings)) {
            if (this.hasPemStoreSettings(transportSettings)) {
                if (transportSettings.hasValue("pemcert_filepath") && transportSettings.hasValue("pemkey_filepath") && transportSettings.hasValue("pemtrustedcas_filepath")) return;
                throw new OpenSearchException("Wrong Transport SSL configuration. " + String.join((CharSequence)",", "plugins.security.ssl.transport.pemcert_filepath", "plugins.security.ssl.transport.pemkey_filepath", "plugins.security.ssl.transport.pemtrustedcas_filepath") + " must be set", new Object[0]);
            }
            if (!this.hasKeyOrTrustStoreSettings(transportSettings)) throw new OpenSearchException("Wrong Transport SSL configuration. One of Keystore and Truststore files or X.509 PEM certificates and PKCS#8 keys groups should be set to configure Transport layer properly", new Object[0]);
            this.verifyKeyAndTrustStoreSettings(transportSettings);
            return;
        } else {
            Settings serverTransportSettings = transportSettings.getByPrefix("server.");
            Settings clientTransportSettings = transportSettings.getByPrefix("client.");
            if (this.hasKeyOrTrustStoreSettings(transportSettings)) {
                this.verifyKeyAndTrustStoreSettings(transportSettings);
                if (serverTransportSettings.hasValue("keystore_alias") && serverTransportSettings.hasValue("truststore_alias") && clientTransportSettings.hasValue("keystore_alias") && clientTransportSettings.hasValue("truststore_alias")) return;
                throw new OpenSearchException("Wrong Transport/Transport Client SSL configuration. " + String.join((CharSequence)",", "plugins.security.ssl.transport.server.keystore_alias", "plugins.security.ssl.transport.server.truststore_alias", "plugins.security.ssl.transport.client.keystore_alias", "plugins.security.ssl.transport.client.truststore_alias") + " must be set if plugins.security.ssl.transport.extended_key_usage_enabled is set", new Object[0]);
            }
            if (this.hasKeyOrTrustStoreSettings(transportSettings)) throw new OpenSearchException("Wrong Transport/Transport Client SSL configuration. One of Keystore and Truststore files or X.509 PEM certificates and PKCS#8 keys groups should be set to configure HTTP layer", new Object[0]);
            if (serverTransportSettings.hasValue("pemcert_filepath") && serverTransportSettings.hasValue("pemkey_filepath") && serverTransportSettings.hasValue("pemtrustedcas_filepath") && clientTransportSettings.hasValue("pemcert_filepath") && clientTransportSettings.hasValue("pemkey_filepath") && clientTransportSettings.hasValue("pemtrustedcas_filepath")) return;
            throw new OpenSearchException("Wrong Transport/Transport Client SSL configuration. " + String.join((CharSequence)",", "plugins.security.ssl.transport.server.pemcert_filepath", "plugins.security.ssl.transport.server.pemkey_filepath", "plugins.security.ssl.transport.server.pemtrustedcas_filepath", "plugins.security.ssl.transport.client.pemcert_filepath", "plugins.security.ssl.transport.client.pemkey_filepath", "plugins.security.ssl.transport.client.pemtrustedcas_filepath") + " must be set if plugins.security.ssl.transport.extended_key_usage_enabled is set", new Object[0]);
        }
    }

    private void verifyKeyAndTrustStoreSettings(Settings settings) {
        if (!settings.hasValue("keystore_filepath") || !settings.hasValue("truststore_filepath")) {
            throw new OpenSearchException("Wrong Transport/Tran SSL configuration. One of Keystore and Truststore files or X.509 PEM certificates and PKCS#8 keys groups should be set to configure Transport layer properly", new Object[0]);
        }
    }

    private boolean hasExtendedKeyUsageEnabled(Settings settings) {
        return settings.getAsBoolean("extended_key_usage_enabled", Boolean.valueOf(false));
    }

    private boolean hasKeyOrTrustStoreSettings(Settings settings) {
        return settings.hasValue("keystore_filepath") || settings.hasValue("truststore_filepath");
    }

    private boolean hasPemStoreSettings(Settings settings) {
        return settings.hasValue("pemkey_filepath") || settings.hasValue("pemcert_filepath") || settings.hasValue("pemtrustedcas_filepath");
    }

    void jceWarnings() {
        try {
            int aesMaxKeyLength = Cipher.getMaxAllowedKeyLength("AES");
            if (aesMaxKeyLength < 256) {
                LOGGER.info("AES-256 not supported, max key length for AES is {} bit. (This is not an issue, it just limits possible encryption strength. To enable AES 256, install 'Java Cryptography Extension (JCE) Unlimited Strength Jurisdiction Policy Files')", (Object)aesMaxKeyLength);
            }
        }
        catch (NoSuchAlgorithmException e) {
            LOGGER.error("AES encryption not supported (SG 1). ", (Throwable)e);
        }
    }
}

