/*
 * ZXC - High-performance lossless compression
 *
 * Copyright (c) 2025-2026 Bertrand Lebonnois and contributors.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/**
 * @file zxc_decompress.c
 * @brief Block-level decompression: NUM / GLO / GHI / RAW decoding with
 *        SIMD-accelerated prefix-sum, bit-unpacking, and overlapping copies.
 *
 * Like @ref zxc_compress.c, this file is compiled multiple times with
 * @c ZXC_FUNCTION_SUFFIX to produce per-ISA variants.
 */

#include "../../include/zxc_error.h"
#include "../../include/zxc_sans_io.h"
#include "zxc_internal.h"

/*
 * Function Multi-Versioning Support
 * If ZXC_FUNCTION_SUFFIX is defined (e.g. _avx2), rename the public entry point.
 */
#ifdef ZXC_FUNCTION_SUFFIX
#define ZXC_CAT_IMPL(x, y) x##y
#define ZXC_CAT(x, y) ZXC_CAT_IMPL(x, y)
#define zxc_decompress_chunk_wrapper ZXC_CAT(zxc_decompress_chunk_wrapper, ZXC_FUNCTION_SUFFIX)
#endif

/**
 * @brief Consumes a specified number of bits from the bit reader buffer without
 * performing safety checks.
 *
 * This function advances the bit reader's state by `n` bits. It is marked as
 * always inline for performance critical paths.
 *
 * @warning This is a "fast" variant, meaning it assumes the buffer has enough
 * bits available. The caller is responsible for ensuring that at least `n` bits
 * are present in the accumulator or buffer before calling this function to
 * avoid undefined behavior or reading past valid memory.
 *
 * @param[in,out] br Pointer to the bit reader instance.
 * @param[in] n The number of bits to consume (must be <= 32, typically <= 24
 * depending on implementation).
 * @return The value of the consumed bits as a 32-bit unsigned integer.
 */
static ZXC_ALWAYS_INLINE uint32_t zxc_br_consume_fast(zxc_bit_reader_t* br, uint8_t n) {
#if defined(__BMI2__) && (defined(__x86_64__) || defined(_M_X64))
    // BMI2 Optimization: _bzhi_u64(x, n) copies the lower n bits of x to dst and
    // clears the rest. It is equivalent to x & ((1ULL << n) - 1) but executes in
    // a single cycle without dependency chains.
    const uint32_t val = (uint32_t)_bzhi_u64(br->accum, n);
#else
    const uint32_t val = (uint32_t)(br->accum & ((1ULL << n) - 1));
#endif
    br->accum >>= n;
    br->bits -= n;
    return val;
}

/**
 * @brief Reads a Prefix Varint encoded integer from a stream.
 *
 * This function decodes a 32-bit unsigned integer encoded in Prefix Varint format
 * from the provided byte stream. Unary prefix bits in the first byte determine
 * the total length (1-5 bytes).
 *
 * Format:
 * - 1 byte (0xxxxxxx): 7 bits (val < 128)
 * - 2 bytes (10xxxxxx ...): 14 bits (val < 16384)
 * - 3 bytes (110xxxxx ...): 21 bits (val < 2M)
 * - 4 bytes (1110xxxx ...): 28 bits (val < 256M)
 * - 5 bytes (11110xxx ...): 32 bits (Full Range)
 *
 * @param[in,out] ptr Pointer to a pointer to the current position in the stream.
 * @param[in] end Pointer to the end of the readable stream (for bounds checking).
 * @return The decoded 32-bit integer, or 0 if reading would overflow bounds (safe default).
 */
static ZXC_ALWAYS_INLINE uint32_t zxc_read_varint(const uint8_t** ptr, const uint8_t* end) {
    const uint8_t* p = *ptr;
    // Bounds check: need at least 1 byte
    if (UNLIKELY(p >= end)) return 0;

    const uint32_t b0 = p[0];

    // 1 Byte: 0xxxxxxx (7 bits) -> val < 128
    if (LIKELY(b0 < 128)) {
        *ptr = p + 1;
        return b0;
    }

    // 2 Bytes: 10xxxxxx xxxxxxxx (14 bits)
    if (LIKELY(b0 < 0xC0)) {
        if (UNLIKELY(p + 1 >= end)) {
            *ptr = end;
            return 0;
        }
        *ptr = p + 2;
        return (b0 & 0x3F) | ((uint32_t)p[1] << 6);
    }

    // 3 Bytes: 110xxxxx xxxxxxxx xxxxxxxx (21 bits)
    if (LIKELY(b0 < 0xE0)) {
        if (UNLIKELY(p + 2 >= end)) {
            *ptr = end;
            return 0;
        }
        *ptr = p + 3;
        return (b0 & 0x1F) | ((uint32_t)p[1] << 5) | ((uint32_t)p[2] << 13);
    }

    // 4 Bytes: 1110xxxx ... (28 bits)
    if (UNLIKELY(b0 < 0xF0)) {
        if (UNLIKELY(p + 3 >= end)) {
            *ptr = end;
            return 0;
        }
        *ptr = p + 4;
        return (b0 & 0x0F) | ((uint32_t)p[1] << 4) | ((uint32_t)p[2] << 12) |
               ((uint32_t)p[3] << 20);
    }

    // 5 Bytes: 11110xxx ... (32 bits)
    if (UNLIKELY(p + 4 >= end)) {
        *ptr = end;
        return 0;
    }

    *ptr = p + 5;
    return (b0 & 0x07) | ((uint32_t)p[1] << 3) | ((uint32_t)p[2] << 11) | ((uint32_t)p[3] << 19) |
           ((uint32_t)p[4] << 27);
}

/**
 * @brief Shuffle masks for overlapping copies with small offsets (0-15).
 *
 * Shared between ARM NEON and x86 SSSE3. Each row defines how to replicate
 * source bytes to fill 16 bytes when offset < 16.
 */
#if defined(ZXC_USE_NEON64) || defined(ZXC_USE_NEON32) || defined(ZXC_USE_AVX2) || \
    defined(ZXC_USE_AVX512)
/**
 * @brief Precomputed masks for handling overlapping data during decompression.
 *
 * This 16x16 lookup table contains 128-bit aligned masks used to efficiently
 * mask off or combine bytes when processing overlapping copy operations or
 * boundary conditions in the ZXC decompression algorithm.
 *
 * The alignment to 16 bytes ensures compatibility with SIMD instructions
 * (like SSE/AVX) for optimized memory operations.
 */
static const ZXC_ALIGN(16) uint8_t zxc_overlap_masks[16][16] = {
    {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},      // off=0 (unused)
    {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},      // off=1 (RLE handled separately)
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},      // off=2
    {0, 1, 2, 0, 1, 2, 0, 1, 2, 0, 1, 2, 0, 1, 2, 0},      // off=3
    {0, 1, 2, 3, 0, 1, 2, 3, 0, 1, 2, 3, 0, 1, 2, 3},      // off=4
    {0, 1, 2, 3, 4, 0, 1, 2, 3, 4, 0, 1, 2, 3, 4, 0},      // off=5
    {0, 1, 2, 3, 4, 5, 0, 1, 2, 3, 4, 5, 0, 1, 2, 3},      // off=6
    {0, 1, 2, 3, 4, 5, 6, 0, 1, 2, 3, 4, 5, 6, 0, 1},      // off=7
    {0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 3, 4, 5, 6, 7},      // off=8
    {0, 1, 2, 3, 4, 5, 6, 7, 8, 0, 1, 2, 3, 4, 5, 6},      // off=9
    {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 1, 2, 3, 4, 5},      // off=10
    {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 0, 1, 2, 3, 4},     // off=11
    {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 0, 1, 2, 3},    // off=12
    {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 0, 1, 2},   // off=13
    {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 0, 1},  // off=14
    {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 0}  // off=15
};
#endif

/**
 * @brief Copies 16 bytes from an overlapping source to the destination.
 *
 * This function is designed to handle memory copies where the source and
 * destination regions might overlap, specifically copying 16 bytes from
 * `dst - off` to `dst`. It is typically used in decompression routines
 * (like LZ77) where repeating a previous sequence is required.
 *
 * Handles NEON64, NEON32, SSSE3/AVX2 and generic scalar fallback.
 *
 * @param[out] dst Pointer to the destination buffer where bytes will be written.
 * @param[in]  off The offset backwards from the destination pointer to read from.
 *                 (i.e., source address is `dst - off`).
 */
// codeql[cpp/unused-static-function] : False positive, used in DECODE_SEQ_SAFE/FAST macros
static ZXC_ALWAYS_INLINE void zxc_copy_overlap16(uint8_t* dst, uint32_t off) {
    // off is always >= ZXC_LZ_OFFSET_BIAS by design (offset bias encoding: stored +
    // ZXC_LZ_OFFSET_BIAS)
#if defined(ZXC_USE_NEON64)
    uint8x16_t mask = vld1q_u8(zxc_overlap_masks[off]);
    uint8x16_t src_data = vld1q_u8(dst - off);
    vst1q_u8(dst, vqtbl1q_u8(src_data, mask));

#elif defined(ZXC_USE_NEON32)
    uint8x8x2_t src_tbl;
    src_tbl.val[0] = vld1_u8(dst - off);
    src_tbl.val[1] = vld1_u8(dst - off + 8);
    uint8x8_t mask_lo = vld1_u8(zxc_overlap_masks[off]);
    uint8x8_t mask_hi = vld1_u8(zxc_overlap_masks[off] + 8);
    vst1_u8(dst, vtbl2_u8(src_tbl, mask_lo));
    vst1_u8(dst + 8, vtbl2_u8(src_tbl, mask_hi));

#elif defined(ZXC_USE_AVX2) || defined(ZXC_USE_AVX512)
    __m128i mask = _mm_load_si128((const __m128i*)zxc_overlap_masks[off]);
    __m128i src_data = _mm_loadu_si128((const __m128i*)(dst - off));
    _mm_storeu_si128((__m128i*)dst, _mm_shuffle_epi8(src_data, mask));

#else
    const uint8_t* src = dst - off;
    for (size_t i = 0; i < 16; i++) {
        dst[i] = src[i % off];
    }
#endif
}

#if defined(ZXC_USE_NEON64) || defined(ZXC_USE_NEON32)
/**
 * @brief Computes the prefix sum of a 128-bit vector of 32-bit unsigned
 * integers using NEON intrinsics.
 *
 * This function calculates the running total of the elements in the input
 * vector `v`. If the input vector is `[a, b, c, d]`, the result will be `[a,
 * a+b, a+b+c, a+b+c+d]`. This operation is typically used for calculating
 * cumulative distributions or offsets in parallel.
 *
 * @param[in] v The input vector containing four 32-bit unsigned integers.
 * @return A uint32x4_t vector containing the prefix sums.
 */
static ZXC_ALWAYS_INLINE uint32x4_t zxc_neon_prefix_sum_u32(uint32x4_t v) {
    const uint32x4_t zero = vdupq_n_u32(0);  // Create a vector of zeros

    // Rotate right by 1 element (shift 4 bytes)
    const uint32x4_t s1 =
        vreinterpretq_u32_u8(vextq_u8(vreinterpretq_u8_u32(zero), vreinterpretq_u8_u32(v), 12));
    v = vaddq_u32(v, s1);  // Add shifted version: [a, b, c, d] + [0, a, b, c] ->
                           // [a, a+b, b+c, c+d]

    // Rotate right by 2 elements (shift 8 bytes)
    const uint32x4_t s2 =
        vreinterpretq_u32_u8(vextq_u8(vreinterpretq_u8_u32(zero), vreinterpretq_u8_u32(v), 8));
    v = vaddq_u32(v, s2);  // Add shifted version to complete prefix sum

    return v;
}
#endif

#if defined(ZXC_USE_AVX2)
/**
 * @brief Computes the prefix sum of 32-bit integers within a 256-bit vector.
 *
 * This function calculates the cumulative sum of the eight 32-bit integers
 * contained in the input vector `v`.
 *
 * Operation logic (conceptually):
 *   out[0] = v[0]
 *   out[1] = v[0] + v[1]
 *   ...
 *   out[7] = v[0] + v[1] + ... + v[7]
 *
 * @param[in] v The input 256-bit vector containing eight 32-bit integers.
 * @return A 256-bit vector containing the prefix sums of the input elements.
 */
// codeql[cpp/unused-static-function] : Used conditionally when ZXC_USE_AVX2 is defined
static ZXC_ALWAYS_INLINE __m256i zxc_mm256_prefix_sum_epi32(__m256i v) {
    v = _mm256_add_epi32(v, _mm256_slli_si256(v, 4));  // Add value shifted by 1 element
    v = _mm256_add_epi32(v, _mm256_slli_si256(v, 8));  // Add value shifted by 2 elements

    // Use permute/shuffle to bridge the 128-bit lane gap
    __m256i v_bridge = _mm256_permute2x128_si256(v, v, 0x00);  // Duplicate lower 128 to upper
    v_bridge = _mm256_shuffle_epi32(v_bridge,
                                    0xFF);  // Broadcast last element of lower 128
    v_bridge = _mm256_blend_epi32(_mm256_setzero_si256(), v_bridge,
                                  0xF0);  // Only apply to upper lane

    return _mm256_add_epi32(v, v_bridge);  // Add bridge value to upper lane
}
#endif

#if defined(ZXC_USE_AVX512)
/**
 * @brief Computes the prefix sum of 32-bit integers within a 512-bit vector.
 *
 * This function calculates the running sum of the 16 packed 32-bit integers
 * in the input vector `v`.
 *
 * For an input vector v = [x0, x1, x2, ... x15], the result will be:
 * [x0, x0+x1, x0+x1+x2, ... , sum(x0..x15)].
 *
 * @note This function is forced inline for performance reasons.
 *
 * @param[in] v The input 512-bit vector containing sixteen 32-bit integers.
 * @return A 512-bit vector containing the prefix sums of the input elements.
 */
static ZXC_ALWAYS_INLINE __m512i zxc_mm512_prefix_sum_epi32(__m512i v) {
    __m512i t = _mm512_bslli_epi128(v, 4);  // Shift left by 4 bytes (1 int)
    v = _mm512_add_epi32(v, t);             // Add shifted value
    t = _mm512_bslli_epi128(v, 8);          // Shift left by 8 bytes (2 ints)
    v = _mm512_add_epi32(v, t);             // Add shifted value

    // Propagate sums across 128-bit lanes (sequential dependency)
    __m512i v_l0 = _mm512_shuffle_i32x4(v, v, 0x00);  // Broadcast lane 0
    v_l0 = _mm512_shuffle_epi32(v_l0, 0xFF);          // Broadcast last element of lane 0
    v = _mm512_mask_add_epi32(v, 0x00F0, v, v_l0);    // Add to lane 1 only

    __m512i v_l1 = _mm512_shuffle_i32x4(v, v, 0x55);  // Broadcast lane 1
    v_l1 = _mm512_shuffle_epi32(v_l1, 0xFF);          // Broadcast last element of lane 1
    v = _mm512_mask_add_epi32(v, 0x0F00, v, v_l1);    // Add to lane 2 only

    __m512i v_l2 = _mm512_shuffle_i32x4(v, v, 0xAA);  // Broadcast lane 2
    v_l2 = _mm512_shuffle_epi32(v_l2, 0xFF);          // Broadcast last element of lane 2
    v = _mm512_mask_add_epi32(v, 0xF000, v, v_l2);    // Add to lane 3 only

    return v;
}
#endif

/**
 * @brief Decodes a block of numerical data compressed with the ZXC format.
 *
 * This function reads a compressed numerical block from the source buffer,
 * parses the header to determine the number of values and encoding parameters,
 * and then decompresses the data into the destination buffer.
 *
 * **Algorithm Details:**
 * 1. **Header Parsing:** Reads the `zxc_num_header_t` to get the count of
 * values.
 * 2. **Bit Unpacking:** For each chunk of values, it initializes a bit reader.
 *    - **Unrolling:** The main loop is unrolled 4x to minimize branch overhead
 *      and maximize instruction throughput.
 * 3. **ZigZag Decoding:** Converts the unsigned unpacked value back to a signed
 * delta using `(n >> 1) ^ -(n & 1)`.
 * 4. **Delta Reconstruction:** Adds the signed delta to a `running_val`
 * accumulator to recover the original integer sequence.
 *
 * @param[in] src Pointer to the source buffer containing compressed data.
 * @param[in] src_size Size of the source buffer in bytes.
 * @param[out] dst Pointer to the destination buffer where decompressed data will be
 * written.
 * @param[in] dst_capacity Maximum capacity of the destination buffer in bytes.
 *
 * @return The number of bytes written to the destination buffer on success,
 *         or a negative zxc_error_t code if an error occurs (e.g., buffer overflow, invalid header,
 *         or malformed compressed stream).
 */
static int zxc_decode_block_num(const uint8_t* RESTRICT src, const size_t src_size,
                                uint8_t* RESTRICT dst, const size_t dst_capacity) {
    zxc_num_header_t nh;
    if (UNLIKELY(zxc_read_num_header(src, src_size, &nh) != 0)) return ZXC_ERROR_BAD_HEADER;

    size_t offset = ZXC_NUM_HEADER_BINARY_SIZE;
    uint8_t* d_ptr = dst;
    const uint8_t* const d_end = dst + dst_capacity;
    uint64_t vals_remaining = nh.n_values;
    uint32_t running_val = 0;

    ZXC_ALIGN(ZXC_CACHE_LINE_SIZE)
    uint32_t deltas[ZXC_NUM_DEC_BATCH];

    while (vals_remaining > 0) {
        if (UNLIKELY(offset + ZXC_NUM_CHUNK_HEADER_SIZE > src_size)) return ZXC_ERROR_SRC_TOO_SMALL;

        const uint16_t nvals = zxc_le16(src + offset);
        const uint16_t bits = zxc_le16(src + offset + 2);
        const uint32_t psize = zxc_le32(src + offset + 12);  // padding + nvals + bits
        offset += ZXC_NUM_CHUNK_HEADER_SIZE;

        if (UNLIKELY(nvals > vals_remaining || src_size < offset + psize ||
                     (size_t)(d_end - d_ptr) < (size_t)nvals * sizeof(uint32_t) ||
                     bits > (sizeof(uint32_t) * ZXC_BITS_PER_BYTE)))
            return ZXC_ERROR_CORRUPT_DATA;

        zxc_bit_reader_t br;
        zxc_br_init(&br, src + offset, psize);
        size_t i = 0;

        for (; i + ZXC_NUM_DEC_BATCH <= nvals; i += ZXC_NUM_DEC_BATCH) {
            for (int k = 0; k < ZXC_NUM_DEC_BATCH; k += 4) {
                zxc_br_ensure(&br, bits);
                deltas[k + 0] = zxc_zigzag_decode(zxc_br_consume_fast(&br, (uint8_t)bits));
                zxc_br_ensure(&br, bits);
                deltas[k + 1] = zxc_zigzag_decode(zxc_br_consume_fast(&br, (uint8_t)bits));
                zxc_br_ensure(&br, bits);
                deltas[k + 2] = zxc_zigzag_decode(zxc_br_consume_fast(&br, (uint8_t)bits));
                zxc_br_ensure(&br, bits);
                deltas[k + 3] = zxc_zigzag_decode(zxc_br_consume_fast(&br, (uint8_t)bits));
            }

            uint32_t* batch_dst = (uint32_t*)d_ptr;

#if defined(ZXC_USE_AVX512)
            for (int k = 0; k < ZXC_NUM_DEC_BATCH; k += 16) {
                __m512i v_deltas = _mm512_load_si512((void*)&deltas[k]);  // Load 16 deltas
                __m512i v_run = _mm512_set1_epi32(running_val);  // Broadcast current running total

                __m512i v_sum = zxc_mm512_prefix_sum_epi32(v_deltas);  // Compute local prefix sums
                v_sum = _mm512_add_epi32(v_sum, v_run);                // Add base running total

                _mm512_storeu_si512((void*)&batch_dst[k],
                                    v_sum);  // Store decoded values

                // Extract the last value (15th element) to update running_val for next
                // batch
                __m128i v_last128 = _mm512_extracti32x4_epi32(v_sum, 3);
                running_val = (uint32_t)_mm_cvtsi128_si32(_mm_shuffle_epi32(v_last128, 0xFF));
            }

#elif defined(ZXC_USE_AVX2)
            for (int k = 0; k < ZXC_NUM_DEC_BATCH; k += 8) {
                __m256i v_deltas = _mm256_load_si256((const __m256i*)&deltas[k]);  // Load 8 deltas
                __m256i v_run = _mm256_set1_epi32(running_val);  // Broadcast running total

                __m256i v_sum = zxc_mm256_prefix_sum_epi32(v_deltas);  // Compute local prefix sums
                v_sum = _mm256_add_epi32(v_sum, v_run);                // Add base

                _mm256_storeu_si256((__m256i*)&batch_dst[k],
                                    v_sum);                   // Store decoded values
                running_val = ((uint32_t*)&batch_dst[k])[7];  // Update running_val
            }

#elif defined(ZXC_USE_NEON64) || defined(ZXC_USE_NEON32)
            uint32x4_t v_run = vdupq_n_u32(running_val);  // Broadcast running total
            for (int k = 0; k < ZXC_NUM_DEC_BATCH; k += 4) {
                uint32x4_t v_deltas = vld1q_u32(&deltas[k]);  // Load 4 deltas

                uint32x4_t v_sum = zxc_neon_prefix_sum_u32(v_deltas);  // Compute local prefix sums
                v_sum = vaddq_u32(v_sum, v_run);                       // Add base

                vst1q_u32(&batch_dst[k], v_sum);  // Store decoded values

                running_val = vgetq_lane_u32(v_sum, 3);  // Extract last element
                v_run = vdupq_n_u32(running_val);        // Update vector for next iter
            }

#else
            for (int k = 0; k < ZXC_NUM_DEC_BATCH; k++) {
                running_val += deltas[k];
#ifdef ZXC_BIG_ENDIAN
                zxc_store_le32(&batch_dst[k], running_val);
#else
                batch_dst[k] = running_val;
#endif
            }
#endif
            d_ptr += ZXC_NUM_DEC_BATCH * sizeof(uint32_t);
        }

        for (; i < nvals; i++) {
            zxc_br_ensure(&br, bits);
            const uint32_t delta = zxc_zigzag_decode(zxc_br_consume_fast(&br, (uint8_t)bits));
            running_val += delta;
            zxc_store_le32(d_ptr, running_val);
            d_ptr += sizeof(uint32_t);
        }

        offset += psize;
        vals_remaining -= nvals;
    }
    return (int)(d_ptr - dst);
}

/**
 * @brief Decodes a General Low (GLO) format compressed block.
 *
 * This function handles the decoding of a compressed block formatted with the
 * internal GLO structure. The decompressed size is derived from Section
 * Descriptors within the compressed payload.
 *
 * @param[in,out] ctx Pointer to the compression context (`zxc_cctx_t`) containing
 * @param[in] src Pointer to the source buffer containing compressed data.
 * @param[in] src_size Size of the source buffer in bytes.
 * @param[out] dst Pointer to the destination buffer for decompressed data.
 * @param[in] dst_capacity Maximum capacity of the destination buffer.
 *
 * @return The number of bytes written to the destination buffer on success, or
 * a negative zxc_error_t code on failure (e.g., invalid header, buffer overflow, or corrupted
 * data).
 */
static int zxc_decode_block_glo(zxc_cctx_t* RESTRICT ctx, const uint8_t* RESTRICT src,
                                const size_t src_size, uint8_t* RESTRICT dst,
                                const size_t dst_capacity) {
    zxc_gnr_header_t gh;
    zxc_section_desc_t desc[ZXC_GLO_SECTIONS];

    const int res = zxc_read_glo_header_and_desc(src, src_size, &gh, desc);
    if (UNLIKELY(res != 0)) return ZXC_ERROR_BAD_HEADER;

    const uint8_t* p_data =
        src + ZXC_GLO_HEADER_BINARY_SIZE + ZXC_GLO_SECTIONS * ZXC_SECTION_DESC_BINARY_SIZE;
    const uint8_t* p_curr = p_data;

    // --- Literal Stream Setup ---
    const uint8_t* l_ptr;
    const uint8_t* l_end;
    uint8_t* rle_buf = NULL;

    size_t lit_stream_size = (size_t)(desc[0].sizes & ZXC_SECTION_SIZE_MASK);

    if (gh.enc_lit == 1) {
        const size_t required_size = (size_t)(desc[0].sizes >> 32);

        if (required_size > 0) {
            if (UNLIKELY(required_size > dst_capacity)) return ZXC_ERROR_DST_TOO_SMALL;

            if (ctx->lit_buffer_cap < required_size + ZXC_PAD_SIZE) {
                uint8_t* new_buf = (uint8_t*)realloc(ctx->lit_buffer, required_size + ZXC_PAD_SIZE);
                if (UNLIKELY(!new_buf)) {
                    free(ctx->lit_buffer);
                    ctx->lit_buffer = NULL;
                    ctx->lit_buffer_cap = 0;
                    return ZXC_ERROR_MEMORY;
                }
                ctx->lit_buffer = new_buf;
                ctx->lit_buffer_cap = required_size + ZXC_PAD_SIZE;
            }

            rle_buf = ctx->lit_buffer;
            if (UNLIKELY(!rle_buf || lit_stream_size > (size_t)(src + src_size - p_curr)))
                return ZXC_ERROR_CORRUPT_DATA;

            const uint8_t* r_ptr = p_curr;
            const uint8_t* r_end = r_ptr + lit_stream_size;
            uint8_t* w_ptr = rle_buf;
            const uint8_t* const w_end = rle_buf + required_size;

            while (r_ptr < r_end && w_ptr < w_end) {
                uint8_t token = *r_ptr++;
                if (LIKELY(!(token & ZXC_LIT_RLE_FLAG))) {
                    // Raw copy (most common path): use ZXC_PAD_SIZE-byte wild copies
                    // token is 7-bit (0-127), so len is 1-128 bytes
                    const uint32_t len = (uint32_t)token + 1;
                    if (UNLIKELY(w_ptr + len > w_end || r_ptr + len > r_end))
                        return ZXC_ERROR_CORRUPT_DATA;

                    // Destination has ZXC_PAD_SIZE bytes of safe overrun space.
                    // Source may not - check before wild copy.
                    // Fast path: source has ZXC_PAD_SIZE-byte read headroom (most common)
                    if (LIKELY(r_ptr + ZXC_PAD_SIZE <= r_end)) {
                        // Single 32-byte copy covers len <= ZXC_PAD_SIZE (most tokens)
                        zxc_copy32(w_ptr, r_ptr);

                        if (UNLIKELY(len > ZXC_PAD_SIZE)) {
                            // Unroll: max len=128, so max 4 copies total
                            // Use unconditional stores with overlap - faster than branches
                            if (len <= 2 * ZXC_PAD_SIZE) {
                                zxc_copy32(w_ptr + len - ZXC_PAD_SIZE, r_ptr + len - ZXC_PAD_SIZE);
                            } else if (len <= 3 * ZXC_PAD_SIZE) {
                                zxc_copy32(w_ptr + ZXC_PAD_SIZE, r_ptr + ZXC_PAD_SIZE);
                                zxc_copy32(w_ptr + len - ZXC_PAD_SIZE, r_ptr + len - ZXC_PAD_SIZE);
                            } else {
                                zxc_copy32(w_ptr + ZXC_PAD_SIZE, r_ptr + ZXC_PAD_SIZE);
                                zxc_copy32(w_ptr + 2 * ZXC_PAD_SIZE, r_ptr + 2 * ZXC_PAD_SIZE);
                                zxc_copy32(w_ptr + len - ZXC_PAD_SIZE, r_ptr + len - ZXC_PAD_SIZE);
                            }
                        }
                    } else {
                        // Near end of source: safe copy (rare cold path)
                        ZXC_MEMCPY(w_ptr, r_ptr, len);
                    }

                    w_ptr += len;
                    r_ptr += len;
                } else {
                    // RLE run: fill with single byte
                    const uint32_t len = (token & ZXC_LIT_LEN_MASK) + 4;
                    if (UNLIKELY(w_ptr + len > w_end || r_ptr >= r_end))
                        return ZXC_ERROR_CORRUPT_DATA;
                    ZXC_MEMSET(w_ptr, *r_ptr++, len);
                    w_ptr += len;
                }
            }
            if (UNLIKELY(w_ptr != w_end)) return ZXC_ERROR_CORRUPT_DATA;
            l_ptr = rle_buf;
            l_end = rle_buf + required_size;
        } else {
            l_ptr = p_curr;
            l_end = p_curr;
        }
    } else {
        l_ptr = p_curr;
        l_end = p_curr + lit_stream_size;
    }

    p_curr += lit_stream_size;

    // --- Stream Pointers & Validation ---
    const size_t sz_tokens = (size_t)(desc[1].sizes & ZXC_SECTION_SIZE_MASK);
    const size_t sz_offsets = (size_t)(desc[2].sizes & ZXC_SECTION_SIZE_MASK);
    const size_t sz_extras = (size_t)(desc[3].sizes & ZXC_SECTION_SIZE_MASK);

    // Validate stream sizes match sequence count (early rejection of malformed data)
    const size_t expected_off_size =
        (gh.enc_off == 1) ? (size_t)gh.n_sequences : (size_t)gh.n_sequences * 2;

    if (UNLIKELY(sz_tokens < gh.n_sequences || sz_offsets < expected_off_size))
        return ZXC_ERROR_CORRUPT_DATA;

    const uint8_t* t_ptr = p_curr;
    const uint8_t* o_ptr = t_ptr + sz_tokens;
    const uint8_t* e_ptr = o_ptr + sz_offsets;
    const uint8_t* const e_end = e_ptr + sz_extras;  // For vbyte overflow detection

    // Validate streams don't overflow source buffer
    if (UNLIKELY(e_end != src + src_size)) return ZXC_ERROR_CORRUPT_DATA;

    uint8_t* d_ptr = dst;
    const uint8_t* const d_end = dst + dst_capacity;
    // Destination safe margin for 4x loop: max output without varint extension.
    // ll_max = 14, ml_max = 14 + 5 = 19, per-seq = 33, 4x = 132 + 4 safety = 136
    const uint8_t* const d_end_safe = d_end - 136;

    // Literal stream safe threshold for 4x-unrolled loops.
    // Without varint extension, max ll per sequence = ZXC_TOKEN_LL_MASK - 1 = 14.
    // For 4 sequences: 4 * 14 = 56. With this margin, l_ptr checks are only needed
    // on the cold varint path, keeping the hot path free of l_ptr overhead.
    const size_t glo_sz_lit = (size_t)(l_end - l_ptr);
    const size_t glo_margin_4x = 4 * (ZXC_TOKEN_LL_MASK - 1);  // 56
    const size_t glo_margin_1x = ZXC_TOKEN_LL_MASK - 1;        // 14
    const uint8_t* const l_end_safe_4x =
        (glo_sz_lit > glo_margin_4x) ? l_end - glo_margin_4x : l_ptr;
    const uint8_t* const l_end_safe_1x =
        (glo_sz_lit > glo_margin_1x) ? l_end - glo_margin_1x : l_ptr;

    uint32_t n_seq = gh.n_sequences;

    // Track bytes written for offset validation
    // For 1-byte offsets (enc_off==1): validate until 256 bytes written (max 8-bit offset)
    // For 2-byte offsets (enc_off==0): validate until 65536 bytes written (max 16-bit offset)
    // After threshold, all offsets are guaranteed valid (can't exceed written bytes)
    size_t written = 0;

// Macro for copy literal + match (uses 32-byte wild copies)
// SAFE version: validates offset against written bytes
#define DECODE_SEQ_SAFE(ll, ml, off)                                     \
    do {                                                                 \
        {                                                                \
            const uint8_t* src_lit = l_ptr;                              \
            uint8_t* dst_lit = d_ptr;                                    \
            zxc_copy32(dst_lit, src_lit);                                \
            if (UNLIKELY(ll > ZXC_PAD_SIZE)) {                           \
                dst_lit += ZXC_PAD_SIZE;                                 \
                src_lit += ZXC_PAD_SIZE;                                 \
                size_t rem = ll - ZXC_PAD_SIZE;                          \
                while (rem > ZXC_PAD_SIZE) {                             \
                    zxc_copy32(dst_lit, src_lit);                        \
                    dst_lit += ZXC_PAD_SIZE;                             \
                    src_lit += ZXC_PAD_SIZE;                             \
                    rem -= ZXC_PAD_SIZE;                                 \
                }                                                        \
                zxc_copy32(dst_lit, src_lit);                            \
            }                                                            \
            l_ptr += ll;                                                 \
            d_ptr += ll;                                                 \
            written += ll;                                               \
        }                                                                \
        {                                                                \
            if (UNLIKELY(off > written)) return ZXC_ERROR_BAD_OFFSET;    \
            const uint8_t* match_src = d_ptr - off;                      \
            if (LIKELY(off >= ZXC_PAD_SIZE)) {                           \
                zxc_copy32(d_ptr, match_src);                            \
                if (UNLIKELY(ml > ZXC_PAD_SIZE)) {                       \
                    uint8_t* out = d_ptr + ZXC_PAD_SIZE;                 \
                    const uint8_t* ref = match_src + ZXC_PAD_SIZE;       \
                    size_t rem = ml - ZXC_PAD_SIZE;                      \
                    while (rem > ZXC_PAD_SIZE) {                         \
                        zxc_copy32(out, ref);                            \
                        out += ZXC_PAD_SIZE;                             \
                        ref += ZXC_PAD_SIZE;                             \
                        rem -= ZXC_PAD_SIZE;                             \
                    }                                                    \
                    zxc_copy32(out, ref);                                \
                }                                                        \
                d_ptr += ml;                                             \
                written += ml;                                           \
            } else if (off >= (ZXC_PAD_SIZE / 2)) {                      \
                zxc_copy16(d_ptr, match_src);                            \
                if (UNLIKELY(ml > (ZXC_PAD_SIZE / 2))) {                 \
                    uint8_t* out = d_ptr + (ZXC_PAD_SIZE / 2);           \
                    const uint8_t* ref = match_src + (ZXC_PAD_SIZE / 2); \
                    size_t rem = ml - (ZXC_PAD_SIZE / 2);                \
                    while (rem > (ZXC_PAD_SIZE / 2)) {                   \
                        zxc_copy16(out, ref);                            \
                        out += (ZXC_PAD_SIZE / 2);                       \
                        ref += (ZXC_PAD_SIZE / 2);                       \
                        rem -= (ZXC_PAD_SIZE / 2);                       \
                    }                                                    \
                    zxc_copy16(out, ref);                                \
                }                                                        \
                d_ptr += ml;                                             \
                written += ml;                                           \
            } else if (off == 1) {                                       \
                ZXC_MEMSET(d_ptr, match_src[0], ml);                     \
                d_ptr += ml;                                             \
                written += ml;                                           \
            } else {                                                     \
                size_t copied = 0;                                       \
                while (copied < ml) {                                    \
                    zxc_copy_overlap16(d_ptr + copied, off);             \
                    copied += (ZXC_PAD_SIZE / 2);                        \
                }                                                        \
                d_ptr += ml;                                             \
                written += ml;                                           \
            }                                                            \
        }                                                                \
    } while (0)

// FAST version: no offset validation (for use after written >= 256 or 65536)
#define DECODE_SEQ_FAST(ll, ml, off)                                     \
    do {                                                                 \
        {                                                                \
            const uint8_t* src_lit = l_ptr;                              \
            uint8_t* dst_lit = d_ptr;                                    \
            zxc_copy32(dst_lit, src_lit);                                \
            if (UNLIKELY(ll > ZXC_PAD_SIZE)) {                           \
                dst_lit += ZXC_PAD_SIZE;                                 \
                src_lit += ZXC_PAD_SIZE;                                 \
                size_t rem = ll - ZXC_PAD_SIZE;                          \
                while (rem > ZXC_PAD_SIZE) {                             \
                    zxc_copy32(dst_lit, src_lit);                        \
                    dst_lit += ZXC_PAD_SIZE;                             \
                    src_lit += ZXC_PAD_SIZE;                             \
                    rem -= ZXC_PAD_SIZE;                                 \
                }                                                        \
                zxc_copy32(dst_lit, src_lit);                            \
            }                                                            \
            l_ptr += ll;                                                 \
            d_ptr += ll;                                                 \
        }                                                                \
        {                                                                \
            const uint8_t* match_src = d_ptr - off;                      \
            if (LIKELY(off >= ZXC_PAD_SIZE)) {                           \
                zxc_copy32(d_ptr, match_src);                            \
                if (UNLIKELY(ml > ZXC_PAD_SIZE)) {                       \
                    uint8_t* out = d_ptr + ZXC_PAD_SIZE;                 \
                    const uint8_t* ref = match_src + ZXC_PAD_SIZE;       \
                    size_t rem = ml - ZXC_PAD_SIZE;                      \
                    while (rem > ZXC_PAD_SIZE) {                         \
                        zxc_copy32(out, ref);                            \
                        out += ZXC_PAD_SIZE;                             \
                        ref += ZXC_PAD_SIZE;                             \
                        rem -= ZXC_PAD_SIZE;                             \
                    }                                                    \
                    zxc_copy32(out, ref);                                \
                }                                                        \
                d_ptr += ml;                                             \
            } else if (off >= (ZXC_PAD_SIZE / 2)) {                      \
                zxc_copy16(d_ptr, match_src);                            \
                if (UNLIKELY(ml > (ZXC_PAD_SIZE / 2))) {                 \
                    uint8_t* out = d_ptr + (ZXC_PAD_SIZE / 2);           \
                    const uint8_t* ref = match_src + (ZXC_PAD_SIZE / 2); \
                    size_t rem = ml - (ZXC_PAD_SIZE / 2);                \
                    while (rem > (ZXC_PAD_SIZE / 2)) {                   \
                        zxc_copy16(out, ref);                            \
                        out += (ZXC_PAD_SIZE / 2);                       \
                        ref += (ZXC_PAD_SIZE / 2);                       \
                        rem -= (ZXC_PAD_SIZE / 2);                       \
                    }                                                    \
                    zxc_copy16(out, ref);                                \
                }                                                        \
                d_ptr += ml;                                             \
            } else if (off == 1) {                                       \
                ZXC_MEMSET(d_ptr, match_src[0], ml);                     \
                d_ptr += ml;                                             \
            } else {                                                     \
                size_t copied = 0;                                       \
                while (copied < ml) {                                    \
                    zxc_copy_overlap16(d_ptr + copied, off);             \
                    copied += (ZXC_PAD_SIZE / 2);                        \
                }                                                        \
                d_ptr += ml;                                             \
            }                                                            \
        }                                                                \
    } while (0)

    // --- SAFE Loop: offset validation until threshold (4x unroll) ---
    // For 1-byte offsets: bounds check until 256 bytes written
    // For 2-byte offsets: bounds check until 65536 bytes written
    const size_t bounds_threshold = (gh.enc_off == 1) ? (1U << 8) : (1U << 16);

    while (n_seq >= 4 && d_ptr < d_end_safe && l_ptr < l_end_safe_4x &&
           written < bounds_threshold) {
        uint32_t tokens = zxc_le32(t_ptr);
        t_ptr += 4;

        uint32_t off1 = ZXC_LZ_OFFSET_BIAS, off2 = ZXC_LZ_OFFSET_BIAS, off3 = ZXC_LZ_OFFSET_BIAS,
                 off4 = ZXC_LZ_OFFSET_BIAS;
        if (gh.enc_off == 1) {
            // Read 4 x 1-byte offsets
            uint32_t offsets = zxc_le32(o_ptr);
            o_ptr += 4;
            off1 += offsets & 0xFF;
            off2 += (offsets >> 8) & 0xFF;
            off3 += (offsets >> 16) & 0xFF;
            off4 += (offsets >> 24) & 0xFF;
        } else {
            // Read 4 x 2-byte offsets
            uint64_t offsets = zxc_le64(o_ptr);
            o_ptr += 8;
            off1 += (uint32_t)(offsets & 0xFFFF);
            off2 += (uint32_t)((offsets >> 16) & 0xFFFF);
            off3 += (uint32_t)((offsets >> 32) & 0xFFFF);
            off4 += (uint32_t)((offsets >> 48) & 0xFFFF);
        }

        uint32_t ll1 = (tokens & 0x0F0) >> 4;
        uint32_t ml1 = (tokens & 0x00F);
        if (UNLIKELY(ll1 == ZXC_TOKEN_LL_MASK)) {
            ll1 += zxc_read_varint(&e_ptr, e_end);
            if (UNLIKELY(l_ptr + ll1 > l_end || d_ptr + ll1 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        if (UNLIKELY(ml1 == ZXC_TOKEN_ML_MASK)) {
            ml1 += zxc_read_varint(&e_ptr, e_end);
            ml1 += ZXC_LZ_MIN_MATCH_LEN;
            if (UNLIKELY(d_ptr + ll1 + ml1 > d_end)) return ZXC_ERROR_OVERFLOW;
        } else {
            ml1 += ZXC_LZ_MIN_MATCH_LEN;
        }
        DECODE_SEQ_SAFE(ll1, ml1, off1);

        uint32_t ll2 = (tokens & 0x0F000) >> 12;
        uint32_t ml2 = (tokens & 0x00F00) >> 8;
        if (UNLIKELY(ll2 == ZXC_TOKEN_LL_MASK)) {
            ll2 += zxc_read_varint(&e_ptr, e_end);
            if (UNLIKELY(l_ptr + ll2 > l_end || d_ptr + ll2 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        if (UNLIKELY(ml2 == ZXC_TOKEN_ML_MASK)) {
            ml2 += zxc_read_varint(&e_ptr, e_end);
            ml2 += ZXC_LZ_MIN_MATCH_LEN;
            if (UNLIKELY(d_ptr + ll2 + ml2 > d_end)) return ZXC_ERROR_OVERFLOW;
        } else {
            ml2 += ZXC_LZ_MIN_MATCH_LEN;
        }
        DECODE_SEQ_SAFE(ll2, ml2, off2);

        uint32_t ll3 = (tokens & 0x0F00000) >> 20;
        uint32_t ml3 = (tokens & 0x00F0000) >> 16;
        if (UNLIKELY(ll3 == ZXC_TOKEN_LL_MASK)) {
            ll3 += zxc_read_varint(&e_ptr, e_end);
            if (UNLIKELY(l_ptr + ll3 > l_end || d_ptr + ll3 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        if (UNLIKELY(ml3 == ZXC_TOKEN_ML_MASK)) {
            ml3 += zxc_read_varint(&e_ptr, e_end);
            ml3 += ZXC_LZ_MIN_MATCH_LEN;
            if (UNLIKELY(d_ptr + ll3 + ml3 > d_end)) return ZXC_ERROR_OVERFLOW;
        } else {
            ml3 += ZXC_LZ_MIN_MATCH_LEN;
        }
        DECODE_SEQ_SAFE(ll3, ml3, off3);

        uint32_t ll4 = (tokens >> 28);
        uint32_t ml4 = (tokens >> 24) & 0x0F;
        if (UNLIKELY(ll4 == ZXC_TOKEN_LL_MASK)) {
            ll4 += zxc_read_varint(&e_ptr, e_end);
            if (UNLIKELY(l_ptr + ll4 > l_end || d_ptr + ll4 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        if (UNLIKELY(ml4 == ZXC_TOKEN_ML_MASK)) {
            ml4 += zxc_read_varint(&e_ptr, e_end);
            ml4 += ZXC_LZ_MIN_MATCH_LEN;
            if (UNLIKELY(d_ptr + ll4 + ml4 > d_end)) return ZXC_ERROR_OVERFLOW;
        } else {
            ml4 += ZXC_LZ_MIN_MATCH_LEN;
        }
        DECODE_SEQ_SAFE(ll4, ml4, off4);

        n_seq -= 4;
    }

    // --- FAST Loop: After threshold, no offset validation needed (4x unroll) ---
    while (n_seq >= 4 && d_ptr < d_end_safe && l_ptr < l_end_safe_4x) {
        uint32_t tokens = zxc_le32(t_ptr);
        t_ptr += 4;

        uint32_t off1 = ZXC_LZ_OFFSET_BIAS, off2 = ZXC_LZ_OFFSET_BIAS, off3 = ZXC_LZ_OFFSET_BIAS,
                 off4 = ZXC_LZ_OFFSET_BIAS;
        if (gh.enc_off == 1) {
            // Read 4 x 1-byte offsets
            uint32_t offsets = zxc_le32(o_ptr);
            o_ptr += 4;
            off1 += offsets & 0xFF;
            off2 += (offsets >> 8) & 0xFF;
            off3 += (offsets >> 16) & 0xFF;
            off4 += (offsets >> 24) & 0xFF;
        } else {
            // Read 4 x 2-byte offsets
            uint64_t offsets = zxc_le64(o_ptr);
            o_ptr += 8;
            off1 += (uint32_t)(offsets & 0xFFFF);
            off2 += (uint32_t)((offsets >> 16) & 0xFFFF);
            off3 += (uint32_t)((offsets >> 32) & 0xFFFF);
            off4 += (uint32_t)((offsets >> 48) & 0xFFFF);
        }

        uint32_t ll1 = (tokens & 0x0F0) >> 4;
        uint32_t ml1 = (tokens & 0x00F);
        if (UNLIKELY(ll1 == ZXC_TOKEN_LL_MASK)) {
            ll1 += zxc_read_varint(&e_ptr, e_end);
            if (UNLIKELY(l_ptr + ll1 > l_end || d_ptr + ll1 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        if (UNLIKELY(ml1 == ZXC_TOKEN_ML_MASK)) {
            ml1 += zxc_read_varint(&e_ptr, e_end);
            ml1 += ZXC_LZ_MIN_MATCH_LEN;
            if (UNLIKELY(d_ptr + ll1 + ml1 > d_end)) return ZXC_ERROR_OVERFLOW;
        } else {
            ml1 += ZXC_LZ_MIN_MATCH_LEN;
        }
        DECODE_SEQ_FAST(ll1, ml1, off1);

        uint32_t ll2 = (tokens & 0x0F000) >> 12;
        uint32_t ml2 = (tokens & 0x00F00) >> 8;
        if (UNLIKELY(ll2 == ZXC_TOKEN_LL_MASK)) {
            ll2 += zxc_read_varint(&e_ptr, e_end);
            if (UNLIKELY(l_ptr + ll2 > l_end || d_ptr + ll2 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        if (UNLIKELY(ml2 == ZXC_TOKEN_ML_MASK)) {
            ml2 += zxc_read_varint(&e_ptr, e_end);
            ml2 += ZXC_LZ_MIN_MATCH_LEN;
            if (UNLIKELY(d_ptr + ll2 + ml2 > d_end)) return ZXC_ERROR_OVERFLOW;
        } else {
            ml2 += ZXC_LZ_MIN_MATCH_LEN;
        }
        DECODE_SEQ_FAST(ll2, ml2, off2);

        uint32_t ll3 = (tokens & 0x0F00000) >> 20;
        uint32_t ml3 = (tokens & 0x00F0000) >> 16;
        if (UNLIKELY(ll3 == ZXC_TOKEN_LL_MASK)) {
            ll3 += zxc_read_varint(&e_ptr, e_end);
            if (UNLIKELY(l_ptr + ll3 > l_end || d_ptr + ll3 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        if (UNLIKELY(ml3 == ZXC_TOKEN_ML_MASK)) {
            ml3 += zxc_read_varint(&e_ptr, e_end);
            ml3 += ZXC_LZ_MIN_MATCH_LEN;
            if (UNLIKELY(d_ptr + ll3 + ml3 > d_end)) return ZXC_ERROR_OVERFLOW;
        } else {
            ml3 += ZXC_LZ_MIN_MATCH_LEN;
        }
        DECODE_SEQ_FAST(ll3, ml3, off3);

        uint32_t ll4 = (tokens >> 28);
        uint32_t ml4 = (tokens >> 24) & 0x0F;
        if (UNLIKELY(ll4 == ZXC_TOKEN_LL_MASK)) {
            ll4 += zxc_read_varint(&e_ptr, e_end);
            if (UNLIKELY(l_ptr + ll4 > l_end || d_ptr + ll4 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        if (UNLIKELY(ml4 == ZXC_TOKEN_ML_MASK)) {
            ml4 += zxc_read_varint(&e_ptr, e_end);
            ml4 += ZXC_LZ_MIN_MATCH_LEN;
            if (UNLIKELY(d_ptr + ll4 + ml4 > d_end)) return ZXC_ERROR_OVERFLOW;
        } else {
            ml4 += ZXC_LZ_MIN_MATCH_LEN;
        }
        DECODE_SEQ_FAST(ll4, ml4, off4);

        n_seq -= 4;
    }

#undef DECODE_SEQ_SAFE
#undef DECODE_SEQ_FAST

    // Validate vbyte reads didn't overflow
    if (UNLIKELY(e_ptr > e_end)) return ZXC_ERROR_CORRUPT_DATA;

    // --- Remaining 1 sequence (Fast Path) ---
    while (n_seq > 0 && d_ptr < d_end_safe && l_ptr < l_end_safe_1x) {
        // Save pointers before reading (in case we need to fall back to Safe Path)
        const uint8_t* t_save = t_ptr;
        const uint8_t* o_save = o_ptr;
        const uint8_t* e_save = e_ptr;

        uint8_t token = *t_ptr++;
        uint32_t ll = token >> ZXC_TOKEN_LIT_BITS;
        uint32_t ml = token & ZXC_TOKEN_ML_MASK;
        uint32_t offset = ZXC_LZ_OFFSET_BIAS;
        if (gh.enc_off == 1) {
            offset += *o_ptr++;  // 1-byte offset (biased)
        } else {
            offset += zxc_le16(o_ptr);  // 2-byte offset (biased)
            o_ptr += 2;
        }

        if (UNLIKELY(ll == ZXC_TOKEN_LL_MASK)) {
            ll += zxc_read_varint(&e_ptr, e_end);
            if (UNLIKELY(l_ptr + ll > l_end)) {
                t_ptr = t_save;
                o_ptr = o_save;
                e_ptr = e_save;
                break;
            }
        }
        if (UNLIKELY(ml == ZXC_TOKEN_ML_MASK)) ml += zxc_read_varint(&e_ptr, e_end);
        ml += ZXC_LZ_MIN_MATCH_LEN;

        // Check bounds before wild copies - if too close to end, fall back to Safe Path
        if (UNLIKELY(d_ptr + ll + ml + ZXC_PAD_SIZE > d_end)) {
            // Restore pointers and let Safe Path handle this sequence
            t_ptr = t_save;
            o_ptr = o_save;
            e_ptr = e_save;
            break;
        }

        {
            const uint8_t* src_lit = l_ptr;
            uint8_t* dst_lit = d_ptr;
            zxc_copy32(dst_lit, src_lit);
            if (UNLIKELY(ll > ZXC_PAD_SIZE)) {
                dst_lit += ZXC_PAD_SIZE;
                src_lit += ZXC_PAD_SIZE;
                size_t rem = ll - ZXC_PAD_SIZE;
                while (rem > ZXC_PAD_SIZE) {
                    zxc_copy32(dst_lit, src_lit);
                    dst_lit += ZXC_PAD_SIZE;
                    src_lit += ZXC_PAD_SIZE;
                    rem -= ZXC_PAD_SIZE;
                }
                zxc_copy32(dst_lit, src_lit);
            }
            l_ptr += ll;
            d_ptr += ll;
            written += ll;
        }

        {
            // Skip check if written >= bounds_threshold (256 for 8-bit, 65536 for 16-bit)
            if (UNLIKELY(written < bounds_threshold && offset > written))
                return ZXC_ERROR_BAD_OFFSET;

            const uint8_t* match_src = d_ptr - offset;
            if (LIKELY(offset >= ZXC_PAD_SIZE)) {
                zxc_copy32(d_ptr, match_src);
                if (UNLIKELY(ml > ZXC_PAD_SIZE)) {
                    uint8_t* out = d_ptr + ZXC_PAD_SIZE;
                    const uint8_t* ref = match_src + ZXC_PAD_SIZE;
                    size_t rem = ml - ZXC_PAD_SIZE;
                    while (rem > ZXC_PAD_SIZE) {
                        zxc_copy32(out, ref);
                        out += ZXC_PAD_SIZE;
                        ref += ZXC_PAD_SIZE;
                        rem -= ZXC_PAD_SIZE;
                    }
                    zxc_copy32(out, ref);
                }
                d_ptr += ml;
                written += ml;
            } else if (offset == 1) {
                ZXC_MEMSET(d_ptr, match_src[0], ml);
                d_ptr += ml;
                written += ml;
            } else {
                for (size_t i = 0; i < ml; i++) d_ptr[i] = match_src[i];
                d_ptr += ml;
                written += ml;
            }
        }
        n_seq--;
    }

    // --- Safe Path for Remaining Sequences ---
    while (n_seq > 0) {
        uint8_t token = *t_ptr++;
        uint32_t ll = token >> ZXC_TOKEN_LIT_BITS;
        uint32_t ml = token & ZXC_TOKEN_ML_MASK;
        uint32_t offset = ZXC_LZ_OFFSET_BIAS;
        if (gh.enc_off == 1) {
            offset += *o_ptr++;  // 1-byte offset (biased)
        } else {
            offset += zxc_le16(o_ptr);  // 2-byte offset (biased)
            o_ptr += 2;
        }

        if (UNLIKELY(ll == ZXC_TOKEN_LL_MASK)) ll += zxc_read_varint(&e_ptr, e_end);
        if (UNLIKELY(ml == ZXC_TOKEN_ML_MASK)) ml += zxc_read_varint(&e_ptr, e_end);
        ml += ZXC_LZ_MIN_MATCH_LEN;

        if (UNLIKELY(d_ptr + ll > d_end || l_ptr + ll > l_end)) return ZXC_ERROR_OVERFLOW;
        ZXC_MEMCPY(d_ptr, l_ptr, ll);
        l_ptr += ll;
        d_ptr += ll;

        const uint8_t* match_src = d_ptr - offset;
        if (UNLIKELY(match_src < dst || d_ptr + ml > d_end)) return ZXC_ERROR_BAD_OFFSET;

        if (offset < ml) {
            for (size_t i = 0; i < ml; i++) d_ptr[i] = match_src[i];
        } else {
            ZXC_MEMCPY(d_ptr, match_src, ml);
        }
        d_ptr += ml;
        n_seq--;
    }

    // --- Trailing Literals ---
    // Copy remaining literals from source stream (literal exhaustion)
    if (UNLIKELY(l_ptr > l_end)) return ZXC_ERROR_CORRUPT_DATA;
    const size_t remaining_literals = (size_t)(l_end - l_ptr);
    if (remaining_literals > 0) {
        if (UNLIKELY(d_ptr + remaining_literals > d_end)) return ZXC_ERROR_OVERFLOW;
        ZXC_MEMCPY(d_ptr, l_ptr, remaining_literals);
        d_ptr += remaining_literals;
    }

    return (int)(d_ptr - dst);
}

/**
 * @brief Decodes a GHI (General High) format compressed block.
 *
 * This function handles the decoding of a compressed block formatted with the
 * internal GHI structure. The decompressed size is derived from Section
 * Descriptors within the compressed payload.
 *
 * @param[in] ctx Pointer to the decompression context (unused in current implementation).
 * @param[in] src Pointer to the source buffer containing compressed data.
 * @param[in] src_size Size of the source buffer in bytes.
 * @param[out] dst Pointer to the destination buffer for decompressed data.
 * @param[in] dst_capacity Capacity of the destination buffer in bytes.
 * @return int Returns the number of bytes written on success, or a negative zxc_error_t code on
 * failure.
 */
static int zxc_decode_block_ghi(zxc_cctx_t* RESTRICT ctx, const uint8_t* RESTRICT src,
                                const size_t src_size, uint8_t* RESTRICT dst,
                                const size_t dst_capacity) {
    (void)ctx;
    zxc_gnr_header_t gh;
    zxc_section_desc_t desc[ZXC_GHI_SECTIONS];

    const int res = zxc_read_ghi_header_and_desc(src, src_size, &gh, desc);
    if (UNLIKELY(res != 0)) return ZXC_ERROR_BAD_HEADER;

    const uint8_t* p_curr =
        src + ZXC_GHI_HEADER_BINARY_SIZE + ZXC_GHI_SECTIONS * ZXC_SECTION_DESC_BINARY_SIZE;

    // --- Stream Pointers & Validation ---
    const size_t sz_lit = (uint32_t)desc[0].sizes;
    const size_t sz_seqs = (uint32_t)desc[1].sizes;
    const size_t sz_exts = (uint32_t)desc[2].sizes;
    const uint8_t* l_ptr = p_curr;
    const uint8_t* l_end = l_ptr + sz_lit;
    p_curr += sz_lit;

    const uint8_t* seq_ptr = p_curr;
    const uint8_t* extras_ptr = p_curr + sz_seqs;
    const uint8_t* const extras_end = extras_ptr + sz_exts;

    // Validate streams don't overflow source buffer
    if (UNLIKELY(extras_end != src + src_size)) return ZXC_ERROR_CORRUPT_DATA;

    uint8_t* d_ptr = dst;
    const uint8_t* const d_end = dst + dst_capacity;
    const uint8_t* const d_end_safe = d_end - (ZXC_PAD_SIZE * 4);  // 128
    // Safety margin for 4x unrolled loop: 4 * (ZXC_SEQ_LL_MASK LL +
    // ZXC_SEQ_ML_MASK+ZXC_LZ_MIN_MATCH_LEN ML) + ZXC_PAD_SIZE Pad = 4 x (255 + 255 + 5) + 32 = 2092
    const uint8_t* const d_end_fast = d_end - (ZXC_PAD_SIZE * 66);  // 2112

    // Literal stream safe thresholds for GHI loops.
    // Without varint extension, max ll per sequence = ZXC_SEQ_LL_MASK - 1 = 254.
    // For 4 sequences: 4 * 254 = 1016. With this margin, l_ptr checks are only needed
    // on the cold varint path, keeping the hot path free of l_ptr overhead.
    const size_t ghi_margin_4x = 4 * (ZXC_SEQ_LL_MASK - 1);  // 1016
    const size_t ghi_margin_1x = ZXC_SEQ_LL_MASK - 1;        // 254
    const uint8_t* const l_end_safe_4x = (sz_lit > ghi_margin_4x) ? l_end - ghi_margin_4x : l_ptr;
    const uint8_t* const l_end_safe_1x = (sz_lit > ghi_margin_1x) ? l_end - ghi_margin_1x : l_ptr;

    uint32_t n_seq = gh.n_sequences;

    // Track bytes written for offset validation
    // For 1-byte offsets (enc_off==1): validate until 256 bytes written (max 8-bit offset)
    // For 2-byte offsets (enc_off==0): validate until 65536 bytes written (max 16-bit offset)
    // After threshold, all offsets are guaranteed valid (can't exceed written bytes)
    size_t written = 0;

// Macro for copy literal + match (uses 32-byte wild copies)
// SAFE version: validates offset against written bytes
#define DECODE_SEQ_SAFE(ll, ml, off)                                     \
    do {                                                                 \
        {                                                                \
            const uint8_t* src_lit = l_ptr;                              \
            uint8_t* dst_lit = d_ptr;                                    \
            zxc_copy32(dst_lit, src_lit);                                \
            if (UNLIKELY(ll > ZXC_PAD_SIZE)) {                           \
                dst_lit += ZXC_PAD_SIZE;                                 \
                src_lit += ZXC_PAD_SIZE;                                 \
                size_t rem = ll - ZXC_PAD_SIZE;                          \
                while (rem > ZXC_PAD_SIZE) {                             \
                    zxc_copy32(dst_lit, src_lit);                        \
                    dst_lit += ZXC_PAD_SIZE;                             \
                    src_lit += ZXC_PAD_SIZE;                             \
                    rem -= ZXC_PAD_SIZE;                                 \
                }                                                        \
                zxc_copy32(dst_lit, src_lit);                            \
            }                                                            \
            l_ptr += ll;                                                 \
            d_ptr += ll;                                                 \
            written += ll;                                               \
        }                                                                \
        {                                                                \
            if (UNLIKELY(off > written)) return ZXC_ERROR_BAD_OFFSET;    \
            const uint8_t* match_src = d_ptr - off;                      \
            if (LIKELY(off >= ZXC_PAD_SIZE)) {                           \
                zxc_copy32(d_ptr, match_src);                            \
                if (UNLIKELY(ml > ZXC_PAD_SIZE)) {                       \
                    uint8_t* out = d_ptr + ZXC_PAD_SIZE;                 \
                    const uint8_t* ref = match_src + ZXC_PAD_SIZE;       \
                    size_t rem = ml - ZXC_PAD_SIZE;                      \
                    while (rem > ZXC_PAD_SIZE) {                         \
                        zxc_copy32(out, ref);                            \
                        out += ZXC_PAD_SIZE;                             \
                        ref += ZXC_PAD_SIZE;                             \
                        rem -= ZXC_PAD_SIZE;                             \
                    }                                                    \
                    zxc_copy32(out, ref);                                \
                }                                                        \
                d_ptr += ml;                                             \
                written += ml;                                           \
            } else if (off >= (ZXC_PAD_SIZE / 2)) {                      \
                zxc_copy16(d_ptr, match_src);                            \
                if (UNLIKELY(ml > (ZXC_PAD_SIZE / 2))) {                 \
                    uint8_t* out = d_ptr + (ZXC_PAD_SIZE / 2);           \
                    const uint8_t* ref = match_src + (ZXC_PAD_SIZE / 2); \
                    size_t rem = ml - (ZXC_PAD_SIZE / 2);                \
                    while (rem > (ZXC_PAD_SIZE / 2)) {                   \
                        zxc_copy16(out, ref);                            \
                        out += (ZXC_PAD_SIZE / 2);                       \
                        ref += (ZXC_PAD_SIZE / 2);                       \
                        rem -= (ZXC_PAD_SIZE / 2);                       \
                    }                                                    \
                    zxc_copy16(out, ref);                                \
                }                                                        \
                d_ptr += ml;                                             \
                written += ml;                                           \
            } else if (off == 1) {                                       \
                ZXC_MEMSET(d_ptr, match_src[0], ml);                     \
                d_ptr += ml;                                             \
                written += ml;                                           \
            } else {                                                     \
                size_t copied = 0;                                       \
                while (copied < ml) {                                    \
                    zxc_copy_overlap16(d_ptr + copied, off);             \
                    copied += (ZXC_PAD_SIZE / 2);                        \
                }                                                        \
                d_ptr += ml;                                             \
                written += ml;                                           \
            }                                                            \
        }                                                                \
    } while (0)

// FAST version: no offset validation (for use after written >= 256 or 65536)
#define DECODE_SEQ_FAST(ll, ml, off)                                     \
    do {                                                                 \
        {                                                                \
            const uint8_t* src_lit = l_ptr;                              \
            uint8_t* dst_lit = d_ptr;                                    \
            zxc_copy32(dst_lit, src_lit);                                \
            if (UNLIKELY(ll > ZXC_PAD_SIZE)) {                           \
                dst_lit += ZXC_PAD_SIZE;                                 \
                src_lit += ZXC_PAD_SIZE;                                 \
                size_t rem = ll - ZXC_PAD_SIZE;                          \
                while (rem > ZXC_PAD_SIZE) {                             \
                    zxc_copy32(dst_lit, src_lit);                        \
                    dst_lit += ZXC_PAD_SIZE;                             \
                    src_lit += ZXC_PAD_SIZE;                             \
                    rem -= ZXC_PAD_SIZE;                                 \
                }                                                        \
                zxc_copy32(dst_lit, src_lit);                            \
            }                                                            \
            l_ptr += ll;                                                 \
            d_ptr += ll;                                                 \
        }                                                                \
        {                                                                \
            const uint8_t* match_src = d_ptr - off;                      \
            if (LIKELY(off >= ZXC_PAD_SIZE)) {                           \
                zxc_copy32(d_ptr, match_src);                            \
                if (UNLIKELY(ml > ZXC_PAD_SIZE)) {                       \
                    uint8_t* out = d_ptr + ZXC_PAD_SIZE;                 \
                    const uint8_t* ref = match_src + ZXC_PAD_SIZE;       \
                    size_t rem = ml - ZXC_PAD_SIZE;                      \
                    while (rem > ZXC_PAD_SIZE) {                         \
                        zxc_copy32(out, ref);                            \
                        out += ZXC_PAD_SIZE;                             \
                        ref += ZXC_PAD_SIZE;                             \
                        rem -= ZXC_PAD_SIZE;                             \
                    }                                                    \
                    zxc_copy32(out, ref);                                \
                }                                                        \
                d_ptr += ml;                                             \
            } else if (off >= (ZXC_PAD_SIZE / 2)) {                      \
                zxc_copy16(d_ptr, match_src);                            \
                if (UNLIKELY(ml > (ZXC_PAD_SIZE / 2))) {                 \
                    uint8_t* out = d_ptr + (ZXC_PAD_SIZE / 2);           \
                    const uint8_t* ref = match_src + (ZXC_PAD_SIZE / 2); \
                    size_t rem = ml - (ZXC_PAD_SIZE / 2);                \
                    while (rem > (ZXC_PAD_SIZE / 2)) {                   \
                        zxc_copy16(out, ref);                            \
                        out += (ZXC_PAD_SIZE / 2);                       \
                        ref += (ZXC_PAD_SIZE / 2);                       \
                        rem -= (ZXC_PAD_SIZE / 2);                       \
                    }                                                    \
                    zxc_copy16(out, ref);                                \
                }                                                        \
                d_ptr += ml;                                             \
            } else if (off == 1) {                                       \
                ZXC_MEMSET(d_ptr, match_src[0], ml);                     \
                d_ptr += ml;                                             \
            } else {                                                     \
                size_t copied = 0;                                       \
                while (copied < ml) {                                    \
                    zxc_copy_overlap16(d_ptr + copied, off);             \
                    copied += (ZXC_PAD_SIZE / 2);                        \
                }                                                        \
                d_ptr += ml;                                             \
            }                                                            \
        }                                                                \
    } while (0)

    // --- SAFE Loop: offset validation until threshold ---
    // Since offset is 16-bit, threshold is 65536.
    // For 1-byte offsets (enc_off==1): validate until 256 bytes written
    // For 2-byte offsets (enc_off==0): validate until 65536 bytes written
    const size_t bounds_threshold = (gh.enc_off == 1) ? (1U << 8) : (1U << 16);

    while (n_seq > 0 && d_ptr < d_end_safe && written < bounds_threshold) {
        uint32_t seq = zxc_le32(seq_ptr);
        seq_ptr += 4;

        uint32_t ll = (uint32_t)(seq >> 24);
        if (UNLIKELY(ll == ZXC_SEQ_LL_MASK)) ll += zxc_read_varint(&extras_ptr, extras_end);

        uint32_t m_bits = (uint32_t)((seq >> 16) & 0xFF);
        uint32_t ml = m_bits + ZXC_LZ_MIN_MATCH_LEN;
        if (UNLIKELY(m_bits == ZXC_SEQ_ML_MASK)) ml += zxc_read_varint(&extras_ptr, extras_end);

        uint32_t offset = (uint32_t)(seq & 0xFFFF) + ZXC_LZ_OFFSET_BIAS;

        // Strict bounds check: sequence must fit, AND wild copies must not overshoot
        if (UNLIKELY(d_ptr + ll + ml + ZXC_PAD_SIZE > d_end)) {
            // Fallback to exact copy (slow but safe)
            if (UNLIKELY(d_ptr + ll > d_end)) return ZXC_ERROR_OVERFLOW;
            ZXC_MEMCPY(d_ptr, l_ptr, ll);
            l_ptr += ll;
            d_ptr += ll;
            written += ll;

            if (UNLIKELY(offset > written || d_ptr + ml > d_end)) return ZXC_ERROR_BAD_OFFSET;
            const uint8_t* match_src = d_ptr - offset;

            if (offset < ml) {
                for (size_t i = 0; i < ml; i++) d_ptr[i] = match_src[i];
            } else {
                ZXC_MEMCPY(d_ptr, match_src, ml);
            }
            d_ptr += ml;
            written += ml;
        } else {
            // Safe to process with wild copies
            DECODE_SEQ_SAFE(ll, ml, offset);
        }
        n_seq--;
    }

    // --- FAST Loop: After threshold, check large margin to avoid individual bounds checks ---
    while (n_seq >= 4 && d_ptr < d_end_fast && l_ptr < l_end_safe_4x) {
        uint32_t s1 = zxc_le32(seq_ptr);
        uint32_t s2 = zxc_le32(seq_ptr + 4);
        uint32_t s3 = zxc_le32(seq_ptr + 8);
        uint32_t s4 = zxc_le32(seq_ptr + 12);
        seq_ptr += 16;

        uint32_t ll1 = (uint32_t)(s1 >> 24);
        if (UNLIKELY(ll1 == ZXC_SEQ_LL_MASK)) {
            ll1 += zxc_read_varint(&extras_ptr, extras_end);
            if (UNLIKELY(l_ptr + ll1 > l_end || d_ptr + ll1 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        uint32_t m1b = (uint32_t)((s1 >> 16) & 0xFF);
        uint32_t ml1 = m1b + ZXC_LZ_MIN_MATCH_LEN;
        if (UNLIKELY(m1b == ZXC_SEQ_ML_MASK)) {
            ml1 += zxc_read_varint(&extras_ptr, extras_end);
            if (UNLIKELY(d_ptr + ll1 + ml1 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        uint32_t off1 = (uint32_t)(s1 & 0xFFFF) + ZXC_LZ_OFFSET_BIAS;
        DECODE_SEQ_FAST(ll1, ml1, off1);

        uint32_t ll2 = (uint32_t)(s2 >> 24);
        if (UNLIKELY(ll2 == ZXC_SEQ_LL_MASK)) {
            ll2 += zxc_read_varint(&extras_ptr, extras_end);
            if (UNLIKELY(l_ptr + ll2 > l_end || d_ptr + ll2 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        uint32_t m2b = (uint32_t)((s2 >> 16) & 0xFF);
        uint32_t ml2 = m2b + ZXC_LZ_MIN_MATCH_LEN;
        if (UNLIKELY(m2b == ZXC_SEQ_ML_MASK)) {
            ml2 += zxc_read_varint(&extras_ptr, extras_end);
            if (UNLIKELY(d_ptr + ll2 + ml2 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        uint32_t off2 = (uint32_t)(s2 & 0xFFFF) + ZXC_LZ_OFFSET_BIAS;
        DECODE_SEQ_FAST(ll2, ml2, off2);

        uint32_t ll3 = (uint32_t)(s3 >> 24);
        if (UNLIKELY(ll3 == ZXC_SEQ_LL_MASK)) {
            ll3 += zxc_read_varint(&extras_ptr, extras_end);
            if (UNLIKELY(l_ptr + ll3 > l_end || d_ptr + ll3 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        uint32_t m3b = (uint32_t)((s3 >> 16) & 0xFF);
        uint32_t ml3 = m3b + ZXC_LZ_MIN_MATCH_LEN;
        if (UNLIKELY(m3b == ZXC_SEQ_ML_MASK)) {
            ml3 += zxc_read_varint(&extras_ptr, extras_end);
            if (UNLIKELY(d_ptr + ll3 + ml3 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        uint32_t off3 = (uint32_t)(s3 & 0xFFFF) + ZXC_LZ_OFFSET_BIAS;
        DECODE_SEQ_FAST(ll3, ml3, off3);

        uint32_t ll4 = (uint32_t)(s4 >> 24);
        if (UNLIKELY(ll4 == ZXC_SEQ_LL_MASK)) {
            ll4 += zxc_read_varint(&extras_ptr, extras_end);
            if (UNLIKELY(l_ptr + ll4 > l_end || d_ptr + ll4 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        uint32_t m4b = (uint32_t)((s4 >> 16) & 0xFF);
        uint32_t ml4 = m4b + ZXC_LZ_MIN_MATCH_LEN;
        if (UNLIKELY(m4b == ZXC_SEQ_ML_MASK)) {
            ml4 += zxc_read_varint(&extras_ptr, extras_end);
            if (UNLIKELY(d_ptr + ll4 + ml4 > d_end)) return ZXC_ERROR_OVERFLOW;
        }
        uint32_t off4 = (uint32_t)(s4 & 0xFFFF) + ZXC_LZ_OFFSET_BIAS;
        DECODE_SEQ_FAST(ll4, ml4, off4);

        n_seq -= 4;
    }

#undef DECODE_SEQ_SAFE
#undef DECODE_SEQ_FAST

    // --- Remaining 1 sequence (Fast Path) ---
    while (n_seq > 0 && d_ptr < d_end_safe && l_ptr < l_end_safe_1x) {
        // Save state for fallback
        const uint8_t* seq_save = seq_ptr;
        const uint8_t* ext_save = extras_ptr;

        const uint32_t seq = zxc_le32(seq_ptr);
        seq_ptr += 4;

        uint32_t ll = (uint32_t)(seq >> 24);
        if (UNLIKELY(ll == ZXC_SEQ_LL_MASK)) {
            ll += zxc_read_varint(&extras_ptr, extras_end);
            if (UNLIKELY(l_ptr + ll > l_end)) {
                seq_ptr = seq_save;
                extras_ptr = ext_save;
                break;
            }
        }

        uint32_t m_bits = (uint32_t)((seq >> 16) & 0xFF);
        uint32_t ml = m_bits + ZXC_LZ_MIN_MATCH_LEN;
        if (UNLIKELY(m_bits == ZXC_SEQ_ML_MASK)) ml += zxc_read_varint(&extras_ptr, extras_end);

        // Strict bounds checks (including wild copy overrun safety)
        if (UNLIKELY(d_ptr + ll + ml + ZXC_PAD_SIZE > d_end)) {
            // Restore state and break to Safe Path
            seq_ptr = seq_save;
            extras_ptr = ext_save;
            break;
        }
        uint32_t offset = (uint32_t)(seq & 0xFFFF) + ZXC_LZ_OFFSET_BIAS;

        {
            const uint8_t* src_lit = l_ptr;
            uint8_t* dst_lit = d_ptr;
            zxc_copy32(dst_lit, src_lit);
            if (UNLIKELY(ll > ZXC_PAD_SIZE)) {
                dst_lit += ZXC_PAD_SIZE;
                src_lit += ZXC_PAD_SIZE;
                size_t rem = ll - ZXC_PAD_SIZE;
                while (rem > ZXC_PAD_SIZE) {
                    zxc_copy32(dst_lit, src_lit);
                    dst_lit += ZXC_PAD_SIZE;
                    src_lit += ZXC_PAD_SIZE;
                    rem -= ZXC_PAD_SIZE;
                }
                zxc_copy32(dst_lit, src_lit);
            }
            l_ptr += ll;
            d_ptr += ll;
            written += ll;
        }

        {
            // Skip check if written >= bounds_threshold (256 for 8-bit, 65536 for 16-bit)
            if (UNLIKELY(written < bounds_threshold && offset > written))
                return ZXC_ERROR_BAD_OFFSET;

            const uint8_t* match_src = d_ptr - offset;
            if (LIKELY(offset >= ZXC_PAD_SIZE)) {
                zxc_copy32(d_ptr, match_src);
                if (UNLIKELY(ml > ZXC_PAD_SIZE)) {
                    uint8_t* out = d_ptr + ZXC_PAD_SIZE;
                    const uint8_t* ref = match_src + ZXC_PAD_SIZE;
                    size_t rem = ml - ZXC_PAD_SIZE;
                    while (rem > ZXC_PAD_SIZE) {
                        zxc_copy32(out, ref);
                        out += ZXC_PAD_SIZE;
                        ref += ZXC_PAD_SIZE;
                        rem -= ZXC_PAD_SIZE;
                    }
                    zxc_copy32(out, ref);
                }
                d_ptr += ml;
                written += ml;
            } else if (offset == 1) {
                ZXC_MEMSET(d_ptr, match_src[0], ml);
                d_ptr += ml;
                written += ml;
            } else {
                for (size_t i = 0; i < ml; i++) d_ptr[i] = match_src[i];
                d_ptr += ml;
                written += ml;
            }
        }
        n_seq--;
    }

    // --- Safe Path for Remaining Sequences ---
    while (n_seq > 0) {
        uint32_t seq = zxc_le32(seq_ptr);
        seq_ptr += 4;

        uint32_t ll = (uint32_t)(seq >> 24);
        if (UNLIKELY(ll == ZXC_SEQ_LL_MASK)) ll += zxc_read_varint(&extras_ptr, extras_end);

        uint32_t m_bits = (uint32_t)((seq >> 16) & 0xFF);
        uint32_t ml = m_bits + ZXC_LZ_MIN_MATCH_LEN;
        if (UNLIKELY(m_bits == ZXC_SEQ_ML_MASK)) ml += zxc_read_varint(&extras_ptr, extras_end);
        uint32_t offset = (uint32_t)(seq & 0xFFFF) + ZXC_LZ_OFFSET_BIAS;

        if (UNLIKELY(d_ptr + ll > d_end || l_ptr + ll > l_end)) return ZXC_ERROR_OVERFLOW;
        ZXC_MEMCPY(d_ptr, l_ptr, ll);
        l_ptr += ll;
        d_ptr += ll;

        const uint8_t* match_src = d_ptr - offset;
        if (UNLIKELY(match_src < dst || d_ptr + ml > d_end)) return ZXC_ERROR_BAD_OFFSET;

        if (offset < ml) {
            for (size_t i = 0; i < ml; i++) d_ptr[i] = match_src[i];
        } else {
            ZXC_MEMCPY(d_ptr, match_src, ml);
        }
        d_ptr += ml;
        n_seq--;
    }

    // --- Trailing Literals ---
    // Copy remaining literals from source stream (literal exhaustion)
    if (UNLIKELY(l_ptr > l_end)) return ZXC_ERROR_CORRUPT_DATA;
    const size_t remaining_literals = (size_t)(l_end - l_ptr);
    if (remaining_literals > 0) {
        if (UNLIKELY(d_ptr + remaining_literals > d_end)) return ZXC_ERROR_OVERFLOW;
        ZXC_MEMCPY(d_ptr, l_ptr, remaining_literals);
        d_ptr += remaining_literals;
    }

    return (int)(d_ptr - dst);
}

// cppcheck-suppress unusedFunction
int zxc_decompress_chunk_wrapper(zxc_cctx_t* RESTRICT ctx, const uint8_t* RESTRICT src,
                                 const size_t src_sz, uint8_t* RESTRICT dst, const size_t dst_cap) {
    if (UNLIKELY(src_sz < ZXC_BLOCK_HEADER_SIZE)) return ZXC_ERROR_SRC_TOO_SMALL;

    const uint8_t type = src[0];
    const uint32_t comp_sz = zxc_le32(src + 3);
    const int has_crc = ctx->checksum_enabled;

    // Check bounds: Header + Body + Checksum(if any)
    const size_t expected_sz =
        (size_t)ZXC_BLOCK_HEADER_SIZE + comp_sz + (has_crc ? ZXC_BLOCK_CHECKSUM_SIZE : 0);
    if (UNLIKELY(src_sz < expected_sz)) return ZXC_ERROR_SRC_TOO_SMALL;

    const uint8_t* data = src + ZXC_BLOCK_HEADER_SIZE;

    if (has_crc) {
        const uint32_t stored = zxc_le32(data + comp_sz);
        const uint32_t calc = zxc_checksum(data, comp_sz, ZXC_CHECKSUM_RAPIDHASH);
        if (UNLIKELY(stored != calc)) return ZXC_ERROR_BAD_CHECKSUM;
    }

    int decoded_sz = ZXC_ERROR_BAD_BLOCK_TYPE;

    switch (type) {
        case ZXC_BLOCK_GLO:
            decoded_sz = zxc_decode_block_glo(ctx, data, comp_sz, dst, dst_cap);
            break;
        case ZXC_BLOCK_GHI:
            decoded_sz = zxc_decode_block_ghi(ctx, data, comp_sz, dst, dst_cap);
            break;
        case ZXC_BLOCK_RAW:
            // For RAW blocks, comp_sz == raw_sz (uncompressed data stored as-is)
            if (UNLIKELY(comp_sz > dst_cap)) return ZXC_ERROR_DST_TOO_SMALL;
            ZXC_MEMCPY(dst, data, comp_sz);
            decoded_sz = (int)comp_sz;
            break;
        case ZXC_BLOCK_NUM:
            decoded_sz = zxc_decode_block_num(data, comp_sz, dst, dst_cap);
            break;
        case ZXC_BLOCK_EOF:
            // EOF should be handled by the dispatcher, not here
            return ZXC_ERROR_CORRUPT_DATA;
        default:
            return ZXC_ERROR_BAD_BLOCK_TYPE;
    }

    return decoded_sz;
}
