/****************************************************************************
Copyright 2017  David Edmundson <kde@davidedmundson.co.uk>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) version 3, or any
later version accepted by the membership of KDE e.V. (or its
successor approved by the membership of KDE e.V.), which shall
act as a proxy defined in Section 6 of version 3 of the license.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library.  If not, see <http://www.gnu.org/licenses/>.
****************************************************************************/
#ifndef WRAPLAND_CLIENT_SERVER_DECORATION_PALETTE_H
#define WRAPLAND_CLIENT_SERVER_DECORATION_PALETTE_H

#include <QObject>
// STD
#include <Wrapland/Client/wraplandclient_export.h>
#include <memory>

struct org_kde_kwin_server_decoration_palette_manager;
struct org_kde_kwin_server_decoration_palette;

namespace Wrapland
{
namespace Client
{

class EventQueue;
class Surface;
class ServerSideDecorationPalette;

/**
 * @short Wrapper for the org_kde_kwin_server_decoration_palette_manager interface.
 *
 * This class provides a convenient wrapper for the org_kde_kwin_server_decoration_palette_manager
 * interface.
 *
 * To use this class one needs to interact with the Registry. There are two
 * possible ways to create the ServerSideDecorationPaletteManager interface:
 * @code
 * ServerSideDecorationPaletteManager *c = registry->createServerSideDecorationPaletteManager(name,
 * version);
 * @endcode
 *
 * This creates the ServerSideDecorationPaletteManager and sets it up directly. As an alternative
 * this can also be done in a more low level way:
 * @code
 * ServerSideDecorationPaletteManager *c = new ServerSideDecorationPaletteManager;
 * c->setup(registry->bindServerSideDecorationPaletteManager(name, version));
 * @endcode
 *
 * The ServerSideDecorationPaletteManager can be used as a drop-in replacement for any
 * org_kde_kwin_server_decoration_palette_manager pointer as it provides matching cast operators.
 *
 * @see Registry
 **/
class WRAPLANDCLIENT_EXPORT ServerSideDecorationPaletteManager : public QObject
{
    Q_OBJECT
public:
    /**
     * Creates a new ServerSideDecorationPaletteManager.
     * Note: after constructing the ServerSideDecorationPaletteManager it is not yet valid and one
     * needs to call setup. In order to get a ready to use ServerSideDecorationPaletteManager prefer
     * using Registry::createServerSideDecorationPaletteManager.
     **/
    explicit ServerSideDecorationPaletteManager(QObject* parent = nullptr);
    virtual ~ServerSideDecorationPaletteManager();

    /**
     * Setup this ServerSideDecorationPaletteManager to manage the @p
     * serverSideDecorationPaletteManager. When using
     * Registry::createServerSideDecorationPaletteManager there is no need to call this method.
     **/
    void setup(org_kde_kwin_server_decoration_palette_manager* serverSideDecorationPaletteManager);
    /**
     * @returns @c true if managing a org_kde_kwin_server_decoration_palette_manager.
     **/
    bool isValid() const;
    /**
     * Releases the org_kde_kwin_server_decoration_palette_manager interface.
     * After the interface has been released the ServerSideDecorationPaletteManager instance is no
     * longer valid and can be setup with another org_kde_kwin_server_decoration_palette_manager
     * interface.
     **/
    void release();

    /**
     * Sets the @p queue to use for creating objects with this ServerSideDecorationPaletteManager.
     **/
    void setEventQueue(EventQueue* queue);
    /**
     * @returns The event queue to use for creating objects with this
     * ServerSideDecorationPaletteManager.
     **/
    EventQueue* eventQueue();

    ServerSideDecorationPalette* create(Surface* surface, QObject* parent = nullptr);

    operator org_kde_kwin_server_decoration_palette_manager*();
    operator org_kde_kwin_server_decoration_palette_manager*() const;

Q_SIGNALS:
    /**
     * The corresponding global for this interface on the Registry got removed.
     *
     * This signal gets only emitted if the ServerSideDecorationPaletteManager got created by
     * Registry::createServerSideDecorationPaletteManager
     **/
    void removed();

private:
    class Private;
    std::unique_ptr<Private> d;
};

class WRAPLANDCLIENT_EXPORT ServerSideDecorationPalette : public QObject
{
    Q_OBJECT
public:
    virtual ~ServerSideDecorationPalette();

    /**
     * Setup this ServerSideDecorationPalette to manage the @p serversidedecorationpalette.
     * When using ServerSideDecorationPaletteManager::create there is no need to call this
     * method.
     **/
    void setup(org_kde_kwin_server_decoration_palette* serversidedecorationpalette);
    /**
     * @returns @c true if managing a org_kde_kwin_server_decoration_palette.
     **/
    bool isValid() const;
    /**
     * Releases the org_kde_kwin_server_decoration_palette interface.
     * After the interface has been released the ServerSideDecorationPalette instance is no
     * longer valid and can be setup with another org_kde_kwin_server_decoration_palette interface.
     **/
    void release();

    /**
     * Sets the palette to be used by the server side decorations.
     * Absolute file path, or name of palette in the user's config directory.
     * If set to empty the default palette will be used.
     */
    void setPalette(QString const& palette);

    operator org_kde_kwin_server_decoration_palette*();
    operator org_kde_kwin_server_decoration_palette*() const;

private:
    friend class ServerSideDecorationPaletteManager;
    explicit ServerSideDecorationPalette(QObject* parent = nullptr);
    class Private;
    std::unique_ptr<Private> d;
};

}
}

#endif
