#serial 1

# Check for C++ Standard Template Library headers
#
# The results are written into dclib-stl-use.h NOT config.h, since they are
# used in dclib's installed headers, and dclib's config.h is not installed.
#
# Written by Edward Sheldrake, this file is part of dclib.
#

AC_DEFUN([DCLIB_CHECK_STL],
[
  AC_MSG_NOTICE([checking for C++ Standard Template Library features])
  
  dnl these may or may not be used if unordered versions are available
  dnl but they always exist, standard part of C++
  AC_CHECK_HEADER([map], [], [AC_MSG_ERROR([No std::map header found!])])
  AC_CHECK_HEADER([set], [], [AC_MSG_ERROR([No std::set header found!])])
  AC_CHECK_HEADER([vector], [], [AC_MSG_ERROR([No std::vector header found!])])
  AC_CHECK_HEADER([algorithm], [], [AC_MSG_ERROR([No std::algorithm header found!])])
  
  AC_ARG_ENABLE([unordered-stl],
    [AS_HELP_STRING([--disable-unordered-stl],[disable use of C++ classes unordered_map and unordered_set (default=used if present)])],
    [ UNORDERED_STL=$enableval ],
    [ UNORDERED_STL="yes"]
  )  
  
  AC_ARG_ENABLE([cstring-key],
    [AS_HELP_STRING([--disable-cstring-key],[disable use of CString as the key for unordered_map and unordered_set (default=enabled)])],
    [ CSTRING_KEY=$enableval ],
    [ CSTRING_KEY="yes"]
  )
  
  dnl if the headers are not found the variables are defined to 0
  dnl for making dclib-stl-use.h
  
  if test "x${UNORDERED_STL}" = "xyes"
  then
    
    dnl C++0x locations - not currently available without -std=c++0x
    AC_CHECK_HEADER([unordered_map],
      [HAVE_UNORDERED_MAP_H=1],
      [HAVE_UNORDERED_MAP_H=0]
    )
    AC_CHECK_HEADER([unordered_set],
      [HAVE_UNORDERED_SET_H=1],
      [HAVE_UNORDERED_SET_H=0]
    )
    
    if test $ac_cv_header_unordered_map != $ac_cv_header_unordered_set
    then
      AC_MSG_ERROR([C++0x STL headers inconsistent, only one of unordered_map / unordered_set available])
      AC_MSG_ERROR([Re-run configure with --disable-unordered-stl])
    fi
    
    if test "x$ac_cv_header_unordered_map" = "xno"
    then
       dnl TR1 locations
       AC_CHECK_HEADER([tr1/unordered_map],
         [HAVE_TR1_UNORDERED_MAP_H=1],
	 [HAVE_TR1_UNORDERED_MAP_H=0]
       )
       AC_CHECK_HEADER([tr1/unordered_set],
         [HAVE_TR1_UNORDERED_SET_H=1],
	 [HAVE_TR1_UNORDERED_SET_H=0]
       )
    else
      HAVE_TR1_UNORDERED_MAP_H=0
      HAVE_TR1_UNORDERED_SET_H=0
    fi
    
    if test x${CSTRING_KEY} = "xyes"
    then
      AC_MSG_CHECKING([for some Fnv_hash])
      if test "x$ac_cv_header_unordered_map" = "xyes"
      then
        AC_LINK_IFELSE([AC_LANG_SOURCE(
[[
#include <unordered_map>
int main()
{
	const char * data = "abcdef";
	size_t hash = std::Fnv_hash<>::hash( data, 6 );
	
	return 0;
}
]]	)],
	[ALLOW_CSTRING_KEY=1
	 FNV_HASH_PREFIXED=0
	 FNV_HASH_IS_TEMPLATE=1
	 AC_MSG_RESULT([Fnv_hash<>])],
	[AC_LINK_IFELSE([AC_LANG_SOURCE(
[[
#include <unordered_map>
int main()
{
	const char * data = "abcdef";
	size_t hash = std::_Fnv_hash<>::hash( data, 6 );
	
	return 0;
}
]]
	)],
	[ALLOW_CSTRING_KEY=1
	 FNV_HASH_PREFIXED=1
	 FNV_HASH_IS_TEMPLATE=1
	 AC_MSG_RESULT([_Fnv_hash<>])],
	[AC_LINK_IFELSE([AC_LANG_SOURCE(
[[
#include <unordered_map>
int main()
{
	const char * data = "abcdef";
	size_t hash = std::_Fnv_hash::hash( data, 6 );
	
	return 0;
}
]]
	)],
	[ALLOW_CSTRING_KEY=1
	 FNV_HASH_PREFIXED=1
	 FNV_HASH_IS_TEMPLATE=0
	 AC_MSG_RESULT([_Fnv_hash])],
	[ALLOW_CSTRING_KEY=0
	 FNV_HASH_PREFIXED=1
	 FNV_HASH_IS_TEMPLATE=0
	 AC_MSG_RESULT([no])])])]
	)
	
	AC_MSG_CHECKING([for unordered_set with a custom key])	
	if test "x$ALLOW_CSTRING_KEY" = "x1"
	then
	  AC_LINK_IFELSE([AC_LANG_SOURCE(
[[
#include <unordered_set>

class MyClass {
public:
	MyClass() {};
	~MyClass() {};
	int x;
};

bool operator == ( const MyClass & lhs, const MyClass & rhs )
{
	return lhs.x == rhs.x;
}

namespace std
{
template<>
struct hash<const MyClass &>
{
	size_t operator() (const MyClass & param) const
	{
		return param.x + 33;
	}
};
template<>
struct hash<MyClass>
{
	size_t operator() (MyClass param) const
	{
		return param.x + 33;
	}
};
}

int main()
{
	std::unordered_set<MyClass> myset;
	MyClass c1;
	c1.x = 90;
	MyClass c2;
	c2.x = 71;
	myset.insert(c1);
	myset.insert(c2);
	return 0;
}
]]
	)],
	[AC_MSG_RESULT([ok])],
	[AC_MSG_RESULT([didn't work])
	 ALLOW_CSTRING_KEY=0]
	)

	else
	  AC_MSG_RESULT([dclib requires some Fnv_hash for this])
	fi
	
      else
        AC_LINK_IFELSE([AC_LANG_SOURCE(
[[
#include <tr1/unordered_map>
int main()
{
	const char * data = "abcdef";
	size_t hash = std::tr1::Fnv_hash<>::hash( data, 6 );
	
	return 0;
}
]]
	)],
	[ALLOW_CSTRING_KEY=1
	 FNV_HASH_PREFIXED=0
	 FNV_HASH_IS_TEMPLATE=1
	 AC_MSG_RESULT([Fnv_hash<>])],
	[AC_LINK_IFELSE([AC_LANG_SOURCE(
[[
#include <tr1/unordered_map>
int main()
{
	const char * data = "abcdef";
	size_t hash = std::tr1::_Fnv_hash<>::hash( data, 6 );
	
	return 0;
}
]]
	)],
	[ALLOW_CSTRING_KEY=1
	 FNV_HASH_PREFIXED=1
	 FNV_HASH_IS_TEMPLATE=1
	 AC_MSG_RESULT([_Fnv_hash<>])],
	[AC_LINK_IFELSE([AC_LANG_SOURCE(
[[
#include <tr1/unordered_map>
int main()
{
	const char * data = "abcdef";
	size_t hash = std::tr1::_Fnv_hash::hash( data, 6 );
	
	return 0;
}
]]
	)],
	[ALLOW_CSTRING_KEY=1
	 FNV_HASH_PREFIXED=1
	 FNV_HASH_IS_TEMPLATE=0
	 AC_MSG_RESULT([_Fnv_hash])],
	[ALLOW_CSTRING_KEY=0
	 FNV_HASH_PREFIXED=1
	 FNV_HASH_IS_TEMPLATE=0
	 AC_MSG_RESULT([no])])])]
	)
	
	AC_MSG_CHECKING([for unordered_set with a custom key])	
	if test "x$ALLOW_CSTRING_KEY" = "x1"
	then
	  AC_LINK_IFELSE([AC_LANG_SOURCE(
[[
#include <tr1/unordered_set>

class MyClass {
public:
	MyClass() {};
	~MyClass() {};
	int x;
};

bool operator == ( const MyClass & lhs, const MyClass & rhs )
{
	return lhs.x == rhs.x;
}

namespace std
{
namespace tr1
{
template<>
struct hash<const MyClass &>
{
	size_t operator() (const MyClass & param) const
	{
		return param.x + 33;
	}
};
template<>
struct hash<MyClass>
{
	size_t operator() (MyClass param) const
	{
		return param.x + 33;
	}
};
}
}

int main()
{
	std::tr1::unordered_set<MyClass> myset;
	MyClass c1;
	c1.x = 90;
	MyClass c2;
	c2.x = 71;
	myset.insert(c1);
	myset.insert(c2);
	return 0;
}
]]
	)],
	[AC_MSG_RESULT([ok])],
	[AC_MSG_RESULT([didn't work])
	 ALLOW_CSTRING_KEY=0]
	)
	else
	  AC_MSG_RESULT([dclib requires some Fnv_hash for this])
	fi
	
      fi
    else
      AC_MSG_CHECKING([custom key class for unordered_set])
      AC_MSG_RESULT([disabled])
      ALLOW_CSTRING_KEY=0
      FNV_HASH_PREFIXED=1
      FNV_HASH_IS_TEMPLATE=0
    fi
    
  else
    HAVE_UNORDERED_MAP_H=0
    HAVE_UNORDERED_SET_H=0
    HAVE_TR1_UNORDERED_MAP_H=0
    HAVE_TR1_UNORDERED_SET_H=0
    ALLOW_CSTRING_KEY=0
    FNV_HASH_PREFIXED=1
    FNV_HASH_IS_TEMPLATE=0
  fi
  
  AC_SUBST(HAVE_UNORDERED_MAP_H)
  AC_SUBST(HAVE_UNORDERED_SET_H)
  AC_SUBST(HAVE_TR1_UNORDERED_MAP_H)
  AC_SUBST(HAVE_TR1_UNORDERED_SET_H)
  AC_SUBST(ALLOW_CSTRING_KEY)
  AC_SUBST(FNV_HASH_PREFIXED)
  AC_SUBST(FNV_HASH_IS_TEMPLATE)
])
