/*
 *       File name:  libcnrun/model/nmlio.cc
 *         Project:  cnrun
 *          Author:  Andrei Zavada <johnhommer@gmail.com>
 * Initial version:  2008-09-02
 *
 *         Purpose:  NeuroML import/export.
 *
 *         License:  GPL-2+
 */

#if HAVE_CONFIG_H && !defined(VERSION)
#  include "config.h"
#endif

#include <string>
#include <iostream>
#include <regex.h>

#include "forward-decls.hh"
#include "model.hh"


using namespace std;
using cnrun::stilton::str::sasprintf;

int
cnrun::CModel::
import_NetworkML( const string& fname, TNMLImportOption import_option)
{
        // LIBXML_TEST_VERSION;

        xmlDoc *doc = xmlReadFile( fname.c_str(), nullptr, 0);
        if ( !doc )
                return TNMLIOResult::nofile;

        int retval = import_NetworkML( doc, fname, import_option);

        xmlFreeDoc( doc);

        return retval;
}





namespace {

xmlNode*
find_named_root_child_elem( xmlNode *node,     // node to start search from
                            const char *elem)  // name of the element searched for
{
        xmlNode *n;
        for ( n = node->children; n; n = n->next ) {
                if ( n->type == XML_ELEMENT_NODE ) {
                        if ( xmlStrEqual( n->name, BAD_CAST elem) )
                                return n;
// the <populations> and <projections> nodes are expected to appear as
// direct children of the root node; so don't go search deeper than that

//                        if ( n->children ) { // go search deeper
//                                ni = find_named_elem( n->children, elem);
//                                if ( ni )
//                                        return ni;
//                        }
                }
        }
        return nullptr;
}

}

int
cnrun::CModel::
import_NetworkML( xmlDoc *doc, const string& fname,
                  TNMLImportOption import_option)
{
        int retval = 0;

        // we pass up on validation (for which we would need to keep a
        // .dtd or Schema at hand), and proceed to extracting elements

        xmlNode *root_node = xmlDocGetRootElement( doc),
                *n;

      // read meta:notes and make out a name for the model
        if ( !root_node ) {
                vp( 0, stderr, "import_NetworkML(\"%s\"): No root element\n", fname.c_str());
                retval = TNMLIOResult::noelem;
                goto out;
        }

      // give it a name: assume it's generated by neuroConstruct for now
        if ( import_option == TNMLImportOption::reset ) {
                reset();
                if ( !(n = find_named_root_child_elem( root_node, "notes")) ) {
                        vp( 1, stderr, "<notes> element not found; model will be unnamed\n");
                        // this is not critical, so just keep the user
                        // informed and proceed
                } else
                        if ( n->type == XML_ELEMENT_NODE ) {  // only concern ourselves with nodes of this type
                                xmlChar *notes_s = xmlNodeGetContent( n);
                                // look for a substring specific to neuroConstruct, which is obviously speculative
                                regex_t RE;
                                regcomp( &RE, ".*project: (\\w*).*", REG_EXTENDED);
                                regmatch_t M[1+1];
                                name = (0 == regexec( &RE, (char*)notes_s, 1+1, M, 0))
                    ? string ((char*)notes_s + M[1].rm_so, M[1].rm_eo - M[1].rm_so)
                    : "(unnamed)";
                                xmlFree( notes_s);
                        } else
                                name = "(unnamed)";
        }

        vp( 1, "Model \"%s\": %s topology from %s\n",
            name.c_str(),
            (import_option == TNMLImportOption::merge) ?"Merging" :"Importing",
            fname.c_str());

        // In the following calls to _process_{populations,instances}
        // functions, the actual order of appearance of these nodes in
        // the xml file doesn't matter, thanks to the xml contents
        // being already wholly read and available to us as a tree.

      // process <populations>
        if ( !(n = find_named_root_child_elem( root_node, "populations")) ) {
                retval = TNMLIOResult::noelem;
                goto out;
        } // assume there is only one <populations> element: don't loop to catch more
        if ( (retval = _process_populations( n->children)) < 0)        // note n->children, which is in fact a pointer to the first child
                goto out;

      // process <projections>
      // don't strictly require any projections as long as there are some neurons
        if ( (n = find_named_root_child_elem( root_node, "projections")) ) {
                if ( (retval = _process_projections( n->children)) < 0 )
                        goto out;
        } else
                vp( 2, "No projections found\n");

out:
        // we are done with topology; now put units' variables on a vector
        finalize_additions();
        // can call time_step only after finalize_additions

        return retval;
}





int
cnrun::CModel::
_process_populations( xmlNode *n)
{
        xmlChar *group_id_s = nullptr,
                *cell_type_s = nullptr;

        int     pop_cnt = 0;

        try {
                for ( ; n; n = n->next ) {  // if is nullptr (parent had no children), we won't do a single loop
                        if ( n->type != XML_ELEMENT_NODE || !xmlStrEqual( n->name, BAD_CAST "population") )
                                continue;

                        group_id_s = xmlGetProp( n, BAD_CAST "name");
                        // BAD_CAST is just a cast to xmlChar*
                        // with a catch that libxml functions
                        // expect strings pointed to to be good UTF
                        if ( !group_id_s ) {
                                vp( 0, stderr, "<population> element missing a \"name\" attribute near line %d\n", n->line);
                                return TNMLIOResult::badattr;
                        }
                      // probably having an unnamed popuation isn't an error so serious as to abort the
                      // operation, but discipline is above all

                        cell_type_s = xmlGetProp( n, BAD_CAST "cell_type");
                        // now we know the type of cells included in this population; remember it to pass on to
                        // _process_population_instances, where it is used to select an appropriate unit type
                        // when actually adding a neuron to the model

                      // but well, let's check if we have units of that species in stock
                        if ( !unit_species_is_neuron((char*)cell_type_s) && !unit_family_is_neuron((char*)cell_type_s) ) {
                                vp( 0, stderr, "Bad cell species or family (\"%s\") in population \"%s\"\n",
                                    (char*)cell_type_s, group_id_s);
                                throw TNMLIOResult::badcelltype;
                        }

                        xmlNode *nin = n->children;  // again, ->children means ->first
                        for ( ; nin; nin = nin->next )  // deal with multiple <instances> nodes
                                if ( nin->type == XML_ELEMENT_NODE && xmlStrEqual( nin->name, BAD_CAST "instances") ) {
                                        int subretval = _process_population_instances(
                                                nin->children,
                                                group_id_s, cell_type_s);
                                        if ( subretval < 0 )
                                                throw subretval;

                                        vp( 2, " %5d instance(s) of type \"%s\" in population \"%s\"\n",
                                            subretval, cell_type_s,  group_id_s);
                                        ++pop_cnt;
                                }

                        xmlFree( cell_type_s), xmlFree( group_id_s);
                }

                vp( 1, "\tTotal %d population(s)\n", pop_cnt);

        } catch (int ex) {
                xmlFree( cell_type_s), xmlFree( group_id_s);

                return ex;
        }

        return pop_cnt;
}






int
cnrun::CModel::
_process_projections( xmlNode *n)
{
        // much the same code as in _process_populations

        xmlChar *prj_name_s = nullptr,
                *prj_src_s = nullptr,
                *prj_tgt_s = nullptr,
                *synapse_type_s = nullptr;

        size_t pop_cnt = 0;

        try {
                for ( ; n; n = n->next ) {
                        if ( n->type != XML_ELEMENT_NODE || !xmlStrEqual( n->name, BAD_CAST "projection") )
                                continue;

                        prj_name_s = xmlGetProp( n, BAD_CAST "name");
                        if ( !prj_name_s ) {
                                fprintf( stderr, "<projection> element missing a \"name\" attribute near line %u\n", n->line);
                                return TNMLIOResult::badattr;
                        }

                        prj_src_s  = xmlGetProp( n, BAD_CAST "source");
                        prj_tgt_s  = xmlGetProp( n, BAD_CAST "target");
                        if ( !prj_src_s || !prj_tgt_s ) {
                                fprintf( stderr, "Projection \"%s\" missing a \"source\" and/or \"target\" attribute near line %u\n",
                                         prj_name_s, n->line);
                                throw TNMLIOResult::badattr;
                        }

                        xmlNode *nin;
                        nin = n->children;
                        if ( !nin )
                                fprintf( stderr, "Empty <projection> node near line %d\n", n->line);

                        for ( ; nin; nin = nin->next )
                                if ( nin->type == XML_ELEMENT_NODE && xmlStrEqual( nin->name, BAD_CAST "synapse_props") ) {
                                        synapse_type_s = xmlGetProp( nin, BAD_CAST "synapse_type");
                                        if ( !unit_species_is_synapse( (char*)synapse_type_s) &&
                                             !unit_family_is_synapse( (char*)synapse_type_s) ) {
                                                fprintf( stderr, "Bad synapse type \"%s\" near line %u\n",
                                                         (char*)synapse_type_s, nin->line);
                                                throw TNMLIOResult::badcelltype;
                                        }
                                }

                        for ( nin = n->children; nin; nin = nin->next )
                                if ( nin->type == XML_ELEMENT_NODE && xmlStrEqual( nin->name, BAD_CAST "connections") ) {
                                        int subretval = _process_projection_connections(
                                                nin->children,
                                                prj_name_s, synapse_type_s,
                                                prj_src_s, prj_tgt_s);
                                        if ( subretval < 0 )
                                                throw subretval;

                                        vp( 2, " %5d connection(s) of type \"%s\" in projection \"%s\"\n",
                                            subretval, synapse_type_s,  prj_name_s);
                                        ++pop_cnt;
                                }
                        xmlFree( prj_name_s), xmlFree( prj_src_s), xmlFree( prj_tgt_s);
                }

                vp( 1, "\tTotal %zd projection(s)\n", pop_cnt);

        } catch (int ex) {
                xmlFree( prj_name_s), xmlFree( prj_src_s), xmlFree( prj_tgt_s);
                return ex;
        }

        return (int)pop_cnt;
}







int
cnrun::CModel::
_process_population_instances(
        xmlNode *n,
        const xmlChar *population,
        const xmlChar *type_s)
{
        int     retval = 0;  // also keeps a count of added neurons

        double  x, y, z;
        string  label;

        xmlNode *nin;

        char    *id_s = nullptr;
        try {
                for ( ; n; n = n->next ) {
                        if ( n->type != XML_ELEMENT_NODE || !xmlStrEqual( n->name, BAD_CAST "instance") )
                                continue;

                        xmlChar *id_s = xmlGetProp( n, BAD_CAST "id");
                        if ( !id_s ) {
                                fprintf( stderr, "<instance> element without an \"id\" attribute near line %u\n", n->line);
                                return TNMLIOResult::badattr;
                        }

                        if ( !(nin = n->children) )
                                return retval;

                        for ( ; nin; nin = nin->next ) {
                                if ( !(nin->type == XML_ELEMENT_NODE &&
                                       xmlStrEqual( nin->name, BAD_CAST "location")) )
                                        continue;

                                xmlChar *x_s = xmlGetProp( nin, BAD_CAST "x"),
                                        *y_s = xmlGetProp( nin, BAD_CAST "y"),
                                        *z_s = xmlGetProp( nin, BAD_CAST "z");
                                // here we do actually insert neurons into the model
                                if ( !(x_s && y_s && z_s) )
                                        vp( 1, stderr, "<location> element missing full set of coordinates near line %d\n", nin->line);
                                // not an error
                                x = strtod( (char*)x_s, nullptr), y = strtod( (char*)y_s, nullptr), z = strtod( (char*)z_s, nullptr);
                                auto maybe_free = [](decltype(x_s) X) { if ( X ) free((void*)X); };
                                maybe_free( x_s), maybe_free( y_s), maybe_free( z_s);

                                C_BaseNeuron *neu = add_neuron_species(
                                        (char*)type_s, (char*)population, (char*)id_s,
                                        TIncludeOption::is_notlast,
                                        x, y, z);

                                xmlFree( id_s);

                                if ( !neu || neu->_status & CN_UERROR ) {
                                        if ( neu )
                                                delete neu;
                                        fprintf( stderr, "Failed to add neuron of type %s, population:id \"%s:%s\" near line %u\n", type_s, population, id_s, n->line);
                                        return TNMLIOResult::structerror;
                                } else {
                                        neu->pos = make_tuple( x, y, z);
                                        ++retval;
                                }
                        }
                }
        } catch (int ex) {
                xmlFree( id_s);
                return ex;
        }

        return retval;
}




int
cnrun::CModel::
_process_projection_connections(
        xmlNode *N,
        const xmlChar *unused_synapse_name,
        const xmlChar *type_s,
        const xmlChar *src_grp_prefix,
        const xmlChar *tgt_grp_prefix)
{
        // similar to _process_population_instances, except that we read some more attributes (source and
        // target units)

        int     retval = 0;  // is also a counter of synapses

        string  label_src, label_tgt;
        double  weight;

        C_BaseSynapse *y;

        xmlChar *src_cell_id_s = nullptr,
                *tgt_cell_id_s = nullptr,
                *weight_s      = nullptr;
        try {
                for ( ; N; N = N->next ) {
                        if ( N->type != XML_ELEMENT_NODE || !xmlStrEqual( N->name, BAD_CAST "connection") )
                                continue;

                        src_cell_id_s = xmlGetProp( N, BAD_CAST "pre_cell_id"),
                        tgt_cell_id_s = xmlGetProp( N, BAD_CAST "post_cell_id"),
                        weight_s      = xmlGetProp( N, BAD_CAST "weight");
                        if ( /*!synapse_id_s || */ !src_cell_id_s || !tgt_cell_id_s ) {
                                fprintf( stderr, "A <connection> element without \"pre_cell_id\" and/or \"post_cell_id\" attribute near line %u\n", N->line);
                                throw TNMLIOResult::badattr;
                        }

                        label_src = C_BaseNeuron::make_nml_name( (char*)src_grp_prefix, (char*)src_cell_id_s);
                        label_tgt = C_BaseNeuron::make_nml_name( (char*)tgt_grp_prefix, (char*)tgt_cell_id_s);
                        if ( label_src.size() > C_BaseUnit::max_label_size ||
                             label_tgt.size() > C_BaseUnit::max_label_size ) {
                                fprintf( stderr, "Source or target label size (%zu) exceeds %zu chars near line %u\n",
                                         max(label_tgt.size(), label_src.size()), C_BaseUnit::max_label_size, N->line);
                                return TNMLIOResult::biglabel;
                        }

                        if ( !weight_s ) {
                                vp( 3, stderr, "Assuming 0 for a synapse of \"%s.%s\" to \"%s%s\" without a \"weight\" attribute near line %u\n",
                                    src_grp_prefix, src_cell_id_s, tgt_grp_prefix, tgt_cell_id_s, N->line);
                                weight = 0.;
                        }
                        /* xmlFree( synapse_id_s), */ xmlFree( src_cell_id_s), xmlFree( tgt_cell_id_s),
                                xmlFree( weight_s);

                        y = add_synapse_species(
                                (char*)type_s,
                                label_src.c_str(), label_tgt.c_str(),
                                weight,
                                TSynapseCloningOption::yes,
                                TIncludeOption::is_notlast);

                        if ( !y || y->_status & CN_UERROR ) {
                                if ( y )
                                        delete y;
                                fprintf( stderr, "Failed to add an \"%s\" synapse from \"%s\" to \"%s\" near line %u\n",
                                         (char*)type_s, label_src.c_str(), label_tgt.c_str(), N->line);
                                return TNMLIOResult::structerror;
                        } else
                                ++retval;
                }

        } catch (int ex) {
                /* xmlFree( synapse_id_s), */ xmlFree( src_cell_id_s), xmlFree( tgt_cell_id_s),
                        xmlFree( weight_s);
                return ex;
        }

        return retval;
}



int
cnrun::CModel::
export_NetworkML( const string& fname)
{
        int retval = 0;

        xmlIndentTreeOutput = 1;

        xmlDoc  *doc = xmlNewDoc( BAD_CAST "1.0");

        xmlNode *root_node = xmlNewDocNode( doc, NULL, BAD_CAST "networkml", NULL);
        xmlDocSetRootElement( doc, root_node);
        // This is from m.nml, originally generated by neuroConstruct ca 2009:
        // <networkml xmlns="http://morphml.org/networkml/schema"
        //            xmlns:meta="http://morphml.org/metadata/schema"
        //            xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
        //            xsi:schemaLocation="http://morphml.org/networkml/schema NetworkML_v1.7.xsd"
        //            lengthUnits="micron">
        // The definitions below are from https://github.com/NeuroML/NeuroML2/blob/master/examples/NML2_FullNeuroML.nml:
        xmlNs *ns = xmlNewNs( root_node, BAD_CAST "http://www.neuroml.org/schema/neuroml2", NULL);
        xmlNewNsProp( root_node, ns, BAD_CAST "xmlns:meta", BAD_CAST "http://morphml.org/metadata/schema");
        xmlNewNsProp( root_node, ns, BAD_CAST "xmlns:xsi", BAD_CAST "http://www.w3.org/2001/XMLSchema-instance");
        xmlNewNsProp( root_node, ns, BAD_CAST "xsi:schemaLocation", BAD_CAST "http://www.neuroml.org/schema/neuroml2 ../Schemas/NeuroML2/NeuroML_v2beta4.xsd");

        {
                time_t t = time(NULL);
                string manifest = sasprintf(
                        "Model %s\n"
                        "Created with CNRun2 v. " VERSION " on %s",
                        name.c_str(), asctime( localtime( &t)));
                xmlNewTextChild( root_node, ns, BAD_CAST "meta:notes", BAD_CAST manifest.c_str());
        }

        xmlNode *populations_node = xmlNewNode( NULL, BAD_CAST "populations"),
                *projections_node = xmlNewNode( NULL, BAD_CAST "projections");
        xmlAddChild( root_node, populations_node);
        xmlAddChild( root_node, projections_node);

        map<string, xmlNode*> PP;
        auto get_or_create_population = [&] (const char *population,
                                             const char *cell_type) -> xmlNode* {
                auto found = PP.find(string(population));
                if ( found == PP.end() ) {
                        xmlNode *N = xmlNewChild( populations_node, NULL, BAD_CAST "population", NULL);
                        xmlNewProp( N, BAD_CAST "name", BAD_CAST population);
                        xmlNewProp( N, BAD_CAST "cell_type", BAD_CAST cell_type);
                        return PP[population] = xmlNewChild( N, NULL, BAD_CAST "instances", NULL);
                }
                return found->second;
        };
        auto get_or_create_projection = [&] (const char *projection,
                                             const char *source,
                                             const char *target,
                                             const char *synapse_type,
                                             const char *weight) -> xmlNode* {
                auto found = PP.find(string(projection));
                if ( found == PP.end() ) {  // is always true
                        xmlNode *N = xmlNewChild( projections_node, NULL, BAD_CAST "projection", NULL);
                        xmlNewProp( N, BAD_CAST "name", BAD_CAST projection);
                        xmlNewProp( N, BAD_CAST "source", BAD_CAST source);
                        xmlNewProp( N, BAD_CAST "target", BAD_CAST target);
                        xmlNode *Nsyn_props = xmlNewChild( N, NULL, BAD_CAST "synapse_props", NULL);
                        xmlNewProp( Nsyn_props, BAD_CAST "synapse_type", BAD_CAST synapse_type);
                        xmlNewProp( Nsyn_props, BAD_CAST "weight", BAD_CAST weight);
                        return PP[projection] = xmlNewChild( N, NULL, BAD_CAST "connections", NULL);
                }
                return found->second;
        };
        for ( const auto& U : units )
                if ( U->is_neuron() ) {
                        const auto& N = reinterpret_cast<C_BaseNeuron*>(U);
                        xmlNode *instances_node =
                                get_or_create_population(
                                        U->population(),
                                        U->species());
                        xmlNode *instance_node = xmlNewChild( instances_node, NULL, BAD_CAST "instance", NULL);
                        xmlNewProp(instance_node, BAD_CAST "id", BAD_CAST U->id());
                        xmlNode *location_node = xmlNewChild( instance_node, NULL, BAD_CAST "location", NULL);
                        xmlNewProp(location_node, BAD_CAST "x", BAD_CAST sasprintf("%g", N->pos.x).c_str());
                        xmlNewProp(location_node, BAD_CAST "y", BAD_CAST sasprintf("%g", N->pos.y).c_str());
                        xmlNewProp(location_node, BAD_CAST "z", BAD_CAST sasprintf("%g", N->pos.z).c_str());
                } else if ( U->is_synapse() ) {
                        const auto& Y = reinterpret_cast<C_BaseSynapse*>(U);
                        string  src_pop, src_id,
                                tgt_pop, tgt_id;
                        C_BaseNeuron::extract_nml_parts(
                                Y->source()->label(),
                                &src_pop, &src_id);
                        for ( const auto T : Y->targets() ) {
                                C_BaseNeuron::extract_nml_parts(
                                        T->label(),
                                        &tgt_pop, &tgt_id);
                                xmlNode *projection_node =
                                        get_or_create_projection(
                                                (src_pop + "-" + tgt_pop).c_str(),
                                                src_pop.c_str(), tgt_pop.c_str(),
                                                Y->type_s(), "1.0");
                                xmlNode *connection_node = xmlNewChild( projection_node, NULL, BAD_CAST "connection", NULL);
                                xmlNewProp( connection_node, BAD_CAST "pre_cell_id", BAD_CAST src_id.c_str());
                                xmlNewProp( connection_node, BAD_CAST "post_cell_id", BAD_CAST tgt_id.c_str());
                        }
                }
        {
                FILE* fout = fopen( fname.c_str(), "w");
                if ( !fout )
                        return TNMLIOResult::nofile;
                xmlDocFormatDump( fout, doc, 1);
                fclose( fout);
        }
        xmlFreeDoc( doc);
        xmlCleanupParser();

        return retval;
}


// Local Variables:
// Mode: c++
// indent-tabs-mode: nil
// tab-width: 8
// c-basic-offset: 8
// End:
